/* -*- linux-c -*- */
/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <asm/errno.h>
#include <linux/evms/evms_user.h>


static int open_device(	char	* device,
			int	* fd )
{
	/* open device */ 
	*fd = open(device, O_RDONLY | O_NONBLOCK);
	if (*fd < 0) {
		fprintf(stderr, "Open failed for `%s': %s\n",
			device, strerror(errno));
		switch(errno) {
		case ENOENT:
			fprintf(stderr, "You may want to create this node.\n");
			fprintf(stderr, "\tType: \"mknod %s b %d 0\", and try again.\n",
				device, EVMS_MAJOR);
			break;
		case EACCES:
			fprintf(stderr, "You must be root to run this program.\n");
			break;
		default:
		}
	}
	return(errno);
}


static int verify_device(int	* fd,
			int	* status )
{
	evms_version_t ver;

	*status = ioctl (*fd, EVMS_GET_IOCTL_VERSION, &ver);
	if (*status) {
		fprintf(stderr, "%s failed : %s\n",
			EVMS_GET_IOCTL_VERSION_STRING, strerror(errno));
	}
	else {
		if ( (ver.major != EVMS_IOCTL_INTERFACE_MAJOR) ||
		     (ver.minor != EVMS_IOCTL_INTERFACE_MINOR) ||
		     (ver.patchlevel != EVMS_IOCTL_INTERFACE_PATCHLEVEL) ) {
			fprintf(stderr, "Bad IOCTL version: Recompile with latest headers!\n");
			errno = EINVAL;
		}
	}
	return(errno);
}


static int close_device(char	* device,
			int	* fd,
			int	* status)
{
	/* close device */
	*status = close (*fd);
	if (*status) {
		fprintf (stderr, "Close failed for `%s': %s\n",
			device, strerror (errno));
	}
	return(errno);
}


int
main ( int argc, char **argv )
{
	int fd;			/* file descriptor for device */
	int status;		/* return status for system calls */
	int rc;			/* program return code */
	char *device = EVMS_DEVICE_NAME;
	int info_level;

	if (argc == 1) {
		printf("Usage: %s [new info level(range: 0 - 10)]\n", argv[0]);
		printf("\tIf no parameter, reports current evms info level.\n");
		printf("\tOptional parameter, sets current evms info level.\n");
	} 

	rc = open_device(device, &fd);
	if (!rc) {
		/* verify device is EVMS */
		rc = verify_device(&fd, &status);
		if (!rc) {
			status = ioctl(fd, EVMS_GET_INFO_LEVEL, &info_level);
			if (status) {
				fprintf(stderr, "%s failed : %s\n",
					"EVMS_GET_INFO_LEVEL",
					strerror(errno));
			} else {
				printf("Current evms info level : %d\n", info_level);
				if (argc > 1) {
					info_level = atoi(argv[1]);
					if ( info_level < 0 || info_level > 10 )
						printf("Specified info level (%d) out of valid range (0 - 10)!\n",
							info_level);
					else {
						status = ioctl(fd, EVMS_SET_INFO_LEVEL, &info_level);
						if ( status ) {
							fprintf(stderr, "%s failed : %s\n",
								"EVMS_SET_INFO_LEVEL",
								strerror(errno));
						} else {
							printf("New evms info level : %d\n", info_level);
						}
					}
				}
			}
		}
		close_device(device, &fd, &status);
	}

	return(rc);
}

