/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: devnode_fixup.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <getopt.h>
#include <libgen.h>
#include <string.h>
#include <semaphore.h>
#include <signal.h>

#include <frontend.h>

static char * cmd = NULL;
static evms_verbose_level_t verbose_level = ERRORS_ONLY;
static BOOLEAN daemon_mode = FALSE;
static sem_t daemon_sem;
static BOOLEAN daemon_running = FALSE;


static void sigcont_handler (int signo) {

    /* Safety check */
    if (signo == SIGCONT) {
        sem_post(&daemon_sem);
    }
}


void show_help(void) {
    printf("Usage: %s [option]\n", cmd);
    printf("Update the EVMS device node tree.\n\n");
    printf("  -d,     --daemon   run in daemon mode\n");
    printf("  -v,     --verbose  display errors and informational messages\n");
    printf("  -q      --quiet    display no messages\n");
    printf("  -h, -?  --help     display this help\n\n");
}


static int parse_options(int argc, char * * argv) {

    int    rc = 0;
    int    c;
    char * short_opts = "dvqh?";
    struct option long_opts[] = {
        { "deamon",  no_argument, NULL, 'd'},
        { "verbose", no_argument, NULL, 'v'},
        { "quiet",   no_argument, NULL, 'q'},
        { "help",    no_argument, NULL, 'h'},
        { NULL,      0,           NULL, 0}
    };

    while ( (c = getopt_long(argc, argv,
                             short_opts, long_opts, NULL)) != EOF ) {
        switch (c) {
            case 'd':
                daemon_mode = TRUE;

                break;

            case 'v':
                if (verbose_level != NO_MESSAGES) {
                    verbose_level = ERRORS_AND_INFO;


                } else {
                    printf("%s -- ERROR: cannot specify both quiet and verbose options.\n\n", cmd);

                    /* Display the help. */
                    rc = 1;
                }
                break;

            case 'q':
                if (verbose_level != ERRORS_AND_INFO) {
                    verbose_level = NO_MESSAGES;

                } else {
                    printf("%s -- ERROR: cannot specify both verbose and quiet options.\n\n", cmd);

                    /* Display the help. */
                    rc = 1;
                }
                break;

            case 'h':
            case '?':
                /* Display the help. */
                rc = 1;
                break;

            default:
                printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);

                /* Display the help. */
                rc = 1;
                break;
        }
    }

    return rc;
}


int main(int argc, char * * argv, char * * env) {

    int rc = 0;
    struct stat statbuf;

    cmd = basename(argv[0]);

    /* Check if devfs is installed.  If it is, there is nothing to do. */
    rc = stat("/dev/.devfsd", &statbuf);

    if (rc == 0) {
        printf("Devfs is running on this system.  Devfs will keep the EVMS device nodes up to date.  %s has nothing to do.\n", cmd);
        return 0;
    }

    rc = parse_options(argc, argv);

    if (rc == 0) {
        pid_t pid;

        if (daemon_mode) {
            int evms_block_dev_handle = 0;

            evms_block_dev_handle = open(EVMS_DEVICE_NAME, O_RDWR|O_NONBLOCK);

            if (evms_block_dev_handle > 0) {
                pid = fork();

                if (pid == 0) {
                    /* I am the spawed process. */
                    int status;
                    evms_notify_t kernel_notify;

                    sem_init(&daemon_sem, 0, 0);

                    signal(SIGCONT, sigcont_handler);

                    /* Tell the EVMS kernel which process to signal. */
                    kernel_notify.command         = 1;      /* register */
                    kernel_notify.eventry.pid     = getpid();
                    kernel_notify.eventry.eventid = EVMS_EVENT_END_OF_DISCOVERY;
                    kernel_notify.eventry.signo   = SIGCONT;
                    kernel_notify.status          = 0;

                    status = ioctl(evms_block_dev_handle, EVMS_PROCESS_NOTIFY_EVENT, &kernel_notify);

                    if ((status == 0) && (kernel_notify.status == 0)) {
                        daemon_running = TRUE;
                        while (daemon_running && (rc == 0)) {
                            sem_wait(&daemon_sem);

                            if (daemon_running) {
                                rc = evms_update_evms_dev_tree(verbose_level);
                            }
                        }

                        /* Tell the EVMS kernel to stop signalling the process. */
                        kernel_notify.command = 0;      /* unregister */

                        ioctl(evms_block_dev_handle, EVMS_PROCESS_NOTIFY_EVENT, &kernel_notify);

                    } else {
                        /*
                         * We failed to register for notifications from the
                         * kernel.
                         */
                        if (status != 0) {
                            printf("%s daemon: error from ioctl to register for notifications from the EVMS kernel: %s\n", cmd, strerror(errno));
                            rc = errno;
                        } else {
                            printf("%s daemon: error code %d returned in the ioctl to register for notifications from the EVMS kernel.\n", cmd, kernel_notify.status);
                            rc = kernel_notify.status;
                        }
                    }

                } else {
                    if (pid != -1) {
                        printf("%s -- daemon processs started.\n", cmd);
                    } else {
                        printf("%s -- error spawning daemon process: %s\n", cmd, strerror(errno));
                        rc = errno;
                    }
                }

            } else {
                printf("%s -- error opening the EVMS block device: %s\n", cmd, strerror(errno));
                rc = errno;
            }

        } else {
            if (verbose_level != NO_MESSAGES) {
                printf("%s -- updating the EVMS device node tree...\n", cmd);
            }

            rc = evms_update_evms_dev_tree(verbose_level);

            if (verbose_level != NO_MESSAGES) {
                if (rc == 0) {
                    printf("%s -- Finished.\n", cmd);

                } else {
                    printf("%s -- ERROR: return code from evms_update_evms_dev_tree() was %d.\n", cmd, rc);
                }
            }
        }

    } else {
        show_help();
    }

    return rc;
}

