/*----------------------------------------------------------------------
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: syslinux.c
 *---------------------------------------------------------------------*/

/*----------------------------------------------------------------------
 * Change History:
 *
 * 9/2001  John Stiles   Initial version.
 *
 *---------------------------------------------------------------------*/


/* Identify this file. */
#define SYSLINUX_C   1

/*----------------------------------------------------------------------
 * Necessary include files
 *---------------------------------------------------------------------*/
#include <stdlib.h>        /* malloc, free */
#include <string.h>        /* strcat, strcpy */
#include <curses.h>        /* For terminal interactive operation  */

#include "evmsn.h"


/*----------------------------------------------------------------------
 * Global Variables
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * There are No Private Constants
 * There are No Private Type Definitions
 * There are No Private Global Variables.
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * Public Functions
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 *
 *   Function Name:  kb_video_setup
 *
 *   Descriptive Name:  Sets up the ncurses environment.
 *
 *   Input:  no inputs
 *
 *   Output:  nothing returned
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  ncurses environment is set up - to return to
 *                   normal command mode, the equivalent terminate
 *                   function must be called.
 *
 *   Notes:  only call once when starting program.
 *
 *---------------------------------------------------------------------*/
void kb_video_setup( void )
{
  (void) initscr();                 /* initialize curses                   */
  (void) flushinp();                /* flush the input buffer              */
  (void) keypad( stdscr, TRUE );    /* return extra keys as int values     */
/* void) cbreak();                   * Take input one char at a time       */
  (void) nonl();                    /* don't do CR/LF on output            */
  (void) noecho();                  /* stop echoing input to screen        */

  // Check for mono or color screen...
  if( !has_colors() )
    GL_use_color = 0;               /* no colors                           */
  else {
    GL_use_color = 1;               /* display colors                      */
    start_color();                  /* set up color environment            */
    init_pair( COLOR_DEFAULT, COLOR_WHITE, COLOR_BLACK );
    init_pair( COLOR_TITLE, COLOR_YELLOW, COLOR_BLACK );
    init_pair( COLOR_BAR, COLOR_BLACK, COLOR_CYAN );
    init_pair( COLOR_FKEYS, COLOR_WHITE, COLOR_BLUE );
    attron( COLOR_PAIR( COLOR_DEFAULT ) );
  }

}


/*----------------------------------------------------------------------
 *
 *   Function Name:  update_screen_info
 *
 *   Descriptive Name:  Check if screen size changed.
 *
 *   Input:  no inputs
 *
 *   Output:  0 == nothing changed, !0 == changed screen dimensions
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  screen size Globals may be changed.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
int update_screen_info( void )
{
  int  rc = 0;

  if( GL_screen_lines != LINES ) {
      GL_screen_lines  = LINES;
      rc++;                          /* set changed indicator           */
  }

  if( GL_screen_cols != COLS ) {
      GL_screen_cols  = COLS;
      rc++;                          /* set changed indicator           */
  }

  return( rc );
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  signal_user
 *
 *   Descriptive Name:  send an audible signal ( beep ).
 *
 *   Input:  no inputs
 *
 *   Output:  nothing returned
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  user jumps upon hearing beep.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
void signal_user( void )
{
  (void) beep();
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  clear_screen
 *
 *   Descriptive Name:  clear the display.
 *
 *   Input:  no inputs
 *
 *   Output:  nothing returned
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  all displayed data is lost.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
void clear_screen( void )
{
  (void) clear();
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  print_string
 *
 *   Descriptive Name:  print a character string on the screen.
 *
 *   Input:  no inputs
 *
 *   Output:  nothing returned
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  string appears on screen.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
void print_string( int x, int y, int color, char * charstring )
{
  NCURSES_ATTR_T attr = COLOR_PAIR( color );
  int size;

  if ( GL_use_color )
    attron( attr );
  else if ( color == COLOR_BAR )
    attron( A_REVERSE );

  if ( x < GL_screen_cols && y < GL_screen_lines ) {
    size = strlen( charstring );
    if ( x + size >= GL_screen_cols )
      size = GL_screen_cols - x;

    mvaddnstr( y, x, charstring, size );
    if ( !GL_use_color && color == COLOR_BAR )
      attroff( A_REVERSE );
    
    /*
     * Flush output now
     */
    refresh ();
  }
  else {
    log_error( "%s: string %s is off the screen:  x == %i (max == %i), y == %i (max == %i).\n", __FUNCTION__, charstring, x, GL_screen_cols, y, GL_screen_lines );
  }
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  move_cursor
 *
 *   Descriptive Name:  move the cursor to a place on the screen.
 *
 *   Input:  no inputs
 *
 *   Output:  nothing returned
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  cursor appears on screen.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
void move_cursor( int x, int y )
{
  move( y, x );
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  get_inputc
 *
 *   Descriptive Name:  get a single input character from the keyboard.
 *
 *   Input:  no inputs
 *
 *   Output:  one key character.
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  none
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
int get_inputc( void )
{
  (void) flushinp();                /* flush the input buffer              */
  return  getch();
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  kb_video_shutdown
 *
 *   Descriptive Name:  Shuts down the ncurses environment.
 *
 *   Input:  no inputs
 *
 *   Output:  nothing returned
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  ncurses environment is terminated.
 *
 *   Notes:  only call once when ending program.
 *
 *---------------------------------------------------------------------*/
void kb_video_shutdown( void )
{
  endwin();           /* shut down curses                    */
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  drawhline, drawvline
 *
 *   Descriptive Name:  Draws horizontal or vertical lines on screen.
 *
 *   Input:  x and y start coordinates, length of line
 *
 *   Output:  nothing returned
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  line drawn on screen.
 *
 *   Notes:  need to make insure it does NOT run off the screen.
 *
 *---------------------------------------------------------------------*/
void drawhline( int x, int y, int count )
{
  mvhline( y, x, ACS_HLINE, count );
}


void drawvline( int x, int y, int count )
{
  mvvline( y, x, ACS_VLINE, count );
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  drawbleft, drawbright
 *
 *   Descriptive Name:  Draws left or right side of box on screen.
 *
 *   Input:  x and y start coordinates, height of box
 *
 *   Output:  nothing returned
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  line drawn on screen.
 *
 *   Notes:  the box left/right is essentially a vertical line w/ corners.
 *           need to make insure it does NOT run off the screen.
 *
 *---------------------------------------------------------------------*/
void drawbleft( int x, int y, int count )
{
  if ( count > 2 ) {
    mvvline( y, x, ACS_ULCORNER, 1 );
    mvvline( y + 1, x, ACS_VLINE, count - 2 );
    mvvline( y + count - 1, x, ACS_LLCORNER, 1 );
  }
}


void drawbright( int x, int y, int count )
{
  if ( count > 2 ) {
    mvvline( y, x, ACS_URCORNER, 1 );
    mvvline( y + 1, x, ACS_VLINE, count - 2 );
    mvvline( y + count - 1, x, ACS_LRCORNER, 1 );
  }
}
