/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: volume_cb.c
 */ 
 
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "create.h"
#include "destroy.h"
#include "resize.h"
#include "revert.h"
#include "volume_cb.h"
#include "selection.h"
#include "selection_cb.h"
#include "modify_volume.h"
#include "thing.h"
#include "readable.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   void on_revert_volume_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the Revert API call using the handle
 *      of the volume associated with the last row selected. It then
 *      creates and displays the results popup which destroys both
 *      windows when dismissed.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_revert_volume_button_clicked (GtkButton *button, gpointer user_data)
{
    gchar *error_msg   = _("An error was encountered attempting to revert the volume.");
    gchar *success_msg = _("The volume was successfully reverted to a storage object.");

    on_revert_thing_button_clicked (button, error_msg, success_msg);
}

/*
 *
 *   void on_revert_volume_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of volumes that can be reverted.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Selection list populated with volumes
 *
 */
void on_revert_volume_clist_realize (GtkWidget *widget, gpointer user_data)
{    
    on_revert_thing_clist_realize (widget, VOLUME);
}

/*
 *
 *   void on_create_compat_volume_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the evms_create_compatibility_volume() API call
 *      using the handle of the object associated with the last row 
 *      selected. It then creates and displays the results popup which
 *      destroys both windows when dismissed.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_create_compat_volume_button_clicked (GtkButton *button, gpointer user_data)
{
    gchar *error_msg   = _("An error was encountered attempting to create the volume.");
    gchar *success_msg = _("The volume was successfully created.");

    on_create_volume_button_clicked (button, error_msg, success_msg, TRUE);
}

/*
 *
 *   void on_create_compat_volume_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of storage objects that can be used to create a compatibility
 *      volume.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Selection list populated with sysdata objects
 *
 */
void on_create_compat_volume_clist_realize (GtkWidget *widget, gpointer user_data)
{
    on_create_volume_clist_realize (widget, TRUE);
}

/*
 *
 *   void on_create_evms_volume_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the evms_create_volume API call
 *      using the handle of the object associated with the last row 
 *      selected. It then creates and displays the results popup which
 *      destroys both windows when dismissed.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_create_evms_volume_button_clicked (GtkButton *button, gpointer user_data)
{
    gchar *error_msg   = _("An error was encountered attempting to create the volume.");
    gchar *success_msg = _("The volume was successfully created.");

    on_create_volume_button_clicked (button, error_msg, success_msg, FALSE);
}

/*
 *
 *   void on_create_evms_volume_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of storage objects that can be used to create an evms
 *      volume.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Selection list populated with suitable toplevel storage objects
 *
 */
void on_create_evms_volume_clist_realize (GtkWidget *widget, gpointer user_data)
{
    on_create_volume_clist_realize (widget, FALSE);
}

/*
 *
 *   void on_destroy_volume_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the destruction of a volume. It uses the
 *      handle of the volume associated with the last row selected. It
 *      then  creates and displays the results popup which destroys both 
 *      windows when dismissed.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_destroy_volume_button_clicked (GtkButton *button, gpointer user_data)
{
    gchar *error_msg   = _("An error was encountered attempting to destroy the volume.");
    gchar *success_msg = _("The volume was successfully destroyed.");

    on_destroy_thing_button_clicked (button, error_msg, success_msg);
}

/*
 *
 *   void on_destroy_volume_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of volumes that can be destroyed.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Selection list populated with suitable volumes
 *
 */
void on_destroy_volume_clist_realize (GtkWidget *widget, gpointer user_data)
{
    on_destroy_thing_clist_realize (widget, VOLUME);
}

/*
 *
 *   GtkWidget *create_create_evms_volume_window (GtkSignalFunc, gpointer)
 *   
 *   Description:
 *      This routine creates the dialog that allows creating an EVMS volume
 *      from the objects in the selection list. It is essentially a standard
 *      selection dialog modified slightly to allow specifying the name of
 *      the volume.
 * 
 *   Entry:
 *      clist_realize_sighandler - function that will populate list when realized
 *      user_data                - address of user data to bind with signal handlers
 *      
 *   Exit:
 *      Returns the id of the dialog window created.
 *
 */
GtkWidget *create_create_evms_volume_window (GtkSignalFunc clist_realize_sighandler, gpointer user_data)
{
    GtkWidget *window;

    window = create_selection_window ();

    if (window != NULL)
    {
        GtkWidget *clist;
        GtkWidget *help_button;
        GtkWidget *cancel_button;
        GtkWidget *prev_button;
        GtkWidget *next_button;
        GtkWidget *vbox;
        GtkWidget *label;
        GtkWidget *hbox;
        GtkWidget *entry;
        GtkWidget *next_button_label;

        /*
         * Before we show the window, setup the signal handlers
         * that will populate the clist when the clist is realized. 
         * Also initialize the "Cancel" button to dismiss the window.
         * Setup the the "Next" button to invoke the task and display
         * a results window. The "Next" button's sensitivity is reliant
         * on a row selection. The "Previous" button should be set
         * to insensitive since now panel preceded it.
         */

        clist             = gtk_object_get_data (GTK_OBJECT (window), "selection_window_clist");
        help_button       = gtk_object_get_data (GTK_OBJECT (window), "help_button");
        prev_button       = gtk_object_get_data (GTK_OBJECT (window), "selection_window_prev_button");
        next_button       = gtk_object_get_data (GTK_OBJECT (window), "selection_window_next_button");
        cancel_button     = gtk_object_get_data (GTK_OBJECT (window), "selection_window_cancel_button");
        vbox              = gtk_object_get_data (GTK_OBJECT (window), "selection_window_vbox");
        label             = gtk_object_get_data (GTK_OBJECT (window), "selection_window_instruction_label");
        next_button_label = gtk_object_get_data (GTK_OBJECT (window), "selection_window_next_button_label");

        /*
         * By default we hide the min/max object size column. Only the resize
         * code fills it in a makes use of it.
         */
        
        gtk_clist_set_column_visibility (GTK_CLIST (clist), SL_MINMAX_SIZE_COLUMN, FALSE);
        gtk_clist_column_titles_passive (GTK_CLIST (clist));
        
        /*
         * We need to modify the selection window slightly in order to
         * add a vertical box to hold the instructions label and a
         * text entry box to hold the name of the volume. We will connect
         * a different row selection signal handle and another one to the
         * text entry changes to ensure that "Next" is only sensitive when
         * both text is in the entry field and a row has been selected.
         */

        gtk_container_remove (GTK_CONTAINER (vbox), label);

        hbox = gtk_hbox_new (FALSE, 10);
        gtk_widget_set_name (hbox, "selection_window_hbox");
        gtk_widget_ref (hbox);
        gtk_object_set_data_full (GTK_OBJECT (window), "selection_window_hbox", hbox,
                                  (GtkDestroyNotify) gtk_widget_unref);
        gtk_widget_show (hbox);
        gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 6);
        gtk_container_set_border_width (GTK_CONTAINER (hbox), 15);

        label = gtk_label_new (_("Name"));
        gtk_widget_set_name (label, "selection_window_name_label");
        gtk_widget_ref (label);
        gtk_object_set_data_full (GTK_OBJECT (window), "selection_window_name_label", label,
                                  (GtkDestroyNotify) gtk_widget_unref);
        gtk_widget_show (label);
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

        entry = gtk_entry_new ();
        gtk_widget_set_name (entry, "selection_window_entry");
        gtk_widget_ref (entry);
        gtk_object_set_data_full (GTK_OBJECT (window), "selection_window_entry", entry,
                                  (GtkDestroyNotify) gtk_widget_unref);
        gtk_widget_show (entry);
        gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);

        gtk_box_reorder_child (GTK_BOX (vbox), hbox, 1);

        gtk_window_set_title (GTK_WINDOW (window), _("Create EVMS Volume"));

        gtk_label_set_text (GTK_LABEL (next_button_label), _("Create"));

        gtk_signal_connect (GTK_OBJECT (clist), "realize",
                            clist_realize_sighandler, user_data);

        gtk_signal_connect (GTK_OBJECT (help_button), "clicked",
                            on_selection_window_help_button_clicked, 
                            create_evms_volume_help_text);

        gtk_signal_connect (GTK_OBJECT (next_button), "clicked",
                            on_create_evms_volume_button_clicked, user_data);

        gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
                            on_button_clicked_destroy_window, NULL);

        gtk_signal_connect (GTK_OBJECT (clist), "select_row",
                            GTK_SIGNAL_FUNC (on_selection_window_clist_select_row_extra),
                            user_data);
                            
        gtk_signal_connect (GTK_OBJECT (clist), "unselect_row",
                            GTK_SIGNAL_FUNC (on_selection_window_clist_unselect_row),
                            NULL);

        gtk_signal_connect (GTK_OBJECT (entry), "changed",
                            GTK_SIGNAL_FUNC (on_selection_window_entry_changed), user_data);
                            
        gtk_signal_connect (GTK_OBJECT (clist), "expose_event",
                            GTK_SIGNAL_FUNC (check_for_empty_list_when_exposed),
                            GUINT_TO_POINTER (SL_DESC_COLUMN));

        gtk_widget_set_sensitive (next_button, FALSE);
        gtk_widget_set_sensitive (prev_button, FALSE);

        GTK_WIDGET_SET_FLAGS (entry, GTK_CAN_DEFAULT);
        gtk_widget_grab_default (entry);
    }

    return window;
}

/*
 *
 *   void on_create_volume_popup_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine initiates the creation of an EVMS volume using the
 *      handle of a suitable storage object. In this case the user_data
 *      is the handle.
 * 
 *   Entry:
 *      menuitem  - the popup menuitem that was selected
 *      user_data - the user data which is the object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_create_volume_popup_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_create_evms_volume_window (add_thing_as_selected_list_item, user_data);

    gtk_widget_show (window);
}

/*
 *
 *   void on_create_compatibility_volume_popup_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine initiates the creation of a compatibility volume.
 *      The handle of the object to place the volume on top of is
 *      provided through the user data.
 * 
 *   Entry:
 *      menuitem  - the popup menuitem that was selected
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_create_compatibility_volume_popup_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Create Compatibility Volume"),
                                               _("Create"),
                                               create_compat_volume_help_text,
                                               add_thing_as_selected_list_item,
                                               on_create_compat_volume_button_clicked,
                                               NULL, NULL, NULL, NULL, user_data);

    gtk_widget_show (window);
}

/*
 *
 *   void on_resize_volume_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine simply calls the function that handles
 *      displaying expand/shrink points for the selected
 *      volume.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - contains id of resize action EVMS_Task_Expand or
 *                  EVMS_Task_Shrink
 *
 *   Exit:
 *      See description.
 *
 */
void on_resize_volume_button_clicked (GtkButton *button, gpointer user_data)
{
    on_resize_selection_button_clicked (button, GPOINTER_TO_UINT (user_data));
}

/*
 *
 *   void on_resize_volume_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of volumes that can be expanded or shrunk depending on
 *      the action.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - contains id of resize action EVMS_Task_Expand
 *                  or EVMS_Task_Shrink
 *
 *   Exit:
 *      Selection list populated with suitable expandable or shrinkable
 *      volumes.
 *
 */
void on_resize_volume_clist_realize (GtkWidget *widget, gpointer user_data)
{
    gint            rc;
    GtkCList       *clist = GTK_CLIST (widget);
    task_action_t   action = GPOINTER_TO_UINT (user_data);
    handle_array_t *volumes;
    
    gtk_clist_set_column_visibility (clist, SL_MINMAX_SIZE_COLUMN, TRUE);
    gtk_clist_set_column_auto_resize (clist, SL_MINMAX_SIZE_COLUMN, TRUE);
    
    set_selection_window_clist_column_titles (clist, _("Size"), 
                                              make_object_type_readable_string (VOLUME),
                                              action == EVMS_Task_Shrink ? _("Minimum Size") : _("Maximum Size"));
    
    rc = evms_get_volume_list (0, &volumes);
    
    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_volume_list() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {        
        if (action == EVMS_Task_Expand)
            add_expandable_objects_to_selection_clist (clist, volumes, TRUE);
        else
            add_shrinkable_objects_to_selection_clist (clist, volumes, TRUE);
        
        evms_free (volumes);
    }
}

/*
 *
 *   void on_modify_volume_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of volumes that can have their properties changed.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Selection list populated with a list of volumes that can have
 *      their properties changed.
 *
 */
void on_modify_volume_clist_realize (GtkWidget *widget, gpointer user_data)
{
    gint            rc;    
    GtkCList       *clist = GTK_CLIST (widget);
    handle_array_t *volumes;
    
    set_selection_window_clist_column_titles (clist, _("Size"), 
                                              make_object_type_readable_string (VOLUME), NULL);
    
    rc = evms_get_volume_list (0, &volumes);
    
    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_volume_list() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        gint     i;
        gboolean is_selected = (volumes->count == 1);
        
        for (i=0; i < volumes->count; i++)
        {
            if (evms_can_set_volume_name (volumes->handle[i]) == 0)
                add_thing_to_selection_list (clist, volumes->handle[i], is_selected);
        }
        
        if (clist->rows == 1)
            gtk_clist_select_row (clist, 0, 0);

        evms_free (volumes);
    }
}

/*
 *
 *   void on_volume_name_entry_changed (GtkEditable *, gpointer)
 *
 *   Description:
 *      This routine is called whenever the editable text
 *      entry field in the modify volume window has changed.
 *      If the entry is void of characters or only contains
 *      whitespace then we disable the Modify button. If it
 *      entry now has text then we ensure that the Modify 
 *      button is enabled.
 *
 *   Entry:
 *      editable  - address of GtkEditable descendant object that 
 *                  received changed signal
 *      user_data - address of user data bound to widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_volume_name_entry_changed (GtkEditable *editable, gpointer user_data)
{    
    GtkWidget *modify_button = lookup_widget (GTK_WIDGET (editable), "modify_button");

    if (editable_field_is_blank (editable))
    {
       if (GTK_WIDGET_IS_SENSITIVE (modify_button))
           gtk_widget_set_sensitive (modify_button, FALSE);
    }
    else
    {
       if (!(GTK_WIDGET_IS_SENSITIVE (modify_button)))
           gtk_widget_set_sensitive (modify_button, TRUE);
    }
}

/*
 *
 *   gchar* get_volume_name_text (GtkWidget *)
 *
 *   Description:
 *      This routine retrieves and returns the text in the volume
 *      name entry field.
 *
 *   Entry:
 *      widget - a widget in the same window as the volume name entry
 *
 *   Exit:
 *      Dynamically allocated string containing volume name which
 *      should be freed with g_free() when no longer needed. NULL
 *      is returned if problem looking up entry field.
 *
 */
gchar* get_volume_name_text (GtkWidget *widget)
{
    gchar     *text = NULL;
    GtkWidget *entry;
    
    entry = lookup_widget (widget, "volume_name_entry");
    
    if (entry)
        text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
    
    return text;
}

/*
 *
 *   void on_modify_volume_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the modification of a volume's properties.
 *      It invokes evms_set_volume_name() to update the volume name.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_modify_volume_button_clicked (GtkButton *button, gpointer user_data)
{
    object_handle_t volume = GPOINTER_TO_UINT (user_data);
    
    if (volume)
    {
        gint   rc;
        gchar *name;
        
        name = get_volume_name_text (GTK_WIDGET (button));
        
        rc = evms_set_volume_name (volume, g_strstrip (name));
        
        if (rc != SUCCESS)
            log_error ("%s: evms_set_volume_name() returned error code %d.\n", __FUNCTION__, rc);
        
        display_selection_window_results (GTK_WIDGET (button), rc,
                                          _("An error was encountered modifying volume properties"),
                                          _("Volume properties were modified successfully."));
        g_free (name);
    }
}

/*
 *
 *   void on_volume_name_entry_realize (GtkEntry *, gpointer)
 *
 *   Description:
 *      This routine sets the initial name displayed in the volume
 *      name entry field to the current volume name for the given
 *      volume handle.
 * 
 *   Entry:
 *      entry     - the id of the volume name entry field
 *      user_data - contains the volume object handle
 *
 *   Exit:
 *      Nothing returned.
 *
 */
void on_volume_name_entry_realize (GtkEntry *entry, gpointer user_data)
{
    gint                  rc;
    object_handle_t       volume = GPOINTER_TO_UINT (user_data);
    handle_object_info_t *object;

    rc = evms_get_info (volume, &object);
    
    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
        
        display_results_window (rc, NULL, _("Error retrieving current volume name.\n"), 
                                NULL, FALSE, gtk_widget_get_toplevel (GTK_WIDGET (entry)));
    }
    else
    {
        gtk_entry_set_text (entry, object->info.volume.name);
        
        evms_free (object);
    }    
}

/*
 *
 *   GtkWidget* build_modify_volume_window (object_handle_t)
 *
 *   Description:
 *      This routine invokes the routine to create the standard
 *      modify volume window connects additional signal handlers
 *      for the modify volume window widgets.
 * 
 *   Entry:
 *      volume - the volume object handle
 *
 *   Exit:
 *      Returns the window id of the modify volume window created
 *
 */
GtkWidget* build_modify_volume_window (object_handle_t volume)
{
    GtkWidget *window;
    GtkWidget *modify_button;
    GtkWidget *volume_name_entry;
    
    window = create_modify_volume_window ();
    
    modify_button     = gtk_object_get_data (GTK_OBJECT (window), "modify_button");
    volume_name_entry = gtk_object_get_data (GTK_OBJECT (window), "volume_name_entry");
    
    gtk_signal_connect (GTK_OBJECT (modify_button), "clicked",
                        GTK_SIGNAL_FUNC (on_modify_volume_button_clicked), 
                        GUINT_TO_POINTER (volume));
    
    gtk_signal_connect (GTK_OBJECT (volume_name_entry), "realize",
                        GTK_SIGNAL_FUNC (on_volume_name_entry_realize),
                        GUINT_TO_POINTER (volume));
    
    return window;
}

/*
 *
 *   void on_modify_volume_selection_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the the modification of a volume's properties.
 *      It creates another window that allows modifying the properties of
 *      of the currently selected volume. 
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_modify_volume_selection_button_clicked (GtkButton *button, gpointer user_data)
{
    GList          *window_list;
    GtkWidget      *clist;
    GtkWidget      *next_window;
    GtkWidget      *selection_window;
    object_handle_t volume;
    object_handle_t prev_volume;

    /*
     * Hide the selection window and either redisplay the next window
     * if it exists and the selection in this window did not change
     * or create it if it does not exist. However, if the selection
     * changed and the next window existed then destroy the next window
     * and create a new one.
     */

    selection_window = gtk_widget_get_toplevel (GTK_WIDGET (button));
    clist            = lookup_widget (GTK_WIDGET (button), "selection_window_clist");
    volume           = GPOINTER_TO_UINT (get_single_select_current_row_data (GTK_CLIST (clist)));
    window_list      = get_window_list (selection_window);
    next_window      = gtk_object_get_data (GTK_OBJECT (selection_window), "next_window_id");
    prev_volume      = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (selection_window), 
                                         "previous_volume"));

    if (window_list == NULL)
        window_list = g_list_append (window_list, selection_window);

    if (volume != prev_volume)
    {
        if (next_window != NULL)
            destroy_window_list (g_list_find (window_list, next_window));

        next_window = build_modify_volume_window (volume);

        window_list = g_list_append (window_list, next_window);

        set_window_list (next_window, window_list);
        set_window_list (selection_window, window_list);

        gtk_object_set_data (GTK_OBJECT (selection_window), "next_window_id" , next_window);
        gtk_object_set_data (GTK_OBJECT (selection_window), "previous_volume", GUINT_TO_POINTER (volume));
        
    }
    
    gtk_widget_show (next_window);
    gtk_widget_hide (selection_window);
}

/*
 *
 *   void on_modify_volume_help_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine displays help on the modify volume properties window.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_modify_volume_help_button_clicked (GtkButton *button, gpointer user_data)
{
    GtkWidget *parent;
    
    parent = gtk_widget_get_toplevel (GTK_WIDGET (button));
    display_help_window (GTK_WINDOW (parent), modify_volume_properties_2_help_text);
}

/*
 *
 *   void on_modify_volume_popup_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine initiates the modification of an EVMS volume's properties.
 * 
 *   Entry:
 *      menuitem  - the popup menuitem that was selected
 *      user_data - the user data which is the object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_modify_volume_popup_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *window;

    window = create_standard_selection_window (_("Modify Logical Volume Properties"),
                                               NULL,
                                               modify_volume_properties_help_text,
                                               add_thing_as_selected_list_item,
                                               on_modify_volume_selection_button_clicked,
                                               NULL, NULL, NULL, NULL, user_data);

    gtk_widget_show (window);    
}

/*
 *
 *   gchar *get_volume_status_string (u_int32_t)
 *
 *   Description:
 *      This routine converts the flags bitmap to a readable string
 *      representing the flags that are enabled.
 * 
 *   Entry:
 *      flags - the volume flags bitmap
 *
 *   Exit:
 *      Returns a dynamically allocated string containing the volume
 *      status. The string should be freed with g_free() when no
 *      longer needed.
 *
 */
gchar *get_volume_status_string (u_int32_t flags)
{
    gchar   *status_string;
    GString *status_gstring;
        
    status_gstring = g_string_new ("");
    
    if (flags & VOLFLAG_DIRTY)
    {
        status_gstring = g_string_append (status_gstring, _("Dirty"));
    }
    
    if (flags & VOLFLAG_NEW)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("New"));        
    }

    if (flags & VOLFLAG_NEEDS_DEV_NODE)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("Needs devnode"));
    }

    if (flags & VOLFLAG_READ_ONLY)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("Read only"));
    }
    
    if (flags & VOLFLAG_COMPATIBILITY)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("Compatibility"));
    }
    
    if (flags & VOLFLAG_FOREIGN)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("Foreign"));
    }
    
    if (flags & VOLFLAG_MKFS)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("Mkfs pending"));
    }
    
    if (flags & VOLFLAG_UNMKFS)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("Unmkfs pending"));
    }

    if (flags & VOLFLAG_FSCK)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("Fsck pending"));
    }
    
    if (flags & VOLFLAG_DEFRAG)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("Defragment pending"));
    }
    
    if (flags & VOLFLAG_EXPAND_FS)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("Expand pending"));
    }

    if (flags & VOLFLAG_SHRINK_FS)
    {
        if (*(status_gstring->str) != '\0')
            status_gstring = g_string_append (status_gstring, ", ");
        
        status_gstring = g_string_append (status_gstring, _("Shrink pending"));
    }
    
    if (*(status_gstring->str) == '\0')
        status_gstring = g_string_append (status_gstring, _("OK"));
    
    status_string = g_strdup (status_gstring->str);
    g_string_free (status_gstring, TRUE);
    
    return status_string;
}

/*
 *
 *   void create_volume_pseudo_extended_info (logical_volume_info_t *, extended_info_array_t **)
 *
 *   Description:
 *      This routine takes the information in a logical volume
 *      info structure and produces a fake extended_info_array_t
 *      from it. The extended info field for the FSIM will allow
 *      a "More" button to get created if an FSIM is associated
 *      with the volume in order to get file system related 
 *      extended info.
 * 
 *   Entry:
 *      volume        - generic info about the volume from the engine
 *      extended_info - address where to store pointer to our pseudo
 *                      extended_info_array_t structure
 *
 *   Exit:
 *      See description.
 *
 */
void create_volume_pseudo_extended_info (logical_volume_info_t *volume, extended_info_array_t **extended_info)
{
    *extended_info = g_malloc0 (sizeof (extended_info_array_t) + (sizeof (extended_info_t) * VOL_EI_FIELD_COUNT));
        
    (*extended_info)->count = VOL_EI_FIELD_COUNT;

    (*extended_info)->info[VOL_EI_FSIM_FIELD].name = "";
    (*extended_info)->info[VOL_EI_FSIM_FIELD].title = _("File System Interface Module");
    (*extended_info)->info[VOL_EI_FSIM_FIELD].desc = _("This is the name of the File System Interface module associated with the volume.");
    (*extended_info)->info[VOL_EI_FSIM_FIELD].type = EVMS_Type_String;
    (*extended_info)->info[VOL_EI_FSIM_FIELD].unit = EVMS_Unit_None;
    (*extended_info)->info[VOL_EI_FSIM_FIELD].format = EVMS_Format_Normal;
    (*extended_info)->info[VOL_EI_FSIM_FIELD].collection_type = EVMS_Collection_None;
    
    if (volume->file_system_manager)
    {    
        (*extended_info)->info[VOL_EI_FSIM_FIELD].value.s = get_volume_fsim_name (volume);
        (*extended_info)->info[VOL_EI_FSIM_FIELD].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
    }
    else
        (*extended_info)->info[VOL_EI_FSIM_FIELD].value.s = g_strdup (_("None"));
        
    (*extended_info)->info[VOL_EI_MOUNT_POINT_FIELD].name = _("mount_point");
    (*extended_info)->info[VOL_EI_MOUNT_POINT_FIELD].title = _("Mount Point");
    (*extended_info)->info[VOL_EI_MOUNT_POINT_FIELD].desc = _("This is the file system mount point for the volume.");
    (*extended_info)->info[VOL_EI_MOUNT_POINT_FIELD].type = EVMS_Type_String;
    (*extended_info)->info[VOL_EI_MOUNT_POINT_FIELD].unit = EVMS_Unit_None;
    (*extended_info)->info[VOL_EI_MOUNT_POINT_FIELD].format = EVMS_Format_Normal;
    (*extended_info)->info[VOL_EI_MOUNT_POINT_FIELD].collection_type = EVMS_Collection_None;
    
    if (volume->mount_point)
        (*extended_info)->info[VOL_EI_MOUNT_POINT_FIELD].value.s = g_strdup (volume->mount_point);
    else
        (*extended_info)->info[VOL_EI_MOUNT_POINT_FIELD].value.s = g_strdup (_("Volume is not mounted."));

    (*extended_info)->info[VOL_EI_SIZE_FIELD].name = _("size");
    (*extended_info)->info[VOL_EI_SIZE_FIELD].title = _("File System Size");
    (*extended_info)->info[VOL_EI_SIZE_FIELD].desc = _("This field describes the size of the file system.");
    (*extended_info)->info[VOL_EI_SIZE_FIELD].type = EVMS_Type_Unsigned_Int64;
    (*extended_info)->info[VOL_EI_SIZE_FIELD].unit = EVMS_Unit_Sectors;
    (*extended_info)->info[VOL_EI_SIZE_FIELD].format = EVMS_Format_Normal;
    (*extended_info)->info[VOL_EI_SIZE_FIELD].collection_type = EVMS_Collection_None;
    (*extended_info)->info[VOL_EI_SIZE_FIELD].value.ui64 = volume->fs_size;
    
    (*extended_info)->info[VOL_EI_MIN_SIZE_FIELD].name = _("min_size");
    (*extended_info)->info[VOL_EI_MIN_SIZE_FIELD].title = _("Minimum File System Size");
    (*extended_info)->info[VOL_EI_MIN_SIZE_FIELD].desc = _("This field describes the minimum size of the file system.");
    (*extended_info)->info[VOL_EI_MIN_SIZE_FIELD].type = EVMS_Type_Unsigned_Int64;
    (*extended_info)->info[VOL_EI_MIN_SIZE_FIELD].unit = EVMS_Unit_Sectors;
    (*extended_info)->info[VOL_EI_MIN_SIZE_FIELD].format = EVMS_Format_Normal;
    (*extended_info)->info[VOL_EI_MIN_SIZE_FIELD].collection_type = EVMS_Collection_None;
    (*extended_info)->info[VOL_EI_MIN_SIZE_FIELD].value.ui64 = volume->min_fs_size;
    
    (*extended_info)->info[VOL_EI_MAX_SIZE_FIELD].name = _("min_size");
    (*extended_info)->info[VOL_EI_MAX_SIZE_FIELD].title = _("Maximum File System Size");
    (*extended_info)->info[VOL_EI_MAX_SIZE_FIELD].desc = _("This field describes the maximum size of the file system.");
    (*extended_info)->info[VOL_EI_MAX_SIZE_FIELD].type = EVMS_Type_Unsigned_Int64;
    (*extended_info)->info[VOL_EI_MAX_SIZE_FIELD].unit = EVMS_Unit_Sectors;
    (*extended_info)->info[VOL_EI_MAX_SIZE_FIELD].format = EVMS_Format_Normal;
    (*extended_info)->info[VOL_EI_MAX_SIZE_FIELD].collection_type = EVMS_Collection_None;
    (*extended_info)->info[VOL_EI_MAX_SIZE_FIELD].value.ui64 = volume->max_fs_size;

    (*extended_info)->info[VOL_EI_MAX_VOL_SIZE_FIELD].name = _("max_vol_size");
    (*extended_info)->info[VOL_EI_MAX_VOL_SIZE_FIELD].title = _("Maximum Volume Size");
    (*extended_info)->info[VOL_EI_MAX_VOL_SIZE_FIELD].desc = _("This field describes the maximum size of the volume object.");
    (*extended_info)->info[VOL_EI_MAX_VOL_SIZE_FIELD].type = EVMS_Type_Unsigned_Int64;
    (*extended_info)->info[VOL_EI_MAX_VOL_SIZE_FIELD].unit = EVMS_Unit_Sectors;
    (*extended_info)->info[VOL_EI_MAX_VOL_SIZE_FIELD].format = EVMS_Format_Normal;
    (*extended_info)->info[VOL_EI_MAX_VOL_SIZE_FIELD].collection_type = EVMS_Collection_None;
    (*extended_info)->info[VOL_EI_MAX_VOL_SIZE_FIELD].value.ui64 = volume->max_vol_size;

    (*extended_info)->info[VOL_EI_TOPMOST_OBJECT_FIELD].name = _("topmost_object");
    (*extended_info)->info[VOL_EI_TOPMOST_OBJECT_FIELD].title = _("Topmost Object");
    (*extended_info)->info[VOL_EI_TOPMOST_OBJECT_FIELD].desc = _("This field describes the storage object immediately below the volume.");
    (*extended_info)->info[VOL_EI_TOPMOST_OBJECT_FIELD].type = EVMS_Type_String;
    (*extended_info)->info[VOL_EI_TOPMOST_OBJECT_FIELD].unit = EVMS_Unit_None;
    (*extended_info)->info[VOL_EI_TOPMOST_OBJECT_FIELD].format = EVMS_Format_Normal;
    (*extended_info)->info[VOL_EI_TOPMOST_OBJECT_FIELD].collection_type = EVMS_Collection_None;
    (*extended_info)->info[VOL_EI_TOPMOST_OBJECT_FIELD].value.s = get_object_name (volume->object);

    (*extended_info)->info[VOL_EI_MINOR_NUMBER_FIELD].name = _("minor_no");
    (*extended_info)->info[VOL_EI_MINOR_NUMBER_FIELD].title = _("Minor Number");
    (*extended_info)->info[VOL_EI_MINOR_NUMBER_FIELD].desc = _("This field describes the volume's device minor number.");
    (*extended_info)->info[VOL_EI_MINOR_NUMBER_FIELD].type = EVMS_Type_Unsigned_Int;
    (*extended_info)->info[VOL_EI_MINOR_NUMBER_FIELD].unit = EVMS_Unit_None;
    (*extended_info)->info[VOL_EI_MINOR_NUMBER_FIELD].format = EVMS_Format_Normal;
    (*extended_info)->info[VOL_EI_MINOR_NUMBER_FIELD].collection_type = EVMS_Collection_None;
    (*extended_info)->info[VOL_EI_MINOR_NUMBER_FIELD].value.ui64 = volume->minor_number;

    (*extended_info)->info[VOL_EI_SERIAL_NUMBER_FIELD].name = _("serial_no");
    (*extended_info)->info[VOL_EI_SERIAL_NUMBER_FIELD].title = _("Serial Number");
    (*extended_info)->info[VOL_EI_SERIAL_NUMBER_FIELD].desc = _("This field describes the volume's serial number.");
    (*extended_info)->info[VOL_EI_SERIAL_NUMBER_FIELD].type = EVMS_Type_Unsigned_Int64;
    (*extended_info)->info[VOL_EI_SERIAL_NUMBER_FIELD].unit = EVMS_Unit_None;
    (*extended_info)->info[VOL_EI_SERIAL_NUMBER_FIELD].format = EVMS_Format_Hex;
    (*extended_info)->info[VOL_EI_SERIAL_NUMBER_FIELD].collection_type = EVMS_Collection_None;
    (*extended_info)->info[VOL_EI_SERIAL_NUMBER_FIELD].value.ui64 = volume->serial_number;    
    
    (*extended_info)->info[VOL_EI_FLAGS_FIELD].name = _("flags");
    (*extended_info)->info[VOL_EI_FLAGS_FIELD].title = _("Status");
    (*extended_info)->info[VOL_EI_FLAGS_FIELD].desc = _("This field provides status about the volume.");
    (*extended_info)->info[VOL_EI_FLAGS_FIELD].type = EVMS_Type_String;
    (*extended_info)->info[VOL_EI_FLAGS_FIELD].unit = EVMS_Unit_None;
    (*extended_info)->info[VOL_EI_FLAGS_FIELD].format = EVMS_Format_Normal;
    (*extended_info)->info[VOL_EI_FLAGS_FIELD].collection_type = EVMS_Collection_None;
    (*extended_info)->info[VOL_EI_FLAGS_FIELD].value.s = get_volume_status_string (volume->flags);
}

/*
 *
 *   void on_modify_volume_popup_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine initiates the modification of an EVMS volume's properties.
 * 
 *   Entry:
 *      menuitem  - the popup menuitem that was selected
 *      user_data - the user data which is the object handle
 *
 *   Exit:
 *      See description.
 *
 */
void free_extended_info (extended_info_array_t *extended_info)
{
    g_free (extended_info->info[VOL_EI_FSIM_FIELD].value.s);
    g_free (extended_info->info[VOL_EI_MOUNT_POINT_FIELD].value.s);
    g_free (extended_info->info[VOL_EI_TOPMOST_OBJECT_FIELD].value.s);
    g_free (extended_info->info[VOL_EI_FLAGS_FIELD].value.s);
    g_free (extended_info);
}

/*
 *
 *   void on_display_volume_details_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine displays the generic volume information as though it
 *      was extended info. If the volume has an associated FSIM then the
 *      user can click on the more button to see file system related info
 *      on the volume.
 * 
 *   Entry:
 *      menuitem  - menu item selected
 *      user_data - contains handle of the volume we want detailed info on
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_display_volume_details_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    gint                  rc;
    object_handle_t       handle = GPOINTER_TO_UINT (user_data);
    handle_object_info_t *object;

    rc = evms_get_info (handle, &object);

    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
    	display_popup_window (_("View Detailed Information"), _("No detailed information is available."));        
    }
    else
    {
        GtkWidget             *window;
        extended_info_array_t *extended_info;
        
        create_volume_pseudo_extended_info (&(object->info.volume), &extended_info);
        
        window = create_thing_details_window (handle, extended_info, NULL);
        bind_object_handle_to_toplevel_widget (window, handle);
        gtk_widget_show (window);

        free_extended_info (extended_info);
        evms_free (object);
    }
}
