/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001, 2002
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: transfer.c
 */ 
 
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "transfer.h"
#include "thing.h"
#include "readable.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   gchar* get_container_name (object_handle_t)
 *
 *   Description:
 *      This routine returns a dynamically allocated
 *      string containing the name of the container
 *      corresponding to the handle given.
 * 
 *   Entry:
 *      handle - object handle
 *
 *   Exit:
 *      Container name string which should be freed
 *      with g_free or NULL if the name could not be
 *      obtained.
 *
 */
gchar* get_container_name (object_handle_t handle)
{
    gchar *name=NULL;
    
    if (handle)
    {
        gint                  rc;
        handle_object_info_t *info;
    
        rc = evms_get_info (handle, &info);
        
        if (rc != SUCCESS)
        {
            log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
        }
        else
        {
            name = g_strdup (info->info.container.name);
            evms_free (info);
        }
    }
    
    return name;
}

/*
 *
 *   void on_remove_thing_from_container_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates removing a storage object from its
 *      consuming container.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_remove_thing_from_container_button_clicked (GtkButton *button, gpointer user_data)
{
    GtkCList       *clist;
    object_handle_t handle;
        
    clist = GTK_CLIST (lookup_widget (GTK_WIDGET (button), "selection_window_clist"));

    handle = GPOINTER_TO_UINT (get_single_select_current_row_data (clist));

    if (handle != 0)
    {    
        gint   rc;
        gchar *error_msg   = _("An error was encountered attempting to remove the storage object.");
        gchar *success_msg = _("The storage object was successfully removed.");
    
        rc = evms_transfer (handle, 0, 0, NULL);
            
        display_selection_window_results (GTK_WIDGET (button), rc, error_msg, success_msg);
    }
}

/*
 *
 *   void on_remove_thing_from_container_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the thing we intend to remove from
 *      a certain container and allow the user to confirm the removal
 *      operation.
 * 
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_remove_thing_from_container_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    gint                  rc;
    object_handle_t       handle = GPOINTER_TO_UINT (user_data);
    handle_object_info_t *info;

    rc = evms_get_info (handle, &info);
    
    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
    }
    else
    {
        gchar        *window_title;
        gchar        *help_text;
        gchar        *container_name;
        GtkWidget    *window;
        object_type_t type;

        container_name = get_container_name (info->info.object.consuming_container);
        
        if (container_name)
        {    
            window_title = g_strdup_printf (_("Remove %s from %s"), 
                                            info->info.object.name,
                                            container_name);
            g_free (container_name);
        }
        else
            window_title = g_strdup_printf (_("Remove %s from unknown container"),
                                            info->info.object.name);
        
        if (evms_get_handle_object_type (handle, &type) == SUCCESS)
        {
            switch (type)
            {
                case REGION:
                    help_text = remove_region_from_container_help_text;
                    break;
                
                case SEGMENT:
                    help_text = remove_segment_from_container_help_text;
                    break;
                
                case DISK:
                    help_text = remove_disk_from_container_help_text;
                    break;
                
                default:
                    help_text = NULL;            
                    break;
            }
        }
        else
        {    
            help_text = NULL;
        }

        window = create_standard_selection_window (window_title,
                                                   _("Remove"),
                                                   help_text,
                                                   add_thing_as_selected_list_item,
                                                   on_remove_thing_from_container_button_clicked,
                                                   NULL, NULL, NULL, NULL, user_data);

        gtk_widget_show (window);
        
        evms_free (info);
        g_free (window_title);
    }
}

/*
 *
 *   void add_consumed_object_to_selection_list (GtkCList *, object_handle_t, gboolean)
 *   
 *   Description:
 *      This routine appends a row to a GtkCList with information
 *      on the given object consumed by a container.
 * 
 *   Entry:
 *      clist       - address of the selections GtkCList widget
 *      object      - the handle to the consumed object
 *      is_selected - whether to mark this item as selected
 *
 *   Exit:
 *      A new row corresponding to the consumed object is added to the 
 *      clist.
 *
 */
void add_consumed_object_to_selection_list (GtkCList *clist, object_handle_t object, gboolean is_selected)
{
    gint row;
    
    row = add_thing_to_selection_list (clist, object, is_selected);    
    
    if (row != -1)
    {
        gint                  rc;
        handle_object_info_t *info;
    
        rc = evms_get_info (object, &info);
        
        if (rc != SUCCESS)
        {
            log_error ("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
        }
        else
        {        
            gchar *container_name;
            
            container_name = get_container_name (info->info.object.consuming_container);
            gtk_clist_set_text (clist, row, SL_MINMAX_SIZE_COLUMN, container_name);
            g_free (container_name);
            evms_free (info);
        }
    }
}

/*
 *
 *   void on_remove_thing_from_container_clist_realize (GtkWidget *, object_type_t)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of storage objects that can be removed from their consuming
 *      container.
 * 
 *   Entry:
 *      widget - address of the selections GtkCList widget
 *      type   - type of the container consumables to list
 *
 *   Exit:
 *      Selection list populated with things in the handle array that can
 *      removed from their corresponding container
 *
 */
void on_remove_thing_from_container_clist_realize (GtkWidget *widget, object_type_t type)
{
    gint            rc=0;
    GtkCList       *clist = GTK_CLIST (widget);
    handle_array_t *things;

    /*
     * Use the normally hidden min/max size column for the name of the consuming
     * container.
     */    
    gtk_clist_set_column_visibility (clist, SL_MINMAX_SIZE_COLUMN, TRUE);
    gtk_clist_set_column_auto_resize (clist, SL_MINMAX_SIZE_COLUMN, TRUE);
    
    set_selection_window_clist_column_titles (clist, _("Size"), 
                                              make_object_type_readable_string (type),
                                              _("Consuming Container"));
    
    switch (type)
    {
        case REGION:
        case SEGMENT:
        case DISK:
            rc = evms_get_object_list (type, DATA_TYPE, 0, 0, &things);
            break;

        default:
            log_error ("%s: Unsupported selection type %d.\n", __FUNCTION__, type);
            rc = EINVAL;
            break;
    }

    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_object_list() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint    i;
        gboolean is_selected = (things->count == 1);
    
        for (i=0; i < things->count; i++)
        {
            if (evms_can_remove_from_container (things->handle[i]) == 0)
                add_consumed_object_to_selection_list (clist, things->handle[i], is_selected);
        }
        
        if (clist->rows == 1)
            gtk_clist_select_row (clist, 0, 0);
        
        evms_free (things);
    }
}
