/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: resize.c
 */ 
 
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "widget.h"
#include "task.h"
#include "resize.h"
#include "thing.h"
#include "readable.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   void set_resize_points_clist_properties (GtkCList *)
 *   
 *   Description:
 *      This routine updates some of the column properties for
 *      a selection list to be used for shrink/expand point
 *      selection.
 * 
 *   Entry:
 *      clist  - address of the selections GtkCList widget
 *      action - EVMS_Task_Expand or EVMS_Task_Shrink
 *
 *   Exit:
 *      Selection list column properties set accordingly
 *
 */
void set_resize_points_clist_properties (GtkCList *clist, task_action_t action)
{
    set_selection_window_clist_column_titles (clist,
                                              _("Object Size"),
                                              action == EVMS_Task_Shrink ? _("Shrink Point") : _("Expansion Point"),
                                              action == EVMS_Task_Shrink ? _("Shrinkable by") : _("Expandable by"));
    
    gtk_clist_set_column_visibility (clist, SL_MINMAX_SIZE_COLUMN, TRUE);
    gtk_clist_set_column_auto_resize (clist, SL_MINMAX_SIZE_COLUMN, TRUE);
    gtk_clist_set_column_justification (clist, SL_MINMAX_SIZE_COLUMN, GTK_JUSTIFY_RIGHT);        
    gtk_clist_set_column_auto_resize (clist, SL_SIZE_COLUMN, TRUE);
    gtk_clist_set_column_justification (clist, SL_SIZE_COLUMN, GTK_JUSTIFY_RIGHT);
}

/*
 *
 *   void add_resize_point_to_selection_list (GtkCList *, object_handle_t, 
 *                                            sector_count_t, gboolean)
 *   
 *   Description:
 *      This routine appends a row to a GtkCList with the given 
 *      shrink/expand point information.
 * 
 *   Entry:
 *      clist       - address of the selections GtkCList widget
 *      object      - the handle to the shrink/expand point object
 *      resize_max  - the max shrink or expand size for the object
 *      is_selected - whether to mark this item as selected
 *
 *   Exit:
 *      A new row corresponding to the shrink/expand is added to the 
 *      clist.
 *
 */
void add_resize_point_to_selection_list (GtkCList *clist, object_handle_t object, 
                                         sector_count_t resize_max, gboolean is_selected)
{
    gint row;
    
    row = add_thing_to_selection_list (clist, object, is_selected);    
    
    if (row != -1)
    {
        gchar *readable_size;
        
        readable_size = make_sectors_readable_string (resize_max);    
        gtk_clist_set_text (clist, row, SL_MINMAX_SIZE_COLUMN, readable_size);    
        g_free (readable_size);
    }
}

/*
 *
 *   void on_shrink_points_clist_realize (GtkCList *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of shrink points for an object.
 * 
 *   Entry:
 *      clist     - address of the selections GtkCList widget
 *      user_data - contains handle of object
 *
 *   Exit:
 *      Selection list populated with shrink points for the given object
 *
 */
void on_shrink_points_clist_realize (GtkCList *clist, gpointer user_data)
{
    gint                   rc;
    object_handle_t        object = GPOINTER_TO_UINT (user_data);
    shrink_handle_array_t *shrink_points;

    rc = evms_get_shrink_points (object, &shrink_points);

    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_shrink_points() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        gint     i;
        gboolean is_selected = (shrink_points->count == 1);
        
        set_resize_points_clist_properties (clist, EVMS_Task_Shrink);
        
        for (i=0; i < shrink_points->count; i++)
            add_resize_point_to_selection_list (clist, shrink_points->shrink_point[i].object,
                                                shrink_points->shrink_point[i].max_shrink_size,
                                                is_selected);

        evms_free (shrink_points);
    }
}

/*
 *
 *   void on_expand_points_clist_realize (GtkCList *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of expand points for an object.
 * 
 *   Entry:
 *      clist     - address of the selections GtkCList widget
 *      user_data - contains handle of object
 *
 *   Exit:
 *      Selection list populated with expansion points for the given object
 *
 */
void on_expand_points_clist_realize (GtkCList *clist, gpointer user_data)
{
    gint                   rc;
    object_handle_t        object = GPOINTER_TO_UINT (user_data);
    expand_handle_array_t *expand_points;
    
    rc = evms_get_expand_points (object, &expand_points);

    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_expand_points() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        gint     i;
        gboolean is_selected = (expand_points->count == 1);

        set_resize_points_clist_properties (clist, EVMS_Task_Expand);

        for (i=0; i < expand_points->count; i++)
            add_resize_point_to_selection_list (clist, expand_points->expand_point[i].object,
                                                expand_points->expand_point[i].max_expand_size,
                                                is_selected);
            
        evms_free (expand_points);
    }    
}

/*
 *
 *   gchar* get_resize_points_window_title_text (object_handle_t, task_action_t)
 *
 *   Description:
 *      This routine creates a string do be used for the window title for
 *      selecting an expand or shrink point for a volume or object.
 * 
 *   Entry:
 *      handle - the volume or object handle
 *      action - action (EVMS_Task_Expand or EVMS_Task_Shrink)
 *
 *   Exit:
 *      Returns NULL if an error encountered or a dynamically allocated
 *      string containing the window title that should be freed with
 *      g_free () when no longer needed.
 *
 */
gchar* get_resize_points_window_title_text (object_handle_t handle, task_action_t action)
{
    gint                  rc;
    gchar                *title_text=NULL;
    handle_object_info_t *object;
    
    rc = evms_get_info (handle, &object);
    
    if (rc != SUCCESS)
        log_error ("%s: evms_get_info() returned error code %d.\n", __FUNCTION__, rc);
    else
    {
        gchar *name = "";
        
        if (object->type == VOLUME)
            name = object->info.volume.name;
        else
            name = object->info.object.name;
        
        if (action == EVMS_Task_Expand)
            title_text = g_strdup_printf (_("Select Expansion Point for %s"), name); 
        else
            title_text = g_strdup_printf (_("Select Shrink Point for %s"), name);
            
        evms_free (object);
    }
    
    return title_text;
}

/*
 *
 *   gboolean resize_point_is_selected_object (object_handle_t, task_action_t)
 *
 *   Description:
 *      This routine determines if the object has one and only one
 *      resize point and whether it is the same object as the resize
 *      selection object.
 * 
 *   Entry:
 *      handle - the volume or object handle
 *      action - action (EVMS_Task_Expand or EVMS_Task_Shrink)
 *
 *   Exit:
 *      Returns TRUE if the single resize point is the same as the object
 *      to resize.
 *
 */
gboolean resize_point_is_selected_object (object_handle_t object, task_action_t action)
{
    gint     rc;
    gboolean result=FALSE;
    
    if (action == EVMS_Task_Expand)
    {    
        expand_handle_array_t *expand_points;
        
        rc = evms_get_expand_points (object, &expand_points);
    
        if (rc != SUCCESS)
        {
            log_error ("%s: evms_get_expand_points() returned error code %d.\n", __FUNCTION__, rc);
        }
        else
        {
            result = (expand_points->count == 1 && object == expand_points->expand_point[0].object);
               
            evms_free (expand_points);
        }
    }
    else
    {
        shrink_handle_array_t *shrink_points;
        
        rc = evms_get_shrink_points (object, &shrink_points);
    
        if (rc != SUCCESS)
        {
            log_error ("%s: evms_get_shrink_points() returned error code %d.\n", __FUNCTION__, rc);
        }
        else
        {
            result = (shrink_points->count == 1 && object == shrink_points->shrink_point[0].object);
               
            evms_free (shrink_points);
        }        
    }
    
    return result;
}

/*
 *
 *   void on_resize_point_selection_button_clicked (GtkButton *, task_action_t)
 *
 *   Description:
 *      This routine responds to resize point selection with creating the
 *      task given and creating the acceptable object selection dialog
 *      window to proceed with acceptable parameter and option selections.
 * 
 *   Entry:
 *      button - address of the GtkButton widget
 *      action - task action; either EVMS_Task_Expand, EVMS_Task_Expand_Container
 *               or EVMS_Task_Shrink
 *      
 *   Exit:
 *      The task context is created with the given resize point and task action.
 *      This window is hidden and another standard selection window is displayed.
 *
 */
void on_resize_point_selection_button_clicked (GtkButton *button, task_action_t action)
{
    GList          *window_list;
    GtkWidget      *clist;
    GtkWidget      *next_window;
    GtkWidget      *selection_window;
    object_handle_t resize_point;
    object_handle_t prev_resize_point;

    /*
     * Hide the selection window and either redisplay the existing task window
     * if it exists and the resize point selection did not change. If the task window
     * does not exist, create it. However, if the resize point selection changed and
     * the task window existed then destroy the current task window and create
     * a new one.
     */

    selection_window  = gtk_widget_get_toplevel (GTK_WIDGET (button));
    clist             = lookup_widget (GTK_WIDGET (button), "selection_window_clist");
    resize_point      = GPOINTER_TO_UINT (get_single_select_current_row_data (GTK_CLIST (clist)));
    window_list       = get_window_list (selection_window);
    next_window       = gtk_object_get_data (GTK_OBJECT (selection_window), "next_window_id");
    prev_resize_point = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (selection_window), "previous_resize_point"));

    if (window_list == NULL)
        window_list = g_list_append (window_list, selection_window);

    if (resize_point != prev_resize_point)
    {
        gint          rc;
        task_handle_t handle;

        rc = evms_create_task (resize_point, action, &handle);

        if (rc != SUCCESS)
        {
            log_error ("%s: evms_create_task() returned error code %d.\n", __FUNCTION__, rc);

            display_results_window (rc, NULL, _("Unable to create the task context."), 
                                    NULL, FALSE, selection_window);
        }
        else
        {
            gint   count;
            gchar *window_title;
            gchar *next_button_text=NULL;            
            
            /*
             * If we already created the next window for selecting
             * the acceptable objects (and maybe even other follow-on 
             * windows like for options), make sure we destroy them
             * all and clean up the list since our plugin selections
             * have apperently changed and so all the following panels
             * need to be recreated.
             */

            if (next_window != NULL)
                destroy_window_list (g_list_find (window_list, next_window));

            if (action == EVMS_Task_Expand)
                window_title = _("Expand Transmogrification Object Selection");
            else if (action == EVMS_Task_Expand_Container)
                window_title = _("Select Additional Storage Objects To Consume");                
            else
                window_title = _("Shrink Transmogrification Object Selection");

            /*
             * If we have no options then set the Next button label to
             * indicate the actual task action rather than "Next".
             */
            
            if (evms_get_option_count (handle, &count) == SUCCESS && count == 0)
                next_button_text = get_task_action_string (handle);
            
            next_window = create_standard_selection_window (window_title, 
                                                            next_button_text,
                                                            acceptable_objects_help_text,
                                                            on_acceptable_objects_clist_realize,
                                                            on_acceptable_objects_button_clicked,
                                                            on_button_clicked_display_prev_window,
                                                            on_button_clicked_destroy_window_list,
                                                            on_acceptable_objects_clist_select_row,
                                                            on_acceptable_objects_clist_unselect_row,
                                                            GUINT_TO_POINTER (handle));

            window_list = g_list_append (window_list, next_window);

            set_window_list (next_window, window_list);
            set_window_list (selection_window, window_list);

            gtk_object_set_data (GTK_OBJECT (selection_window), "next_window_id", next_window);
            gtk_object_set_data (GTK_OBJECT (selection_window), "previous_resize_point", GUINT_TO_POINTER (resize_point));

            /*
             * Connect the destroy signal to the next window. If the window is
             * destroyed, so is the task context.
             */

            gtk_signal_connect (GTK_OBJECT (next_window), "destroy", 
                                GTK_SIGNAL_FUNC (on_task_window_destroy), GUINT_TO_POINTER (handle));
                                
            if (next_button_text)
                g_free (next_button_text);
            
            gtk_widget_show (next_window);
            gtk_widget_hide (selection_window);            
        }
    }
    else
    {
        gtk_widget_show (next_window);
        gtk_widget_hide (selection_window);
    }
}

/*
 *
 *   void on_expand_point_selection_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine responds to expand point selection with creating the
 *      task given and creating the selectable objects selection dialog
 *      window to proceed with acceptable parameter and option selections.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - not used
 *
 *   Exit:
 *      The task context is created with the selected expand point object and task action.
 *      This window is hidden and another standard selection window is displayed.
 *
 */
void on_expand_point_selection_button_clicked (GtkButton *button, gpointer user_data)
{
    on_resize_point_selection_button_clicked (button, EVMS_Task_Expand);
}

/*
 *
 *   void on_shrink_point_selection_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine responds to shrink point selection with creating the
 *      task given and creating the selectable objects selection dialog
 *      window to proceed with acceptable parameter and option selections.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - not used
 *
 *   Exit:
 *      The task context is created with the selected shrink point object and task action.
 *      This window is hidden and another standard selection window is displayed.
 *
 */
void on_shrink_point_selection_button_clicked (GtkButton *button, gpointer user_data)
{
    on_resize_point_selection_button_clicked (button, EVMS_Task_Shrink);
}

/*
 *
 *   void on_resize_selection_button_clicked (GtkButton *, task_action_t)
 *
 *   Description:
 *      This routine responds to volume/objects selection for an EVMS_Task_Expand
 *      or EVMS_Task_Shrink action by getting the the expand/shrink points
 *      for the volume/object. Once the expand/shrink point is selected and
 *      the user continues, a task context will be started for normal
 *      task selectable objects and options processing.
 * 
 *   Entry:
 *      button - address of the GtkButton widget
 *      action - action (EVMS_Task_Expand or EVMS_Task_Shrink)
 *
 *   Exit:
 *      This window is hidden and another standard selection window is displayed.
 *
 */
void on_resize_selection_button_clicked (GtkButton *button, task_action_t action)
{
    GList          *window_list;
    GtkWidget      *clist;
    GtkWidget      *next_window;
    GtkWidget      *selection_window;
    object_handle_t object;
    object_handle_t prev_object;

    /*
     * Hide the selection window and either redisplay the existing resize
     * points selection window if it exists and the shrink/expand object 
     * selection did not change. If the resize points selection window
     * does not exist, create it. However, if the shrink/expand object 
     * selection changed and the expand/shrink points window existed
     * then destroy the current expand/shrink points window and create
     * a new one.
     */

    selection_window = gtk_widget_get_toplevel (GTK_WIDGET (button));
    clist            = lookup_widget (GTK_WIDGET (button), "selection_window_clist");
    object           = GPOINTER_TO_UINT (get_single_select_current_row_data (GTK_CLIST (clist)));
    window_list      = get_window_list (selection_window);
    next_window      = gtk_object_get_data (GTK_OBJECT (selection_window), "next_window_id");
    prev_object      = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (selection_window), 
                                         "previous_object"));

    if (window_list == NULL)
        window_list = g_list_append (window_list, selection_window);

    if (object != prev_object)
    {
        /*
         * Check to see if we can skip past the resize points. This
         * would be the case when there is only one resize point and
         * its object handle matches that of our selection then we 
         * can skip the resize point panel and start the task.
         */

        if (resize_point_is_selected_object (object, action))
        {
            on_resize_point_selection_button_clicked (button, action);
        }
        else
        {    
            gchar         *title_text;
            gchar         *resize_points_help_text;
            GtkSignalFunc  resize_points_clist_realize_cb;
            GtkSignalFunc  resize_points_button_clicked_cb;
                    
            /*
             * If we already created the next window for selecting
             * the shrink/expand points (and maybe even other follow-on 
             * windows like for options), make sure we destroy them
             * all and clean up the list since our object selections
             * have apperently changed and so all the following panels
             * need to be recreated.
             */
    
            if (next_window != NULL)
                destroy_window_list (g_list_find (window_list, next_window));
            
            if (action == EVMS_Task_Expand)
            {
                log_debug ("%s: Object to expand is %#x.\n", __FUNCTION__, object);
                resize_points_help_text = expansion_points_help_text;
                resize_points_clist_realize_cb  = on_expand_points_clist_realize;
                resize_points_button_clicked_cb = on_expand_point_selection_button_clicked;
            }
            else
            {
                log_debug ("%s: Object to shrink is %#x.\n", __FUNCTION__, object);            
                resize_points_help_text = shrink_points_help_text;
                resize_points_clist_realize_cb  = on_shrink_points_clist_realize;
                resize_points_button_clicked_cb = on_shrink_point_selection_button_clicked;
            }
            
            title_text = get_resize_points_window_title_text (object, action);
            
            next_window = create_standard_selection_window (title_text, NULL,
                                                            resize_points_help_text,
                                                            resize_points_clist_realize_cb,
                                                            resize_points_button_clicked_cb,
                                                            on_button_clicked_display_prev_window,
                                                            on_button_clicked_destroy_window_list,
                                                            NULL, NULL,
                                                            GUINT_TO_POINTER (object));
    
            window_list = g_list_append (window_list, next_window);
    
            set_window_list (next_window, window_list);
            set_window_list (selection_window, window_list);
    
            gtk_object_set_data (GTK_OBJECT (selection_window), "next_window_id" , next_window);
            gtk_object_set_data (GTK_OBJECT (selection_window), "previous_object", GUINT_TO_POINTER (object));
            
            g_free (title_text);

            gtk_widget_show (next_window);
            gtk_widget_hide (selection_window);
        }
    }
    else
    {    
        gtk_widget_show (next_window);
        gtk_widget_hide (selection_window);
    }
}

/*
 *
 *   inline void show_volume_minmax_size (GtkCList *, gint, object_handle_t, gboolean)
 *   
 *   Description:
 *      This routine displays either the volume minimum or maximum
 *      size in human readable form in the selection window clist 
 *      minmax column at the row specified.
 * 
 *   Entry:
 *      clist    - address of the selections GtkCList widget
 *      row      - the row in the list to update
 *      handle   - the volume's handle
 *      show_max - TRUE if show volume maximum otherwise show minimum size
 *
 *   Exit:
 *      Selection list populated with objects that say they can
 *      expand.
 *
 */
inline void show_volume_minmax_size (GtkCList *clist, gint row, object_handle_t handle, gboolean show_max)
{
    handle_object_info_t *object;
    
    if (evms_get_info (handle, &object) == SUCCESS)
    {
        gchar *text;
        
        if (show_max)
            text = make_sectors_readable_string (object->info.volume.max_fs_size);
        else
            text = make_sectors_readable_string (object->info.volume.min_fs_size);
        
        gtk_clist_set_text (clist, row, SL_MINMAX_SIZE_COLUMN, text);
        
        g_free (text);
        evms_free (object);
    }    
}

/*
 *
 *   void add_expandable_objects_to_selection_clist (GtkCList *, handle_array_t objects, gboolean)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of objects that evms_can_expand () says are expandable.
 * 
 *   Entry:
 *      clist               - address of the selections GtkCList widget
 *      objects             - address if array of object handles (typically
 *                            volumes or topmost objects)
 *      objects_are_volumes - TRUE is objects are supposedly volumes so
 *                            can attempt to show max volume size
 *
 *   Exit:
 *      Selection list populated with objects that say they can
 *      expand.
 *
 */
void add_expandable_objects_to_selection_clist (GtkCList *clist, handle_array_t *objects, 
                                                gboolean objects_are_volumes)
{
    gint     i;
    gint     row;
    gboolean is_selected = (objects->count == 1);
    
    for (i=0; i < objects->count; i++)
    {
        if (evms_can_expand (objects->handle[i]) == 0)
        {            
            row = add_thing_to_selection_list (clist, objects->handle[i], is_selected);
            
            if (row != -1 && objects_are_volumes)
                show_volume_minmax_size (clist, row, objects->handle[i], TRUE);
        }
    }
    
    if (clist->rows == 1)
        gtk_clist_select_row (clist, 0, 0);
    else
        gtk_clist_unselect_all (clist);    
}

/*
 *
 *   void add_shrinkable_objects_to_selection_clist (GtkCList *, handle_array_t objects, gboolean)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of objects that evms_can_shrink () says are shrinkable.
 * 
 *   Entry:
 *      clist               - address of the selections GtkCList widget
 *      objects             - address if array of object handles (typically
 *                            volumes or topmost objects)
 *      objects_are_volumes - TRUE is objects are supposedly volumes so
 *                            can attempt to show min volume size
 *
 *   Exit:
 *      Selection list populated with objects that say they can
 *      shrink.
 *
 */
void add_shrinkable_objects_to_selection_clist (GtkCList *clist, handle_array_t *objects,
                                                gboolean objects_are_volumes)
{
    gint     i;
    gint     row;
    gboolean is_selected = (objects->count == 1);

    for (i=0; i < objects->count; i++)
    {
        if (evms_can_shrink (objects->handle[i]) == 0)
        {    
            row = add_thing_to_selection_list (clist, objects->handle[i], is_selected);
            
            if (row != -1 && objects_are_volumes)
                show_volume_minmax_size (clist, row, objects->handle[i], FALSE);                
        }
    }
    
    if (clist->rows == 1)
        gtk_clist_select_row (clist, 0, 0);
    else
        gtk_clist_unselect_all (clist);    
}

/*
 *
 *   void on_expand_thing_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine calls the function that creates the expand
 *      points panel.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - not used
 *
 *   Exit:
 *      The expand points window or some other selection window is displayed
 *      and this window is hidden.
 *
 */
void on_expand_thing_button_clicked (GtkButton *button, gpointer user_data)
{
    on_resize_selection_button_clicked (button, EVMS_Task_Expand);
}

/*
 *
 *   void on_shrink_thing_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine calls the function that creates the shrink
 *      points panel.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - not used
 *
 *   Exit:
 *      The shrink points window or some other selection window is displayed
 *      and this window is hidden.
 *
 */
void on_shrink_thing_button_clicked (GtkButton *button, gpointer user_data)
{
    on_resize_selection_button_clicked (button, EVMS_Task_Shrink);
}

/*
 *
 *   void add_volume_as_selected_resize_item (GtkCList *, gpointer, gboolean)
 *   
 *   Description:
 *      This routine populates the standard selection window GtkCList
 *      with the information from one volume handle. It also selects
 *      this single entry in the list so that the "Next" button is
 *      made active to complete an operation on the row.
 * 
 *   Entry:
 *      widget    - address of the selection window GtkCList widget
 *      user_data - contains handle of volume to place in selection window
 *      is_expand - TRUE if this is an EVMS_Task_Expand
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void add_volume_as_selected_resize_item (GtkCList *clist, gpointer user_data, gboolean is_expand)
{
    gint            row;
    object_handle_t handle = GPOINTER_TO_UINT (user_data);
    
    gtk_clist_set_column_visibility (clist, SL_MINMAX_SIZE_COLUMN, TRUE);
    gtk_clist_set_column_auto_resize (clist, SL_MINMAX_SIZE_COLUMN, TRUE);
    
    set_selection_window_clist_column_titles (clist, _("Size"), 
                                              make_object_type_readable_string (VOLUME),
                                              is_expand ?  _("Maximum Size") : _("Minimum Size"));
    
    row = add_thing_to_selection_list (clist, handle, TRUE);
    
    if (row != -1)
        show_volume_minmax_size (clist, row, handle, is_expand);
}

/*
 *
 *   void add_volume_as_selected_expansion_item (GtkCList *, gpointer)
 *   
 *   Description:
 *      This routine populates the standard selection window GtkCList
 *      with the information from one volume handle. It also selects
 *      this single entry in the list so that the "Next" button is
 *      made active to complete an operation on the row.
 * 
 *   Entry:
 *      widget    - address of the selection window GtkCList widget
 *      user_data - contains handle of volume to place in selection window
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void add_volume_as_selected_expansion_item (GtkCList *clist, gpointer user_data)
{
    add_volume_as_selected_resize_item (clist, user_data, TRUE);
}

/*
 *
 *   void add_volume_as_selected_shrink_item (GtkCList *, gpointer)
 *   
 *   Description:
 *      This routine populates the standard selection window GtkCList
 *      with the information from one volume handle. It also selects
 *      this single entry in the list so that the "Next" button is
 *      made active to complete an operation on the row.
 * 
 *   Entry:
 *      widget    - address of the selection window GtkCList widget
 *      user_data - contains handle of volume to place in selection window
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void add_volume_as_selected_shrink_item (GtkCList *clist, gpointer user_data)
{
    add_volume_as_selected_resize_item (clist, user_data, FALSE);
}

/*
 *
 *   void on_expand_thing_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the thing we intend to expand and
 *      allow the user to proceed through the expansion process.
 * 
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_expand_thing_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    object_type_t type;

    if (evms_get_handle_object_type (GPOINTER_TO_UINT (user_data), &type) == SUCCESS)
    {
        gchar     *window_title;
        gchar     *help_text;
        GtkWidget *window;

        switch (type)
        {
            case VOLUME:
                window_title = _("Expand Logical Volume");
                help_text = expand_volume_help_text;
                break;

            case EVMS_OBJECT:
                window_title = _("Expand Feature Object");
                help_text = expand_object_help_text;
                break;
            
            case REGION:
                window_title = _("Expand Storage Region");
                help_text = expand_region_help_text;
                break;
            
            case SEGMENT:
                window_title = _("Expand Disk Segment");
                help_text = expand_segment_help_text;
                break;

            case CONTAINER:
                window_title = _("Expand Storage Container");
                help_text = expand_container_help_text;
                break;

            default:
                window_title = _("Expand");
                help_text = NULL;
                break;
        }

        window = create_standard_selection_window (window_title,
                                                   NULL,
                                                   help_text,
                                                   type == VOLUME ? add_volume_as_selected_expansion_item :
                                                                    add_thing_as_selected_list_item,
                                                   on_expand_thing_button_clicked,
                                                   NULL, NULL, NULL, NULL, user_data);

        gtk_widget_show (window);
    }
}

/*
 *
 *   void on_shrink_thing_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the thing we intend to shrink and
 *      allow the user to proceed through the shrink process.
 * 
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_shrink_thing_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    object_type_t type;

    if (evms_get_handle_object_type (GPOINTER_TO_UINT (user_data), &type) == SUCCESS)
    {
        gchar     *window_title;
        gchar     *help_text;
        GtkWidget *window;

        switch (type)
        {
            case VOLUME:
                window_title = _("Shrink Logical Volume");
                help_text = shrink_volume_help_text;
                break;

            case EVMS_OBJECT:
                window_title = _("Shrink Feature Object");
                help_text = shrink_object_help_text;
                break;
            
            case REGION:
                window_title = _("Shrink Storage Region");
                help_text = shrink_region_help_text;
                break;
            
            case SEGMENT:
                window_title = _("Shrink Disk Segment");
                help_text = shrink_segment_help_text;
                break;

            case CONTAINER:
                window_title = _("Shrink Storage Container");
                help_text = shrink_container_help_text;
                break;

            default:
                window_title = _("Shrink");
                help_text = NULL;
                break;
        }

        window = create_standard_selection_window (window_title,
                                                   NULL,
                                                   help_text,
                                                   type == VOLUME ? add_volume_as_selected_shrink_item :
                                                                    add_thing_as_selected_list_item,        
                                                   on_shrink_thing_button_clicked,
                                                   NULL, NULL, NULL, NULL, user_data);

        gtk_widget_show (window);
    }
}
