/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils/helpers
 * File: translate_name.c
 */

#ifndef _LVM_HELP_TRANSLATE_NAME_
#define _LVM_HELP_TRANSLATE_NAME_ 1

#define DEV_DIRECTORY		"/dev/"
#define EVMS_DEV_DIRECTORY	"evms/"
#define LVM_DEV_DIRECTORY	"lvm/"


/* Function: translate_dev_name
 *
 *	These utilities will allow a user to specify PVs and LVs using the
 *	"/dev/name" or "/dev/evms/name". However, we need to translate these
 *	to get rid of the "/dev/" and "/dev/evms" in order to properly
 *	match the object names in the engine.
 */
char * translate_dev_name( char * input )
{
	char * ptr = input;

	// Does ptr start with "/dev/"?
	if ( strstr(ptr, DEV_DIRECTORY) == ptr ) {
		ptr = &(ptr[strlen(DEV_DIRECTORY)]);
	}

	// Does ptr start with "evms/"?
	if ( strstr(ptr, EVMS_DEV_DIRECTORY) == ptr ) {
		ptr = &(ptr[strlen(EVMS_DEV_DIRECTORY)]);
	}

	return ptr;
}


/* Function: translate_vg_name_to_container_name
 *
 *	Since these utilities only deal with LVM containers, we can allow the
 *	user to only specify the LVM name. If the given name does not have the
 *	LVM directory prefix, add it.
 */
int translate_vg_name_to_container_name( char * vg_name, char * cont_name )
{
	char * ptr = vg_name;

	memset(cont_name, 0, EVMS_VOLUME_NAME_SIZE);

	// Strip off "/dev/" or "/dev/evms/"
	ptr = translate_dev_name(vg_name);

	if ( strstr(ptr, LVM_DEV_DIRECTORY) != ptr ) {
		// If the vg_name doesn't start with "lvm/", add it.
		strcpy(cont_name, LVM_DEV_DIRECTORY);
		strncat(cont_name, ptr, EVMS_VOLUME_NAME_SIZE-strlen(cont_name));
	}
	else {
		// Otherwise, just copy vg_name.
		strncpy(cont_name, ptr, EVMS_VOLUME_NAME_SIZE);
	}

	// Remove trailing slashes.
	if ( cont_name[strlen(cont_name)] == '/' ) {
		cont_name[strlen(cont_name)] = 0;
	}

	return 0;
}


/* Function: translate_lv_name_to_region_name
 *
 *	Since these utilities only deal with LVM volumes, we can allow the
 *	user to only specify the LVM name. If the given name does not have the
 *	LVM directory prefix, add it.
 */
int translate_lv_name_to_region_name( char * lv_name, char * region_name )
{
	char * ptr;

	memset(region_name, 0, EVMS_VOLUME_NAME_SIZE);

	// Strip off "/dev/" or "/dev/evms/"
	ptr = translate_dev_name(lv_name);

	if ( strstr(ptr, LVM_DEV_DIRECTORY) != ptr ) {
		// If the lv_name doesn't start with "lvm/", add it.
		strcpy(region_name, LVM_DEV_DIRECTORY);
		strncat(region_name, ptr, EVMS_VOLUME_NAME_SIZE-strlen(region_name));
	}
	else {
		// Otherwise, just copy lv_name.
		strncpy(region_name, ptr, EVMS_VOLUME_NAME_SIZE);
	}

	return 0;
}

#endif

