/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_lvextend.c
 *
 *	Emulates LVM's 'lvextend' utility using the EVMS Engine. All options
 *	and several status messages are based on the original lvextend command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <ctype.h>
#include <frontend.h>

typedef struct cmd_options_s {
	u_int32_t	add_extents;
	u_int32_t	add_size;
	int		debug;
	int		help;
	u_int32_t	total_extents;
	u_int32_t	total_size;
	int		verbose;
	int		version;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

// Must be included after above definitions.
#include "helpers/get_freespace.c"
#include "helpers/get_number.c"
#include "helpers/get_region_handle.c"
#include "helpers/open_engine.c"
#include "helpers/remove_duplicates.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	printf("%s -- Logical Volume Extend\n\n",cmd);
	return 0;
}


static int showhelp( void )
{
	showheader();
	printf("Synopsis:\n");
	printf("---------\n\n");
	printf("%s\n", cmd);
	printf("\t[-A/--autobackup y/n]\n");
	printf("\t[-d/--debug]\n");
	printf("\t[-h/-?/--help]\n");
	printf("\t{-l/--extents [+]LogicalExtentsNumber |\n");
	printf("\t -L/--size [+]LogicalVolumeSize[kKmMgGtT]}\n");
	printf("\t[-v/--verbose]\n");
	printf("\tLogicalVolume[Path] [ PhysicalVolumePath... ]\n\n");
	return 0;
}


static int lvextend_end(option_array_t		* a,
			handle_object_info_t	* b,
			handle_object_info_t	* c,
			extended_info_array_t	* d )
{
	if (a) free(a);
	if (b) evms_free(b);
	if (c) evms_free(c);
	if (d) evms_free(d);
	evms_close_engine();
	return 0;
}


static int parse_options( int argc, char ** argv )
{
	int		c;
	char		* short_opts = "A:dh?l:L:vV:";
	char		* leftover = NULL;
	struct option	long_opts[] = {
				{ "autobackup",	required_argument, NULL, 'A'},
				{ "debug",	no_argument,       NULL, 'd'},
				{ "help",	no_argument,       NULL, 'h'},
				{ "extents",	required_argument, NULL, 'l'},
				{ "size",	required_argument, NULL, 'L'},
				{ "verbose",	no_argument,       NULL, 'v'},
				{ "version",	no_argument,       NULL, 'V'},
				{ NULL, 0, NULL, 0} };

	while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'A':
			// -A is ignored by EVMS.
			if ( optarg[0] != 'n' && optarg[0] != 'y' ) {
				printf( "%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
				return EINVAL;
			}
			break;
		case 'd':
			opts.debug = TRUE;
			opts.verbose = TRUE;
			break;
		case 'h':
		case '?':
			opts.help = TRUE;
			break;
		case 'l':
			// Only specify -l or -L
			if ( opts.add_size || opts.total_size ) {
				printf("%s -- -L option already given\n\n", cmd);
				return EINVAL;
			}
			if ( opts.add_extents || opts.total_extents ) {
				printf("%s -- -l option already given\n\n", cmd);
				return EINVAL;
			}
			if ( optarg[0] == '+' ) {
				opts.add_extents = strtoul(&(optarg[1]), &leftover, 0);
				if ( opts.add_extents <= 0 ) {
					printf("%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
					return EINVAL;
				}
			}
			else {
				opts.total_extents = strtoul(optarg, &leftover, 0);
				if ( opts.total_extents <= 0 ) {
					printf("%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
					return EINVAL;
				}
			}
			break;
		case 'L':
			// Only specify -l or -L
			if ( opts.add_extents || opts.total_extents ) {
				printf("%s -- -l option already given\n\n", cmd);
				return EINVAL;
			}
			if ( opts.add_size || opts.total_size ) {
				printf("%s -- -L option already given\n\n", cmd);
				return EINVAL;
			}
			if ( optarg[0] == '+' ) {
				opts.add_size = get_number(&(optarg[1]));
				if ( opts.add_size <= 0 ) {
					printf("%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
					return EINVAL;
				}
				// Convert kB to sectors
				opts.add_size = (opts.add_size * 1024) / EVMS_VSECTOR_SIZE;
			}
			else {
				opts.total_size = get_number(optarg);
				if ( opts.total_size <= 0 ) {
					printf("%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
					return EINVAL;
				}
				// Convert kB to sectors
				opts.total_size = (opts.total_size * 1024) / EVMS_VSECTOR_SIZE;
			}
			break;
		case 'v':
			opts.verbose = TRUE;
			break;
		case 'V':
			opts.version = TRUE;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	// Must specify a size
	if ( ! opts.add_extents && ! opts.add_size &&
	     ! opts.total_extents && ! opts.total_size ) {
		printf("%s -- please enter either option -l or -L\n\n", cmd);
		return EINVAL;
	}

	return 0;
}


int set_lvextend_options(	option_array_t	* creation_options,
				cmd_options_t	* opts,
				char		** pv_names )
{
	int i = 0;
	int j;

	// Set the appropriate size option.
	if ( opts->add_extents ) {
		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 0;	// Expand Extents
		creation_options->option[i].value.ui32		= opts->add_extents;
		if ( opts->verbose ) {
			printf("%s -- Setting Expand Extents option: %d\n", cmd, opts->add_extents);
		}
		i++;
	}
	else {
		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 1;	// Expand Size
		creation_options->option[i].value.ui32		= opts->add_size;
		if ( opts->verbose ) {
			printf("%s -- Setting Expand Size option: %d sectors\n", cmd, opts->add_size);
		}
		i++;
	}

	if ( pv_names[0] ) {
		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 2;
		creation_options->option[i].value.list = malloc(2500);
		for ( j = 0; pv_names[j]; j++ ) {
			creation_options->option[i].value.list->count++;
			creation_options->option[i].value.list->value[j].s = malloc(strlen(pv_names[j])+1);
			pv_names[j] = translate_dev_name(pv_names[j]);
			strcpy(creation_options->option[i].value.list->value[j].s, pv_names[j]);
			if ( opts->verbose ) {
				printf("%s -- Setting PV Name %d option: %s\n", cmd, j, pv_names[j]);
			}
		}
		i++;
	}

	return 0;
}


int main( int argc, char * argv[] )
{
	option_array_t		* extend_options	= NULL;
	object_handle_t		container		= 0;
	object_handle_t		region			= 0;
	handle_object_info_t	* region_info		= NULL;
	handle_object_info_t	* volume_info		= NULL;
	extended_info_array_t	* region_ext_info	= NULL;
	handle_array_t		freespace_region;
	char			* lv_name		= NULL;
	char			* pv_names[256]		= {NULL};
	int			number_of_pvs		= 0;
	int			log_level		= DEFAULT;
	int			rc;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}

	// Check for target volume name.
	if ( optind == argc ) {
		printf("%s -- please enter a logical volume path.\n", cmd);
		return EINVAL;
	}
	lv_name = argv[optind];
	if ( opts.verbose ) {
		printf("%s -- Extending volume %s\n", cmd, lv_name);
	}
	optind++;

	// Check for PV names
	if ( optind != argc) {
		number_of_pvs = argc - optind;
	
		// Copy the PV names from the command line, detecting any duplicates.
		rc = remove_duplicates( &argv[optind], pv_names, number_of_pvs);
		if (rc) {
			printf("%s -- Duplicate PVs specified. Please only specify each PV once\n", cmd);
			return rc;
		}
	}

	// Open the EVMS Engine.
	rc = open_engine(ENGINE_READWRITE, log_level);
	if (rc) {
		return rc;
	}

	// Get the handle for the target region.
	rc = get_region_handle(lv_name, &region);
	if (rc) {
		lvextend_end(extend_options, region_info, volume_info, region_ext_info);
		return rc;
	}

	// Get the basic info for the target region.
	rc = evms_get_info(region, &region_info);
	if (rc) {
		printf("%s -- Error getting info for region %s (%d)\n", cmd, lv_name, rc);
		lvextend_end(extend_options, region_info, volume_info, region_ext_info);
		return rc;
	}

	// Make sure this region is a compatibility volume. If it is not, then
	// there is another object on top of this region. Must use the GUI to
	// do the expansion in that case.
	rc = evms_get_info(region_info->info.region.volume, &volume_info);
	if (rc) {
		printf("%s -- Error getting volume info for region %s (%d)\n", cmd, lv_name, rc);
		lvextend_end(extend_options, region_info, volume_info, region_ext_info);
		return rc;
	}
	if ( volume_info->info.volume.object != region ) {
		printf("%s -- Region %s is not a top-level object. Cannot expand.\n", cmd, lv_name);
		printf("%s -- Use evmsgui to expand this region.\n", cmd);
		lvextend_end(extend_options, region_info, volume_info, region_ext_info);
		return EINVAL;
	}

	// Get the handle for the target freespace region
	container = region_info->info.region.producing_container;
	rc = get_freespace_for_container(container, &(freespace_region.handle[0]));
	if (rc) {
		lvextend_end(extend_options, region_info, volume_info, region_ext_info);
		return rc;
	}
	freespace_region.count = 1;

	// Convert total_extents to add_extents, or total_size to add_size.
	if ( opts.total_extents ) {
		rc = evms_get_extended_info(region, NULL, &region_ext_info);
		if (rc) {
			lvextend_end(extend_options, region_info, volume_info, region_ext_info);
			return rc;
		}

		// The 5th element is current number of LEs. See lvm_info.c.
		// That's right, this is a hack. I don't want to deal with
		// both inside the plugin. :)
		opts.add_extents = opts.total_extents - region_ext_info->info[4].value.ui32;
		if ( opts.add_extents < 0 ) {
			printf("%s -- New extents specified is less than existing extents.\n", cmd);
			printf("%s -- New: %d extents. Existing: %d extents.\n", cmd, opts.total_extents, region_ext_info->info[4].value.ui32);
			lvextend_end(extend_options, region_info, volume_info, region_ext_info);
			return EINVAL;
		}
	}
	else if ( opts.total_size ) {
		opts.add_size = opts.total_size - region_info->info.region.size;
		if ( opts.add_size < 0 ) {
			printf("%s -- New size specified is less than existing size.\n", cmd);
			printf("%s -- New: %d sectors. Existing: %lld sectors.\n", cmd, opts.total_size, region_info->info.region.size);
			lvextend_end(extend_options, region_info, volume_info, region_ext_info);
			return EINVAL;
		}
	}

	// Memory for the segment and option arrays.
	extend_options = (option_array_t*)malloc(1000);
	if ( ! extend_options ) {
		printf("%s -- Memory allocation error.\n", cmd);
		lvextend_end(extend_options, region_info, volume_info, region_ext_info);
		return ENOMEM;
	}
	memset(extend_options, 0, 1000);

	// Set up the option array
	set_lvextend_options(extend_options, &opts, pv_names);

	// Extend the volume.
	rc = evms_expand(region,
			&freespace_region,
			extend_options);
	if (rc) {
		printf("%s -- Error extending region %s (%d)\n", cmd, lv_name, rc);
		lvextend_end(extend_options, region_info, volume_info, region_ext_info);
		return rc;
	}
	if ( opts.verbose ) {
		printf("%s -- Region extended\n", cmd);
	}


	// Write everything to disk.
	rc = evms_commit_changes(NULL);
	if (rc) {
		printf("%s -- Error committing changes to disk (%d)\n", cmd, rc);
		lvextend_end(extend_options, region_info, volume_info, region_ext_info);
		return rc;
	}
	if ( opts.verbose ) {
		printf("%s -- Extended region committed to disk\n", cmd);
	}


	printf("%s -- logical volume \"%s\" successfully extended\n\n", cmd, lv_name);
	lvextend_end(extend_options, region_info, volume_info, region_ext_info);
	return 0;
}

