/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_discover.c
 *
 * Description: This file contains all functions related to the initial
 *              discovery of MD physical volumes, volume groups, and logical
 *              volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <time.h>
#include <sys/ioctl.h>
#define MY_PLUGIN my_plugin
#include "md.h"


// function to check if the uuids of 2 super block are equal,
// returns 1 if they are equal, 0 otherwise.
int md_uuids_equal(mdp_super_t * sb1, mdp_super_t * sb2)
{
	LOG_ENTRY;
	if (sb1->set_uuid0 == sb2->set_uuid0 &&
	    sb1->set_uuid1 == sb2->set_uuid1 &&
	    sb1->set_uuid2 == sb2->set_uuid2 &&
	    sb1->set_uuid3 == sb2->set_uuid3 ) {
		RETURN(1);
	} else {
		RETURN(0);
	}
}


int  md_read_sb_from_disk(storage_object_t *object, mdp_super_t * buffer) {
	int rc = 0;
	u_int64_t location;
	LOG_ENTRY;


	if (object->data_type != DATA_TYPE) {
		LOG_DETAILS("Object not data type, skipping %s\n", object->name);
		RETURN(rc);
	}

	if (object->size > MD_RESERVED_SECTORS) {
		location = MD_NEW_SIZE_SECTORS(object->size);
	} else {
		LOG_DETAILS("Object too small for MD, skipping %s\n", object->name);
		RETURN(rc);

	}
	LOG_DEBUG("Looking for MD Superblock at %lld on %s\n",location, object->name);

	if ( READ(object, location, MD_SB_SECTORS, (char*)buffer )) {
		LOG_SERIOUS("Error reading MD SUperBlock from object %s\n", object->name);
		RETURN(EIO);
	}

	RETURN (rc);

}


/* write all of the super block for an MD region to disk */
int  md_write_sbs_to_disk(md_volume_t * volume) {
	int rc = 0;
	u_int64_t location;
	int i,j;
	int nr_disks;
	mdp_super_t *super;
	time_t utime;
	LOG_ENTRY;

	// update utime
	utime = time(NULL);
	volume->super_block->utime = utime;
	volume->super_block->events_lo++;
	if (volume->super_block->events_lo == 0) {
		volume->super_block->events_hi++;
	}

	for (i = 0, nr_disks = 0; (i < MAX_MD_DEVICES) && (nr_disks < volume->nr_disks); i++) {
		if (volume->super_array[i] != NULL) {
			// copy superblock form master, and fix up this_disk
			memcpy(volume->super_array[i], volume->super_block, MD_SB_BYTES);

			if (volume->super_array[i]->disks[i].state & (1<<MD_DISK_NEW)) {
				volume->super_array[i]->events_lo = 0; // null out event counter
				volume->super_array[i]->events_hi = 0; // as signal to kernel on new devices
			}
			// turn off new bit in master SB
			for (j = 0; j < MAX_MD_DEVICES; j++) {
				volume->super_array[i]->disks[j].state &= ~(1<<MD_DISK_NEW);
			}
			volume->super_block->disks[i].state &= ~(1<<MD_DISK_NEW);

			volume->super_array[i]->this_disk = volume->super_array[i]->disks[i];

			// see if we need to read the current superblock state from disk.
			if (volume->commit_flag & MD_COMMIT_USE_DISK) {
				LOG_DETAILS("reading state info for disk %d of region %s\n",i,volume->name);
				rc = md_check_for_pv(volume->child_object[i], &super);
				if (!rc) {
					if (md_uuids_equal(super, volume->super_block)) {
						volume->super_array[i]->state = super->state;
					}
					md_deallocate_memory(super);
				} else {
					LOG_ERROR("Error reading state info for disk %d of region %s\n",i,volume->name);
				}
			}
			location = MD_NEW_SIZE_SECTORS(volume->child_object[i]->size);
			LOG_DEBUG("Writing MD Superblock at %lld on %s\n",location, volume->child_object[i]->name);


			volume->super_array[i]->sb_csum = 0;
			rc = EngFncs->calculate_checksum((char *)volume->super_array[i],
							 MD_SB_BYTES,
							 0,
							 &volume->super_array[i]->sb_csum);

			if ( WRITE(volume->child_object[i], location, MD_SB_SECTORS, (char*)volume->super_array[i] )) {
				LOG_SERIOUS("Error writing MD SUperBlock from object %s\n", volume->child_object[i]->name);
				RETURN(EIO);
			}

			nr_disks++;
		}
	}
	RETURN (rc);

}

// the input object has been validated as a pv for this personality.
// using the list head for this personality, search to see if other PVs
// for this volume already exist, if so add this supreblock to the list for that volume
// else create a new volume structure for it and add it to the volume list

int md_find_volume_for_object(storage_object_t * object, mdp_super_t *md_super_buffer) {

	int rc = 0;
	int i;
	md_volume_t * volume = volume_list_head;
	mdp_super_t * tmp_super = NULL;
	storage_object_t * tmp_object = NULL;
	evms_md_array_info_t  md_info;

	LOG_ENTRY;
	while (volume) {
		if (md_uuids_equal(volume->super_block, md_super_buffer)) {
			if (md_super_buffer->utime > volume->super_block->utime) {
				memcpy(volume->super_block,
				       md_super_buffer,
				       MD_SB_BYTES);
			}
			// check to make sure slot is empty
			if (!volume->super_array[md_super_buffer->this_disk.number]) {
				volume->super_array[md_super_buffer->this_disk.number] = md_super_buffer;
				volume->child_object[md_super_buffer->this_disk.number] = object;
			} else {
				if (md_super_buffer->utime > volume->super_array[md_super_buffer->this_disk.number]->utime) {
					// new buffer is newer, use it.
					// save off the old super and object
					volume->flags |= MD_DIRTY;
					tmp_super = volume->super_array[md_super_buffer->this_disk.number];
					tmp_object = volume->child_object[md_super_buffer->this_disk.number];
					// fill in the new info
					volume->super_array[md_super_buffer->this_disk.number] = md_super_buffer;
					volume->child_object[md_super_buffer->this_disk.number] = object;
				} else if (md_super_buffer->utime < volume->super_array[md_super_buffer->this_disk.number]->utime) {
					//existing buffer the correct one, reassign this one into tmp
					volume->flags |= MD_DIRTY;
					tmp_super = md_super_buffer;
					tmp_object = object;
				} else {
					// time stamps are equal, bad news.
					MESSAGE("Multiple superblocks found for region %s index %d\n",volume->name,md_super_buffer->this_disk.number);
					MESSAGE("Object 1 %s, Object 2 %s\n",object->name,volume->child_object[md_super_buffer->this_disk.number]->name);
					volume->flags |= MD_CORRUPT;
					// don't want to write this out on commit.
					tmp_super = md_super_buffer;
					tmp_object = object;
				}
				// stick in next available slot, figure out later
				for (i = 0; i< MAX_MD_DEVICES; i++) {
					if (volume->super_array[i] == NULL) {
						volume->super_array[i] = tmp_super;
						volume->child_object[i] = tmp_object;
						// fix up the this_disk field
						memcpy(&tmp_super->this_disk, &tmp_super->disks[i], sizeof(tmp_super->this_disk));
						break;
					}
				}
			}

			volume->nr_disks++;
			break;
		}
		volume = volume->next;

	}
	if (!volume) {
		// add new entry
		if ( md_allocate_memory((void**)&volume, sizeof(md_volume_t) ) ) {
			LOG_CRITICAL("Memory error creating buffer to read super block.\n");
			RETURN(ENOMEM);
		}

		volume->super_array[md_super_buffer->this_disk.number] = md_super_buffer;
		volume->child_object[md_super_buffer->this_disk.number] = object;
		volume->personality = level_to_pers(md_super_buffer->level);
		volume->nr_disks = 1;
		volume->removed_disks = CreateList();
		volume->added_disks = CreateList();
		volume->activated_disks = CreateList();
		volume->deactivated_disks = CreateList();
		volume->commit_flag = MD_COMMIT_USE_DISK; // use on disk state for 'discovered' volumes
		md_allocate_memory((void **)&volume->super_block, MD_SB_BYTES);
		memcpy(volume->super_block,
		       md_super_buffer,
		       MD_SB_BYTES);
		md_add_volume_to_list(volume);
		sprintf(volume->name, "md/md%d",md_super_buffer->md_minor);
		rc = md_get_kernel_info(volume, &md_info);
		if (!rc) {
			if (md_uuids_equal(md_info.sb, volume->super_block)) { // sanity check same volume
				volume->flags |= MD_ACTIVE;
				md_deallocate_memory(md_info.sb);
			}
		}
	}
	RETURN (0);
}
/* Function: md_check_for_pv
 *
 *	This function is used during discovery as an argument to PruneList.
 *	When this function is called, one object is passed in. The first
 *	sector of that object is examined to determine if it is an MD PV.
 *      If it is a PV, then a 0 return code is returned along with the super_block
 *      buffer allocated.
 */
int  md_check_for_pv(storage_object_t * object,mdp_super_t ** md_super_buffer) {
	int                     rc;
	int old_csum, new_csum;

	LOG_ENTRY;


	if ( md_allocate_memory((void**)md_super_buffer, MD_SB_BYTES)) {
		LOG_CRITICAL("Memory error creating buffer to read super block.\n");
		RETURN(ENOMEM);
	}

	// Read the first block and look for a PV signature.
	if ( md_read_sb_from_disk(object, *md_super_buffer) ) {
		LOG_SERIOUS("I/O error on object %s.\n", object->name);
		md_deallocate_memory(*md_super_buffer);
		RETURN(EIO);
	}
	if ( ! ( (*md_super_buffer)->md_magic == MD_SB_MAGIC &&
		 (*md_super_buffer)->major_version == 0 &&
		 (*md_super_buffer)->minor_version == 90 ) ) {
		LOG_EXTRA("Object %s is not an MD PV - bad signature or version\n", object->name);
		md_deallocate_memory(*md_super_buffer);
		RETURN(ENXIO);
	}

	old_csum = (*md_super_buffer)->sb_csum;
	(*md_super_buffer)->sb_csum = 0;
	rc = EngFncs->calculate_checksum((char *)(*md_super_buffer),
					 MD_SB_BYTES,
					 0,
					 &new_csum);

	if (!rc) {
		if (new_csum != old_csum) {
			LOG("Object %s is not an MD PV - bad Checksum\n", object->name);
			md_deallocate_memory(*md_super_buffer);
			RETURN(ENXIO);
		}
	}


	RETURN(0);
}


/* Function: md_discover_volume_groups
 *
 *	This function is the entry point into the first half of discovery,
 *	which examines all objects to find PVs and assigns them to the
 *	appropriate groups.
 */
int md_discover_volumes( dlist_t input_list, dlist_t output_list) {
	void * waste;
	storage_object_t * object;
	int tag,size;
	int rc = 0;
	mdp_super_t             * md_super_buffer;

	LOG_ENTRY;
	LOG_DETAILS("Searching for MD Super Blocks.\n");

	// A buffer for md_check_object_for_pv to use for reading
	// the PV metadata.

	while (!(rc = BlindExtractObject(input_list, &size, (TAG *)&tag, NULL, (void *)&object))) {
		if (object->data_type == DATA_TYPE) {
			rc = md_check_for_pv(object, &md_super_buffer);
			if (rc) {
				// wasn't ours, put it on the output list now.
				InsertObject(output_list, size, object, tag, NULL, AppendToList, FALSE, &waste);
			} else {
				if ( md_find_volume_for_object(object, md_super_buffer)) {
					md_deallocate_memory(md_super_buffer);
					LOG_WARNING("Error finding volume minor %d for PV %s\n", md_super_buffer->md_minor, object->name);
				}

			}
		} else {
			LOG_DETAILS("Skipping object %s because not DATA_TYPE\n",object->name);
		}

	}
	if ((rc == DLIST_EMPTY) || (rc == DLIST_END_OF_LIST)) {
		rc = 0;
	} else {
		LOG_WARNING("Error processing input list rc = %d\n", rc);
	}
	RETURN(rc);
}

// Get information for MD device from kernel
//Caller is responsible for freeing superblock memory on 0 rc, otherwise memory not allocated.

int md_get_kernel_info(md_volume_t * volume,evms_md_array_info_t* md_info) {

	int rc = 0;
	evms_md_ioctl_t         md_ioctl = {0};
	evms_plugin_ioctl_t     plugin_ioctl = {0};

	LOG_ENTRY;

	if ( md_allocate_memory((void**)&md_info->sb, MD_SB_BYTES ) ) {
		RETURN (ENOMEM);
	}

       	plugin_ioctl.feature_id = SetPluginID(IBM_OEM_ID, EVMS_REGION_MANAGER, 0x4),
       				  plugin_ioctl.feature_command = EVMS_MD_GET_ARRAY_INFO,
       	plugin_ioctl.feature_ioctl_data = &md_ioctl;

       	md_ioctl.mddev_idx = volume->super_block->md_minor;
       	md_ioctl.cmd = 0; //not used here
       	md_ioctl.arg = md_info;

       	rc = EngFncs->ioctl_evms_kernel(EVMS_PLUGIN_IOCTL, &plugin_ioctl);

       	/* If the ioctl failed we can try do to the write by hand. */
       	if (rc != 0) {
       		/*
       		 * Get a real error code other than the -1 which ioctl()
       		 * returns.
       		 */
       		if (plugin_ioctl.status != 0) {
       			rc = plugin_ioctl.status;
       		} else {
       			rc = errno;
       		}
		md_deallocate_memory(md_info->sb);
       	}
	RETURN(rc);

}


static BOOLEAN  md_namespace_registered = FALSE;

int md_register_name_space(void) {

        int rc = 0;

        LOG_ENTRY;

        if (!md_namespace_registered) {
                rc = EngFncs->register_name(MD_NAME_SPACE);

                if (rc == 0) {
                        md_namespace_registered = TRUE;
                } else {
                        LOG_SERIOUS("Error registering the MD name space \"%s\".\n", MD_NAME_SPACE);
                }
        }

        RETURN(rc);
}
