/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: drivelink.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>
#include <linux/evms/evms_drivelink.h>

#include "drivelink.h"
#include "dloptions.h"
#include "dlsn.h"
#include "dlrescue.h"

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                         PRIVATE DATA AREAS AND SUBROUTINES                           +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

static plugin_record_t drivelink_plugin_record;

plugin_record_t              *DL_PluginRecord_Ptr=&drivelink_plugin_record; // our feature plugin record ptr
struct engine_functions_s    *DLEngFncs=NULL;                               // engine function table ptr



dlist_t                       MetaDataList=NULL;                            // list of saved metadata buffers
#define METADATA_TAG          0x10101010

dlist_t                       PSN_List=NULL;                                // list of parent serial numbers we have
#define PSN_TAG               0x20202020                                    // discovered




/*
 *  Debug routine - called to display the feature header by writing it
 *                  to the log file.
 */
static void display_feature_header(evms_feature_header_t  * feature_header)
{

    LOG_DEBUG("Feature Header ...\n");
    LOG_DEBUG("\t       FH      flags: 0x%X\n", feature_header->flags );
    LOG_DEBUG("\t       FH    feat id: 0x%X\n", feature_header->feature_id );
    LOG_DEBUG("\t       FH      major: 0x%X\n", feature_header->version.major );
    LOG_DEBUG("\t       FH      minor: 0x%X\n", feature_header->version.minor );
    LOG_DEBUG("\t       FH      patch: 0x%X\n", feature_header->version.patchlevel );
    LOG_DEBUG("\t       FH   sequence: %lld\n", feature_header->sequence_number );
    LOG_DEBUG("\t       FH    padding: %lld\n", feature_header->alignment_padding );
    LOG_DEBUG("\t       FH       lsn1: %lld\n", feature_header->feature_data1_start_lsn );
    LOG_DEBUG("\t       FH data size1: %lld\n", feature_header->feature_data1_size );
    LOG_DEBUG("\t       FH       lsn2: %lld\n", feature_header->feature_data2_start_lsn );
    LOG_DEBUG("\t       FH data size2: %lld\n", feature_header->feature_data2_size );
    LOG_DEBUG("\t       FH   obj name: %s\n",   feature_header->object_name);

}



/*
 *  Called to determine if the storage object is one of
 *  our bogus missing child objects, produced by dlrescue.
 */
BOOLEAN isa_missing_child(storage_object_t *child)
{
    BOOLEAN  result=FALSE;

    LOGENTRY();
    if (child) {

        if (child->plugin == DL_PluginRecord_Ptr) {

            if ( child->private_data ) {

                if ( ((Drive_Link_Private_Data *)child->private_data)->signature==MISSING_CHILD_SIGNATURE) {

                    result = TRUE;

                }

            }

        }

    }

    LOGEXIT();
    return result;
}


/*
 *  Called to test if the specified drivelink is missing
 *  child objects and is READ ONLY.
 */
BOOLEAN  isa_RDONLY_drivelink( storage_object_t *object )
{
    storage_object_t         *drive_link_parent=NULL;
    BOOLEAN                   result=FALSE;

    LOGENTRY();

    if (object) {

        drive_link_parent = get_parent(object);

        if (drive_link_parent) {

            if ( drive_link_parent->flags & SOFLAG_READ_ONLY ) {

              result = TRUE;

            }

        }

    }


    LOGEXIT();
    return result;
}



/*
 *  Called to test if the specified drivelink is missing
 *  some child objects and will make sure the RDONLY flag
 *  bit is set in the parent private data area.
 */
void  test_and_set_RDONLY_flag( storage_object_t *object )
{
    storage_object_t         *drive_link_parent=NULL;
    Drive_Link_Private_Data  *pdata;
    int                       i;

    LOGENTRY();

    if (object) {

        drive_link_parent = get_parent(object);

        if (drive_link_parent) {

            pdata =  (Drive_Link_Private_Data *) drive_link_parent->private_data;

            if (pdata) {

                drive_link_parent->flags &= ~SOFLAG_READ_ONLY;

                for (i=0; i < pdata->drive_link_count; i++) {

                    if ( pdata->drive_link[i].flags & DL_FLAG_MISSING ) {

                        drive_link_parent->flags |= SOFLAG_READ_ONLY;

                    }

                }

            }

        }

    }

    LOGEXIT();

}


/*
 *  Returns the private saved metadata area for the specified object.
 */
SAVED_METADATA * get_saved_metadata( storage_object_t *obj )
{
    int                 rc;
    SAVED_METADATA     *mdata = NULL;


    LOGENTRY();

    // make sure dlist is created
    if ( MetaDataList ) {

        rc = GoToStartOfList( MetaDataList );
        if ( rc == DLIST_SUCCESS ) {

            rc = GetObject( MetaDataList,sizeof(SAVED_METADATA),METADATA_TAG,NULL,TRUE,(void **)&mdata );

            while (rc == DLIST_SUCCESS) {

                // compare storage object ptr to key field in private data area
                if ( mdata->object == obj ) {
                    LOGEXIT();
                    return mdata;

                }

                rc = GetNextObject( MetaDataList,sizeof(SAVED_METADATA), METADATA_TAG,(void **) &mdata );

            };

        }

    }

    LOGEXIT();
    return NULL;
}



/*
 *  Called to delete the specified saved metadata
 */
int delete_saved_metadata( storage_object_t *obj )
{
    int                rc = EINVAL;
    SAVED_METADATA    *mdata = get_saved_metadata( obj );

    LOGENTRY();

    if ( MetaDataList ) {

        if (mdata) {

            DeleteObject( MetaDataList, mdata );

            DLEngFncs->engine_free(mdata);

            rc = 0;
        }
        else {
            rc = EINVAL;
        }
    }

    LOGEXITRC();
    return rc;
}



/*
 *  Called to allocate a private copy of drivelink metadata. This
 *  copy is kept around during discovery till a drivelink object is completed.
 */
int create_saved_metadata( storage_object_t  *obj )
{
    int rc=ENOMEM;
    SAVED_METADATA *mdata=NULL;
    void *handle;


    LOGENTRY();

    // make sure dlist is created
    if ( MetaDataList == NULL) {

        MetaDataList = CreateList();

        if (MetaDataList == NULL) {
            LOGEXITRC();
            return rc;
        }

    }

    if ( get_saved_metadata(obj ) == NULL ) {

        mdata = (SAVED_METADATA *) DLEngFncs->engine_alloc( sizeof(SAVED_METADATA) );

        if ( mdata ) {

            mdata->signature   = SAVED_METADATA_SIGNATURE;
            mdata->object      = (void *)obj;

            rc = InsertObject ( (dlist_t)          MetaDataList,
                                (int)              sizeof(SAVED_METADATA),
                                (void *)           mdata,
                                (TAG)              METADATA_TAG,
                                (void *)           NULL,
                                (Insertion_Modes)  AppendToList,
                                (BOOLEAN)          TRUE,
                                (void **)         &handle );

            if ( rc ) {
                free(mdata);
            }

        }
        else {
            rc = ENOMEM;
        }
    }
    else {
        rc = 0;
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Called to delete ALL saved metadata at the conclusion of Discovery.
 */
static void delete_all_saved_metadata( void )
{
    int                rc = EINVAL;
    SAVED_METADATA    *mdata = NULL;

    LOGENTRY();

    if ( MetaDataList ) {

        rc = GoToStartOfList( MetaDataList );
        if ( rc==DLIST_SUCCESS ) {

            rc = GetObject( MetaDataList,sizeof(SAVED_METADATA),METADATA_TAG,NULL,TRUE, (void **) &mdata );

            while (rc == DLIST_SUCCESS) {

                if (mdata->metadata) DLEngFncs->engine_free(mdata->metadata);

                rc = GetNextObject( MetaDataList, sizeof(SAVED_METADATA),METADATA_TAG, (void **) &mdata );
            }

            DestroyList( &MetaDataList, TRUE );
        }

    }

    LOGEXIT();
}



/*
 *  Returns first object found in a dlist, with no regard as to
 *  what kind of object it may be.
 */
storage_object_t *  get_first_object_in_list( dlist_t list )
{
    int rc;
    storage_object_t *obj;
    storage_object_t *object_from_list = NULL;
    uint size;
    TAG  tag;

    LOGENTRY();

    rc = GoToStartOfList( list );
    if ( rc == DLIST_SUCCESS ) {

        rc = BlindGetObject( list, &size, &tag, NULL, FALSE, (void **)&obj );
        if ( rc == DLIST_SUCCESS ) {

            object_from_list = obj;

        }

    }

    LOGEXIT();
    return  object_from_list;
}



/*
 *  Called to find the last child storage object in a drivelink and return it.
 *
 *  This routine can be called with any object in the drivelink, i.e. the parent
 *  drivelink storage object or any of the children objects.
 *
 *  Returns: pointer to last child storage object if successful
 *           NULL pointer otherwise.
 *
 */
storage_object_t *  get_last_drivelink_child( storage_object_t *object )
{
    storage_object_t         *last_child=NULL;
    storage_object_t         *drive_link_parent=NULL;
    Drive_Link_Private_Data  *pdata;


    LOGENTRY();

    if (object) {

        //  Find the parent storage object for the drivelink
        if (object->plugin == DL_PluginRecord_Ptr ) {

            drive_link_parent = object;

        }
        else {  // We must have been passed a child drivelink object

            drive_link_parent = get_first_object_in_list(object->parent_objects);

        }

        // If we got a parent object ... then validate private data and get last child
        if (drive_link_parent) {

            pdata =  (Drive_Link_Private_Data *) drive_link_parent->private_data;

            if (pdata) {

                // validate private data
                if (pdata->signature == EVMS_DRIVELINK_SIGNATURE ) {

                    last_child = pdata->drive_link[pdata->drive_link_count-1].object;

                }
            }
        }

    }


    LOGEXIT();
    return last_child;
}


/*
 *  Called to get the drive link parent object.
 *
 *  Can be called with any object in the drivelink.
 *
 */
storage_object_t * get_parent( storage_object_t  *object )
{
    storage_object_t *parent_object=NULL;
    storage_object_t *parent_drivelink_object=NULL;

    LOGENTRY();

    if (object) {

        //  Find the parent storage object for the drivelink
        if (object->plugin == DL_PluginRecord_Ptr ) {

            parent_object = object;

        }
        else {  // We must have been passed a child drivelink object

            parent_object = get_first_object_in_list(object->parent_objects);

        }
    }

    // validate that we have the parent by looking for our signature in the private data area
    if ( parent_object ) {

        if( parent_object->private_data ) {

            if ( ((Drive_Link_Private_Data *)parent_object->private_data)->signature == EVMS_DRIVELINK_SIGNATURE ) {

                parent_drivelink_object = parent_object;

            }
        }

    }

    LOGEXIT();
    return parent_drivelink_object;
}



/*
 *  Called to test if the specified storage object is a completed
 *  drive link. Meaning ... we have found all the child storage
 *  objects.
 *
 *  If the link count says there are N children in the drivelink
 *  but when walking the ordering table we discover a missing
 *  child ... no storage object for a link in the ordering table ...
 *  then the drivelink is incomplete.
 *
 *  Returns: TRUE only if we have all the child objects needed.
 *
 */
static BOOLEAN isa_complete_aggregate( storage_object_t *parent )
{
    BOOLEAN rc = TRUE;
    int     i;
    Drive_Link_Private_Data *pdata = (Drive_Link_Private_Data *)parent->private_data;


    for (i=0; i<pdata->drive_link_count; i++) {

        if ( pdata->drive_link[i].object == NULL ) {
            rc = FALSE;
            break;
        }

    }


    return rc;
}



/*
 *  Called to test if we own the specified storage object.  This is also
 *  an opportunity to place code here to further inspect an object prior
 *  to making any changes to it.
 *
 *  Returns: TRUE only if we own the object and it appears Ok at first glance
 *
 */

static BOOLEAN i_can_modify_object( storage_object_t *object )
{

    if (object) {

        if (object->plugin == DL_PluginRecord_Ptr) {

            if ( object->private_data ) {

                if ( ((Drive_Link_Private_Data *)object->private_data)->signature==EVMS_DRIVELINK_SIGNATURE) {

                    return TRUE;

                }

            }

        }

    }

    return FALSE;
}



/*
 *  This routine is called to figure out the position of a storage object in a
 *  drive link.  The position is determined by the ordering_table array. This
 *  array holds serial numbers of storage objects and their position in the array
 *  determines their position in the drive link.
 *
 *  Returns: -1 if the object is not found in the ordering_table
 *                         ( otherwise )
 *           the index into the drive link ordering table where the storage object
 *           was found.
 *
 */
int get_drivelink_index_by_sn( evms_dl_ordering_table_entry_t *table,         // ordering table
                               u_int32_t                       serial_number, // serial # of storage object
                               int                             count)         // count of objects in table
{
    int  i;
    evms_dl_ordering_table_entry_t *dot = table;

    for (i=0; i < count; i++) {

        if ( dot->child_serial_number == serial_number ) {
            return i;
        }

        ++dot;
    }

    return -1;
}



/*
 *  Called to get a child serial number that is Ok to use in the specified
 *  ordering table, i.e. it is unused so far and not a duplicate.
 *
 *  We start with a guess... 0x100 plus the current drive link count. Then,
 *  increment the guess till we hit a serial number that is not a dup of
 *  any existing serial numbers.
 *
 *  Returns: Zero ... if unable to provide a serial number
 *                        ( otherwise )
 *           a new child serial number that is Ok to use in the specified
 *           drive link ordering table
 */
static u_int32_t create_child_serial_number( evms_dl_ordering_table_entry_t *table,        // ordering table
                                             u_int32_t                       link_count )  // count of links in table
{
    u_int32_t      guess = 0x100 + link_count;
    u_int32_t      child_serial_number = 0;
    int            i;
    BOOLEAN        duplicate;

    if (link_count < EVMS_DRIVELINK_MAX_ENTRIES) {

        while (child_serial_number == 0) {

            duplicate = FALSE;

            for (i=0; i<link_count; i++) {

                if ( ((evms_dl_ordering_table_entry_t *)table+i)->child_serial_number == guess ) {
                    duplicate=TRUE;
                    break;
                }

            }

            if (duplicate==TRUE) {
                ++guess;
            }
            else {
                child_serial_number = guess;
            }

        };

    }

    return child_serial_number;
}


/*
 *  Called to see if a drive link childs serial number appears in the
 *  specified ordering table.
 *
 *  This routine is called from discovery code. The first parm is an
 *  ordering table that is being built by adding drivelink children to
 *  it as they are discovered.  So ... the child we are trying to add
 *  to the ordering table should not be found in the table yet.  If it
 *  appears in the table, it means that we have 2 storage objects that
 *  claim to be the same drivelink child object!
 *
 *  Returns: TRUE if we find that the child serial number is already in the table
 *
 */
static BOOLEAN  isa_dup_child_serial_number( evms_dl_ordering_table_entry_t *table,         // ordering table
                                             u_int32_t                       serial_number, // object serial number
                                             int                             count )        // count of objects in table
{
    int i;
    evms_dl_ordering_table_entry_t *dot = table;

    for (i=0; i<count; i++) {

        if ( dot->child_serial_number == serial_number ) {

            return TRUE;

        }

        ++dot;
    }

    return FALSE;
}


/*
 *  Called to run some simple integrity checks on a drive link ordering
 *  table. The checks are for:
 *
 *  (1) no NULL entries allowed ... cant have holes in the address space
 *  (2) no duplicate entries ... cant have same object at different virtual addresses
 *
 *  Returns: FALSE if any checks fail
 *
 */
static BOOLEAN ordering_table_ok( evms_dl_ordering_table_entry_t *table, int linkcount )
{
    int i,j,count;
    u_int32_t  sn1=0, sn2=0;

    for (i=0; i<linkcount; i++) {

        sn1 = ((evms_dl_ordering_table_entry_t *)table+i)->child_serial_number;


        // check for a NULL entry in the table ... 0 is an invalid serial number
        if ( sn1  == 0 ) {
            return FALSE;
        }


        // look for duplicate entries in table
        count = 0;
        for (j=0; j<linkcount; j++) {

            sn2 = ((evms_dl_ordering_table_entry_t *)table+j)->child_serial_number;

            if ( sn1 == sn2) ++count;
        }

        if (count != 1) {
            return FALSE;
        }
    }

    return TRUE;
}



/*
 *  Called to build a drive link feature header for the specified
 *  child object link.
 *
 *
 *
 *  A picture is worth a thousand words
 *  ----------------------------------------------------------------------------------------------------+
 *           |  2nd copy of metadata  | 1st copy of metadata | 2nd feature header  | 1st feature header |
 *  ---------------------------------------------------------------------------------------------------+
 *                                                            LSN=n-1               LSN=n
 *
 *
 *  Returns: feature header in the supplied buffer if RC=0
 *
 */
static int BuildFeatureHeader( storage_object_t      *object,          // drive link object
                               drive_link_t          *link,            // child link
                               evms_feature_header_t *feature_header ) // buffer
{
    int rc=0;
    Drive_Link_Private_Data *pdata;
    u_int32_t  feature_header_sequence_number=0;
    u_int32_t  feature_header_data1_start_lsn=0;
    u_int32_t  feature_header_data1_size     =0;
    u_int32_t  feature_header_data2_start_lsn=0;
    u_int32_t  feature_header_data2_size     =0;

    LOGENTRY();
    LOG_DEBUG("building feature header for object %s\n", link->object->name );

    pdata = (Drive_Link_Private_Data *)object->private_data;

    if ( ( link ) &&
         ( pdata ) &&
         ( feature_header )) {

        memset(feature_header, 0, sizeof(evms_feature_header_t));

        // provide an incremented feature header sequence number
        feature_header_sequence_number   = pdata->fh_sequence_number;

        // 1st copy of metadata is found immediately before feature header sectors
        feature_header_data1_start_lsn = link->object->size - (FEATURE_HEADER_SECTOR_COUNT*2) - DRIVELINK_METADATA_SECTOR_COUNT;
        feature_header_data1_size      = DRIVELINK_METADATA_SECTOR_COUNT;

        // 2nd copy of metadata is found immediately before 1st copy of metadata
        feature_header_data2_start_lsn = feature_header_data1_start_lsn - DRIVELINK_METADATA_SECTOR_COUNT;
        feature_header_data2_size      = DRIVELINK_METADATA_SECTOR_COUNT;

        if (rc == 0) {

            feature_header->signature               = EVMS_FEATURE_HEADER_SIGNATURE;
            feature_header->feature_id              = SetPluginID(EVMS_OEM_IBM, EVMS_FEATURE, EVMS_DRIVELINK_FEATURE_ID );
            feature_header->sequence_number         = feature_header_sequence_number;
            feature_header->feature_data1_size      = feature_header_data1_size;
            feature_header->feature_data2_size      = feature_header_data2_size;
            feature_header->feature_data1_start_lsn = feature_header_data1_start_lsn;
            feature_header->feature_data2_start_lsn = feature_header_data2_start_lsn;

            strncpy( feature_header->object_name,
                     object->name,
                     EVMS_VOLUME_NAME_SIZE );

            link->object->flags |= SOFLAG_FEATURE_HEADER_DIRTY;

        }

    }
    else {
        rc = ENOMEM;
    }

    LOGEXITRC();
    return rc;
}


static void Disk_Metadata_To_CPU( evms_drivelink_metadata_t *metadata )
{
    int i;

    metadata->signature              = DISK_TO_CPU32(metadata->signature);
    metadata->crc                    = DISK_TO_CPU32(metadata->crc);
    metadata->version.major          = DISK_TO_CPU32(metadata->version.major);
    metadata->version.minor          = DISK_TO_CPU32(metadata->version.minor);
    metadata->version.patchlevel     = DISK_TO_CPU32(metadata->version.patchlevel);
    metadata->flags                  = DISK_TO_CPU32(metadata->flags);
    metadata->sequence_number        = DISK_TO_CPU64(metadata->sequence_number);
    metadata->child_serial_number    = DISK_TO_CPU64(metadata->child_serial_number);
    metadata->parent_serial_number   = DISK_TO_CPU64(metadata->parent_serial_number);
    metadata->child_count            = DISK_TO_CPU64(metadata->child_count);

    for (i=0; i<EVMS_DRIVELINK_MAX_ENTRIES; i++) {
        metadata->ordering_table[i].child_serial_number = DISK_TO_CPU64(metadata->ordering_table[i].child_serial_number);
        metadata->ordering_table[i].child_vsize         = DISK_TO_CPU64(metadata->ordering_table[i].child_vsize );
    }

}

static void CPU_Metadata_To_Disk( evms_drivelink_metadata_t *metadata )
{
    int i;

    metadata->signature              = CPU_TO_DISK32(metadata->signature);
    metadata->crc                    = CPU_TO_DISK32(metadata->crc);
    metadata->version.major          = CPU_TO_DISK32(metadata->version.major);
    metadata->version.minor          = CPU_TO_DISK32(metadata->version.minor);
    metadata->version.patchlevel     = CPU_TO_DISK32(metadata->version.patchlevel);
    metadata->flags                  = CPU_TO_DISK32(metadata->flags);
    metadata->sequence_number        = CPU_TO_DISK64(metadata->sequence_number);
    metadata->child_serial_number    = CPU_TO_DISK64(metadata->child_serial_number);
    metadata->parent_serial_number   = CPU_TO_DISK64(metadata->parent_serial_number);
    metadata->child_count            = CPU_TO_DISK64(metadata->child_count);

    for (i=0; i<EVMS_DRIVELINK_MAX_ENTRIES; i++) {
        metadata->ordering_table[i].child_serial_number = CPU_TO_DISK64(metadata->ordering_table[i].child_serial_number);
        metadata->ordering_table[i].child_vsize         = CPU_TO_DISK64(metadata->ordering_table[i].child_vsize );
    }

}



/*
 *  Called to read the meta data for a feature into the specified
 *  buffer. If the feature header ptr is NULL it means we need to
 *  get the feature header for the storage object by reading it off
 *  the disk. We need the feature header because it tells us where
 *  the feature data is located in the storage object.
 *
 *  Returns: feature meta data in the supplied buffer if RC=0
 *
 */
static int ReadMetaData( storage_object_t          * object,
                         evms_drivelink_metadata_t * metadata,
                         evms_feature_header_t     * feature_header )
{
    int                        rc=ENODATA;
    int                        rc1,rc2;
    evms_drivelink_metadata_t  metadata2;
    u_int32_t                  crc;
    u_int32_t                  calculated_crc;
    lsn_t                      lsn1=0;
    lsn_t                      lsn2=0;

    LOGENTRY();


    // rc==0 means we have a feature header which tells us where
    // the feature data is located on the storage object.
    if (object && metadata && feature_header) {

        lsn1 = feature_header->feature_data1_start_lsn;
        lsn2 = feature_header->feature_data2_start_lsn;

        rc1 = READ( object, lsn1, DRIVELINK_METADATA_SECTOR_COUNT, metadata );

        if (rc1 == 0) {

            if ( DISK_TO_CPU32(metadata->signature) == EVMS_DRIVELINK_SIGNATURE ) {

                crc = DISK_TO_CPU32(metadata->crc);

                metadata->crc = 0;

                calculated_crc = DLEngFncs->calculate_CRC(  EVMS_INITIAL_CRC,
                                                            (void *)metadata,
                                                            sizeof(evms_drivelink_metadata_t) );

                metadata->crc = CPU_TO_DISK32(crc);

                if ( crc == calculated_crc ) {
                    Disk_Metadata_To_CPU( metadata );
                    rc1 = 0;
                }
                else {
                    rc1 = ENODATA;
                }
            }
            else {
                rc1 = ENODATA;
            }
        }


        // look for second copy of metadata
        if ( feature_header->feature_data2_size != 0 ) {

            rc2 = READ( object, lsn2, DRIVELINK_METADATA_SECTOR_COUNT, &metadata2 );

            if (rc2 ==0 ) {

                if ( DISK_TO_CPU32(metadata2.signature) == EVMS_DRIVELINK_SIGNATURE) {

                    crc = DISK_TO_CPU32(metadata2.crc);

                    metadata2.crc = 0;

                    calculated_crc = DLEngFncs->calculate_CRC(  EVMS_INITIAL_CRC,
                                                                (void *)&metadata2,
                                                                sizeof(evms_drivelink_metadata_t) );

                    metadata2.crc = CPU_TO_DISK32(crc);

                    if ( crc == calculated_crc ) {
                        Disk_Metadata_To_CPU( &metadata2 );
                        rc2 = 0;
                    }
                    else {
                        rc2 = ENODATA;
                    }
                }
                else {
                    rc2 = ENODATA;
                }

            }

        }
        else {
            rc2 = ENODATA;
        }


        if ( (rc1==0)&&(rc2==0) ) {
            if ( metadata->sequence_number >= metadata2.sequence_number) {
                LOG_DEBUG("\tusing 1st copy cuz seq numbers are same or 1st is > 2nd\n");
            }
            else {
                LOG_DEBUG("\tusing 2nd copy of metadata cuz of seq numbers\n");
                memcpy(metadata, &metadata2, sizeof(evms_drivelink_metadata_t ));
            }
            rc = 0;
        }
        else if (rc1 == 0) {
            rc = 0;
        }
        else if (rc2 == 0) {
            memcpy(metadata, &metadata2, sizeof(evms_drivelink_metadata_t) );
            rc = 0;
        }
        else {
            rc = ENODATA;
        }

    }
    else {
        rc = EINVAL;
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Called to write meta data for a feature out to disk.  If the
 *  feature header ptr is NULL it means we need to get the feature
 *  header by reading it off the disk. We need the feature header
 *  because it tells us where the feature data is located in the
 *  storage object.
 *
 *  Returns: RC=0 if successful
 *
 */
static int WriteMetaData( storage_object_t          * object,
                          evms_drivelink_metadata_t * metadata,
                          evms_feature_header_t     * feature_header,
                          uint                        commit_phase )
{
    int                      rc=EINVAL;
    u_int32_t                crc;
    lsn_t                    lsn1=0;
    lsn_t                    lsn2=0;

    LOGENTRY();
    LOG_DEBUG("commit phase= %d\n", commit_phase);

    // rc==0 means we have a feature header which tells us where
    // the feature data is located on the storage object.
    if (object && metadata && feature_header) {

        // get lsn values from feature header
        lsn1 = feature_header->feature_data1_start_lsn;
        lsn2 = feature_header->feature_data2_start_lsn;

        // convert metadata to disk format
        CPU_Metadata_To_Disk( metadata );

        // crc the data
        metadata->crc = 0;
        crc = DLEngFncs->calculate_CRC(  EVMS_INITIAL_CRC,
                                         (void *)metadata,
                                         sizeof(evms_drivelink_metadata_t) );
        metadata->crc = CPU_TO_DISK32(crc);


        // write out the metadata
        if (commit_phase == 1) {
LOG_DEBUG("writing metadata for commit phase 1 @ lsn %lld\n", lsn1);
            rc = WRITE( object, lsn1, DRIVELINK_METADATA_SECTOR_COUNT, metadata );
            if (rc) {
                LOG_ERROR("error, failed to write 1st copy of metadata ... rc= %d\n", rc);
            }
        }
        else if (commit_phase == 2) {
LOG_DEBUG("writing metadata for commit phase 2 @ lsn %lld\n", lsn2);
            // write 2nd copy if feature header says to do so
            if ( feature_header->feature_data2_size > 0)  {

                rc = WRITE( object, lsn2, DRIVELINK_METADATA_SECTOR_COUNT, metadata );
                if (rc) {
                    LOG_ERROR("error, failed to write 2nd copy of metadata ... rc= %d\n", rc);
                }

            }
            else {
                rc = 0;
            }
        }
        else {
            LOG_ERROR("error, unknown commit phase ... not 1 or 2\n");
            rc = EINVAL;
        }

    }
    else {
        LOG_ERROR("error, invalid parms were passed\n");
        rc = EINVAL;
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Only called during feature discovery!!
 *
 *  This is a wrapper routine for ReadMetaData()
 *
 *  Because we make several passes at the objects in the engine DLIST
 *  we would have to read the metadata over and over ... so temp space is
 *  allocated at the start of feature discovery to save object metadata
 *  so it wont have to be read more than once.  At the conclusion of
 *  feature discovery ... the temp space and metadata buffers are freed.
 *
 *  Called to get drivelink metadata for the specified storage object by
 *  searching in TEMP_SPACE. If we dont find metadata in the TEMP_SPACE
 *  object we'll read the metadata and then save it in the TEMP_SPACE
 *  object for later use ... if we were successful.
 *
 *  Returns: rc=0 if successful and metadata ptr is set
 *           rc=errno if unsuccessful
 *
 */
int get_metadata( storage_object_t *object, evms_drivelink_metadata_t * *metadata )

{
    SAVED_METADATA             *mdata;
    int                         rc;


    LOGENTRY();

    rc        = ENOMEM;
    *metadata = NULL;

    mdata = get_saved_metadata( object );
    if (mdata == NULL) {
        create_saved_metadata(object);
        mdata = get_saved_metadata(object);
    }

    if (mdata) {

        // if we havent read the metadata yet, the ptr will be NULL
        if (mdata->metadata == NULL) {

            // malloc an I/O buffer
            mdata->metadata = malloc(sizeof(evms_drivelink_metadata_t));
            if (mdata->metadata) {

                // read storage objects metadata
                rc = ReadMetaData( object , mdata->metadata, object->feature_header);
                if (rc==0) {

                    // set callers metadata ptr
                    *metadata = (evms_drivelink_metadata_t *) mdata->metadata;

                    // return good rc
                    rc = 0;
                }
            }

        }
        else {  // we are here cuz the temp space data ptr is not NULL!
                // meaning that we have already read in the metadata.

            // set callers metadata ptr
            *metadata = (evms_drivelink_metadata_t *) mdata->metadata;

            // return good rc
            rc = 0;

        }

    }

    LOGEXITRC();
    return rc;
}




/*
 *  Frees up a drive link object that I allocated using the
 *  malloc_drive_link_object() routine.
 *
 *  Returns: nothing
 *
 */
static void free_drive_link_object( storage_object_t  * drivelink )
{

    if (drivelink) {

        DLEngFncs->unregister_name( drivelink->name);

        if (drivelink->private_data) {
            unregister_parent_serial_number(((Drive_Link_Private_Data *)drivelink->private_data)->parent_serial_number);
            free(drivelink->private_data);
        }

        DLEngFncs->free_evms_object(drivelink);
    }

}


/*
 *  Routine will allocate memory for a new parent drive link storage object
 *  and initialize the new object for the caller.  The only parm is
 *  the parent's drive link serial number.
 *
 *  Returns: address of new storage object if successful, NULL otherwise
 *
 */
static storage_object_t * malloc_drive_link_object( u_int32_t parent_serial_number )
{
    int                      rc;
    storage_object_t        *drivelink = NULL;
    Drive_Link_Private_Data *pdata;

    rc = DLEngFncs->allocate_evms_object(NULL, &drivelink);
    if ( rc == 0 ) {

        pdata = calloc(1,sizeof(Drive_Link_Private_Data));
        if ( pdata == NULL ) {
            DLEngFncs->free_evms_object( drivelink );
            drivelink = NULL;
        }
        else {

            drivelink->plugin         = DL_PluginRecord_Ptr;
            drivelink->private_data   = pdata;
            pdata->signature          = EVMS_DRIVELINK_SIGNATURE;

        }
    }

    return drivelink;
}


/*
 *  Called to commmit metadata for the specified drive link child object.
 *
 *  Returns: RC=0 if feature metadata was written to the storage object
 *
 */
static int Commit_DriveLink_Object( storage_object_t   *parent,
                                    drive_link_t       *link,
                                    uint                commit_phase )
{
    Drive_Link_Private_Data   *pdata;
    evms_drivelink_metadata_t  metadata;
    int                        rc=EINVAL;


    LOGENTRY();
    LOG_DEBUG("commit phase= %d\n", commit_phase );

    if ( i_can_modify_object( parent ) == TRUE ) {

        pdata = (Drive_Link_Private_Data *) parent->private_data;

        if ((pdata)&&(link->object->feature_header)) {

            // only rebuild the feature header for phase 1
            if (commit_phase == 1) {
                rc =  BuildFeatureHeader( parent, link, link->object->feature_header );
            }
            else {
                rc = 0;
            }

            if (rc==0) {

                memset(&metadata, 0, sizeof(evms_drivelink_metadata_t));

                metadata.signature            = EVMS_DRIVELINK_SIGNATURE;
                metadata.parent_serial_number = pdata->parent_serial_number;
                metadata.child_serial_number  = link->serial_number;
                metadata.child_count          = pdata->drive_link_count;

                metadata.version.major        = EVMS_DRIVELINK_VERSION_MAJOR;
                metadata.version.minor        = EVMS_DRIVELINK_VERSION_MINOR;
                metadata.version.patchlevel   = EVMS_DRIVELINK_VERSION_PATCHLEVEL;

                memcpy( &metadata.ordering_table[0],
                        &pdata->ordering_table[0],
                        sizeof(evms_dl_ordering_table_entry_t)*pdata->drive_link_count );

                rc = WriteMetaData( link->object, &metadata, link->object->feature_header, commit_phase );

            }

        }

    }


    LOGEXITRC();
    return rc;
}


/*
 *   Called to build a child_objects dlist from the drivelink
 *   ordering table.
 *
 *   Returns DLIST_SUCCESS if successful and the child_objects
 *   dlist should contain all the objects from the drive link
 *   in the proper order.
 *
 *   Returns
 */
static int build_ordered_child_object_list( storage_object_t  *parent, dlist_t *target_list )
{
    void                     *handle=NULL;
    int                       rc=EINVAL;
    int                       i;
    dlist_t                   objects;
    Drive_Link_Private_Data  *pdata = (Drive_Link_Private_Data  *) parent->private_data;


    LOGENTRY();

    if ( parent && target_list ) {

        objects = *target_list;

        if ( objects ) {

            DeleteAllItems ( objects, FALSE );

            for(i=0,rc=0; (i<pdata->drive_link_count)&&(rc==0); i++) {

                if ( pdata->drive_link[i].object != NULL ) {

                    rc = InsertObject ( objects,
                                        sizeof(storage_object_t),
                                        pdata->drive_link[i].object,
                                        pdata->drive_link[i].object->object_type,
                                        NULL,
                                        AppendToList,
                                        TRUE,
                                        &handle );
                }

            }

        }
        else {
            LOG_ERROR("error, caller passed NULL target dlist ptr\n");
        }

    }
    else {
        LOG_ERROR("error, invalid parms passed\n");
    }

    LOGEXITRC();
    return rc;
}



/*
 *  Called to insert a drive link child object into the parent drive
 *  link object's DLIST.
 *
 *  Information about each child object is placed in the private data
 *  area of the parent drive link object.  The child objects are links
 *  and the links are kept ordered by their link sequence number.  This
 *  number is found in the child object meta data area.
 *
 *  Returns: RC=0 if child object was added to the drive link
 *
 */
static int consume_drive_link_child( storage_object_t       * parent,
                                     storage_object_t       * child,
                                     u_int32_t                child_serial_number,
                                     evms_feature_header_t  * feature_header )
{
    int               rc;
    int               i;
    void             *handle;
    u_int32_t         padding;
    u_int64_t         lsn=0;
    u_int64_t         child_useable_size=0;
    int               child_index=0;
    evms_dl_ordering_table_entry_t  *table_entry=NULL;
    Drive_Link_Private_Data *pdata = (Drive_Link_Private_Data *) parent->private_data;

    LOGENTRY();
    LOG_DEBUG("\nchild= %s  child sn= %d  size= %lld\n", child->name, child_serial_number, child->size);

    //
    // Make sure the child is in the ordering table
    //
    rc = EINVAL;
    for (i=0; i<pdata->drive_link_count; i++) {

        table_entry = &pdata->ordering_table[i];

        if ( table_entry->child_serial_number == child_serial_number ) {
            child_index = i;
            rc = DLIST_SUCCESS;
            break;
        }

    }

    if (rc) {
        LOG_ERROR("error, child serial number not found in parent ordering table, child SN= %d\n",child_serial_number);
        LOGEXITRC();
        return rc;
    }

    //
    // We will only have the feature header parm if called during discovery.
    //
    if (feature_header) {

        // during discovery we'll pick the greatest sequence number from our
        // child objects and use that number.
        if (feature_header->sequence_number > pdata->fh_sequence_number) {
            pdata->fh_sequence_number = feature_header->sequence_number;
        }

    }


    //
    // Calculate the useable area of this storage object by subtracting from the size
    // of the object ... the sectors we are using for drivelink metadata and feature headers.
    //
    if ( isa_missing_child(child) == TRUE ) {
        child_useable_size = child->size;
        padding            = 0;
    }
    else {

        // useable area = size - 2 feature header sectors - 2 copies of metadata
        child_useable_size = child->size - (DRIVELINK_METADATA_SECTOR_COUNT*2) - (FEATURE_HEADER_SECTOR_COUNT*2);

        //
        // useable area must an 8k (16 512 byte sectors) multiple, remaining sectors will
        // be placed in the padding area.
        //
        padding = child_useable_size % 16;
    }

    child_useable_size -= padding;

    //
    // check the useable size against what the ordering table says is
    // the size of this link.
    //
    if ( table_entry->child_vsize ) {

        if ( table_entry->child_vsize != child_useable_size ) {

            LOG_ERROR("error, child metadata says size should be %lld sectors but actual is %lld sectors\n",
                      table_entry->child_vsize, child_useable_size );

            table_entry->child_vsize = child_useable_size;
        }

    }
    else {
        table_entry->child_vsize = child_useable_size;
    }


    // children storage objects hang off parent storage object
    rc = InsertObject( parent->child_objects,
                       sizeof(storage_object_t),
                       child,
                       EVMS_OBJECT_TAG,
                       NULL,
                       AppendToList,
                       FALSE,
                       (void**) &handle );

    if (rc == DLIST_SUCCESS) {

        // the parent storage object hangs off the child object
        rc = InsertObject( child->parent_objects,
                           sizeof(storage_object_t),
                           parent,
                           EVMS_OBJECT_TAG,
                           NULL,
                           AppendToList,
                           FALSE,
                           (void**) &handle );
    }


    if (rc == DLIST_SUCCESS) {

        // fill in the information for this link in the drive link array
        pdata->drive_link[child_index].padding            = padding;
        pdata->drive_link[child_index].sector_count       = child_useable_size;
        pdata->drive_link[child_index].serial_number      = child_serial_number;
        pdata->drive_link[child_index].sequence_number    = child_index;
        pdata->drive_link[child_index].object             = child;

        // if this child is among the missing then mark is missing in
        // our drive_link array and also mark the parent object as RDONLY.
        if ( isa_missing_child(child) == TRUE ) {
            pdata->drive_link[child_index].flags |= DL_FLAG_MISSING;
            parent->flags |= SOFLAG_READ_ONLY;
        }


        LOG_DEBUG("Added child to drive link: index= %d  child sn= 0x%X  size= %lld\n",
                   child_index, child_serial_number, child_useable_size );

        // ... UPDATE THE DRIVE LINK ADDRESS SPACE ...
        // if this drive link object is complete, i.e. has all its children
        // storage objects, then we now need to determine the starting logical
        // sector number (lsn) for each child storage object.
        //
        // We can also figure out the total size of the parent drive link object.
        // at this time.
        //
        if ( ( rc == DLIST_SUCCESS ) &&
             ( isa_complete_aggregate( parent ) == TRUE ) ) {

            lsn          = 0;
            parent->size = 0;

            LOG_DEBUG("Completed Ordering Table...\n");

            for (i=0; i<pdata->drive_link_count; i++) {

                parent->size += pdata->drive_link[i].sector_count;
                pdata->drive_link[i].start_lsn = lsn;
                pdata->drive_link[i].end_lsn   = lsn + pdata->drive_link[i].sector_count - 1;

                lsn = pdata->drive_link[i].end_lsn + 1;

                LOG_DEBUG("\t Child: name= %s  start_lsn= %lld   end_lsn= %lld   size= %lld\n",
                            pdata->drive_link[i].object->name,
                            pdata->drive_link[i].start_lsn,
                            pdata->drive_link[i].end_lsn,
                            pdata->drive_link[i].sector_count );

            }

        }

    }
    else {
        memset( &pdata->drive_link[child_index], 0, sizeof(drive_link_t) );
    }


    LOGEXITRC();
    return rc;
}


/*
 *  Each time called, we see if we can remove the child storage object from the parent
 *  drive link storage object.
 *
 *  This routine can be called only from DL_Shrink().
 *
 *  The job is simply to remove the last child object from the drive link storage
 *  object and clean up the ordering table.
 *
 *  Returns: TRUE if we removed the specified child storage object from
 *           the parent objects DLIST.
 *
 */
static int Shrink_Drive_Link_Object( storage_object_t   *parent,  storage_object_t   *child )
{
    int                       rc = EINVAL;
    int                       index;
    Drive_Link_Private_Data  *pdata;
    struct plugin_functions_s *fncs=NULL;
    sector_count_t            kill_sector_count=0;

    LOGENTRY();

    // get parent object info
    pdata = (Drive_Link_Private_Data *)parent->private_data;

    // get info about last child in drive link
    index = pdata->drive_link_count - 1;

    // remove child from parent object child DLIST
    rc = DeleteObject( parent->child_objects, child );

    // remove parent from child parent DLIST
    if (rc == DLIST_SUCCESS) {
        rc = DeleteObject( child->parent_objects, parent );
    }

    if (rc == DLIST_SUCCESS) {

        // shrink drive link size
        parent->size -= pdata->drive_link[index].sector_count;

        // decrement child link count
        --pdata->drive_link_count;

        // zap our metadata and feature headers
        if ( isa_missing_child(child) == FALSE ) {

            fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;

            kill_sector_count = (DRIVELINK_METADATA_SECTOR_COUNT*2) + (FEATURE_HEADER_SECTOR_COUNT*2);

            rc = fncs->add_sectors_to_kill_list( child,                           // child object
                                                 child->size - kill_sector_count, // lba
                                                 kill_sector_count );             // sector count
        }

        // clear position in the drive link ordering table
        (pdata->ordering_table+index)->child_serial_number = 0;
        (pdata->ordering_table+index)->child_vsize         = 0;

        // clear the drive link entry
        memset( &pdata->drive_link[index], 0, sizeof(drive_link_t) );

    }


    LOGEXITRC();
    return rc;
}




/*
 *  Each time called, we add the child storage object to the parent
 *  drive link storage object.
 *
 *  This routine can be called from either DL_Create() or DL_Expand().
 *  with objects that have NO feature headers and NO feature data.
 *
 *  The job is simply to add child objects to the drive link storage
 *  object who dont have child serial numbers or positions in the
 *  ordering table.
 *
 *  Called with exactly the number of objects we need to add to the
 *  drivelink!
 *
 *  Returns: TRUE if we were successful in adding the child to the
 *           parent object.
 *
 */
static int Expand_Drive_Link_Object( storage_object_t *parent, dlist_t children )
{
    storage_object_t        *child;
    u_int32_t                child_serial_number;
    Drive_Link_Private_Data *pdata;
    int                      rc = EINVAL;
    int                      i;
    int                      j;
    int                      original_drive_link_count=0;
    u_int32_t                new_children[EVMS_DRIVELINK_MAX_ENTRIES];
    int                      new_children_count=0;
    TAG                      tag;
    uint                     size;


    LOGENTRY();

    pdata = (Drive_Link_Private_Data *)parent->private_data;

    //
    // get setup for any possible unwinding of the expand
    //
    original_drive_link_count = pdata->drive_link_count;
    memset(&new_children, 0, (sizeof(u_int32_t)*EVMS_DRIVELINK_MAX_ENTRIES) );

    //
    // walk through the supplied list of objects and try to add them all
    // to the drivelink.
    //
    rc = GoToStartOfList( children );
    if ( rc==DLIST_SUCCESS ) {

        // get child from list
        rc = BlindGetObject( children, &size, &tag, NULL, TRUE, (void **)&child );
        if (rc == DLIST_SUCCESS) {

            do {

                // create a feature header for the child object
                child->feature_header = (evms_feature_header_t *) DLEngFncs->engine_alloc( sizeof(evms_feature_header_t) );

                if (child->feature_header) {

                    // create a drive link serial number for the child object we are consuming .
                    // The routine will check that it is unique for the ordering table.
                    child_serial_number = create_child_serial_number( &pdata->ordering_table[0],
                                                                      pdata->drive_link_count );

                    if (child_serial_number != 0) {

                        // update the drive link ordering table to maintain a
                        // sequence for child links.
                        (pdata->ordering_table+pdata->drive_link_count)->child_serial_number = child_serial_number;
                        ++pdata->drive_link_count;
                        ++new_children_count;

                        // remember child object for unwinding the expand
                        new_children[new_children_count] = child_serial_number;

                        // add child to the drive link storage object
                        rc = consume_drive_link_child( parent, child, child_serial_number, NULL );
                        if (rc == DLIST_SUCCESS) {

                            drive_link_t  *dlink = &pdata->drive_link[pdata->drive_link_count-1];

                            rc =  BuildFeatureHeader( parent, dlink, dlink->object->feature_header );

                            if (rc == DLIST_SUCCESS) {
                                rc = NextItem( children );
                                if (rc == DLIST_SUCCESS) {
                                    rc = BlindGetObject( children, &size, &tag, NULL, TRUE, (void **)&child );
                                }
                            }
                        }
                        else {

                            --pdata->drive_link_count;
                            --new_children_count;

                            LOG_ERROR("failed to consume child storage object\n");
                            rc = ENOMEM;
                        }
                    }
                    else {
                        LOG_ERROR("unable to get unique serial number for child\n");
                        rc = EINVAL;
                    }
                }
                else {
                    rc = ENOMEM;
                }
            } while ( rc == DLIST_SUCCESS  );

        }

    }


    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY ) || ( rc == DLIST_END_OF_LIST ) ) {
        rc = DLIST_SUCCESS;
    }


    // if there were any errors and we consumed some new children objects
    // then we need to unwind the expand
    if ( (rc) && ( new_children_count > 0 )) {

        for (i=0; i < new_children_count; i++) {

            for (j=0; j < pdata->drive_link_count; j++) {

                if ( (pdata->ordering_table+j)->child_serial_number == new_children[i] ) {

                    if (pdata->drive_link[j].object->feature_header) {
                        DLEngFncs->engine_free(pdata->drive_link[j].object->feature_header);
                    }

                    DeleteObject( parent->child_objects, pdata->drive_link[j].object );
                    DeleteObject( pdata->drive_link[j].object->parent_objects, parent );
                    memset(&pdata->drive_link[i], 0, sizeof(drive_link_t));
                    (pdata->ordering_table+j)->child_serial_number = 0;
                    (pdata->ordering_table+j)->child_vsize         = 0;
                }

            }

        }

        // restore original drive_link_count
        pdata->drive_link_count = original_drive_link_count;
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Called to get a parent drive link storage object for the child drive link
 *  objects in the DLIST.  This routine will identify all the unique parent
 *  drive link objects by inspecting each child in the DLIST to see who their
 *  parent object is.  Then, using this list of parent objects we'll see
 *  if we have all the child objects needed to complete one of the parent
 *  drive links and return the parent object if successful.
 *
 *  Returns: New Parent Drive Link Storage Object for aggregate if we can
 *           build it from the list of child objects.
 *
 */
static storage_object_t * find_complete_drive_link( dlist_t child_object_list)
{

    storage_object_t        *dlink = NULL;
    Drive_Link_Private_Data *dlink_pdata;

    storage_object_t *obj;

    u_int32_t        *parents=NULL;
    int               links[EVMS_DRIVELINK_MAX_ENTRIES];

    evms_dl_ordering_table_entry_t  ordering_table[EVMS_DRIVELINK_MAX_ENTRIES];
    evms_dl_ordering_table_entry_t  dup_ordering_table[EVMS_DRIVELINK_MAX_ENTRIES];

    char              parent_object_name[EVMS_VOLUME_NAME_SIZE];
    int               parent_name_length=0;

    int               parent_count=0;
    int               child_count=0;

    evms_drivelink_metadata_t  *metadata;

    int               link_count;
    int               i;
    int               rc;
    uint              count;
    int               index;
    BOOLEAN           need_table;

    BOOLEAN           drive_link_rescued=FALSE;



    LOGENTRY();

    //
    // Make sure we have some storage objects in the dlist to inspect
    rc = GetListSize( child_object_list, &count );
    if (rc || count==0) {
        LOG_ERROR("error, no objects in child_object_list or bad GetListSize call\n");
        LOGEXIT();
        return NULL;
    }

    //
    //  Malloc a vector needed for building a list of parent serial numbers
    //
    parents = (u_int32_t *) calloc(1, sizeof(u_int32_t)*count );
    if (parents==NULL) {
        LOGEXIT();
        return NULL;
    }

    // DISCOVER DRIVELINKS ...
    //
    // Identify all drive link parents by walking the list of storage
    // objects we were passed, building a list of unique drivelink
    // parent serial numbers that child drivelink objects are reporting.
    //
    rc = GoToStartOfList( child_object_list );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( child_object_list, sizeof(storage_object_t), EVMS_OBJECT_TAG, NULL, TRUE,(ADDRESS *) &obj );
        if (rc == DLIST_SUCCESS ) {

            do {
               if (obj) {

                 // get childs meta data
                 rc = get_metadata(obj, &metadata);
                 if (rc) {
                     LOG_DEBUG("failed reading childs metadata\n");
                     LOGEXIT();
                     return NULL;
                 }

                 for(i=0; i<count; i++) {

                     if ( *(parents+i) == 0 ) {
                         LOG_DEBUG("found new parent, parent_serial_number = %d\n", metadata->parent_serial_number);
                         *(parents+i) = metadata->parent_serial_number;
                         links[i]   = metadata->child_count;
                         ++parent_count;
                         break;
                     }
                     else if ( *(parents+i) == metadata->parent_serial_number ) {
                         break;
                     }

                 }

                 rc = GetNextObject( child_object_list, sizeof(storage_object_t), EVMS_OBJECT_TAG, (ADDRESS *) &obj );
             }

            } while ( rc == DLIST_SUCCESS );
        }
    }


    // DISCOVER A COMPLETE DRIVELINK ...
    //
    // now, loop through all the parents we found and find one that has
    // all the child objects in the DLIST that are needed to complete the drivelink.
    //
    for (i=0; i<parent_count; i++) {

        LOG_DEBUG("\tTesting parent: SN= %d LinkCount= %d\n", parents[i], links[i]);

        // zero count of discovered children
        child_count = 0;

        // get number of child objects needed for this drive link
        link_count  = links[i];

        // clear the ordering tables for testing this drive link
        memset(&dup_ordering_table[0], 0, sizeof(evms_dl_ordering_table_entry_t)*link_count);
        memset(&ordering_table[0],     0, sizeof(evms_dl_ordering_table_entry_t)*link_count);

        // set flag to tell us to get a copy of the ordering table
        need_table = TRUE;

        // find children reporting to belong to this drive link
        rc = GoToStartOfList( child_object_list );
        if (rc == DLIST_SUCCESS) {

            rc = GetObject( child_object_list, sizeof(storage_object_t), EVMS_OBJECT_TAG, NULL, TRUE,(ADDRESS *) &obj );
            if (rc == DLIST_SUCCESS ) {

                do {

                    // get childs metadata
                    rc = get_metadata(obj, &metadata);
                    if (rc) {
                        LOG_ERROR("error, failed reading metadata for child %s\n",obj->name);
                        continue;  // skip to other child objects
                    }

                    LOG_DEBUG("\tTesting child: ChildName=%s  ParentSerial=%d  ChildSerial= %d\n",
                                 obj->name, metadata->parent_serial_number, metadata->child_serial_number );

                    // test if child says it belongs to this drivelink
                    if ( *(parents+i) == metadata->parent_serial_number ) {

                        LOG_DEBUG("\tChild reports that it belongs to this parent\n");

                        // get copy of ordering table for this link if we dont have one yet.
                        if ( need_table == TRUE ) {

                            // validate table as best as we can ... no holes and no duplicate serial numbers
                            if ( ordering_table_ok( &metadata->ordering_table[0], links[i]) == FALSE ) {
                                LOG_ERROR("error, child %s has bad ordering table\n", obj->name);
                                continue;  // skip to other child objects
                            }
                            else {

                                // build working copy of ordering table
                                memcpy( &ordering_table[0],
                                        &metadata->ordering_table[0],
                                        sizeof(evms_dl_ordering_table_entry_t)*links[i] );

                                // get parents object name
                                strncpy(parent_object_name,
                                        obj->feature_header->object_name,
                                        EVMS_VOLUME_NAME_SIZE);

                                parent_name_length = strlen(parent_object_name);

                                need_table = FALSE;

                            }
                        }


                        /*
                         *  TEST CHILD OBJECT TO ELIMINATE INCOMPLETE OR BROKEN DRIVELINKS
                         */
                        if (rc == DLIST_SUCCESS) {

                            /*
                             *  TEST 1: children must all report the same number of links in the drivelink
                             */
                            if ( metadata->child_count != links[i]) {
                                LOG_ERROR("error, drive link child reports diff number of links\n");
                                rc = EINVAL;
                            }

                            /*
                             *  TEST 2: all children must have matching ordering tables
                             */
                            else if ( memcmp( &ordering_table[0],
                                              &metadata->ordering_table[0],
                                              (sizeof(evms_dl_ordering_table_entry_t)*links[i]) ) !=0 ) {
                                LOG_ERROR("error, drive link child has diff ordering table for drive link\n");
                                rc = EINVAL;
                            }

                            /*
                             *  TEST 3: a child reporting to belong to a drivelink parent must not be dup of
                             *          one we already added to the table.
                             */
                            else if ( isa_dup_child_serial_number( &dup_ordering_table[0],
                                                                   metadata->child_serial_number,
                                                                   links[i] ) == TRUE ) {
                                LOG_ERROR("error, child serial number is a duplicate of another found in the ordering table\n");
                                rc = EINVAL;
                            }

                            /*
                             *  TEST 4:  all children must agree on the parent drivelink object name
                             */
                            else if ( strncmp( parent_object_name,
                                               obj->feature_header->object_name,
                                               parent_name_length )!=0) {
                                LOG_ERROR("error, child doesnt have same parent storage object name\n");
                                rc = EINVAL;
                            }

                            /*
                             *  TEST 5:  a childs serial number must actually occur in the ordering table
                             */
                            else if (get_drivelink_index_by_sn( &metadata->ordering_table[0],
                                                                metadata->child_serial_number,
                                                                metadata->child_count ) < 0 ) {
                                LOG_ERROR("error, child serial number not found in link table\n");
                                rc = EINVAL;
                            }

                            /*
                             *  SUCCESS ... add this child to the ordering table being built.
                             */
                            else {

                                index = get_drivelink_index_by_sn( metadata->ordering_table,
                                                                   metadata->child_serial_number,
                                                                   metadata->child_count );
                                ++child_count;

                                (dup_ordering_table+index)->child_serial_number = metadata->child_serial_number;
                            }
                        }

                    }

                    if (rc == DLIST_SUCCESS) {
                        rc = GetNextObject( child_object_list, sizeof(storage_object_t), EVMS_OBJECT_TAG, (void **) &obj );
                    }

                } while ( rc == DLIST_SUCCESS );
            }
        }

        // check for errors with child objects
        // ... dup child serial numbers
        // ... children disagree on the ordering of dlink child objects
        // ... children report differnt number of links in drive link object
        if ( rc == EINVAL ) {
            LOG_DEBUG("errors looking for children, looping to next parent\n");
            continue;         // skip this parent
        }

        // check if we found just the right number of child objects and
        // attempt a rescue if we can't find all the child objects.
        if (child_count == link_count) {
            LOG_DEBUG("found all drivelink child objects\n");
        }
        else if (child_count < link_count){

            LOG_DEBUG("wrong number of children found, found %d children while expecting %d children\n", child_count, link_count);
            LOG_DEBUG("will try and rescue this drivelink\n");

            rc = rescue_drive_link( *(parents+i),         // parent sn
                                    parent_object_name,   // parent name
                                    link_count,           // link count
                                    &ordering_table[0],   // ordering table
                                    child_object_list );  // dlist of child storage objects

            if (rc == 0) {
                POPUP_MSG(NULL,NULL,
                          "\nWarning, Drivelink %s is missing child objects but was recovered for READ_ONLY access.\n"
                          "However, any i/o to a missing child object will produce an i/o error.\n"
                          "You should determine the cause of the missing child objects and take corrective action.\n",
                          parent_object_name);
                drive_link_rescued = TRUE;
            }
            else {
                POPUP_MSG(NULL,NULL,
                          "\nError, Drivelink %s is missing child objects and cannot be recovered.\n"
                          "Determine the cause of the missing child objects and take corrective action\n",
                          parent_object_name);
            }

        }
        else {
            LOG_DEBUG("wrong number of children found, found %d children while expecting %d children\n", child_count, link_count);
            LOG_DEBUG("skipping discovery for this drivelink\n");
            continue;
        }

        LOG_DEBUG("All child objects exist for this parent, building parent object\n");

        // Ok, seems fine so build and return a drive link object
        dlink = malloc_drive_link_object( *(parents+i) );
        if ( dlink ) {

           dlink_pdata = (Drive_Link_Private_Data *)dlink->private_data;

           dlink_pdata->parent_serial_number = *(parents+i);
           dlink_pdata->drive_link_count     = links[i];

           // do we need to mark the aggregate read_only ?
           if ( drive_link_rescued == TRUE ){
               dlink->flags |= SOFLAG_READ_ONLY;
           }

           // ordering table
           memcpy( &dlink_pdata->ordering_table[0],
                   &ordering_table[0],
                   sizeof(evms_dl_ordering_table_entry_t)*links[i] );

           // drive link storage object name
           strncpy( dlink->name, parent_object_name, EVMS_VOLUME_NAME_SIZE);

           // register drive link object name
           rc = DLEngFncs->register_name( dlink->name );
           if (rc) {
               free_drive_link_object(dlink);
               LOG_ERROR("failed to register new parent drivelink storage object name\n");
               dlink = NULL;
           }

           // register drive link parent serial number
           rc = register_parent_serial_number( *(parents+i) );
           if (rc) {
               free_drive_link_object(dlink);
               LOG_ERROR("failed to register new drivelink parent serial number\n");
               dlink = NULL;
           }
        }
        else {
            LOG_DEBUG("\tfailed to malloc a parent drivelink storage object\n");
            dlink = NULL;
        }

        break;
    }

    if (parents) free(parents);

    LOGEXIT();
    return dlink;
}



/*
 *  Each time called, we add the specified child storage object to
 *  the parent's drive link storage object.
 *
 *  This routine is called during feature discovery with objects
 *  that have feature headers and feature data.
 *
 *  Returns: TRUE if we consumed the drive link child object by
 *           adding it to the parents DLIST and want it removed
 *           from its original child DLIST.
 */
static BOOLEAN Build_Drive_Link_Object( void     *Object,
                                        TAG       ObjectTag,
                                        uint      ObjectSize,
                                        void     *ObjectHandle,
                                        void     *Parameters,
                                        BOOLEAN  *FreeMemory,
                                        uint     *Error )
{
    storage_object_t        *child            = (storage_object_t *) Object;
    storage_object_t        *parent           = (storage_object_t *) Parameters;
    Drive_Link_Private_Data *pdata;
    u_int32_t                psn = 0;
    BOOLEAN                  rc;
    evms_drivelink_metadata_t *metadata;


    LOGENTRY();

    LOG_DEBUG("child name=%s\n", child->name );

    *FreeMemory = FALSE;           /* tells dlist not to free any memory */
    *Error      = DLIST_SUCCESS;   /* tells dlist we were successful     */
    rc          = FALSE;           /* tells dlist not to prune item from list */

    // get parent info
    if (parent) {

        pdata = (Drive_Link_Private_Data *)parent->private_data;
        psn   = pdata->parent_serial_number;

    }
    else {
        LOG_ERROR("parent storage object ptr was NULL\n");
        *Error = EINVAL;
        LOGEXIT();
        return rc;
    }

    // get child info
    *Error = get_metadata(child, &metadata);
    if (*Error) {
        LOG_ERROR("get child metadata failed\n");
        LOGEXIT();
        return rc;
    }


    // check if child belongs to parent drive link object and consume it
    if ( metadata->parent_serial_number == psn ) {

        LOG_DEBUG("adding child to drive link aggregate\n");

        // consume this drive link storage object, adding it to the
        // aggregating storage object.
        *Error = consume_drive_link_child( parent, child, metadata->child_serial_number, child->feature_header );
        if (*Error == DLIST_SUCCESS) {
            rc = TRUE;   // yes ... dlist can prune object from list
        }
        else {
            LOG_ERROR("failed to consume child storage object\n");
        }

    }

    LOGEXIT();
    return rc;
}



/*
 *  Called during feature discovery to validate a storage object. Meaning
 *  that the object is owned by the drive link feature and that the object
 *  appears Ok.  If Ok, the object will be added to the valid_object DLIST
 *  we are passed.
 *
 *  Returns: TRUE if object is Ok and can be removed from its original child
 *           object DLIST.
 *
 */
static BOOLEAN Validate_Drive_Link_Object( void     *Object,
                                           TAG       ObjectTag,
                                           uint      ObjectSize,
                                           void     *ObjectHandle,
                                           void     *Parameters,
                                           BOOLEAN  *FreeMemory,
                                           uint     *Error )
{
    storage_object_t          * obj                = (storage_object_t *) Object;
    dlist_t                     valid_object_list  = (dlist_t) Parameters;

    evms_drivelink_metadata_t * metadata=NULL;

    BOOLEAN    rc;
    void *     tmp;



    LOGENTRY();
    if (obj->name) LOG_DEBUG("\tValidating storage object, Name=%s\n", obj->name );


    *FreeMemory = FALSE;           /* tells dlist not to free any memory */
    *Error      = DLIST_SUCCESS;   /* tells dlist we were successful     */
    rc          = FALSE;           /* tells dlist not to prune item from list */


    // feature header ptr could be NULL
    if (obj->feature_header) {

        // validate amout of feature data against expected amount and read
        // the meta data into a buffer.
        if ( obj->feature_header->feature_data1_size != DRIVELINK_METADATA_SECTOR_COUNT ) {

            LOG_ERROR("calculated feature data size != size reported by feature header\n");
            display_feature_header(obj->feature_header);
            *Error = EINVAL;

        }
        else {

            *Error = get_metadata( obj, &metadata );

        }

        if (*Error == 0) {

            if ( metadata->signature == EVMS_DRIVELINK_SIGNATURE ) {

                LOG_DEBUG("\tfound drive link feature installed on this object\n");

                display_feature_header(obj->feature_header);

                *Error = InsertObject( valid_object_list,
                                       sizeof(storage_object_t),
                                       obj,
                                       EVMS_OBJECT_TAG,
                                       NULL,
                                       InsertAtStart ,
                                       0,
                                       (void**) &tmp );

                if (*Error==DLIST_SUCCESS) {
                    rc = TRUE;   // yes ... dlist can prune object from list
                }

            }
            else {
                LOG_ERROR("error, not our feature signature\n");
            }
        }
        else {
            LOG_ERROR("error, I/O Error reading from child storage object\n");
        }

    }
    else {
        LOG_ERROR("error, child object is missing feature header ptr\n");
        rc = TRUE;
    }


    LOGEXIT();
    return rc;
}




/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

/*
 *  Called by EVMS, after inspecting and validating our plugin record.  This is the final
 *  step when loading a plugin and means we have been validated by EVMS and will be used.
 *  Most important item here is to get the Engine's function table so we can call engine
 *  API(s) later.
 */
static int DL_SetupEVMSPlugin( engine_mode_t        mode,
                               engine_functions_t * engine_functions)
{
    int rc = 0;


    if (engine_functions) {

        engine_functions->write_log_entry(ENTRY_EXIT,DL_PluginRecord_Ptr,"%s: entry\n",__FUNCTION__);

        DLEngFncs  = engine_functions;

        MetaDataList = CreateList();

        if ( MetaDataList == NULL ) {
            rc = ENOMEM;
        }

        LOGEXITRC();
    }

    return rc;
}


static void DL_Cleanup( void )
{

    if (MetaDataList != NULL) DestroyList( &MetaDataList, TRUE );

}




/*
 *  I can allow an object to be a volume if:
 *
 *  - I own the object
 *
 */
static int DL_CanSetVolume( storage_object_t * object, BOOLEAN  flag )
{
    int rc = EINVAL;


    LOGENTRY();

    if ( i_can_modify_object(object)==TRUE ) {


        rc = 0;

    }

    LOGEXITRC();
    return rc;
}






/*
 *  I can delete a drive link storage object if:
 *
 *  - I own the object
 *  - lower layer plugins say Ok to destroy
 */
static int DL_CanDelete( storage_object_t * object)
{
    int    rc = EINVAL;

    LOGENTRY();

    if ( ( i_can_modify_object(object)==TRUE ) ||
         ( isa_missing_child(object) == TRUE) ) {
        rc = 0;
    }

    LOGEXITRC();
    return rc;
}

/*
 *  I can expand a drive link if:
 *
 *  - adding in another object wont exceed MAX links for a drive link
 *
 *  Since I will only expand by adding another child link, then I'll
 *  place my drive link object in the DLIST as an expansion point. Do not
 *  pass this command down the stack because we only expand right now
 *  by adding child links.
 *
 */
static int DL_CanExpand( storage_object_t   *object,             // drivelink parent object
                         sector_count_t     *expand_limit,       // ?
                         dlist_t             expansion_points )  // dlist to place expand object on
{
    int                      rc = EINVAL;
    void                    *handle;
    storage_object_t        *last_child;
    storage_object_t        *x_obj;
    Drive_Link_Private_Data *pdata;
    expand_object_info_t    *expand_object;
    dlist_t                  acceptable_objects;
    sector_count_t           max_expand_size=0;
    uint                     size;
    TAG                      tag;

    LOGENTRY();

    acceptable_objects = CreateList();

    if ( ( acceptable_objects ) &&
         ( expansion_points ) &&
         ( i_can_modify_object(object)==TRUE )&&
         ( isa_RDONLY_drivelink(object)==FALSE) ) {

        pdata = (Drive_Link_Private_Data *)object->private_data;

        if ( (pdata->drive_link_count+1) < EVMS_DRIVELINK_MAX_ENTRIES ) {

            last_child = pdata->drive_link[pdata->drive_links_found-1].object;

            rc = DLEngFncs->get_object_list( 0,
                                             DATA_TYPE,
                                             NULL,
                                             VALID_INPUT_OBJECT,
                                             &acceptable_objects );

            if (rc == DLIST_SUCCESS) {

                rc = GoToStartOfList( acceptable_objects );
                if ( rc==DLIST_SUCCESS ) {

                    rc = BlindGetObject( acceptable_objects, &size, &tag, NULL, FALSE, (void **)&x_obj );
                    while (rc == DLIST_SUCCESS) {

                        // dont reuse the drivelink parent object and always prepare to lay down
                        // 2 copies of our drive link metadata.
                        if ( (x_obj != object)&&
                             (x_obj->size > ((DRIVELINK_METADATA_SECTOR_COUNT*2)+(FEATURE_HEADER_SECTOR_COUNT*2)))) {

                            max_expand_size += x_obj->size - (DRIVELINK_METADATA_SECTOR_COUNT*2) - (FEATURE_HEADER_SECTOR_COUNT*2);

                        }

                        rc = NextItem(acceptable_objects);
                        if (rc == DLIST_SUCCESS) {
                            rc = BlindGetObject( acceptable_objects, &size, &tag, NULL, FALSE, (void **)&x_obj );
                        }
                    }

                    rc = 0;
                }

                if (max_expand_size) {

                    expand_object = (expand_object_info_t *) DLEngFncs->engine_alloc( sizeof(expand_object_info_t) );
                    if (expand_object) {

                        expand_object->object          = object;
                        expand_object->max_expand_size = max_expand_size;

                        rc = InsertObject ( (dlist_t)          expansion_points,
                                            (int)              sizeof(expand_object_info_t),
                                            (void *)           expand_object,
                                            (TAG)              EXPAND_OBJECT_TAG,
                                            (void *)           NULL,
                                            (Insertion_Modes)  AppendToList,
                                            (BOOLEAN)          TRUE,
                                            (void **)         &handle );
                    }

                }
                else {
                    LOG_DEBUG("max expand size resulted in 0 sectors\n");
                    rc = EINVAL;
                }

            }
            else {
                LOG_DEBUG("engine get_object_list call failed\n");
                rc = ENODATA;
            }

        }
        else {
            LOG_DEBUG("cant expand cuz of too many links\n");
            rc = EPERM;
        }

    }

    LOGEXITRC();
    return rc;
}



/*
 *  I can allow expansion below me if:
 *
 *  - I own the parent object
 *  - there is room for another link in the drive link object
 *
 */
static int DL_CanExpandBy( storage_object_t * object, sector_count_t *size)
{
    int                      rc = EINVAL;
    Drive_Link_Private_Data *pdata;

    LOGENTRY();
    if ( ( i_can_modify_object(object)==TRUE ) &&
         ( isa_RDONLY_drivelink(object)==FALSE) ) {

        pdata = (Drive_Link_Private_Data *)object->private_data;

        // can we add another link ?
        if ( (pdata->drive_link_count+1) <= EVMS_DRIVELINK_MAX_ENTRIES ) {

            rc = 0;

        }

    }

    LOGEXITRC();
    return rc;
}

/*
 *  I can shrink the storage object if ...
 *
 *  - I own it
 *  - There is more than just a single child in the drive link
 *
 *  If so, build an shrink_object_info_t and add it to the
 *  shrink_points list.  The shrink object should be the last
 *  child in the drive link.
 *
 *  Do not pass this command down the plugin stack
 */
static int DL_CanShrink( storage_object_t  *object,
                         sector_count_t    *shrink_limit,
                         dlist_t            shrink_points )
{
    int                        rc = EINVAL;
    Drive_Link_Private_Data   *pdata;
    shrink_object_info_t      *shrink_object;
    void                      *handle;
    storage_object_t          *child;
    struct plugin_functions_s *fncs;

    LOGENTRY();

    LOG_DEBUG("object = %s\n", object->name );

    if ( i_can_modify_object(object)==TRUE ) {

        pdata = (Drive_Link_Private_Data *)object->private_data;

        if ( pdata->drive_link_count > 1 ) {

            if (shrink_points) {

                shrink_object = (shrink_object_info_t *) DLEngFncs->engine_alloc( sizeof(shrink_object_info_t) );
                if (shrink_object) {

                    // max shrink size is the sum child objects 1-N
                    // this would leave us with a minimum drivelink
                    // consisting of just the 1st child object.

                    shrink_object->object          = object;
                    shrink_object->max_shrink_size = object->size - pdata->drive_link[0].sector_count;

                    rc = InsertObject (  (dlist_t)          shrink_points,
                                         (int)              sizeof(shrink_object_info_t),
                                         (void *)           shrink_object,
                                         (TAG)              SHRINK_OBJECT_TAG,
                                         (void *)           NULL,
                                         (Insertion_Modes)  AppendToList,
                                         (BOOLEAN)          TRUE,
                                         (void **)         &handle );

                    // dont return any DLIST error codes
                    if (rc) rc = EPERM;
                }
                else {
                    LOG_ERROR("error, engine alloc of shrink object failed\n");
                    rc = ENOMEM;
                }

            }
            else {
                LOG_ERROR("error, shrink points is NULL dlist\n");
                rc = EINVAL;
            }
        }
        else if ( pdata->drive_link_count == 1 ) {

             // get last child in the drive link
             child = pdata->drive_link[0].object;

             if (child) {

                // get child plugin function table
                fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;

                // pass cmd down feature stack
                rc = fncs->can_shrink(child, shrink_limit, shrink_points);

             }

        }


    }

    LOGEXITRC();
    return rc;

}

/*
 *  I can shrink the storage object by the specified amount if ...
 *
 *  - I own it
 *  - The drive link child has at least SIZE number of sectors of useable
 *    space that can be freed.
 *
 *  If the drive link child does not have enough useable sectors we will
 *  return an error and report the amount that we could shrink by.
 *
 */
static int DL_CanShrinkBy( storage_object_t * object, sector_count_t  *size)
{
    int                        rc = EINVAL;
    Drive_Link_Private_Data   *pdata;
    sector_count_t             shrink_size=0;
    sector_count_t             min_object_size=0;


    LOGENTRY();

    if ( i_can_modify_object(object) == TRUE ) {

        pdata = (Drive_Link_Private_Data *) object->private_data;

        if (pdata->drive_link_count == 1) {

            // max size we can shrink an individual object is really
            // 8k of data to keep kernel happy
            // sectors for feature headers
            // sectors for 2 copies of metadata

            min_object_size = (8192/EVMS_VSECTOR_SIZE) + (DRIVELINK_METADATA_SECTOR_COUNT*2) + (FEATURE_HEADER_SECTOR_COUNT*2);

            if ( object->size > min_object_size ) {
                shrink_size = object->size - min_object_size;
            }
            else {
                shrink_size = 0;
            }

        }
        else {

            // max shrink size is the sum child objects 1-N
            // this would leave us with a minimum drivelink
            // consisting of just the 1st child object.

            shrink_size = object->size - pdata->drive_link[0].sector_count;
        }


        // test that the child link is sufficiently large to shrink by size sectors
        if ( shrink_size >= *size) {
            rc = 0;
        }
        else {
            *size = shrink_size;
        }

    }

    LOGEXITRC();
    return rc;
}



/*
 *  Called by the engine to discover drive link child objects. The objects
 *  in the DLIST have all been given a once over by the engine and the engine
 *  belives they ALL belong to drive link.
 *
 *  Our discovery process requires 4 steps.
 *
 *  Step 1 - validate the storage objects
 *
 *      - run integrity checks on feature header - crc, signature, etc
 *      - prune good objects from original DLIST, placing them in the
 *        valid_object DLIST.
 *
 *  Step 2 - construct drive link objects
 *
 *      - find a drive link we can construct from the objects found in
 *        the valid_object DLIST.
 *      - malloc a new storage object for the drive link
 *      - consume child objects belonging to this drive link from the
 *        valid_object DLIST.
 *      - place new drive link object in the drive_link DLIST, removing
 *        consumed child objects from the valid_object DLIST.
 *
 *  Step 3 - merge all DLIST objects back into the output_objects DLIST
 *
 *
 *  Upon Return: the original DLIST may contain 3 different objects.
 *
 *  (1) original objects that failed integrity checks
 *  (2) original objects that we could not use yet because we didnt find all
 *      the child objects needed to construct a complete drive link object
 *  (3) newly constructed drive link objects that consumed some of the
 *      original child objects from the DLIST.
 *
 *  Best Case:  we return only drive link storage objects, consuming all the original
 *              child objects.
 *
 *  Worst Case: child objects were invalid due to I/O errors, or bad crc or something
 *              and they are all returned to the engine.
 *
 */
static int DL_Feature_Discovery(  dlist_t input_objects,
                                  dlist_t output_objects,
                                  BOOLEAN FinalCall )
{
    int      rc;
    void    *handle;
    uint     valid_object_count;

    storage_object_t *parent;

    dlist_t  valid_object_list = CreateList();
    dlist_t  drive_link_list   = CreateList();
    dlist_t  ordered_list;

    LOGENTRY();


    if ( ( valid_object_list != NULL ) &&
         ( drive_link_list   != NULL )) {

        // validate storage objects found in our discovery DLIST
        // place all valid drive link child objects in the ... child_object_list
        rc = PruneList( input_objects, Validate_Drive_Link_Object, valid_object_list );

        if (rc == DLIST_SUCCESS) {

            // Identify Drive Link Objects that can be completely built from the
            // list of Valid drive link objects. Then build new parent drive link
            // objects for each Drive Link that can be completed.  Leave drive link
            // objects, belonging to incomplete drive link parents, in the valid
            // object list.
            while ( (parent=find_complete_drive_link(valid_object_list)) !=NULL ) {

                // insert new Drive Link Parent Object into ... drive link object list
                rc = InsertObject( drive_link_list,
                                   sizeof(storage_object_t),
                                   parent,
                                   EVMS_OBJECT_TAG,
                                   NULL,
                                   InsertAtStart ,
                                   0,
                                   (void**) &handle );

                if (rc==DLIST_SUCCESS) {

                    // saved parent in drive_link list ... now consume child link objects
                    // from the list of valid drive link objects.
                    rc = PruneList( valid_object_list, Build_Drive_Link_Object, (ADDRESS) parent );
                    if (rc == 0) {

                        // drive link is complete but now reorder the child objects to
                        // match the link ordering table ... mostly for GUI applications.
                        // failing is not an error ... well just use the unordered list.
                        ordered_list=CreateList();

                        if (ordered_list) {
                            rc = build_ordered_child_object_list( parent, &ordered_list );
                            if (rc ==0) {
                                rc = DeleteAllItems ( parent->child_objects, FALSE );
                                if (rc == 0) {
                                    rc = CopyList(parent->child_objects, ordered_list, AppendToList);
                                }
                            }
                            DestroyList(&ordered_list, FALSE);
                        }

                    }
                    else {
                        CopyList( valid_object_list, parent->child_objects, InsertAtStart );
                    }

                }

                if (rc) {
                    free_drive_link_object( parent );
                    break;
                }

            };


            if (rc == DLIST_SUCCESS) {

                CopyList( output_objects, input_objects, InsertAtStart );
                CopyList( output_objects, valid_object_list, InsertAtStart );
                CopyList( output_objects, drive_link_list, InsertAtStart );

                rc = GetListSize( valid_object_list, &valid_object_count );
                if (rc) {
                    valid_object_count = 0;
                }

                // if any children are left in the valid_drivelink_object list
                // then we have an incomplete object.  Else... RC=0 ... done!
                if (valid_object_count) {
                    rc = EVMS_FEATURE_INCOMPLETE_ERROR;
                }
                else {
                    rc = 0;
                }

            }
            else {
                CopyList( output_objects, input_objects, InsertAtStart );
                CopyList( output_objects, valid_object_list, InsertAtStart );
            }

            DestroyList(&valid_object_list, FALSE);
            DestroyList(&drive_link_list, FALSE);
        }

        // dont need metadata dlist after discovery ... destroy it and free saved metadata buffers
        delete_all_saved_metadata();
        MetaDataList = NULL;

    }
    else {
        LOG_ERROR("error, failed to create Dlist or Temp Space for new drive link objects\n");
        rc = ENOMEM;
    }


    LOGEXITRC();
    return rc;
}

/*
 *  To create a drive link I need:
 *
 *  - a name for the drive link
 *  - a serial number for the parent of this link
 *  - a link count to start with
 */
static void GetCreateOptions( option_array_t * options,  char  * drive_link_name  )
{
    int i;

    LOGENTRY();

    for (i = 0; i < options->count; i++) {

        if (options->option[i].is_number_based) {

            if (options->option[i].number == DL_CREATE_LINKNAME_INDEX ) {
                strncpy( drive_link_name, options->option[i].value.s, EVMS_VOLUME_NAME_SIZE );
            }

        }
        else {

            if (strcmp(options->option[i].name,  DL_CREATE_LINKNAME_NAME ) == 0) {
                strncpy( drive_link_name, options->option[i].value.s, EVMS_VOLUME_NAME_SIZE );
            }

        }

    }

    LOGEXIT();
}


/*
 *  To create a drive link object I check that:
 *
 *  - the request link count doesnt exceed MAX drive link count
 *  - that there are enough storage objects in the DLIST to satisfy the
 *    requested number of links
 *
 *  I then create a new drive link storage object and consume child
 *  objects found in the objects DLIST till we reach the requested
 *  link count.
 *
 *  Returns: the address of the new drive link object if successful
 *
 */
static int DL_Create( dlist_t              input_objects,
                      option_array_t     * options,
                      dlist_t              output_objects )
{
    int                        rc=0;
    storage_object_t          *parent;
    char                       drive_link_name[EVMS_VOLUME_NAME_SIZE+1]="";
    u_int32_t                  parent_serial_number = 0;
    int                        drive_link_count=0;
    Drive_Link_Private_Data  * pdata=NULL;
    void                     * handle=NULL;


    LOGENTRY();

    // collect create drivelink info
    GetCreateOptions( options, drive_link_name );
    GetListSize( input_objects, &drive_link_count );

    // test create parms
    if ( (drive_link_count <= 0) ||
         (drive_link_count > EVMS_DRIVELINK_MAX_ENTRIES) ||
         (strlen(drive_link_name)==0)) {
        LOG_ERROR("error, invalid drive_link count or object name==NULL\n");
        LOGEXIT();
        return EINVAL;
    }

    // allocate a new drive link storage object
    parent = malloc_drive_link_object( parent_serial_number );

    if ( parent ) {

        pdata = (Drive_Link_Private_Data *) parent->private_data;

        // drive link storage object name
        strncpy( parent->name, drive_link_name, EVMS_VOLUME_NAME_SIZE );

        // no links yet
        pdata->drive_link_count  = 0;

        // geta drive link parent serial number
        pdata->parent_serial_number = gen_parent_serial_number( (u_int32_t) parent );
        if (pdata->parent_serial_number == BAD_SERIAL_NUMBER) {
            rc = ENOMEM;
            free_drive_link_object( parent );
        }

        // register parent drivelink object name
        if (rc == DLIST_SUCCESS) {
            if ( DLEngFncs->register_name( parent->name ) ) {
                free_drive_link_object( parent );
                rc = EINVAL;
            }
        }

        // Now, consume objects from the engine DLIST
        if (rc==DLIST_SUCCESS) {

            rc = Expand_Drive_Link_Object( parent, input_objects );     // , drive_link_count );

            if ( ( rc == DLIST_SUCCESS ) &&                          // did expand succeed ...
                 ( pdata->drive_link_count == drive_link_count ) &&  // did we wind up with the expected link count
                 ( isa_complete_aggregate(parent) == TRUE ) ) {      // and does the ordering table seem Ok ...

                rc = InsertObject( output_objects,
                                   sizeof(storage_object_t),
                                   parent,
                                   EVMS_OBJECT_TAG,
                                   NULL,
                                   InsertAtStart ,
                                   FALSE,
                                   (void**) &handle );
            }
            else {
                rc = EINVAL;
            }

            if ( rc==0 ) {
                parent->flags |= SOFLAG_DIRTY;     // mark object dirty so it can be committed
            }
            else {
                DLEngFncs->unregister_name( parent->name );
                free_drive_link_object( parent );  // free up drive link object memory
            }

        }

    }
    else {
        rc = ENOMEM;
    }

    LOGEXITRC();
    return rc;
}


/*
 *  This is a non-destructive remove.  Essentially, we do all the same work
 *  as a destroy only we dont tell lower features to also destroy.
 *
 *  - Remove the feature from each child object by blasting the feature header sector
 *  - Free any privately allocated data.
 *  - Remove the parent pointer from each child object
 *  - Put child onto the child_objects DLIST provided in the second parameter.
 *
 *  Returns: RC=0 if successful
 *
 */
static int DL_Delete( storage_object_t * parent, dlist_t  child_objects )
{

    int     rc=0;
    int     i;
    void    *handle;
    Drive_Link_Private_Data   *pdata;
    storage_object_t          *child;
    struct plugin_functions_s *fncs;
    sector_count_t             kill_sector_count=0;

    LOGENTRY();

    if ( i_can_modify_object(parent) == TRUE ) {

        pdata = (Drive_Link_Private_Data *) parent->private_data;

        for (i=0; i<pdata->drive_link_count && rc==0; i++ ) {

            child = pdata->drive_link[i].object;

            if (child) {

                // zap feature headers ... IF ... not a missing child object
                if ( pdata->drive_link[i].flags & DL_FLAG_MISSING) {

                    if (child->feature_header) free(child->feature_header);
                    child->feature_header = NULL;
                    DLEngFncs->free_evms_object(child);
                    rc = 0;
                }
                else {
                    // zap our metadata and feature headers
                    fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;

                    kill_sector_count = (DRIVELINK_METADATA_SECTOR_COUNT*2) + (FEATURE_HEADER_SECTOR_COUNT*2);

                    rc = fncs->add_sectors_to_kill_list( child,                           // child object
                                                         child->size - kill_sector_count, // lba
                                                         kill_sector_count );             // sector count

                    // record the error but dont let it fail the delete
                    if (rc) {
                        LOG_ERROR("error, add kill sectors call failed\n");
                    }

                    DeleteObject(child->parent_objects, parent);

                    rc = InsertObject(  child_objects,
                                        sizeof(storage_object_t),
                                        child,
                                        EVMS_OBJECT_TAG,
                                        NULL,
                                        InsertAtStart ,
                                        FALSE,
                                        (void**) &handle );

                    if (rc) {
                        LOG_ERROR("error, failed to insert object into engine child object dlist\n");
                        rc = EPERM;
                    }

                }

            }
            else {
                rc = EINVAL;
            }

        }

        // free drive link object if we successfully freed up child objects.
        if (rc==0) {
            free_drive_link_object( parent );
        }

    }
    else {
        rc = EINVAL;
    }


    LOGEXITRC();
    return rc;
}


/*
 *  A drive link is expanded by adding all the objects in the
 *  objects DLIST to the end of an existing drive link.
 *
 *  I check that:
 *
 *  - I own the object
 *  - there are objects in the DLIST we can add to the drive link
 *  - adding the objects wont exceed the capacity of a drive link
 *
 *  I expand the drive link by consuming the child objects from the
 *  objects DLIST and adding them to the drive link object. They are
 *  removed from the objects DLIST in no particular order but are
 *  added at the end of the drive link object ... expanding by adding
 *  sectors at the end of the drive link.
 *
 *  Returns: RC=0 if we successfully expanded the drive link object
 *           by consuming all child objects from the DLIST.
 *
 */
static int DL_Expand( storage_object_t * parent,
                      storage_object_t * expand_object,
                      dlist_t            objects,
                      option_array_t   * options )
{
    int                        rc = EINVAL;
    Drive_Link_Private_Data   *pdata;
    uint                       expand_link_count;
    int                        drive_link_count;


    LOGENTRY();

    if ( ( parent )&&
         ( expand_object )&&
         ( parent == expand_object ) &&  // we dont allow child objects to expand ... yet
         ( objects ) ) {

        // get drive link private data
        pdata = (Drive_Link_Private_Data *) parent->private_data;

        // get count of links we are adding
        rc = GetListSize(objects, &expand_link_count);
        if (rc) {
            expand_link_count = 0;
        }

        // get new drive link count
        drive_link_count = pdata->drive_link_count + expand_link_count;

        // test that we have links to add ... and not too many of them
        if ( ( expand_link_count > 0 ) &&
             ( drive_link_count  <= EVMS_DRIVELINK_MAX_ENTRIES ) ) {

            rc = Expand_Drive_Link_Object( parent, objects );        //  , expand_link_count );

            if ( ( rc == DLIST_SUCCESS ) &&                          // did expand succeed ...
                 ( pdata->drive_link_count == drive_link_count ) &&  // did we wind up with the expected link count
                 ( isa_complete_aggregate(parent) == TRUE ) ) {      // and does the ordering table seem Ok ...

                parent->flags |= SOFLAG_DIRTY;             // mark object dirty so it can be committed

            }
            else {
                if (rc == DLIST_SUCCESS) rc = EINVAL;
            }

        }

    }


    LOGEXITRC();
    return rc;
}



/*
 *  Called by the engine to shrink a drive link object by removing child
 *  objects from the parent storage object and reflecting this in the
 *  drive link private data.
 *
 *  I check that:
 *
 *  - I own the object
 *  - Only 1 shrink point is specified
 *
 *  I shrink by removing the last child in the drivelink and placing it
 *  in the objects dlist -OR- if there is only 1 child in the link then
 *  I pass the shrink command down to the child object and see if it can
 *  do the shrink.
 *
 *
 *  Returns:  RC=0 if the drive link has been shrunk by removing the
 *            specified objects from it.
 */

static int DL_Shrink( storage_object_t * parent,
                      storage_object_t * shrink_object,
                      dlist_t            objects,
                      option_array_t   * options )
{
    int                        rc = EINVAL;
    storage_object_t          *child;
    Drive_Link_Private_Data   *pdata=NULL;
    BOOLEAN                    found_good_object=FALSE;
    uint                       size;
    TAG                        tag;
    int                        i;
    int                        search_starting_index=0;
    uint                       selected_objects_count;
    struct plugin_functions_s *fncs;
    sector_count_t             child_useable_size;
    sector_count_t             padding_sectors;
    lba_t                      lsn;

    LOGENTRY();
    LOG_DEBUG("parent = %s\n", parent->name );
    LOG_DEBUG("shrink object = %s\n", shrink_object->name );

    if ( i_can_modify_object( parent ) == TRUE ) {

        pdata = ( Drive_Link_Private_Data * ) parent->private_data;

        // check if we are the shrink point ...
        if ( parent != shrink_object ) {

            i = pdata->drive_link_count-1;

            child = pdata->drive_link[i].object;

            fncs = (struct plugin_functions_s *) child->plugin->functions.plugin;

            rc = fncs->shrink(child, shrink_object, objects, options );
            if (rc == 0) {

                child_useable_size  = child->size - (DRIVELINK_METADATA_SECTOR_COUNT*2) - (FEATURE_HEADER_SECTOR_COUNT*2);

                // drive link child objects must be sized to fall
                // on an 8k boundary ... kernel requirement so
                // individual i/o requests wont span links.
                padding_sectors     = child_useable_size % 16;

                child_useable_size -= padding_sectors;

                pdata->drive_link[i].sector_count       = child_useable_size;
                pdata->drive_link[i].padding            = padding_sectors;

                // recalc the drive link address space
                lsn          = 0;
                parent->size = 0;

                for (i=0; i<pdata->drive_link_count; i++) {

                    parent->size += pdata->drive_link[i].sector_count;
                    pdata->drive_link[i].start_lsn = lsn;
                    pdata->drive_link[i].end_lsn   = lsn + pdata->drive_link[i].sector_count - 1;

                    lsn = pdata->drive_link[i].end_lsn + 1;

                }

                test_and_set_RDONLY_flag( parent ); // update flags

                parent->flags |= SOFLAG_DIRTY;      // mark it dirty now
            }

            LOGEXITRC();
            return rc;
        }

        // we are the shrink point ... init for shrink work
        rc = GetListSize(objects, &selected_objects_count);
        if (rc || selected_objects_count==0) {
            rc = EINVAL;
            LOG_ERROR("error, GetListSize error or no selected shrink objects\n");
            LOGEXITRC();
            return rc;
        }

        // make sure we were not called with too many objects to
        // delete from the drivelink. we can delete all child objects
        // except for the very 1st one in the link.
        if (pdata->drive_link_count >= selected_objects_count) {

            search_starting_index = pdata->drive_link_count - selected_objects_count;

            rc = GoToStartOfList( objects );
        }
        else {
            LOG_ERROR("error, too many objects selected for shrinking this drivelink\n");
            rc = EINVAL;
        }


        // make sure all the selected shrink child objects are at the end of the
        // drive link. we cant remove objects from the middle or front of the
        // aggregating parent object.
        while (rc == DLIST_SUCCESS) {

            rc = BlindGetObject( objects, &size, &tag, NULL, FALSE, (void **)&child );

            if ( rc == DLIST_SUCCESS ) {

                found_good_object = FALSE;

                for (i=search_starting_index; i<pdata->drive_link_count; i++) {
                    if (child == pdata->drive_link[i].object) {
                        found_good_object = TRUE;
                        break;
                    }
                }

                if (found_good_object == FALSE) {
                    LOG_ERROR("error, declining object because it is in the middle/front of the drivelink, object name= %s\n", child->name);
                    rc = EINVAL;
                }
                else {
                    rc = NextItem(objects);
                }

            }

        }


        // if objects are all good then shrink parent.
        if ( ( rc != EINVAL ) &&
             ( selected_objects_count ) ) {

            for (i=0,rc=0; (i<selected_objects_count)&&(rc==0); i++) {

                child = get_last_drivelink_child( parent );

                if ( child  ) {

                    rc = Shrink_Drive_Link_Object( parent, child );

                    // if this was a missing child object then we dont
                    // need to keep it around any longer.
                    if ( ( rc == 0) &&
                         ( isa_missing_child(child) == TRUE )) {

                         // remove a missing child object from dlist of objects
                         // we removed from the drivelink because we are about
                         // to free the storage object and dont want the
                         // engine to get its hands on it later and trap.
                         DeleteObject(objects, child);

                         if (child->feature_header) free(child->feature_header);
                         child->feature_header = NULL;
                         DLEngFncs->free_evms_object(child);

                    }


                }
                else {
                    LOG_ERROR("error, ran out of drive link child objects while shrinking\n");
                    rc = ENODEV;
                }

            }


            if ( rc==0 ) {

                 test_and_set_RDONLY_flag( parent ); // update flags

                 parent->flags |= SOFLAG_DIRTY;      // mark it dirty now

            }

        }

    }


    LOGEXITRC();
    return rc;
}


/*
 *  Passes the API call down to the appropriate drive link child object.
 *
 *  I check that:
 *
 *  - I own the object
 *  - the logical sectors are found in the drive link
 *
 *  I do it by walking the drive link and determining the child storage
 *  object that contains the specified sectors and then forwarding the
 *  API call to that object.
 *
 *  Returns: RC=0 if sectors were added to the kill list successfully
 *
 */
static int DL_AddSectorsToKillList( storage_object_t * object,
                                    lsn_t              lsn,
                                    sector_count_t     count)
{
    int rc = EINVAL;
    int i;
    lsn_t                      io_lsn = lsn;
    lsn_t                      link_lsn;
    sector_count_t             io_sector_count;
    sector_count_t             sectors_left_to_write = count;
    sector_count_t             max_sector_count;
    Drive_Link_Private_Data   *pdata;
    storage_object_t          *child;
    struct plugin_functions_s *fncs;


    LOGENTRY();

    if ( ( i_can_modify_object(object)==TRUE) &&
         ( lsn+count <= object->size ) ) {

        pdata = (Drive_Link_Private_Data *) object->private_data;

        for (i=0; i < pdata->drive_link_count; i++) {

            if ( pdata->drive_link[i].end_lsn >= io_lsn ) {

                child = pdata->drive_link[i].object;

                max_sector_count = pdata->drive_link[i].end_lsn - io_lsn + 1;

                if ( max_sector_count >= sectors_left_to_write ) {
                    io_sector_count = sectors_left_to_write;
                }
                else {
                    io_sector_count = max_sector_count;
                }

                link_lsn = io_lsn - pdata->drive_link[i].start_lsn;

                fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;
                rc = fncs->add_sectors_to_kill_list( child, link_lsn, io_sector_count);

                io_lsn                += io_sector_count;
                sectors_left_to_write -= io_sector_count;

                if ((sectors_left_to_write == 0) || (rc)) break;
            }
        }
    }

    LOGEXITRC();
    return rc;

}

/*
 *  Called to commit changes to the drive link storage object.
 *
 *  This is done by committing each of the child storage objects
 *  that are aggregated by the single drive link storage object.
 *
 *  I check that:
 *
 *  - I own the object
 *
 *  I do it by walking the drive link and calling commit for every
 *  object in the drive link array.
 *
 *  Returns: RC=0 if all objects commit feature data successfully
 *
 */
static int DL_CommitChanges( storage_object_t * object, uint commit_phase )
{
    int rc = EINVAL;
    int i;
    Drive_Link_Private_Data   *pdata;

    LOGENTRY();
    LOG_DEBUG("object: name= %s  flags= %d  commit_phase= %d\n", object->name, object->flags, commit_phase );

    if ( i_can_modify_object(object)==TRUE ) {

        if ( ( commit_phase == 1  || commit_phase == 2 )  &&
             ( object->flags & SOFLAG_DIRTY )) {

            pdata = (Drive_Link_Private_Data *)object->private_data;

            if (pdata) {

                test_and_set_RDONLY_flag( object );

                if ( isa_RDONLY_drivelink(object) == FALSE ) {

                    for (i=0,rc=0; (i<pdata->drive_link_count)&&(rc==0); i++) {

                        rc = Commit_DriveLink_Object( object, &pdata->drive_link[i], commit_phase );

                        if (( rc==0 )&&(commit_phase==2)) {
                            object->flags &= ~SOFLAG_DIRTY;    // mark child object clean
                        }

                    }

                }
                else {
                    POPUP_MSG(NULL,NULL,
                    "\nError, Drivelink %s is missing child objects and was recovered for READ_ONLY access.\n"
                    "You made some changes to it but it still appears to have some missing child objects and changes cannot be committed.\n"
                    "You should remove all missing child objects from the drivelink before committing changes.\nYou might try shrinking the drivelink to remove missing child objects.\n"
                    "You could also remove the topmost drivelink object and build a new drivelink.\n",
                    object->name);
                }

            }

        }
        else {
            rc = 0;
        }

    }

    LOGEXITRC();
    return rc;
}




/*
 *  Passes the API call down to the appropriate drive link child object.
 *
 *  I check that:
 *
 *  - I own the object
 *  - I have a valid buffer
 *  - the logical sectors are found in the drive link
 *
 *  I do the I/O by walking the drive link and determining the child storage
 *  object that contains the specified sectors and then forwarding the
 *  API call to that object. The I/O may span multiple objects and this is
 *  Ok because the I/O will be broken up and multiple Reads performed as
 *  needed.
 *
 *  Returns: RC=0 if all sectors were read successfully
 *
 */
static int DL_Read( storage_object_t * object,
                    lsn_t              lsn,
                    sector_count_t     count,
                    void             * buffer)
{
    int rc = EINVAL;
    int i;
    char                      *io_buffer_ptr = (char *)buffer;
    lsn_t                      io_lsn = lsn;
    lsn_t                      link_lsn;
    sector_count_t             io_sector_count;
    sector_count_t             sectors_left_to_read = count;
    sector_count_t             max_sector_count;
    Drive_Link_Private_Data   *pdata;
    storage_object_t          *child;


    LOGENTRY();
    LOG_DEBUG("\tname= %s   size = %lld    lsn= %lld    count= %lld\n", object->name, object->size, lsn, count);

    if ( ( buffer ) &&
         ( i_can_modify_object(object)==TRUE) &&
         ( lsn+count <= object->size ) ) {

        pdata = (Drive_Link_Private_Data *) object->private_data;

        rc = 0;
        for (i=0; (i < pdata->drive_link_count) && (rc==0); i++) {

            if ( pdata->drive_link[i].end_lsn >= io_lsn ) {

                LOG_DEBUG("\tlsn is in link %d cux link has end_lsn of %d\n", i, (u_int32_t)pdata->drive_link[i].end_lsn);

                child = pdata->drive_link[i].object;


                max_sector_count = pdata->drive_link[i].end_lsn - io_lsn + 1;

                if ( max_sector_count >= sectors_left_to_read ) {
                    io_sector_count = sectors_left_to_read;
                }
                else {
                    io_sector_count = max_sector_count;
                }

                link_lsn = io_lsn - pdata->drive_link[i].start_lsn;

                // test for missing drive link child in aggregate
                if ( pdata->drive_link[i].flags & DL_FLAG_MISSING ) {
                    rc = EIO;
                }
                else {
                    rc = READ( child,
                               link_lsn,
                               io_sector_count,
                               (void *) io_buffer_ptr );
                }

                io_lsn               += io_sector_count;
                io_buffer_ptr        += io_sector_count*EVMS_VSECTOR_SIZE;
                sectors_left_to_read -= io_sector_count;

                if ((sectors_left_to_read == 0) || (rc)) break;

            }

        }

    }


    LOGEXITRC();
    return rc;
}

/*
 *  Passes the API call down to the appropriate drive link child object.
 *
 *  I check that:
 *
 *  - I own the object
 *  - I have a valid buffer
 *  - the logical sectors are found in the drive link
 *
 *  I do the I/O by walking the drive link and determining the child storage
 *  object that contains the specified sectors and then forwarding the
 *  API call to that object. The I/O may span 2 or more child objects and
 *  this is Ok because the I/O will be broken up into multiple Writes as
 *  needed.
 *
 *  Returns: RC=0 if all sectors were written successfully
 *
 */

static int DL_Write( storage_object_t * object,
                     lsn_t              lsn,
                     sector_count_t     count,
                     void             * buffer)
{
    int rc = EINVAL;
    int i;
    char                      *io_buffer_ptr = (char *)buffer;
    lsn_t                      io_lsn = lsn;
    lsn_t                      link_lsn;
    sector_count_t             io_sector_count;
    sector_count_t             sectors_left_to_write = count;
    sector_count_t             max_sector_count;
    Drive_Link_Private_Data   *pdata;
    storage_object_t          *child;


    LOGENTRY();
    LOG_DEBUG("\tobject->size = %08d  LSN= %d  count= %08d\n", object->size, lsn, count);

    if ( ( buffer ) &&
         ( i_can_modify_object(object)==TRUE) &&
         ( lsn+count <= object->size ) ) {

        pdata = (Drive_Link_Private_Data *) object->private_data;

        // test the READ ONLY flag bit that marks a broken drive link
        if ( isa_RDONLY_drivelink(object) == TRUE ) {
            rc = EROFS;  // only read only error I could find
            LOGEXITRC();
            return rc;
        }

        for (i=0; i < pdata->drive_link_count; i++) {

            if ( pdata->drive_link[i].end_lsn >= io_lsn ) {

                LOG_DEBUG("\tlsn is in link %d cux link has end_lsn of %d\n", i, (u_int32_t)pdata->drive_link[i].end_lsn);

                child = pdata->drive_link[i].object;

                max_sector_count = pdata->drive_link[i].end_lsn - io_lsn + 1;

                if ( max_sector_count >= sectors_left_to_write ) {
                    io_sector_count = sectors_left_to_write;
                }
                else {
                    io_sector_count = max_sector_count;
                }


                link_lsn = io_lsn - pdata->drive_link[i].start_lsn;

                rc=WRITE( child,
                          link_lsn ,
                          io_sector_count,
                          (void *) io_buffer_ptr );

                io_lsn                += io_sector_count;
                io_buffer_ptr         += io_sector_count*EVMS_VSECTOR_SIZE;
                sectors_left_to_write -= io_sector_count;

                if ((sectors_left_to_write == 0) || (rc)) break;
            }
        }
    }


    LOGEXITRC();
    return rc;

}



/*
 *  I can allow an object to be a volume if:
 *
 *  - I own the object
 *
 */
static void DL_SetVolume( storage_object_t * object, BOOLEAN  flag )
{

    LOGENTRY();
    // Nothing to do yet.
    LOGEXIT();

}



/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                              PLUGIN FUNCTION TABLE                                   +
+                                                                                      +
+--------------------------------------------------------------------------------------*/
static struct plugin_functions_s fft={

    // routines found above
    setup_evms_plugin:                      DL_SetupEVMSPlugin,
    cleanup_evms_plugin:                    DL_Cleanup,
    can_set_volume:                         DL_CanSetVolume,
    can_delete:                             DL_CanDelete,
    can_expand:                             DL_CanExpand,
    can_expand_by:                          DL_CanExpandBy,
    can_shrink:                             DL_CanShrink,
    can_shrink_by:                          DL_CanShrinkBy,
    discover:                               DL_Feature_Discovery,
    create:                                 DL_Create,
    delete:                                 DL_Delete,
    expand:                                 DL_Expand,
    shrink:                                 DL_Shrink,
    add_sectors_to_kill_list:               DL_AddSectorsToKillList,
    commit_changes:                         DL_CommitChanges,
    read:                                   DL_Read,
    write:                                  DL_Write,
    set_volume:                             DL_SetVolume,

    // routines found in dloptions.c
    get_option_count:                       DL_GetOptionCount,
    init_task:                              DL_InitTask,
    set_option:                             DL_SetOption,
    set_objects:                            DL_SetObjects,
    get_info:                               DL_GetInfo,
    get_plugin_info:                        DL_GetPluginInfo
};


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                       BUILD AND EXPORT AN EVMS PLUGIN RECORD                         +
+                                                                                      +
+--------------------------------------------------------------------------------------*/

static plugin_record_t drivelink_plugin_record = {

    id:                               SetPluginID(EVMS_OEM_IBM, EVMS_FEATURE, EVMS_DRIVELINK_FEATURE_ID ),

    version:                          {MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL},

    required_api_version:             {3, 0, 0},

    short_name:                       "DriveLink",
    long_name:                        "Drive Linking Feature",
    oem_name:                         "IBM",

    functions:                        {plugin: &fft},

    container_functions:              NULL

};


// Vector of plugin record ptrs that we export for the EVMS Engine.
plugin_record_t *evms_plugin_records[] = {
    &drivelink_plugin_record,
    NULL
};

