/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: dlrescue.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>
#include <linux/evms/evms_drivelink.h>

#include "drivelink.h"
#include "dloptions.h"
#include "dlsn.h"
#include "dlrescue.h"



/*
 * Called to build a simple feature header for a missing child object.
 */
static int build_missing_feature_header(storage_object_t  *child,
                                        char              *parent_object_name,
                                        u_int32_t          feature_header_flags )
{
    int                    rc;
    evms_feature_header_t *feature_header;

    LOGENTRY();

    feature_header = (evms_feature_header_t *) calloc(1, sizeof(evms_feature_header_t) );
    if (feature_header) {

        child->feature_header = feature_header;

        feature_header->signature   = EVMS_FEATURE_HEADER_SIGNATURE;
        feature_header->feature_id  = SetPluginID(EVMS_OEM_IBM, EVMS_FEATURE, EVMS_DRIVELINK_FEATURE_ID );
        feature_header->flags       = feature_header_flags;
        strncpy( feature_header->object_name,
                 parent_object_name,
                 EVMS_VOLUME_NAME_SIZE );

        rc = 0;

    }
    else {
        rc = ENOMEM;
    }

    LOGEXITRC();
    return rc;
}

/*
 * Called to build a simple metadata struct for a missing child object.
 */
static int build_missing_metadata( storage_object_t                   *child,
                                   u_int32_t                           parent_serial_number,
                                   int                                 link_count,
                                   u_int32_t                           child_serial_number,
                                   evms_dl_ordering_table_entry_t     *ordering_table )
{
    int rc;
    evms_drivelink_metadata_t *metadata=NULL;
    SAVED_METADATA            *mdata = NULL;


    LOGENTRY();

    // test if child already has metadata ...
    if ( get_saved_metadata(child) ) {
        rc = EINVAL;
        LOG_DEBUG("error, new child object should not have metadata but we found some\n");
        LOGEXITRC();
        return rc;
    }

    // create saved metadata
    rc = create_saved_metadata( child );
    if (rc) {
        rc = EINVAL;
        LOG_DEBUG("error, create saved metadata call failed.\n");
        LOGEXITRC();
        return rc;
    }

    // now get ptr to the metadata
    mdata = get_saved_metadata( child );
    if (mdata==NULL) {
        rc = ENOMEM;
        LOG_DEBUG("error, created child object saved metadata object but could not retrieve it.\n");
        LOGEXITRC();
        return rc;
    }

    // malloc the actual metadata buffer
    mdata->metadata = calloc( 1, sizeof(evms_drivelink_metadata_t) );
    if (mdata->metadata == NULL) {
        rc = ENOMEM;
        LOG_DEBUG("error, malloc metadata buffer failed.\n");
        LOGEXITRC();
        return rc;
    }
    else {
        metadata = (evms_drivelink_metadata_t *) mdata->metadata;
    }


    // lastly, initialize the metadata
    metadata->signature            = EVMS_DRIVELINK_SIGNATURE;
    metadata->parent_serial_number = parent_serial_number;
    metadata->child_serial_number  = child_serial_number;
    metadata->child_count          = link_count;

    metadata->version.major        = EVMS_DRIVELINK_VERSION_MAJOR;
    metadata->version.minor        = EVMS_DRIVELINK_VERSION_MINOR;
    metadata->version.patchlevel   = EVMS_DRIVELINK_VERSION_PATCHLEVEL;

    memcpy( &metadata->ordering_table[0],
            ordering_table,
            sizeof(evms_dl_ordering_table_entry_t)*link_count );

    rc = 0;
    LOGEXITRC();
    return rc;
}


/*
 * Called to build a missing child object.
 */
static storage_object_t *  build_missing_child( u_int32_t                           parent_serial_number,
                                                char                               *parent_object_name,
                                                int                                 link_count,
                                                int                                 index,
                                                u_int32_t                           child_serial_number,
                                                evms_dl_ordering_table_entry_t     *ordering_table,
                                                u_int32_t                           feature_header_flags )
 {
     storage_object_t        *child=NULL;
     int                      rc = EINVAL;
     char                     name[EVMS_VOLUME_NAME_SIZE+48];

     LOGENTRY();

     rc = DLEngFncs->allocate_evms_object(NULL, &child);
     if ( rc == 0 ) {

        rc = build_missing_metadata( child,
                                     parent_serial_number,
                                     link_count,
                                     child_serial_number,
                                     ordering_table );

        if (rc == 0) {

            rc = build_missing_feature_header( child, parent_object_name, feature_header_flags );
            if (rc == 0) {

                child->private_data = malloc( sizeof(u_int32_t) );  // private data is just MISSING_CHILD signature
                if (child->private_data) {

                    child->start        = 0;
                    child->size         = (ordering_table+index)->child_vsize;
                    child->plugin       = DL_PluginRecord_Ptr;

                    ((Drive_Link_Private_Data *)child->private_data)->signature = MISSING_CHILD_SIGNATURE;

                    sprintf( name, "%s-missing_child_%d", parent_object_name, index );
                    strncpy( child->name, name, EVMS_VOLUME_NAME_SIZE );

                }
                else {
                    DLEngFncs->free_evms_object(child);
                    child = NULL;
                }

            }
            else {
                DLEngFncs->free_evms_object(child);
                child = NULL;
            }

        }
        else {
            DLEngFncs->free_evms_object(child);
            child = NULL;
        }

     }

     LOGEXIT();
     return child;
 }





int  rescue_drive_link( u_int32_t                         parent_serial_number,
                        char                             *parent_object_name,
                        int                               link_count,
                        evms_dl_ordering_table_entry_t   *ordering_table,
                        dlist_t                           child_object_list )
{
    int                             rc = EINVAL;
    int                             index;
    int                             i;
    storage_object_t               *child;
    evms_dl_ordering_table_entry_t  tmp_ordering_table[EVMS_DRIVELINK_MAX_ENTRIES];
    dlist_t                         missing_child_objects = CreateList();
    evms_drivelink_metadata_t      *metadata;
    void                           *handle;
    u_int32_t                       child_serial_number;
    u_int32_t                       feature_header_flags=0;


    LOGENTRY();

    // is working dlist ok?
    if (missing_child_objects == NULL) {
        rc = ENOMEM;
        LOGEXITRC();
        return rc;
    }

    // clear working ordering table
    memset(&tmp_ordering_table[0], 0, sizeof(evms_dl_ordering_table_entry_t)*EVMS_DRIVELINK_MAX_ENTRIES );

    // find out which child objects are missing
    rc = GoToStartOfList( child_object_list );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( child_object_list, sizeof(storage_object_t), EVMS_OBJECT_TAG, NULL, TRUE,(ADDRESS *) &child );
        while (rc == DLIST_SUCCESS ) {

            // get childs metadata
            rc = get_metadata(child, &metadata);
            if (rc) {
                LOG_DEBUG("error, failed reading child %s metadata\n", child->name );
                LOGEXITRC();
                return rc;
            }

            // pick out child objects belonging to this parent drivelink
            if (metadata->parent_serial_number == parent_serial_number) {

                index = get_drivelink_index_by_sn( ordering_table,
                                                   metadata->child_serial_number,
                                                   link_count);

                // A broken drivelink might be a volume ... it might not.
                // The safe thing to do is to replicate the same flag field
                // across all missing child objects we create.
                if (feature_header_flags == 0) {
                    feature_header_flags = child->feature_header->flags;
                }

                if (index < 0) {
                    LOG_DEBUG("error, search for child %s by SN failed\n", child->name );
                    LOGEXITRC();
                    return rc;
                }

                tmp_ordering_table[index].child_serial_number = metadata->child_serial_number;
            }

            rc = GetNextObject( child_object_list, sizeof(storage_object_t), EVMS_OBJECT_TAG, (ADDRESS *) &child );
        }

    }


    // build a child object for each missing link
    for (i=0, rc=0; (i < link_count) && (rc==0); i++ ) {

        if (tmp_ordering_table[i].child_serial_number == 0) {

            child_serial_number = (ordering_table+i)->child_serial_number;

            child = build_missing_child( parent_serial_number,
                                         parent_object_name,
                                         link_count,
                                         i,
                                         child_serial_number,
                                         ordering_table,
                                         feature_header_flags );
            if (child) {

                rc = InsertObject( missing_child_objects,
                                   sizeof(storage_object_t),
                                   child,
                                   EVMS_OBJECT_TAG,
                                   NULL,
                                   AppendToList,
                                   FALSE,
                                   (void**) &handle );

                if (rc) {
                    if (child->feature_header) free((void *)child->feature_header);
                    child->feature_header = NULL;
                    DLEngFncs->free_evms_object(child);
                }

            }
            else {
                rc = ENOMEM;
            }
        }

    }


    // success ... so transfer new missing child objects to engine discovery dlist
    if (rc == 0) {
        CopyList( child_object_list, missing_child_objects, AppendToList );
    }
    else {  // failed ... so unwind the rescue operation
        int rc2;

        rc2 = GoToStartOfList( missing_child_objects );
        if (rc2 == DLIST_SUCCESS) {

            rc2 = GetObject( missing_child_objects, sizeof(storage_object_t), EVMS_OBJECT_TAG, NULL, TRUE,(ADDRESS *) &child );
            while (rc2 == DLIST_SUCCESS ) {

                if (child->feature_header) free((void *)child->feature_header);
                child->feature_header = NULL;
                DLEngFncs->free_evms_object(child);

                rc2 = GetNextObject( missing_child_objects, sizeof(storage_object_t), EVMS_OBJECT_TAG, (ADDRESS *) &child );
            }

        }

    }

    // free the working dlist and return
    DestroyList(&missing_child_objects, FALSE);
    LOGEXITRC();
    return rc;
}

