/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: solarisX86.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>

#include "ptables.h"
#include "segs.h"
#include "embedded.h"
#include "solarisX86.h"


/*
 * Builds a disk segment matching the solaris partition info
 */
static DISKSEG * build_solaris_segment( LOGICALDISK       *ld,
                                        DISKSEG           *msdos_seg,
                                        u_int32_t          start,
                                        u_int32_t          size,
                                        u_int32_t          sys_id,
                                        u_int32_t          ptable_index,
                                        u_int32_t          minor,
                                        u_int32_t          tag,
                                        u_int32_t          flags,
                                        dlist_t            recovery_list )
{
    DISKSEG           *seg    = NULL;
    SEG_PRIVATE_DATA  *pdata  = NULL;
    void              *handle = NULL;
    int                rc;


    LOGENTRY();

    seg = build_segment_for_embedded_partition( ld, msdos_seg, start, size, sys_id, ptable_index, minor );
    if (seg) {

        pdata = (SEG_PRIVATE_DATA *) seg->private_data;

        pdata->flags = SEG_IS_SOLARIS_X86_PARTITION | SEG_IS_EMBEDDED;

        pdata->tag         = tag;
        pdata->permissions = flags;

        rc = InsertObject ( (dlist_t)          recovery_list,
                            (int)              sizeof(DISKSEG),
                            (void *)           seg,
                            (TAG)              SEGMENT_TAG,
                            (void *)           NULL,
                            (Insertion_Modes)  InsertBefore,
                            (BOOLEAN)          TRUE,
                            (void **)         &handle );

        if (rc != DLIST_SUCCESS) {
            free_disk_segment( seg );
            seg = NULL;
        }

    }


    LOGEXIT();
    return seg;
}



#define SOLARIS_X86_PART_TABLE_SECTOR_OFFSET 1

int do_solaris_x86_partition_discover( LOGICALDISK *ld, Partition_Record *part  )
{
    u_int32_t                   offset = DISK_TO_CPU32( START_LBA(part) );
    struct solaris_x86_vtoc    *v;
    struct solaris_x86_slice   *s;
    char                        data[EVMS_VSECTOR_SIZE];
    int                         i;
    int                         rc = 0;
    struct plugin_functions_s  *dft;
    DISK_PRIVATE_DATA          *disk_pdata;
    DISKSEG                    *seg;
    DISKSEG                    *s_seg;
    int                         s_partition_count=0;
    u_int32_t                   minor;
    dlist_t                     recovery_list;

    LOGENTRY();

    // get disk info
    disk_pdata = get_disk_private_data(ld);
    if (disk_pdata == NULL) {
        LOGEXIT();
        return ENODEV;
    }

     // get disk manager function table
    dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
    if (dft==NULL) {
        LOGEXIT();
        return ENODEV;
    }

    // read Solaris info
    rc = dft->read( ld,
                    DISK_TO_CPU32( START_LBA(part) ) + SOLARIS_X86_PART_TABLE_SECTOR_OFFSET,
                    1,
                    (void *) data );
    if (rc) {
        LOGEXIT();
        return rc;
    }

    // solaris partitions use a type field of 0x82 which
    // is also the same value as a linux swap partition.
    // so, we need to check that were not actually looking
    // at a linux swap partition.
    v = (struct solaris_x86_vtoc *)data;
    if (v->v_sanity != SOLARIS_X86_VTOC_SANE) {
        LOG_DEBUG("vtoc sanity invalid ... not a solaris x86 partition\n");
        LOGEXIT();
        return 0;
    }

    // check version info
    if (v->v_version != 1) {
        LOG_DEBUG("cannot handle solaris vtoc version %d.\n", v->v_version);
        LOGEXIT();
        return 0;
    }

    // create a dlist
    recovery_list = CreateList();
    if (recovery_list == NULL) {
        LOGEXIT();
        return ENOMEM;
    }

    // get evms segment storage object from disk segment list
    // and remove it from the list, making room for partitions.
    seg = get_matching_segment( (dlist_t)        ld->parent_objects,
                                (lba_t)          DISK_TO_CPU32( START_LBA(part) ),
                                (sector_count_t) DISK_TO_CPU32( NR_SECTS(part)  ));

    if (seg) {

        rc = remove_diskseg_from_list( ld->parent_objects, seg );
        if (rc) {
            DestroyList(&recovery_list, FALSE);
            LOGEXIT();
            return rc;
        }

    }
    else {
        DestroyList(&recovery_list, FALSE);
        LOGEXIT();
        return ENODEV;
    }


    // calculate the first minor to be used by UnixWare embedded partitions
    minor = disk_pdata->logical_drive_count + disk_pdata->embedded_partition_count + 5;


    // Solaris Debug Info
    {
    char   volume_name[16];

    strncpy(volume_name, &v->v_volume[0], 8);

    LOG_DEBUG("Solaris X86 Info:\n");
    LOG_DEBUG("     volume:  %s\n", volume_name);
    LOG_DEBUG("     sector size = %d\n", DISK_TO_CPU16(v->v_sectorsz) );
    LOG_DEBUG("     number of solaris partition table entries: %d\n", DISK_TO_CPU16(v->v_nparts) );

    }


    rc = 0;
    for (i=0; i<DISK_TO_CPU16(v->v_nparts); i++) {

        s = &v->v_slice[i];

LOG_DEBUG("  Slice %d: start:%08u)  size:%08u  tag: 0x%04X  flag: 0x%02X\n", i, DISK_TO_CPU32(s->s_start), DISK_TO_CPU32(s->s_size), DISK_TO_CPU16(s->s_tag), s->s_flag );

        if ( (s->s_size == 0) ||
             (s->s_tag  == SOLARISX86_TAG_ENTIRE_DISK ) ){
             continue;
        }

        s_seg = build_solaris_segment( ld,
                                       seg,
                                       DISK_TO_CPU32( s->s_start )+offset,
                                       DISK_TO_CPU32( s->s_size ),
                                       SOLARIS_X86_PARTITION,
                                       i,
                                       minor,
                                       (u_int32_t) DISK_TO_CPU16(s->s_tag),
                                       (u_int32_t) s->s_flag,
                                       recovery_list );

        if (s_seg) {

            if ( insert_diskseg_into_list( ld->parent_objects, s_seg) ){
                ++minor;
                ++s_partition_count;
                ++disk_pdata->embedded_partition_count;
            }
            else {
                rc = DLIST_CORRUPTED;
            }
        }
        else {
            rc = ENOMEM;
        }

        if (rc) break;

    }


    if (rc) {
        LOG_ERROR("error, problems adding solaris partitions for disk %s.", ld->name );
        remove_embedded_partitions_from_disk( ld, recovery_list );
        insert_diskseg_into_list( ld->parent_objects, seg);
        POPUP_MSG(NULL,NULL,"\nAbandoning effort with embedded solaris partitions found in %s\n", seg->name);
        rc = 0;
    }
    else {

        // if we produced -ANY- solaris segments ... then consume the container segment
        if ( s_partition_count > 0 ) {
            diskseg_to_container_segment( seg );
            CopyList(seg->parent_objects, recovery_list, AppendToList);
            LOG_DEBUG("Info, found %d embedded solaris x86 partitions in %s\n", s_partition_count, seg->name);
        }
        else {
            insert_diskseg_into_list( ld->parent_objects, seg);
        }

    }

    DestroyList(&recovery_list, FALSE);
    LOGEXIT();
    return rc;
}


int do_solaris_x86_partition_commit( LOGICALDISK *ld, DISKSEG *seg )
{
    struct solaris_x86_vtoc    *v;
    struct solaris_x86_slice   *s;
    DISKSEG                    *tseg;
    char                        data[EVMS_VSECTOR_SIZE];
    int                         rc = 0;
    struct plugin_functions_s  *dft;
    DISK_PRIVATE_DATA          *disk_pdata;
    SEG_PRIVATE_DATA           *pdata;
    int                         i;

    LOGENTRY();

    // get disk info
    disk_pdata = get_disk_private_data(ld);
    if (disk_pdata == NULL) {
        LOGEXIT();
        return ENODEV;
    }

    // get disk manager function table
    dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
    if (dft==NULL) {
        LOGEXIT();
        return ENODEV;
    }

    // read Solaris info
    rc = dft->read( ld,
                    seg->start + SOLARIS_X86_PART_TABLE_SECTOR_OFFSET,
                    1,
                    (void *) data );
    if (rc) {
        LOGEXIT();
        return rc;
    }

    // solaris partitions use a type field of 0x82 which
    // is also the same value as a linux swap partition.
    // so, we need to check that were not actually looking
    // at a linux swap partition.
    v = (struct solaris_x86_vtoc *) data;
    if (v->v_sanity != SOLARIS_X86_VTOC_SANE) {
        LOGEXIT();
        return 0;
    }

    // check version info
    if (v->v_version != 1) {
        LOGEXIT();
        return 0;
    }

    // walk through the partition table, removing all the data
    // partitions we find.
    for (i=0; i<DISK_TO_CPU16(v->v_nparts); i++) {

         s = &v->v_slice[i];

         if ( (s->s_size != 0) &&
              (s->s_tag  != SOLARISX86_TAG_ENTIRE_DISK ) ){

             memset( s, 0, sizeof(struct solaris_x86_slice) );

         }

    }

    // now walk through the segment list on this drive and find all
    // the solaris partitions that are embedded in this container seg.
    // and add them to the solaris partition table.
    rc = GoToStartOfList( ld->parent_objects );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (void **) &tseg );
        while (rc == DLIST_SUCCESS) {

            pdata = (SEG_PRIVATE_DATA *)tseg->private_data;

            if ( ( pdata->flags & SEG_IS_SOLARIS_X86_PARTITION ) &&
                 ( only_child( tseg ) == seg ) ) {

                s = &v->v_slice[pdata->ptable_index];

                s->s_start = CPU_TO_DISK32( tseg->start - seg->start );
                s->s_size  = CPU_TO_DISK32( tseg->size );
                s->s_tag   = CPU_TO_DISK16( (u_int16_t) pdata->tag );
                s->s_flag  = CPU_TO_DISK16( (u_int16_t) pdata->permissions );

            }

            if (rc == DLIST_SUCCESS) {
                rc = GetNextObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG, (ADDRESS *) &tseg );
            }

        }
    }


    // update Solaris info
    rc = dft->write( ld,
                     seg->start + SOLARIS_X86_PART_TABLE_SECTOR_OFFSET,
                     1,
                     (void *) data );


    LOGEXIT();
    return rc;
}

