/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: display.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>

#include "ptables.h"
#include "display.h"
#include "segs.h"
#include "bsd.h"
#include "unixware.h"
#include "solarisX86.h"

int   DisplayDiskSeg( void *Object, TAG ObjectTag, u_int32_t ObjectSize, void *ObjectHandle, void *Parameters )
{
 DISKSEG *seg = (DISKSEG *) Object;
 char       Type[21];
 char       Boot[5] = "    ";
 u_int16_t  Id;
 int        rc;
 u_int32_t  StartLBA;
 u_int32_t  EndLBA;
 u_int32_t  SectorCount;
 SEG_PRIVATE_DATA  *pdata;
 char       Name[128];



    if (seg) {

        pdata       = (SEG_PRIVATE_DATA *) seg->private_data;

        StartLBA    = seg->start;
        EndLBA      = seg->start + seg->size -1;
        SectorCount = seg->size;
        Id          = pdata->sys_id;

        if (seg->name) {
            strncpy(Name, seg->name, EVMS_NAME_SIZE );
        }
        else {
            strcpy(Name, "n/a");
        }


        if ( pdata->boot_ind & ACTIVE_PARTITION   ) strcpy(Boot,"  * ");

        switch (seg->data_type)
        {
        case   META_DATA_TYPE:
            if ( pdata->flags & SEG_IS_EBR )
                strcpy(Type, "Meta Data: EBR");
            else if ( pdata->flags & SEG_IS_MBR )
                strcpy(Type, "Meta Data: MBR");
            else
                strcpy(Type, "Meta Data");
            break;

        case   DATA_TYPE:
            if ( pdata->flags & SEG_IS_LINUX_SWAP_PARTITION)
                strcpy(Type, "Data Seg: Linux Swap");
            else
                strcpy(Type, "Data Seg");
            break;

        case   FREE_SPACE_TYPE:
            strcpy(Type, "Free Space");
            break;

        default:
            strcpy(Type, "Unknown ");
            break;
        }

        LOG_DEBUG("%-20s   %-4s   %02X    %08d   %08d   %08d   %s\n", Type, Boot, Id, StartLBA, EndLBA, SectorCount, Name);

        rc = DLIST_SUCCESS;
    }
    else {
        rc = DLIST_CORRUPTED ;
    }

    return rc;
}

void  DisplayDiskSegmentList(LOGICALDISK *ld)
{
 int rc;

    if (ld==NULL) return;
    if (ld->parent_objects==NULL) return;


    LOG_DEBUG("\t\tLogical Disk Segment List ... \n");
    LOG_DEBUG("Type                   Boot   Id   Start LBA    End LBA    Sectors   SegName\n");

    rc = ForEachItem( ld->parent_objects, DisplayDiskSeg, NULL, TRUE );

}


void  DisplayPartitionRecord( Partition_Record          *part ,
                              int                        table_index,
                              BOOLEAN                    logical_partition )
{
    char          Type[21];
    char          Boot[5] = "    ";
    u_int16_t     Id;
    u_int32_t     StartLBA;
    u_int32_t     EndLBA;
    u_int32_t     SectorCount;



    StartLBA    = (u_int32_t) DISK_TO_CPU32(START_LBA(part));
    EndLBA      = (u_int32_t) DISK_TO_CPU32(START_LBA(part))+DISK_TO_CPU32(NR_SECTS(part));
    SectorCount = DISK_TO_CPU32(NR_SECTS(part));
    Id          = SYS_IND(part);

    if (EndLBA) --EndLBA;

    if (BOOT_IND(part))   strcpy(Boot,"  * ");

    switch (Id) {

        case OPENBSD_PARTITION:
        case FREEBSD_PARTITION:
        case NETBSD_PARTITION:
        case BSDI_PARTITION:
            strcpy(Type, "BSD");
            break;
        case LINUX_SWAP_PARTITION:
            strcpy(Type, "Linux Swap/Solaris");
            break;
        case LINUX_RAID_PARTITION:
            strcpy(Type, "Linux RAID");
            break;
        case LINUX_LVM_PARTITION:
            strcpy(Type, "Linux LVM");
            break;
        case LINUX_EXT2_PARTITION:
            strcpy(Type, "Linux");
            break;
        case LINUX_EXTENDED_PARTITION:
            strcpy(Type, "Linux Extd");
            break;
        case UNIXWARE_PARTITION:
            strcpy(Type, "UnixWare");
            break;
        case DOS_EXTENDED_PARTITION:
            strcpy(Type, "DOS Extd");
            break;
        case WIN98_EXTENDED_PARTITION:
            strcpy(Type, "Win95 Extd");
            break;
        case HPFSNTFS_PARTITION:
            strcpy(Type, "Hpfs/Ntfs");
            break;
        case HIDDEN_OS2_PARTITION:
            strcpy(Type, "OS2 Hidden C");
            break;
        case OS2_BOOT_MGR_PARTITION:
            strcpy(Type,"OS2 Boot Mgr");
            break;
        case OS2_LVM_SIGNATURE_PARTITION:
            strcpy(Type,"OS2 LVM");
            break;
        case HIDDEN_OS2_LVM_SIGNATURE_PARTITION:
            strcpy(Type,"OS2 Hidden LVM");
            break;
        case SMALL_FAT16_PARTITION:
        case LARGE_FAT16_PARTITION:
            strcpy(Type,"FAT16");
            break;
        case UNUSED_PARTITION_ENTRY:
            strcpy(Type, "Unused Entry");
            break;
        case GPT_PARTITION:
            strcpy(Type, "GPT");
            break;
        default:
            strcpy(Type,"n/a");
            break;
    }

    LOG_DEBUG("%-20s   %-4s   %02X    %08d   %08d   %08d\n", Type, Boot, Id, StartLBA, EndLBA, SectorCount);
}


void  DisplayPartitionTable(LOGICALDISK *ld, Partition_Record *partptr , BOOLEAN mbr)
{
    int  i;
    Partition_Record *part = partptr;


    if (mbr==TRUE)
        LOG_DEBUG("\t\tMaster Boot Record \n");
    else
        LOG_DEBUG("\t\tExtended Boot Record\n");

    LOG_DEBUG("Type                   Boot   Id   Start LBA    End LBA    Sectors\n");

    for(i=0; i<4; i++) {

        if (mbr==TRUE)
            DisplayPartitionRecord( part, i, FALSE );
        else
            DisplayPartitionRecord( part, i, TRUE  );
        ++part;
    }


}



