/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: aixregmgr
 * File: aix_io.c
 *
 * Description: This file contains functions related to I/O in the AIX 
 *		region manager.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "aixregmgr.h"


/* aix_read_ipl
 *
 *	Read the AIX IPL record from the specified object, and check for a valid
 *	IPL signature. The IPL record is always the first sector of an object.
 */
int aix_read_ipl(storage_object_t	* object,
		aix_ipl_rec_t		** ipl )
{
	aix_ipl_rec_t	* ipl_buffer;
	int		rc;

	LOG_ENTRY;

	// Allocate a new IPL record. This structure is padded to one
	// sector, so we don't need a separate buffer.
	ipl_buffer = aix_engine->engine_alloc(sizeof(aix_ipl_rec_t));
	if ( ! ipl_buffer ) {
		LOG_CRITICAL("Memory error creating buffer to read IPL record for %s\n", object->name);
		RETURN(ENOMEM);
	}

	// Read the first sector.
	rc = READ(object, PSN_IPL_REC, 1, ipl_buffer);
	if (rc) {
		aix_engine->engine_free(ipl_buffer);
		LOG_SERIOUS("Error reading IPL record from %s\n", object->name);
		RETURN(EIO);
	}

	// Check for a valid signature.
	if ( ipl_buffer->IPL_record_id != IPL_REC_ID ) {
		aix_engine->engine_free(ipl_buffer);
		LOG_EXTRA("%s is not an AIX PV.\n", object->name);
		RETURN(EINVAL);
	}

	// Return the IPL structure.
	*ipl = ipl_buffer;

	RETURN(0);
}


/* aix_write_ipl
 *
 *	Write the IPL structure to the first sector of the specified object.
 */
int aix_write_ipl( storage_object_t * object )
{
	aix_pv_data_t	* pv = object->consuming_private_data;
	int		rc;

	LOG_ENTRY;

	// Write the first sector. The IPL structure is sector-size padded,
	// so we don't need a separate I/O buffer.
	rc = WRITE(object, PSN_IPL_REC, 1, pv->ipl);
	if (rc) {
		LOG_SERIOUS("Error writing IPL record to %s\n", object->name);
		RETURN(EIO);
	}

	RETURN(0);
}


/* aix_read_lvm
 *
 *	Read the AIX LVM record from the specified object, and check for a valid
 *	LVM signature. The LVM record is always at sector 7 of an object.
 */
int aix_read_lvm(storage_object_t	* object,
		aix_lvm_rec_t		** lvm )
{
	aix_lvm_rec_t	* lvm_buffer;
	int		rc;

	LOG_ENTRY;

	// Allocate a new LVM record. This structure is padded to one
	// sector, so we only need one alloction.
	lvm_buffer = aix_engine->engine_alloc(sizeof(aix_lvm_rec_t));
	if ( ! lvm_buffer ) {
		LOG_CRITICAL("Memory error creating buffer to read LVM record for %s\n", object->name);
		RETURN(ENOMEM);
	}

	// Read the LVM sector.
	rc = READ(object, PSN_LVM_REC, 1, lvm_buffer);
	if (rc) {
		aix_engine->engine_free(lvm_buffer);
		LOG_SERIOUS("Error reading LVM record from %s\n", object->name);
		RETURN(EIO);
	}

	// Check for a valid signature
	if ( lvm_buffer->lvm_id != AIX_LVM_LVMID ) {
		aix_engine->engine_free(lvm_buffer);
		LOG_EXTRA("%s is not managed by the AIX LVM.\n", object->name);
		RETURN(EINVAL);
	}

	// Return the LVM structure.
	*lvm = lvm_buffer;

	RETURN(0);
}


/* aix_write_lvm
 *
 *	Write the LVM sector to the specified object. The LVM is located at
 *	sector 7.
 */
int aix_write_lvm( storage_object_t * object )
{
	aix_pv_data_t	* pv = object->consuming_private_data;
	int		rc;

	LOG_ENTRY;

	// Write the LVM sector. The LVM structure is sector-size padded,
	// so we don't need a separate I/O buffer.
	rc = WRITE(object, PSN_LVM_REC, 1, pv->lvm);
	if (rc) {
		LOG_SERIOUS("Error writing LVM record to %s\n", object->name);
		RETURN(EIO);
	}

	RETURN(0);
}


/* aix_read_vg_header
 *
 *	Read the VG header from the specified PV. The LVM record in the PV has
 *	a pointer to the beginning of the VGDA, and the VG header is the first
 *	sector in the VGDA. Copy specifies the first (0) or second (1) VGDA.
 */
int aix_read_vg_header(	storage_object_t	* object,
			vg_header		** vgh,
			unsigned int		copy )
{
	aix_pv_data_t	* pv = object->consuming_private_data;
	vg_header	* vgh_buffer;
	int		rc;

	LOG_ENTRY;

	// Allocate a buffer to read the header. The header structure is not
	// sector-size padded, so we need a separate buffer.
	vgh_buffer = aix_engine->engine_alloc(EVMS_VSECTOR_SIZE);
	if ( ! vgh_buffer ) {
		LOG_CRITICAL("Memory error creating buffer to read VG header from %s\n", object->name);
		RETURN(ENOMEM);
	}

	// Read the header.
	rc = READ(object, pv->lvm->vgda_psn[copy], 1, vgh_buffer);
	if (rc) {
		aix_engine->engine_free(vgh_buffer);
		LOG_SERIOUS("Error reading VG header (copy %d) from %s\n", copy, object->name);
		RETURN(rc);
	}

	// Allocate a VG header structure to return.
	*vgh = aix_engine->engine_alloc(sizeof(vg_header));
	if ( ! *vgh ) {
		aix_engine->engine_free(vgh_buffer);
		LOG_CRITICAL("Memory error creating new VG header for %s\n", object->name);
		RETURN(ENOMEM);
	}

	// Copy the structure
	memcpy(*vgh, vgh_buffer, sizeof(vg_header));
	aix_engine->engine_free(vgh_buffer);
	RETURN(0);
}


/* aix_read_vg_headers
 *
 *	Read both copies of the VG header from the specified PV.
 */
int aix_read_vg_headers( storage_object_t * object )
{
	aix_pv_data_t	* pv = object->consuming_private_data;
	vg_header	* vgh;
	int		rc;

	rc = aix_read_vg_header(object, &vgh, 0);
	if (rc) {
		RETURN(rc);
	}
	pv->vg_head[0] = vgh;

	rc = aix_read_vg_header(object, &vgh, 1);
	if (rc) {
		RETURN(rc);
	}
	pv->vg_head[1] = vgh;

	RETURN(0);
}


/* aix_write_vg_header
 *
 *	Write the VG header to the specified copy of the VGDA on this object.
 */
int aix_write_vg_header(storage_object_t	* object,
			unsigned int		copy )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	storage_container_t	* container = object->consuming_container;
	aix_container_data_t	* c_data = container->private_data;
	vg_header		* vgh_buffer;
	int			rc;

	LOG_ENTRY;

	// Need a separate I/O buffer for the VG header.
	vgh_buffer = aix_engine->engine_alloc(EVMS_VSECTOR_SIZE);
	if ( ! vgh_buffer ) {
		LOG_CRITICAL("Memory error creating buffer to write VG header to %s\n", object->name);
		RETURN(ENOMEM);
	}

	// Copy the data to the buffer.
	memcpy(vgh_buffer, c_data->vgda->vg_head, sizeof(vg_header));

	// Write the buffer..
	rc = WRITE(object, pv->lvm->vgda_psn[copy], 1, vgh_buffer);
	if (rc) {
		aix_engine->engine_free(vgh_buffer);
		LOG_SERIOUS("Error writing VG header (copy %d) to %s\n", copy, object->name);
		RETURN(rc);
	}

	aix_engine->engine_free(vgh_buffer);
	RETURN(0);
}


/* aix_read_vg_trailer
 *
 *	Read the VG trailer from the specified PV. The VG trailer is the
 *	last sector in the VGDA. Copy specifies the first (0) or second (1)
 *	VGDA.
 */
int aix_read_vg_trailer(storage_object_t	* object,
			vg_trailer		** vgt,
			unsigned int		copy )
{
	aix_pv_data_t	* pv = object->consuming_private_data;
	vg_trailer	* vgt_buffer;
	int		rc;

	LOG_ENTRY;

	// Allocate a buffer to read the trailer. The trailer structure is not
	// sector-size padded, so we need a separate buffer.
	vgt_buffer = aix_engine->engine_alloc(EVMS_VSECTOR_SIZE);
	if ( ! vgt_buffer ) {
		LOG_CRITICAL("Memory error creating buffer to read VG trailer from %s\n", object->name);
		RETURN(ENOMEM);
	}

	// Read the trailer.
	rc = READ(object, pv->lvm->vgda_psn[copy] + pv->lvm->vgda_len - 1, 1, vgt_buffer);
	if (rc) {
		aix_engine->engine_free(vgt_buffer);
		LOG_SERIOUS("Error reading VG trailer (copy %d) from %s\n", copy, object->name);
		RETURN(EIO);
	}

	// Allocate a VG trailer structure to return.
	*vgt = aix_engine->engine_alloc(sizeof(vg_trailer));
	if ( ! *vgt ) {
		aix_engine->engine_free(vgt_buffer);
		LOG_CRITICAL("Memory error creating new VG trailer for %s\n", object->name);
		RETURN(ENOMEM);
	}

	// Copy the structure
	memcpy(*vgt, vgt_buffer, sizeof(vg_trailer));
	aix_engine->engine_free(vgt_buffer);
	RETURN(0);
}


/* aix_read_vg_trailers
 *
 *	Read both copies of the VG trailer from the specified PV.
 */
int aix_read_vg_trailers( storage_object_t * object )
{
	aix_pv_data_t	* pv = object->consuming_private_data;
	vg_trailer	* vgt;
	int		rc;

	rc = aix_read_vg_trailer(object, &vgt, 0);
	if (rc) {
		RETURN(rc);
	}
	pv->vg_tail[0] = vgt;

	rc = aix_read_vg_trailer(object, &vgt, 1);
	if (rc) {
		RETURN(rc);
	}
	pv->vg_tail[1] = vgt;

	RETURN(0);
}


/* aix_write_vg_trailer
 *
 *	Write the VG trailer to the specified copy of the VGDA on this object.
 */
int aix_write_vg_trailer(storage_object_t	* object,
			unsigned int		copy )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	storage_container_t	* container = object->consuming_container;
	aix_container_data_t	* c_data = container->private_data;
	vg_trailer		* vgt_buffer;
	int			rc;

	LOG_ENTRY;

	// Need a separate I/O buffer for the VG trailer.
	vgt_buffer = aix_engine->engine_alloc(EVMS_VSECTOR_SIZE);
	if ( ! vgt_buffer ) {
		LOG_CRITICAL("Memory error creating buffer to write VG trailer to %s\n", object->name);
		RETURN(ENOMEM);
	}

	// Copy the data to the buffer.
	memcpy(vgt_buffer, c_data->vgda->vg_tail, sizeof(vg_trailer));

	// Write the buffer..
	rc = WRITE(object, pv->lvm->vgda_psn[copy] + pv->lvm->vgda_len - 1, 1, vgt_buffer);
	if (rc) {
		aix_engine->engine_free(vgt_buffer);
		LOG_SERIOUS("Error writing VG trailer (copy %d) to %s\n", copy, object->name);
		RETURN(EIO);
	}

	aix_engine->engine_free(vgt_buffer);
	RETURN(0);
}


/* aix_read_lv_array
 *
 *	Read the LV array from the specified VGDA copy on the specified PV. The
 *	LV array is the second item in the VGDA, and is 16 sectors long. There
 *	are 16 LV structs per sector, for a max of 256 LVs per VG.
 */
int aix_read_lv_array(	storage_object_t	* object,
			unsigned int		copy )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	storage_container_t	* container = object->consuming_container;
	aix_container_data_t	* c_data = container->private_data;
	aix_vgda_t		* vgda = c_data->vgda;
	int			rc;

	LOG_ENTRY;

	// Has the LV array already been read?
	if ( vgda->lv_array ) {
		LOG_EXTRA("Already read LV array for container %s\n", container->name);
		RETURN(0);
	}

	// Allocate the buffer to hold the array
	vgda->lv_array = aix_engine->engine_alloc(MAX_SECTORS_LV_ENTRIES * EVMS_VSECTOR_SIZE);
	if ( ! vgda->lv_array ) {
		LOG_CRITICAL("Memory error creating buffer for LV array for container %s\n", container->name);
		RETURN(ENOMEM);
	}

	// Read in the entire array.
	rc = READ(object, pv->lvm->vgda_psn[copy] + PSN_LVE_OFFSET, MAX_SECTORS_LV_ENTRIES, vgda->lv_array);
	if (rc) {
		LOG_SERIOUS("Error reading LV array from %s\n", object->name);
		aix_engine->engine_free(vgda->lv_array);
		vgda->lv_array = NULL;
		RETURN(EIO);
	}

	RETURN(0);
}


/* aix_write_lv_array
 *
 *	Write the LV array to the specified VGDA copy on the specified PV. The
 *	LV array is the second item in the VGDA, and is 16 sectors long. There
 *	are 16 LV structs per sector, for a max of 256 LVs per VG.
 */
int aix_write_lv_array( storage_object_t	* object,
			unsigned int		copy )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	storage_container_t	* container = object->consuming_container;
	aix_container_data_t	* c_data = container->private_data;
	int			rc;

	LOG_ENTRY;

	// Write the entire array. The array in memory is max-allocated, so
	// no separate I/O buffer is needed.
	rc = WRITE(object, pv->lvm->vgda_psn[copy] + PSN_LVE_OFFSET, MAX_SECTORS_LV_ENTRIES, c_data->vgda->lv_array);
	if (rc) {
		LOG_SERIOUS("Error writing LV array to %s\n", object->name);
		RETURN(EIO);
	}

	RETURN(0);
}


/* aix_read_lv_names
 *
 *	Read the LV namelist from the specified copy of the VGDA on the
 *	specified PV. The LV namelist is the second-to-last item in the VGDA,
 *	and is 32 sectors long. There are 256 LVs per VG, so each name is 64
 *	characters.
 */
int aix_read_lv_names(	storage_object_t	* object,
			unsigned int		copy )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	storage_container_t	* container = object->consuming_container;
	aix_container_data_t	* c_data = container->private_data;
	aix_vgda_t		* vgda = c_data->vgda;
	int			rc;

	LOG_ENTRY;

	// Has the namelist already been read?
	if ( vgda->lv_names ) {
		LOG_EXTRA("Already read LV namelist for container %s\n", container->name);
		RETURN(0);
	}

	// Allocate the buffer to hold the array
	vgda->lv_names = aix_engine->engine_alloc(MAX_SECTORS_NAMELIST * EVMS_VSECTOR_SIZE);
	if ( ! vgda->lv_names ) {
		LOG_CRITICAL("Memory error creating buffer for LV namelist for container %s\n", container->name);
		RETURN(ENOMEM);
	}

	// Read in the entire list
	rc = READ(object, pv->lvm->vgda_psn[copy] + pv->lvm->vgda_len - 1 - MAX_SECTORS_NAMELIST, MAX_SECTORS_NAMELIST, vgda->lv_names);
	if (rc) {
		LOG_SERIOUS("Error reading LV namelist from %s\n", object->name);
		aix_engine->engine_free(vgda->lv_names);
		vgda->lv_names = NULL;
		RETURN(rc);
	}

	RETURN(0);
}


/* aix_write_lv_names
 *
 *	Write the LV namelist to the specified copy of the VGDA on the
 *	specified PV. The LV namelist is the second-to-last item in the VGDA,
 *	and is 32 sectors long. There are 256 LVs per VG, so each name is 64
 *	characters.
 */
int aix_write_lv_names( storage_object_t	* object,
			unsigned int		copy )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	storage_container_t	* container = object->consuming_container;
	aix_container_data_t	* c_data = container->private_data;
	int			rc;

	LOG_ENTRY;

	// Write the entire list. Like the lv_array, this is stored max-allocated in
	// memory, so no extra I/O buffer is necessary.
	rc = WRITE(object, pv->lvm->vgda_psn[copy] + pv->lvm->vgda_len - 1 - MAX_SECTORS_NAMELIST, MAX_SECTORS_NAMELIST, c_data->vgda->lv_names);
	if (rc) {
		LOG_SERIOUS("Error writing LV namelist to %s\n", object->name);
		RETURN(rc);
	}

	RETURN(0);
}


/* aix_read_pvs
 *
 *	Read all of the PV headers and PP maps from the specified copy of the
 *	VGDA on the specified PV. Each PV header/PP map is 34 sectors long,
 *	and the first header starts 17 sectors from the start of the VGDA.
 */
int aix_read_pvs(storage_object_t	* object,
		unsigned int		copy )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	storage_container_t	* container = object->consuming_container;
	aix_container_data_t	* c_data = container->private_data;
	aix_vgda_t		* vgda = c_data->vgda;
	pv_header		* pvh_buffer;
	unsigned int		pv_num;
	int			rc;

	LOG_ENTRY;

	// Create a buffer for reading the PV headers.
	pvh_buffer = aix_engine->engine_alloc(PSN_PVH_INCREMENT * EVMS_VSECTOR_SIZE);
	if ( ! pvh_buffer ) {
		LOG_CRITICAL("Memory error creating buffer to read PV headers for container %s\n", container->name);
		RETURN(ENOMEM);
	}

	for ( pv_num = 0; pv_num < LVM_MAXPVS; pv_num++ ) {
		if ( vgda->pv_headers[pv_num] ) {
			LOG_EXTRA("Already read PV header %d for container %s\n", pv_num, container->name);
			continue;
		}

		// Read the PV header for the current PV number.
		rc = READ(object, pv->lvm->vgda_psn[copy] + PSN_PVH_OFFSET + pv_num * PSN_PVH_INCREMENT, PSN_PVH_INCREMENT, pvh_buffer);
		if (rc) {
			LOG_SERIOUS("Error reading PV header %d from %s\n", pv_num, object->name);
			continue;
		}

		// Check to see if this PV header is valid.
		if ( ! pvh_buffer->pv_state ) {
			continue;
		}

		// Allocate a new PV header entry in the VGDA.
		vgda->pv_headers[pv_num] = aix_engine->engine_alloc(PSN_PVH_INCREMENT * EVMS_VSECTOR_SIZE);
		if ( ! vgda->pv_headers[pv_num] ) {
			LOG_CRITICAL("Memory error creating buffer for PV header for container %s\n", container->name);
			RETURN(ENOMEM);
		}

		// Copy the data.
		memcpy(vgda->pv_headers[pv_num], pvh_buffer, PSN_PVH_INCREMENT * EVMS_VSECTOR_SIZE);
		vgda->pp_array[pv_num] = (pp_entries*)&(vgda->pv_headers[pv_num][1]);
	}

	aix_engine->engine_free(pvh_buffer);

	RETURN(0);
}


/* aix_write_pvs
 *
 *	Write all of the PV headers and PP maps to the specified copy of the
 *	VGDA on the specified PV. Each PV header/PP map is 34 sectors long,
 *	and the first header starts 17 sectors from the start of the VGDA.
 */
int aix_write_pvs(	storage_object_t	* object,
			unsigned int		copy )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	storage_container_t	* container = object->consuming_container;
	aix_container_data_t	* c_data = container->private_data;
	aix_vgda_t		* vgda = c_data->vgda;
	pv_header		* pvh_buffer;
	unsigned int		pv_num;
	int			rc;

	LOG_ENTRY;

	// Allocate a buffer to use for I/O.
	pvh_buffer = aix_engine->engine_alloc(PSN_PVH_INCREMENT * EVMS_VSECTOR_SIZE);
	if ( ! pvh_buffer ) {
		LOG_CRITICAL("Memory error creating buffer to write PV headers for container %s\n", container->name);
		RETURN(ENOMEM);
	}

	// Go through each PV header entry in the VGDA.
	for ( pv_num = 0; pv_num < LVM_MAXPVS; pv_num++ ) {

		// If this PV exists, copy the PV data to the buffer. Otherwise,
		// zero the buffer.
		if ( vgda->pv_headers[pv_num] ) {
			memcpy(pvh_buffer, vgda->pv_headers[pv_num], PSN_PVH_INCREMENT * EVMS_VSECTOR_SIZE);
		}
		else {
			memset(pvh_buffer, 0, PSN_PVH_INCREMENT * EVMS_VSECTOR_SIZE);
		}

		// Write the PV header for the current PV number.
		rc = WRITE(object, pv->lvm->vgda_psn[copy] + PSN_PVH_OFFSET + pv_num * PSN_PVH_INCREMENT, PSN_PVH_INCREMENT, pvh_buffer);
		if (rc) {
			LOG_SERIOUS("Error writing PV header %d to %s\n", pv_num, object->name);
			continue;
		}
	}

	aix_engine->engine_free(pvh_buffer);

	RETURN(0);
}


/* aix_read_vgsa
 *
 *	Read the specified copy of the Volume Group Status Area from the
 *	specified PV.
 */
int aix_read_vgsa(	storage_object_t	* object,
			unsigned int		copy )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	storage_container_t	* container = object->consuming_container;
	aix_container_data_t	* c_data = container->private_data;
	int rc;

	LOG_ENTRY;

	if ( c_data->vgsa ) {
		LOG_EXTRA("Already read VGSA for container %s\n", container->name);
		RETURN(0);
	}

	// The VGSA structure is padded to 8 sectors, so we don't need
	// a separate buffer.
	c_data->vgsa = aix_engine->engine_alloc(sizeof(vgsa_area));
	if ( ! c_data->vgsa ) {
		LOG_CRITICAL("Memory error creating VGSA for container %s\n", container->name);
		RETURN(ENOMEM);
	}

	rc = READ(object, pv->lvm->vgsa_psn[copy], pv->lvm->vgsa_len, c_data->vgsa);
	if (rc) {
		LOG_SERIOUS("Error reading VGSA from %s\n", object->name);
		aix_engine->engine_free(c_data->vgsa);
		c_data->vgsa = NULL;
		RETURN(rc);
	}

	RETURN(0);
}


/* aix_write_vgsa
 *
 *	Write the specified copy of the Volume Group Status Area to the
 *	specified PV.
 */
int aix_write_vgsa(	storage_object_t	* object,
			unsigned int		copy )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	storage_container_t	* container = object->consuming_container;
	aix_container_data_t	* c_data = container->private_data;
	int			rc;

	LOG_ENTRY;

	// The VGSA structure is padded to 8 sectors, so we don't need
	// a separate I/O buffer.
	rc = WRITE(object, pv->lvm->vgsa_psn[copy], pv->lvm->vgsa_len, c_data->vgsa);
	if (rc) {
		LOG_SERIOUS("Error writing VGSA to %s\n", object->name);
		RETURN(rc);
	}

	RETURN(0);
}

