/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: aixregmgr
 * File: aix_containers.c
 *
 * Description: This file contains functions related to managing AIX containers.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "aixregmgr.h"


static unsigned long aix_container_name = 0;


/* aix_find_container_by_id
 *
 *	Search through the global AIX container list for the specified VG id.
 */
storage_container_t * aix_find_container_by_id( unique_id vg_id )
{
	storage_container_t	* container;
	aix_container_data_t	* c_data;
	int			rc;

	LOG_ENTRY;

	FOR_EACH(container, aix_container_list) {
		c_data = container->private_data;
		if ( COMPARE_UNIQUE_IDS(c_data->vgda->vg_head->vg_id, vg_id) ) {
			RETURN(container);
		}
	}

	RETURN(NULL);
}


/* aix_deallocate_vgda
 *
 *	Free all memory used by the Volume Group Desriptor Area structure.
 */
void aix_deallocate_vgda( aix_vgda_t * vgda )
{
	unsigned int i;

	LOG_ENTRY;

	if ( vgda->vg_head ) {
		aix_engine->engine_free(vgda->vg_head);
	}
	if ( vgda->lv_array ) {
		aix_engine->engine_free(vgda->lv_array);
	}
	for ( i = 0; i < LVM_MAXPVS; i++ ) {
		if ( vgda->pv_headers[i] ) {
			aix_engine->engine_free(vgda->pv_headers[i]);
		}
	}
	if ( vgda->lv_names ) {
		aix_engine->engine_free(vgda->lv_names);
	}
	if ( vgda->vg_tail ) {
		aix_engine->engine_free(vgda->vg_tail);
	}

	LOG_EXIT(0);
}


/* aix_deallocate_container
 *
 *	Free all memory for the specified AIX container.
 */
void aix_deallocate_container( storage_container_t * container )
{
	aix_container_data_t	* c_data = container->private_data;
	storage_object_t	* object;
	storage_object_t	* region;
	TAG			tag;
	unsigned int		size;

	LOG_ENTRY;
	LOG_DEBUG("Deallocating container %s.\n", container->name);

	aix_remove_container_from_list(container);

	// Free all the regions.
	GoToStartOfList(container->objects_produced);
	while ( ! BlindExtractObject(container->objects_produced, &size, &tag, NULL, (void**)&region) ) {
		aix_deallocate_region(region);
	}

	// Free all the PVs.
	GoToStartOfList(container->objects_consumed);
	while ( ! BlindExtractObject(container->objects_consumed, &size, &tag, NULL, (void**)&object) ) {
		aix_deallocate_pv(object);
	}

	// Free the container private data.
	if ( c_data ) {
		if ( c_data->vgda ) {
			aix_deallocate_vgda(c_data->vgda);
		}

		if ( c_data->vgsa ) {
			aix_engine->engine_free(c_data->vgsa);
		}

		aix_engine->engine_free(c_data);
	}

	aix_engine->free_container(container);

	LOG_EXIT(0);
}


/* aix_allocate_container
 *
 *	Allocate a storage container and all of the necessary private data. The
 *	names of AIX volume groups are not stored anywhere in the metadata, so
 *	we will just come up with a numerical-based name for each container as
 *	it is created.
 */
storage_container_t * aix_allocate_container(	vg_header	* vgh,
						vg_trailer	* vgt )
{
	storage_container_t	* new_container;
	aix_container_data_t	* c_data;
	unsigned char		name[EVMS_NAME_SIZE] = {0};
	int			rc;

	LOG_ENTRY;

	// Generate a name for this container. AIX does not store VG names in
	// its metadata, so these container will not have persistent names.
	snprintf(name, EVMS_NAME_SIZE, "%s/container%ld", AIX_NAME, aix_container_name++);

	// Allocate the EVMS container.
	rc = aix_engine->allocate_container(name, &new_container);
	if (rc) {
		LOG_CRITICAL("Error allocating EVMS container %s.\n", name);
		RETURN(NULL);
	}

	// Allocate the container private data.
	c_data = aix_engine->engine_alloc(sizeof(aix_container_data_t));
	if ( ! c_data ) {
		LOG_CRITICAL("Error allocating private data for container %s.\n", name);
		aix_deallocate_container(new_container);
		RETURN(NULL);
	}
	new_container->private_data = c_data;

	// Allocate a VGDA parent structure
	c_data->vgda = aix_engine->engine_alloc(sizeof(aix_vgda_t));
	if ( ! c_data->vgda ) {
		LOG_CRITICAL("Memory error creating VGDA for container %s.\n", name);
		aix_deallocate_container(new_container);
		RETURN(NULL);
	}

	// Initialize remaining entries.
	new_container->plugin = aix_plugin;
	c_data->vgda->vg_head = vgh;
	c_data->vgda->vg_tail = vgt;
	
	rc = aix_add_container_to_list(new_container);
	if (rc) {
		aix_deallocate_container(new_container);
		RETURN(NULL);
	}

	LOG_DETAILS("Allocated new container %s\n", new_container->name);
	RETURN(new_container);
}



/* aix_find_container_for_pv
 *
 *	Check the VG header on the specified object, and determine if the
 *	container has already been created. If not, create a new one.
 */
static storage_container_t * aix_find_container_for_pv( storage_object_t * object )
{
	aix_pv_data_t		* pv = object->consuming_private_data;
	vg_header		* vgh = NULL;
	vg_trailer		* vgt = NULL;
	storage_container_t	* container;
	aix_container_data_t	* c_data;

	LOG_ENTRY;

	// Determine which VG header to get the VG ID from.
	switch(pv->pv_state) {

	case AIX_PV_STATE_VALID:
	case AIX_PV_STATE_FIRST_VGDA:
		vgh = pv->vg_head[0];
		vgt = pv->vg_tail[0];
		aix_engine->engine_free(pv->vg_head[1]);
		aix_engine->engine_free(pv->vg_tail[1]);
		pv->vg_head[0] = NULL;
		pv->vg_tail[0] = NULL;
		pv->vg_head[1] = NULL;
		pv->vg_tail[1] = NULL;
		break;

	case AIX_PV_STATE_SECOND_VGDA:
		vgh = pv->vg_head[1];
		vgt = pv->vg_tail[1];
		aix_engine->engine_free(pv->vg_head[0]);
		aix_engine->engine_free(pv->vg_tail[0]);
		pv->vg_head[0] = NULL;
		pv->vg_tail[0] = NULL;
		pv->vg_head[1] = NULL;
		pv->vg_tail[1] = NULL;
		break;

	case AIX_PV_STATE_EITHER_VGDA:
		if ( COMPARE_UNIQUE_IDS(pv->vg_head[0]->vg_id, pv->vg_head[1]->vg_id) ) {
			vgh = pv->vg_head[0];
			vgt = pv->vg_tail[0];
		}
		else {
			// Not sure where this PV belongs. It thinks it is
			// supposed to be in two different containers. We will
			// probably need to put this on a separate, temporary
			// list, and determine later which container is missing
			// a PV.
		}
		break;

	default:
		LOG_ERROR("Invalid PV state (%d) for %s\n", pv->pv_state, object->name);
		RETURN(NULL);
	}

	// Search for the container using the VG ID.
	container = aix_find_container_by_id(vgh->vg_id);

	if ( ! container ) {
		// Didn't find the container. Create a new one.
		container = aix_allocate_container(vgh, vgt);
		if ( ! container ) {
			RETURN(NULL);
		}

		if ( pv->pv_state == AIX_PV_STATE_EITHER_VGDA ) {
			c_data = container->private_data;
			c_data->flags |= AIX_CONTAINER_INVALID_VG_HEADER;
		}
	}
	else {
		// Found the container. Make sure its copy of the VG header
		// is valid.
		c_data = container->private_data;
		if ( c_data->flags & AIX_CONTAINER_INVALID_VG_HEADER &&
		     pv->pv_state != AIX_PV_STATE_EITHER_VGDA ) {
			aix_engine->engine_free(c_data->vgda->vg_head);
			c_data->vgda->vg_head = vgh;
			c_data->flags &= ~AIX_CONTAINER_INVALID_VG_HEADER;
		}
		else {
			aix_engine->engine_free(vgh);
			aix_engine->engine_free(vgt);
		}
	}

	RETURN(container);
}


/* aix_insert_pv_in_container
 *
 *	Insert the specified PV into the specified container. The aix_pv_t
 *	must be inserted into the aix_container_t pv_list in order by
 *	pv->lvm->pv_num. pv->object can be inserted into the container's
 *	objects_consumed list in any order, since there is no way to insert
 *	into a dlist in a specified order.
 */
static int aix_insert_pv_in_container(	storage_object_t	* object,
					storage_container_t	* container )
{
	storage_object_t	* target_object;
	aix_pv_data_t		* target_pv;
	aix_pv_data_t		* pv = object->consuming_private_data;
	void			* handle;
	int			rc;

	LOG_ENTRY;

	// Insert the object into the objects_consumed list in the container.
	// This insertion must be done in order by pv_num.
	FOR_EACH(target_object, container->objects_consumed) {
		target_pv = target_object->consuming_private_data;
		if ( target_pv->lvm->pv_num > pv->lvm->pv_num ) {
			rc = InsertObject(container->objects_consumed,
					sizeof(storage_object_t),
					object,
					object->object_type,
					NULL,
					InsertBefore,
					FALSE,
					&handle);
			if (rc) {
				LOG_SERIOUS("Error inserting PV %s in container %s\n", object->name, container->name);
				RETURN(rc);
			}

			object->consuming_container = container;
			RETURN(0);
		}
	}

	// Insert at end of the list
	rc = aix_add_object_to_list(object, container->objects_consumed);
	if (rc) {
		LOG_SERIOUS("Error inserting PV %s in container %s\n", object->name, container->name);
		RETURN(rc);
	}

	object->consuming_container = container;
	RETURN(0);
}


/* aix_discover_containers
 *
 *	Examine the list of input storage objects. For each object, read the IPL
 *	and LVM records to see if it is an AIX PV. If so, find/create the
 *	appropriate container for that PV, and add the PV to the container's
 *	list.
 */
int aix_discover_containers(	dlist_t	input_objects,
				dlist_t	output_objects )
{
	storage_object_t	* object;
	storage_container_t	* container;
	TAG			tag;
	unsigned int		size;
	int			rc;

	LOG_ENTRY;
	LOG_DETAILS("Searching for AIX PVs.\n");

	GoToStartOfList(input_objects);
	while ( ! BlindExtractObject(input_objects, &size, &tag, NULL, (void**)&object) ) {

		if ( object->data_type != DATA_TYPE ) {
			// Filter out metadata and freespace objects.
			LOG_EXTRA("Skipping %s - not DATA_TYPE.\n", object->name);
			aix_add_object_to_list(object, output_objects);
			continue;
		}

		rc = aix_read_pv_metadata(object);
		if (rc) {
			// This object is not a PV, or an error occurred.
			aix_add_object_to_list(object, output_objects);
			continue;
		}

		container = aix_find_container_for_pv(object);
		if ( ! container ) {
			// Error finding container for this PV.
			aix_deallocate_pv(object);
			aix_add_object_to_list(object, output_objects);
			continue;
		}

		rc = aix_insert_pv_in_container(object, container);
		if (rc) {
			// Error adding this PV to the container list.
			aix_deallocate_pv(object);
			aix_add_object_to_list(object, output_objects);
			continue;
		}
	}

	LOG_DETAILS("Container discovery complete.\n");
	RETURN(0);
}


/* aix_read_vgdas
 *
 *	Read the Volume Group Descriptor Area for each container.
 */
int aix_read_vgdas( void )
{
	storage_container_t	* container;
	aix_container_data_t	* c_data;
	storage_object_t	* object;
	aix_pv_data_t		* pv;
	boolean			done;
	int			copy;
	int			i, rc;

	LOG_ENTRY;

	FOR_EACH(container, aix_container_list) {
		c_data = container->private_data;
		done = FALSE;

		// Get the first valid PV from the container.
		FOR_EACH(object, container->objects_consumed) {
			pv = object->consuming_private_data;

			switch (pv->pv_state) {
			case AIX_PV_STATE_VALID:
			case AIX_PV_STATE_FIRST_VGDA:
				copy = 0;
				break;
			case AIX_PV_STATE_SECOND_VGDA:
				copy = 1;
				break;
			default:
				continue;
			}

			// Read in the VGSA, and all remaining sections of the VGDA.
			rc = aix_read_vgsa(object, copy);
			if (!rc) rc = aix_read_lv_array(object, copy);
			if (!rc) rc = aix_read_pvs(object, copy);
			if (!rc) rc = aix_read_lv_names(object, copy);
			if (!rc) {
				done = TRUE;
				break;
			}
		}

		if ( ! done ) {
			// All PVs are STATE_EITHER_VGDA, or there was some
			// IO error.
			LOG_SERIOUS("Missing metadata for container %s\n", container->name);
			continue;
		}

		// Set up the remaining pointers in the PV structs.
		FOR_EACH(object, container->objects_consumed) {
			pv = object->consuming_private_data;
			if ( ! pv->pv_head ) {
				pv->pv_head = c_data->vgda->pv_headers[pv->lvm->pv_num - 1];
			}
			if ( ! pv->pp_map ) {
				pv->pp_map = c_data->vgda->pp_array[pv->lvm->pv_num - 1];
			}
		}

		// Calculate the size of the container, since that info is
		// not stored in the VG header.
		if ( c_data->pp_count == 0 ) {
			for ( i = 0; i < LVM_MAXPVS; i++ ) {
				if ( c_data->vgda->pv_headers[i] ) {
					c_data->pp_count += c_data->vgda->pv_headers[i]->pp_count;
				}
			}
			container->size = c_data->pp_count * PP_SIZE_TO_VSECTORS(c_data->vgda->vg_head->pp_size);
		}
	}

	RETURN(0);
}

