/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: commit.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <dirent.h>
#include <mntent.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <sys/reboot.h>

#include <fullengine.h>
#include "commit.h"
#include "engine.h"
#include "message.h"
#include "internalAPI.h"
#include "discover.h"
#include "volume.h"
#include "crc.h"

BOOLEAN commit_in_progress = FALSE;
BOOLEAN need_reboot        = FALSE;
commit_status_func_t status_callback = NULL;


static char status_msg[256];

static void status_message(char * fmt, ...) {

    int len;
    va_list args;

    va_start(args, fmt);
    len += vsprintf(status_msg, fmt, args);
    va_end(args);

    LOG_DEBUG("%s", status_msg);

    if (status_callback != NULL) {
        status_callback(status_msg);
    }
}

/*
 * After a certain point (the deleting of volumes in the kernel) the commit
 * process is not allowed to abort due to errors.  If it were to abort, the
 * disks could be left in an inconsistent state and the system could be
 * unusable.  The commit process must try to get as much to disk as possible.
 * However, that presents a problem for error codes.  The commit process could
 * generate several errors.  Which one is to be returned to the user?
 * The following functions provide services for error reporting.  The logic is:
 * - Errors are recorded by class; the debug levels are used for the classes.
 * - The first error for each class is the one that is kept.
 * - When asked for the final overall error, the non-zero error code of the
 *    most critical class (lowest debug level) is returned as the error.
 */

static int commit_error[WARNING + 1] = {0};


static void clear_commit_errors() {

    int i;

    for (i = 0; i <= WARNING; i++) {
        commit_error[i] = 0;
    }
}


static void set_commit_error(debug_level_t level, int error) {

    if (level <= WARNING) {
        if (commit_error[level] == 0) {
            commit_error[level] = error;
        }

    } else {
        LOG_DEBUG("Attempt to set error code %d at invalid error level %d.\n", error, level);
    }
}


static int get_commit_error() {

    int rc = 0;
    int i;

    for (i = 0; (i <= WARNING) && (rc == 0); i++) {
        rc = commit_error[i];
    }

    return(rc);
}


/*
 * Check to see if the kernel knows about this volume.
 */
static BOOLEAN is_kernel_volume(logical_volume_t * volume) {

    BOOLEAN found = FALSE;
    int rc = 0;

    LOG_PROC_ENTRY();

    /*
     * Loop through the VolumeDataList and see if there is a kernel volume with
     * a minor number that matches this volume.
     */

    rc = GoToStartOfList(VolumeDataList);

    if (rc == DLIST_SUCCESS) {
        evms_volume_data_t * vol_data;

        rc = GetObject(VolumeDataList,
                       sizeof(evms_volume_data_t),
                       VOLUME_DATA_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &vol_data);

        while ((rc == DLIST_SUCCESS) && (vol_data != NULL) && !found) {
            if (volume->minor_number == vol_data->minor) {
                found = TRUE;

            } else {
                rc = GetNextObject(VolumeDataList,
                                   sizeof(evms_volume_data_t),
                                   VOLUME_DATA_TAG,
                                   (ADDRESS *) &vol_data);
            }
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the VolumeDataList.\n", rc);
    }

    LOG_PROC_EXIT_BOOLEAN(found);
    return found;
}


#define MOUNTED_ON_STRING   " mounted on "
/*
 * check_mounted will call is_volume_mounted() for the volume which will update
 * the mount_point string in the logical_volume_t to the current mount point of
 * the volume.
 */
static int check_mounted(ADDRESS object,
                         TAG     object_tag,
                         uint    object_tize,
                         ADDRESS object_tandle,
                         ADDRESS parameters) {

    logical_volume_t * vol = (logical_volume_t *) object;
    char * volume_name_list = (char *) parameters;

    LOG_PROC_ENTRY();

    /* Safety check */
    if (object_tag == VOLUME_TAG) {
        if (is_volume_mounted(vol)) {
            /*
             * The calling function guarantees that there is enough space in
             * the buffer for the volume name.
             */
            strcat(volume_name_list, vol->name);
            strcat(volume_name_list, MOUNTED_ON_STRING);
            strcat(volume_name_list, vol->mount_point);
            strcat(volume_name_list, "\n");
        }
    }

    LOG_PROC_EXIT_INT(DLIST_SUCCESS);
    return DLIST_SUCCESS;
}


int remove_volume(ADDRESS object,
                  TAG     object_tag,
                  uint    object_tize,
                  ADDRESS object_tandle,
                  ADDRESS parameters) {

    int rc = 0;
    logical_volume_t * vol = (logical_volume_t *) object;

    LOG_PROC_ENTRY();

    if (vol->mount_point == NULL) {
        int status;

        if (is_kernel_volume(vol)) {
            evms_delete_volume_t  kernel_delete_volume;

            kernel_delete_volume.command = EVMS_HARD_DELETE;
            kernel_delete_volume.minor = vol->minor_number;
            kernel_delete_volume.status = 0;

            if (vol->flags & VOLFLAG_SYNC_FS) {
                kernel_delete_volume.do_vfs = TRUE;
            } else {
                kernel_delete_volume.do_vfs = FALSE;
            }

            /* Initialize to no associated volume. */
            kernel_delete_volume.associative_minor = 0;

            /*
             * If the volume has an associated volume that the kernel knows,
             * then fill in the associative_minor field for the delete.
             */
            if (vol->associated_volume != NULL) {
                if (is_kernel_volume(vol->associated_volume)) {
                    LOG_DEBUG("Adding minor number %d for volume %s to be soft deleted along with the hard delete of volume %s.\n", vol->associated_volume->minor_number, vol->associated_volume->name, vol->name);
                    kernel_delete_volume.associative_minor = vol->associated_volume->minor_number;
                }
            }

            status_message("Hard deleting volume %s...\n", vol->name);
            status = ioctl(evms_block_dev_handle, EVMS_DELETE_VOLUME, &kernel_delete_volume);

            if (status) {
                LOG_WARNING("Error %d issuing ioctl to kernel to delete volume minor number %d.\n", errno, vol->minor_number);
                rc = errno;
                need_reboot = TRUE;
            } else {
                rc = kernel_delete_volume.status;

                if (rc != 0) {
                    LOG_WARNING("Error %d from ioctl to kernel to delete volume minor number %d.\n", errno, vol->minor_number);
                    need_reboot = TRUE;
                }
            }
        }

    } else {
        /*
         * This volume is mounted and cannot be deleted.  A reboot is required to
         * get the volumes to show up in the right place.
         */
        need_reboot = TRUE;
    }

    /*
     * Always process the whole list. Any errors will set the need_reboot
     * flag.
     */
    LOG_PROC_EXIT_INT(DLIST_SUCCESS);
    return DLIST_SUCCESS;
}


/*
 * Process the VolumeRemoveList.
 */
static int remove_volumes() {

    int rc = 0;
    uint count = 0;

    LOG_PROC_ENTRY();


    GetListSize(VolumeRemoveList, &count);

    if (count != 0) {
        char * mounted_volume_names = malloc((EVMS_VOLUME_NAME_SIZE + strlen(MOUNTED_ON_STRING) + MAXNAMLEN + 1) * count + 1);

        if (mounted_volume_names != NULL) {
            BOOLEAN retry = TRUE;

            while (retry && (rc == 0)) {
                /* "Clear" the buffer. */
                *mounted_volume_names = '\0';

                /* Check if any of the volumes is currently mounted. */
                ForEachItem(VolumeRemoveList, check_mounted, mounted_volume_names, TRUE);

                if (*mounted_volume_names != '\0') {
                    /*
                     * We have mounted volumes that should be removed.  Warn the
                     * user.
                     */
                    char * choices[] = {"Retry", "Continue", "Cancel commit", NULL};
                    int answer = 0;     /* Default is "Retry". */

                    engine_user_message(&answer, choices,
                                        "The names of the following compatibility volumes have changed.  "
                                        "The name change cannot be made since they are mounted.  "
                                        "The volumes should be unmounted before the changes are committed.  "
                                        "You can unmount the volumes and then press \"%s\".  "
                                        "If you press \"%s\" the changes will be committed but the volumes and their device nodes "
                                        "will be in an inconsistent state until you reboot the system.  "
                                        "If you press \"%s\" the commit process will be aborted.  "
                                        "\n\n%s",
                                        choices[0], choices[1], choices[2],
                                        mounted_volume_names);
                    switch (answer) {
                        case 0:
                            /*
                             * Retry.
                             * Leave retry == TRUE so we loop back and try again.
                             */
                            break;

                        case 1:
                            /*
                             * Continue:
                             * Warn the user.
                             * Set retry = FALSE so that we fall though and
                             * continue.  Warn the user.
                             */
                            {
                                char * choices[] = {"OK", NULL};
                                int answer = 0;     /* Must hit "OK". */

                                engine_user_message(&answer, choices,
                                                    "You must reboot the system after the changes have been committed.\n");
                            }
                            retry = FALSE;
                            break;

                        case 2:
                            /*
                             * Abort.
                             * set an error code for aborting.
                             */
                            rc = EINTR;
                            break;

                        default:
                            /*
                             * Shouldn't get here.  If we do, handle it like a
                             * retry
                             */
                            break;
                    }


                } else {
                    /* No mounted volumes.  No retry. */
                    retry = FALSE;
                }
            }

            if (rc == 0) {
                ForEachItem(VolumeRemoveList, remove_volume, NULL, TRUE);
            }

            free(mounted_volume_names);

        } else {
            LOG_CRITICAL("Error allocating memory for mounted volume names.\n");
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Find any volumes that are marked for unmkfs and run unmkfs on them.  Volumes
 * marked for unmkfs can be in the VolumeList and in the HardVolumeDeleteList.
 */
static int unmkfs_volumes() {

    int rc = 0;
    logical_volume_t * vol;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(VolumeList);

    if (rc == DLIST_SUCCESS) {

        rc = GetObject(VolumeList,
                       sizeof(logical_volume_t),
                       VOLUME_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &vol);

        while ((rc == 0) && (vol != NULL)) {
            if (vol->flags & VOLFLAG_UNMKFS) {

                /* Make sure there is an FSIM to do the unmkfs. */
                if (vol->original_fsim != NULL) {
                    status_message("Running unmkfs on volume %s...\n", vol->name);
                    rc = vol->original_fsim->functions.fsim->unmkfs(vol);
                }

                if (rc == 0) {
                    vol->flags &= ~VOLFLAG_UNMKFS;
                    vol->original_fsim = NULL;
                }
            }

            if (rc == 0) {
                rc = GetNextObject(VolumeList,
                                   sizeof(logical_volume_t),
                                   VOLUME_TAG,
                                   (ADDRESS *) &vol);
            }
        }

        /* Clear up acceptable return codes */
        if ((rc == DLIST_END_OF_LIST) ||
            (rc == DLIST_EMPTY)) {
            rc = DLIST_SUCCESS;
        }

        if (IS_DLIST_ERROR(rc)) {
            LOG_SERIOUS("Error code %d from when getting an object off the VolumeList.\n", rc);
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the VolumeList.\n", rc);
    }

    if (rc == 0) {
        rc = GoToStartOfList(HardVolumeDeleteList);

        if (rc == DLIST_SUCCESS) {

            rc = GetObject(HardVolumeDeleteList,
                           sizeof(logical_volume_t),
                           VOLUME_TAG,
                           NULL,
                           TRUE,
                           (ADDRESS *) &vol);

            while ((rc == 0) && (vol != NULL)) {
                if (vol->flags & VOLFLAG_UNMKFS) {

                    /* Make sure there is an FSIM to do the unmkfs. */
                    if (vol->original_fsim != NULL) {
                        status_message("Running unmkfs on volume %s...\n", vol->name);
                        rc = vol->original_fsim->functions.fsim->unmkfs(vol);
                    }

                    if (rc == 0) {
                        vol->flags &= ~VOLFLAG_UNMKFS;
                        vol->original_fsim = NULL;
                    }
                }

                if (rc == 0) {
                    rc = GetNextObject(HardVolumeDeleteList,
                                       sizeof(logical_volume_t),
                                       VOLUME_TAG,
                                       (ADDRESS *) &vol);
                }
            }

            /* Clear up acceptable return codes */
            if ((rc == DLIST_END_OF_LIST) ||
                (rc == DLIST_EMPTY)) {
                rc = DLIST_SUCCESS;
            }

            if (IS_DLIST_ERROR(rc)) {
                LOG_SERIOUS("Error code %d from when getting an object off the HardVolumeDeleteList.\n", rc);
            }

        } else {
            /* GoToStartOfList shouldn't fail.  Log an error if it does. */
            LOG_WARNING("Error code %d when going to the start of the HardVolumeDeleteList.\n", rc);
        }

    }

    LOG_PROC_EXIT_INT(rc);
    return(rc);
}


/*
 * Find any volumes that have shrunk in size and call the FSIM for
 * the volume to do the shrink.
 */
static int shrink_volumes() {

    int rc = 0;
    logical_volume_t * vol;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(VolumeList);

    if (rc == DLIST_SUCCESS) {

        rc = GetObject(VolumeList,
                       sizeof(logical_volume_t),
                       VOLUME_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &vol);

        while ((rc == 0) && (vol != NULL)) {
            if (vol->vol_size < vol->original_vol_size) {

                LOG_DEBUG("Shrink volume \"%s\" from %lld sectors to %lld sectors.\n", vol->name, vol->original_vol_size, vol->vol_size);
                if (vol->original_fsim != NULL) {
                    u_int64_t new_size;

                    status_message("Shrinking volume %s...\n", vol->name);
                    LOG_DEBUG("Calling %s FSIM to do the shrink.\n", vol->original_fsim->short_name);
                    rc = vol->original_fsim->functions.fsim->shrink(vol, vol->vol_size, &new_size);

                    if (rc == 0) {
                        /* Save the new file system size. */
                        vol->fs_size = new_size;

                    } else {
                        engine_user_message(NULL, NULL, "FSIM plug-in %s returned error code %d when called to shrink volume %s to %lld sectors.\n", vol->original_fsim->short_name, rc, vol->name, vol->vol_size);
                        set_commit_error(WARNING, rc);
                    }

                } else {
                    LOG_DEBUG("Volume \"%s\" has no FSIM.  The volume's size will be set to %lld sectors.\n", vol->name, vol->vol_size);
                }

                if (rc == 0) {
                    /* Set the volume's new original size. */
                    vol->original_vol_size = vol->vol_size;
                }
            }

            if (rc == 0) {
                rc = GetNextObject(VolumeList,
                                   sizeof(logical_volume_t),
                                   VOLUME_TAG,
                                   (ADDRESS *) &vol);
            }
        }

        /* Clear up acceptable return codes */
        if ((rc == DLIST_END_OF_LIST) ||
            (rc == DLIST_EMPTY)) {
            rc = DLIST_SUCCESS;
        }

        if (IS_DLIST_ERROR(rc)) {
            LOG_SERIOUS("Error code %d from when getting an object off the VolumeList.\n", rc);
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the VolumeList.\n", rc);
    }

    LOG_PROC_EXIT_INT(rc);
    return(rc);
}


/*
 * Delete a volume.
 * This function is structured so that it can be called by ForEachItem()
 * to process a list of volumes.
 * The "Parameters" parameter points to a evms_delete_volume_t control block for
 * the ioctl.  Most of the evms_delete_volume_t is filled in by the caller,
 * including the "command" field which tells whether this is a hard delete or a
 * soft delete.  This function fills in the minor number for the volume in the
 * control block before calling the ioctl.
 */
static int delete_volume(ADDRESS   object,
                         TAG       object_tag,
                         uint      object_size,
                         ADDRESS   object_handle,
                         ADDRESS   parameters) {

    int rc = 0;
    int status;
    logical_volume_t     * volume = (logical_volume_t *) object;
    evms_delete_volume_t * kernel_delete_volume = (evms_delete_volume_t *) parameters;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Request to %s delete volume %s.\n", (kernel_delete_volume->command == 1) ? "hard" : "soft", volume->name);

    if (is_kernel_volume(volume)) {
        kernel_delete_volume->minor = volume->minor_number;
        kernel_delete_volume->status = 0;

        if (volume->flags & VOLFLAG_SYNC_FS) {
            kernel_delete_volume->do_vfs = TRUE;
        } else {
            kernel_delete_volume->do_vfs = FALSE;
        }

        /* Initialize to no associated volume. */
        kernel_delete_volume->associative_minor = 0;

        /*
         * If this is a hard delete and the volume has an associated volume that
         * the kernel knows, then fill in the associative_minor field for the
         * delete.
         */
        if (kernel_delete_volume->command == EVMS_HARD_DELETE) {
            if (volume->associated_volume != NULL) {
                if (is_kernel_volume(volume->associated_volume)) {
                    LOG_DEBUG("Adding minor number %d for volume %s to be soft deleted along with the hard delete of volume %s.\n", volume->associated_volume->minor_number, volume->associated_volume->name, volume->name);
                    kernel_delete_volume->associative_minor = volume->associated_volume->minor_number;
                }
            }
        }

        status_message("%s deleting volume %s...\n", (kernel_delete_volume->command == EVMS_SOFT_DELETE) ? "Soft" : "Hard", volume->name);
        status = ioctl(evms_block_dev_handle, EVMS_DELETE_VOLUME, kernel_delete_volume);

        if (status) {
            LOG_WARNING("Error %d issuing ioctl to kernel to delete volume minor number %d.\n", errno, volume->minor_number);
            rc = errno;
        } else {
            rc = kernel_delete_volume->status;

            if (rc == 0) {
                volume->flags &= ~VOLFLAG_SYNC_FS;
            } else {
                LOG_WARNING("Error %d from ioctl to kernel to delete volume minor number %d.\n", errno, volume->minor_number);
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Call the kernel runtime code to delete volumes.  This includes "hard"
 * deletes where the volume is being permanently deleted from the system
 * and "soft" deletes where the volume is being deleted because its
 * underlying structure changed and so must be rediscovered.
 */
static int delete_volumes() {
    int rc = 0;
    evms_delete_volume_t  kernel_delete_volume;

    LOG_PROC_ENTRY();

    /* Call the kernel to do permanent deletion of volumes. */

    kernel_delete_volume.command = EVMS_HARD_DELETE;
    rc = ForEachItem(HardVolumeDeleteList,
                     delete_volume,
                     (ADDRESS) &kernel_delete_volume,
                     TRUE);

    if (rc == DLIST_SUCCESS) {
        /* Call the kernel to do temporary deletion of volumes. */
        kernel_delete_volume.command = EVMS_SOFT_DELETE;
        rc = ForEachItem(SoftVolumeDeleteList,
                         delete_volume,
                         (ADDRESS) &kernel_delete_volume,
                         TRUE);
    }

    if (rc == 0) {
        /*
         * All the deletes succeeded.  Delete the volumes from the
         * HardVolumeDeleteList and the SoftVolumeDeleteList.
         */
        DeleteAllItems(HardVolumeDeleteList,
                       TRUE);           /* The logical volumes are gone so */
                                        /* free the volume structures.     */
        DeleteAllItems(SoftVolumeDeleteList,
                       FALSE);          /* The logical volumes still exist */
                                        /* so don't free the volumes.      */
    } else {
        /*
         * One of the deletes failed.  Leave all the volumes on the
         * HardVolumeDeleteList and on the SoftVolumeDeleteList.  Issue a kernel
         * rediscover to discover any volumes that were deleted (since no
         * changes were written to the volumes they will be rediscovered) and to
         * flush any soft deletes that are pending.  The system should now be in
         * the same state it was prior to the calling of this function.
         */
        status_message("Rediscovering volumes in the kernel...\n");
        kernel_rediscover();
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Process the KillSectorList.  The KillSectorList contains a list of
 * kill_sector_record_ts.  Each kill_sector_record_t contains a disk, a starting
 * LBA, and a count of sectors to be zeroed out.
 */
static void kill_sectors() {
    int rc = 0;
    kill_sector_record_t * kill_sector;
    char                 * buffer = calloc(1, 4096);
    int                    current_buffer_size = 4096;
    int                    buffer_size_needed = 0;

    LOG_PROC_ENTRY();

    if (buffer != NULL) {

        rc = GoToStartOfList(KillSectorList);
        if (rc == DLIST_SUCCESS) {

            rc = ExtractObject(KillSectorList,
                               sizeof(kill_sector_record_t),
                               KILL_SECTOR_TAG,
                               NULL,
                               (ADDRESS) &kill_sector);

            while ((rc == 0) && (kill_sector != NULL)) {

                /* Test validity.  Ignore bad records. */
                if (kill_sector->logical_disk != NULL) {
                    if (kill_sector->logical_disk->plugin != NULL) {

                        /* Grow our zero filled buffer if needed. */
                        buffer_size_needed = kill_sector->sector_count * EVMS_VSECTOR_SIZE;
                        if (current_buffer_size < buffer_size_needed) {
                            free(buffer);
                            buffer = calloc(1, buffer_size_needed);
                            if (buffer != NULL) {
                                current_buffer_size = buffer_size_needed;

                            } else {
                                LOG_CRITICAL("Error allocating memory for a zero filled buffer for killing sectors.\n");
                                rc = ENOMEM;
                                set_commit_error(CRITICAL, rc);
                            }
                        }

                        /* Zap the sectors. */
                        if (rc == 0) {
                            LOG_DEBUG("Writing %lld sector%s of zeros to disk %s at sector %lld.\n", kill_sector->sector_count, (kill_sector->sector_count == 1) ? "" : "s", kill_sector->logical_disk->name, kill_sector->sector_offset);
                            rc = kill_sector->logical_disk->plugin->functions.plugin->write(kill_sector->logical_disk, kill_sector->sector_offset, kill_sector->sector_count, buffer);

                            if (rc != 0) {
                                engine_user_message(NULL, NULL, "Error code %d from call to write zeroes to %lld sectors starting at sector %lld on disk %s.\n", rc, kill_sector->sector_count, kill_sector->sector_offset, kill_sector->logical_disk->name);
                                set_commit_error(WARNING, rc);

                                /* Don't let non-critical errors stop the commit. */
                                rc = 0;
                            }
                        }

                    } else {
                        LOG_WARNING("Kill sector record does not have a valid disk.  The disk's plug-in pointer is NULL.\n");
                        set_commit_error(WARNING, ENXIO);
                    }

                } else {
                    LOG_WARNING("Kill sector record does not have a valid disk.  The disk pointer is NULL.\n");
                    set_commit_error(WARNING, ENXIO);
                }

                if (rc == 0) {
                    rc = ExtractObject(KillSectorList,
                                       sizeof(kill_sector_record_t),
                                       KILL_SECTOR_TAG,
                                       NULL,
                                       (ADDRESS) &kill_sector);
                }
            }

            if ((rc != DLIST_END_OF_LIST) &&
                (rc != DLIST_EMPTY) &&
                (rc != ENOMEM)) {
                LOG_SERIOUS("Error code %d from ExtractObject() when getting next object off the KillSectorList.\n", rc);
                set_commit_error(SERIOUS, rc);
            }

        } else {
            /* GoToStartOfList shouldn't fail.  Log an error if it does. */
            LOG_WARNING("Error code %d when going to the start of the KillSectorList.\n", rc);
            set_commit_error(WARNING, rc);
        }

        if (buffer != NULL) {
            free(buffer);
        }

    } else {
        LOG_CRITICAL("Error allocating memory for a zero filled buffer for killing sectors.\n");
        set_commit_error(CRITICAL, ENOMEM);
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Slide any segments marked to be slid.  TBD.
 */
static int slide_segments(uint phase) {
    int rc = 0;

    LOG_PROC_ENTRY();

    /* Slide segments in phase one. */
    if (phase == FIRST_METADATA_WRITE) {
        /* Tell the kernel that slide mode is beginning. */

        /* Run a loop touching all the sectors on the segment. */

        /* Tell the kernel that slide mode is finished. */
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Find any dirty segments and call their plug-ins to commit the changes
 * to disk.
 */
static void commit_segments(uint phase) {
    int rc = 0;
    storage_object_t * segment;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(SegmentList);

    if (rc == DLIST_SUCCESS) {
        rc = GetObject(SegmentList,
                       sizeof(storage_object_t),
                       SEGMENT_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &segment);

        while ((rc == 0) && (segment != NULL)) {

            if ((segment->flags & SOFLAG_DIRTY) &&
                (segment->plugin != NULL) &&
                (segment->plugin->functions.plugin != NULL)) {

                status_message("Phase %d:  Committing changes on segment %s...\n", phase, segment->name);
                rc = segment->plugin->functions.plugin->commit_changes(segment, phase);

                if (rc != 0) {
                    engine_user_message(NULL, NULL, "Plug-in %s returned error %d when committing changes for segment %s during phase %d.\n", segment->plugin->short_name, rc, segment->name, phase);
                    set_commit_error(WARNING, rc);

                    /*
                     * We logged the error.  Commit must be allowed to
                     * continue.
                     */
                    rc = 0;
                }
            }

            if (rc == 0) {
                rc = GetNextObject(SegmentList,
                                   sizeof(storage_object_t),
                                   SEGMENT_TAG,
                                   (ADDRESS *) &segment);
            }
        }

        /* Any errors here came from GetObject() or GetNextObject(). */
        if ((rc != DLIST_END_OF_LIST) &&
            (rc != DLIST_EMPTY)) {
            LOG_SERIOUS("Error code %d from when getting an object off the SegmentList.\n", rc);
            set_commit_error(SERIOUS, rc);
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the SegmentList.\n", rc);
        set_commit_error(WARNING, rc);
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Move any segments marked to be moved.  TBD.
 */
static void move_segments(uint phase) {

    LOG_PROC_ENTRY();

    /* Move segments in phase one. */
    if (phase == FIRST_METADATA_WRITE) {
        /* Tell the kernel that move mode is beginning. */

        /* Run a loop touching all the sectors on the segment. */

        /* Tell the kernel that move mode is finished. */
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Find any dirty containers and call their plug-ins to commit the changes
 * to disk.
 */
static void commit_containers(uint phase) {
    int rc = 0;
    storage_container_t * con;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(ContainerList);

    if (rc == DLIST_SUCCESS) {
        rc = GetObject(ContainerList,
                       sizeof(storage_container_t),
                       CONTAINER_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &con);

        while ((rc == 0) && (con != NULL)) {

            if ((con->flags & SCFLAG_DIRTY) &&
                (con->plugin != NULL) &&
                (con->plugin->container_functions != NULL)) {

                status_message("Phase %d:  Committing changes on container %s...\n", phase, con->name);
                rc = con->plugin->container_functions->commit_container_changes(con, phase);

                if (rc != 0) {
                    engine_user_message(NULL, NULL, "Plug-in %s returned error %d when committing changes for container %s during phase %d.\n", con->plugin->short_name, rc, con->name, phase);
                    set_commit_error(WARNING, rc);

                    /*
                     * We logged the error.  Commit must be allowed to
                     * continue.
                     */
                    rc = 0;
                }
            }

            if (rc == 0) {
                rc = GetNextObject(ContainerList,
                                   sizeof(storage_container_t),
                                   CONTAINER_TAG,
                                   (ADDRESS *) &con);
            }
        }

        /* Any errors here came from GetObject() or GetNextObject(). */
        if ((rc != DLIST_END_OF_LIST) &&
            (rc != DLIST_EMPTY)) {
            LOG_SERIOUS("Error code %d from when getting an object off the ContainerList.\n", rc);
            set_commit_error(SERIOUS, rc);
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the ContainerList.\n", rc);
        set_commit_error(WARNING, rc);
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Find any dirty regions and call their plug-ins to commit the changes
 * to disk.
 */
static void commit_regions(uint phase) {
    int rc = 0;
    storage_object_t * region;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(RegionList);

    if (rc == DLIST_SUCCESS) {
        rc = GetObject(RegionList,
                       sizeof(storage_object_t),
                       REGION_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &region);

        while ((rc == 0) && (region != NULL)) {

            if ((region->flags & SOFLAG_DIRTY) &&
                (region->plugin != NULL) &&
                (region->plugin->functions.plugin != NULL)) {

                status_message("Phase %d:  Committing changes on region %s...\n", phase, region->name);
                rc = region->plugin->functions.plugin->commit_changes(region, phase);

                if (rc != 0) {
                    engine_user_message(NULL, NULL, "Plug-in %s returned error %d when committing changes for region %s during phase %d.\n", region->plugin->short_name, rc, region->name, phase);
                    set_commit_error(WARNING, rc);

                    /*
                     * We logged the error.  Commit must be allowed to
                     * continue.
                     */
                    rc = 0;
                }
            }

            if (rc == 0) {
                rc = GetNextObject(RegionList,
                                   sizeof(storage_object_t),
                                   REGION_TAG,
                                   (ADDRESS *) &region);
            }
        }

        /* Any errors here came from GetObject() or GetNextObject(). */
        if ((rc != DLIST_END_OF_LIST) &&
            (rc != DLIST_EMPTY)) {
            LOG_SERIOUS("Error code %d from when getting an object off the RegionList.\n", rc);
            set_commit_error(SERIOUS, rc);
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the RegiontList.\n", rc);
        set_commit_error(WARNING, rc);
    }

    LOG_PROC_EXIT_VOID();
}

/*
 * Check if this object's feature header has a higher sequence number than
 * the one given.  If so, update the given sequence number.
 * This function has its parameters structures so that it can be called by
 * the ForEachItem() dlist processor.  "parameters" points to the sequence
 * number.
 */
static int get_highest_sequence_number(ADDRESS object,
                                       TAG     object_tag,
                                       uint    object_size,
                                       ADDRESS object_handle,
                                       ADDRESS parameters) {

    storage_object_t * obj = (storage_object_t *) object;
    u_int64_t * sequence_number = (u_int64_t *) parameters;

    if (obj->feature_header != NULL) {
        if (obj->feature_header->sequence_number < *sequence_number) {
            *sequence_number = obj->feature_header->sequence_number;
        }
    }

    return DLIST_SUCCESS;
}


/*
 * Set the sequence number in an object's feature header.  If the new
 * sequence number is different from the current sequence number, mark the
 * feature header dirty.
 * This function has its parameters structures so that it can be called by
 * the ForEachItem() dlist processor.  "parameters" points to the sequence
 * number.
 */
static int set_sequence_number(ADDRESS object,
                               TAG     object_tag,
                               uint    object_size,
                               ADDRESS object_handle,
                               ADDRESS parameters) {

    storage_object_t * obj = (storage_object_t *) object;
    u_int64_t * sequence_number = (u_int64_t *) parameters;

    if (obj->feature_header != NULL) {

        /*
         * Set the feature header sequence number if it is not already at the
         * specified number.
         */
        if (obj->feature_header->sequence_number != *sequence_number) {
            obj->feature_header->sequence_number = *sequence_number;

            /* Mark the feature header dirty so it gets rewritten to disk. */
            obj->flags |= SOFLAG_FEATURE_HEADER_DIRTY;
        }
    }

    return DLIST_SUCCESS;
}


/*
 * Convert a feature header in the CPU endian format to the disk endian
 * (little endian) format.
 */
void inline feature_header_cpu_to_disk(evms_feature_header_t * fh) {

    fh->signature                 = CPU_TO_DISK32(fh->signature);
    fh->version.major             = CPU_TO_DISK32(fh->version.major);
    fh->version.minor             = CPU_TO_DISK32(fh->version.minor);
    fh->version.patchlevel        = CPU_TO_DISK32(fh->version.patchlevel);
    fh->engine_version.major      = CPU_TO_DISK32(fh->engine_version.major);
    fh->engine_version.minor      = CPU_TO_DISK32(fh->engine_version.minor);
    fh->engine_version.patchlevel = CPU_TO_DISK32(fh->engine_version.patchlevel);
    fh->flags                     = CPU_TO_DISK32(fh->flags);
    fh->feature_id                = CPU_TO_DISK32(fh->feature_id);
    fh->sequence_number           = CPU_TO_DISK64(fh->sequence_number);
    fh->alignment_padding         = CPU_TO_DISK64(fh->alignment_padding);
    fh->feature_data1_start_lsn   = CPU_TO_DISK64(fh->feature_data1_start_lsn);
    fh->feature_data1_size        = CPU_TO_DISK64(fh->feature_data1_size);
    fh->feature_data2_start_lsn   = CPU_TO_DISK64(fh->feature_data2_start_lsn);
    fh->feature_data2_size        = CPU_TO_DISK64(fh->feature_data2_size);
    fh->volume_serial_number      = CPU_TO_DISK64(fh->volume_serial_number);
    fh->volume_system_id          = CPU_TO_DISK32(fh->volume_system_id);
    fh->object_depth              = CPU_TO_DISK32(fh->object_depth);
}


typedef struct commit_parms_s {
    u_int32_t   depth;
    uint        phase;
} commit_parms_t;


/*
 * Write this object's feature header to disk, if the header is marked dirty.
 * "parameters" points to the parent object.
 * This function has its parameters structures so that it can be called by
 * the ForEachItem() dlist processor.
 */
static int write_feature_header(ADDRESS object,
                                TAG     object_tag,
                                uint    object_size,
                                ADDRESS object_handle,
                                ADDRESS parameters) {
    int rc = 0;
    storage_object_t * obj = (storage_object_t *) object;
    commit_parms_t * parms = (commit_parms_t *) parameters;

    LOG_PROC_ENTRY();

    if ((parms->phase == FIRST_METADATA_WRITE) ||
        (parms->phase == SECOND_METADATA_WRITE)) {

        /* Only write the feature header if it is marked dirty. */
        if (obj->flags & SOFLAG_FEATURE_HEADER_DIRTY) {
            sector_count_t  fh_lsn;

            /* Setup the feature header in phase one. */
            if (parms->phase == FIRST_METADATA_WRITE) {
                evms_feature_header_t * fh = obj->feature_header;

                /* Make sure the signature is correct. */

                fh->signature = EVMS_FEATURE_HEADER_SIGNATURE;

                /* Set the feature header version. */
                fh->version.major = EVMS_FEATURE_HEADER_MAJOR;
                fh->version.minor = EVMS_FEATURE_HEADER_MINOR;
                fh->version.patchlevel = EVMS_FEATURE_HEADER_PATCHLEVEL;

                /*
                 * Set the version of the Engine that wrote the feature header.
                 */
                fh->engine_version.major = MAJOR_VERSION;
                fh->engine_version.minor = MINOR_VERSION;
                fh->engine_version.patchlevel = PATCH_LEVEL;

                /* Set the object depth. */
                fh->object_depth = parms->depth;

                /*
                 * If the object is part of a volume, set the volume
                 * information.
                 */
                if (obj->volume != NULL) {
                    fh->flags &= ~EVMS_VOLUME_DATA_OBJECT;
                    fh->volume_serial_number = obj->volume->serial_number;
                    fh->volume_system_id = obj->volume->minor_number;
                    memset(fh->volume_name, 0, sizeof(fh->volume_name));

                    /*
                     * Strip off the dev node root when copying the volume
                     * name.  Volume names in the feature header are
                     * relative to the dev node root.
                     */
                    strcpy(fh->volume_name, obj->volume->name + strlen(EVMS_DEV_NODE_PATH));
                } else {
                    fh->flags |= EVMS_VOLUME_DATA_OBJECT;
                    fh->volume_serial_number = 0;
                    fh->volume_system_id = 0;
                    memset(fh->volume_name, 0, sizeof(fh->volume_name));
                }
            }

            /*
             * Convert the feature header to disk endian format before
             * calculating the CRC.
             */
            feature_header_cpu_to_disk(obj->feature_header);
            obj->feature_header->crc = 0;
            obj->feature_header->crc = CPU_TO_DISK32(calculate_CRC(0xFFFFFFFF, obj->feature_header, sizeof(evms_feature_header_t)));

            if (parms->phase == FIRST_METADATA_WRITE) {
                fh_lsn = obj->size - FEATURE_HEADER_SECTORS;
            } else {
                fh_lsn = obj->size - (FEATURE_HEADER_SECTORS * 2);
            }

            /* Write the feature header for the object. */
            status_message("Phase %d:  Writing feature header on object %s...\n", parms->phase, obj->name);
            rc = obj->plugin->functions.plugin->write(obj, fh_lsn, FEATURE_HEADER_SECTORS, obj->feature_header);

            /* Convert the feature header back to CPU endian format. */
            feature_header_disk_to_cpu(obj->feature_header);

            if (rc != 0) {
                engine_user_message(NULL, NULL, "Error code %d when writing phase %d feature header to object %s.\n", rc, parms->phase, obj->name);
                set_commit_error(WARNING, rc);
            }

            /* Committing of a feature header is finished after phase 2. */
            if (parms->phase >= SECOND_METADATA_WRITE) {

                /* Turn off the SOFLAG_FEATURE_HEADER_DIRTY flag. */
                obj->flags &= ~SOFLAG_FEATURE_HEADER_DIRTY;
            }

        } else {
            LOG_DEBUG("Object %s does not have its feature header marked dirty.\n", obj->name);
        }
    }

    /* Don't let errors stop the commit process. */
    LOG_PROC_EXIT_INT(DLIST_SUCCESS);
    return DLIST_SUCCESS;
}


static void commit_feature_header(storage_object_t * object,
                                  storage_object_t * parent_object,
                                  commit_parms_t   * commit_parms) {

    LOG_PROC_ENTRY();

    if ((commit_parms->phase == FIRST_METADATA_WRITE) ||
        (commit_parms->phase == SECOND_METADATA_WRITE)) {

        if (object->flags & SOFLAG_FEATURE_HEADER_DIRTY) {

            /*
             * Setup the feature header for this object and its
             * siblings in phase 1.  Other phases just write the same
             * feature header in different location(s).
             */
            if (commit_parms->phase == FIRST_METADATA_WRITE) {
                u_int64_t sequence_number = 0;

                /* Get the highest sequence number of the siblings. */
                ForEachItem(parent_object->child_objects,
                            get_highest_sequence_number,
                            &sequence_number,
                            TRUE);

                /* Bump the sequence number. */
                sequence_number ++;

                /* Set the new sequence number in the siblings. */
                ForEachItem(parent_object->child_objects,
                            set_sequence_number,
                            &sequence_number,
                            TRUE);
            }

            /*
             * Step up to the parent object and commit all of its child
             * objects' feature headers together.  All the child objects'
             * feature headers should have the same sequence number and
             * volume complete flag.
             */
            ForEachItem(parent_object->child_objects,
                        write_feature_header,
                        commit_parms,
                        TRUE);
        }
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Given an object, commit the object and its children to disk.  If the object
 * is an EVMS object, first call this function recursively on the child objects
 * to commit any potential dirty children or any potentially dirty feature
 * headers on the children.  The recursion goes down until we hit a System Data
 * object, which is always the bottom object of a feature stack.  Once the
 * recursion is finished, then if the object is dirty, call its plug-in to
 * commit the changes to disk.
 * This function has its parameters structured so that it can be called by
 * ForEachItem().
 */
static int commit_object(ADDRESS Object,
                         TAG     ObjectTag,
                         uint    ObjectSize,
                         ADDRESS ObjectHandle,
                         ADDRESS Parameters) {
    int rc = 0;
    storage_object_t * object = (storage_object_t *) Object;
    commit_parms_t * parms = (commit_parms_t *) Parameters;
    uint size;
    TAG tag;
    storage_object_t * child_object = NULL;

    LOG_PROC_ENTRY();

    /* If this is an EVMS object, first commit the child objects. */
    if (object->object_type == EVMS_OBJECT) {

        /*
         * Pick one of the children for examining and writing this object's feature
         * header(s), which is/are hanging off the child object(s).
         */
        rc = BlindGetObject(object->child_objects,
                            &size,
                            &tag,
                            NULL,
                            FALSE,
                            (ADDRESS *) &child_object);

        if (rc == DLIST_SUCCESS) {

            /*
             * Increase the depth for the child objects if this object has
             * a feature header hanging off its child.
             */
            if (child_object->feature_header != NULL) {
                parms->depth++;
            }

            /*
             * Call myself recursively to commit any changes on my child
             * objects.
             */
            ForEachItem(object->child_objects,
                        commit_object,
                        parms,
                        TRUE);

            /*
             * Reset the depth for the commit of this object's feature
             * header(s).
             */
            if (child_object->feature_header != NULL) {
                parms->depth--;
            }
        }

        /* Call the feature to commit the object if the object is dirty. */
        if (object->flags & SOFLAG_DIRTY) {
            status_message("Phase %d:  Committing changes to object %s...\n", parms->phase, object->name);
            rc = object->plugin->functions.plugin->commit_changes(object, parms->phase);

            if (rc != 0) {
                engine_user_message(NULL, NULL, "Plug-in %s returned error %d when committing changes for object %s during phase %d.\n", object->plugin->short_name, rc, object->name, parms->phase);
                set_commit_error(WARNING, rc);
            }
        }

        if ((object->object_type == EVMS_OBJECT) &&
            (child_object != NULL)) {
            /* Commit any feature header for this object. */
            commit_feature_header(child_object, object, parms);
        }
    }

    /* Don't allow errors to stop the commit process. */
    LOG_PROC_EXIT_INT(DLIST_SUCCESS);
    return DLIST_SUCCESS;
}


/*
 * Mark all the feature headers in an object stack dirty so that they will
 * be re-written with updated data.
 */

int mark_feature_headers_dirty(ADDRESS object,
                               TAG     object_tag,
                               uint    object_size,
                               ADDRESS object_handle,
                               ADDRESS parameters) {
    LOG_PROC_ENTRY();

    if ((object_tag == DISK_TAG) ||
        (object_tag == SEGMENT_TAG) ||
        (object_tag == REGION_TAG) ||
        (object_tag == EVMS_OBJECT_TAG)) {

        storage_object_t * obj = (storage_object_t *) object;

        /* If this object has a feature header, mark it dirty. */
        if (obj->feature_header != NULL) {
            obj->flags |= SOFLAG_FEATURE_HEADER_DIRTY;
        }

        /*
         * If the object is an EVMS object, call myself recursively on
         * all my children to set their feature headers dirty.
         */
        if (obj->object_type == EVMS_OBJECT) {
            ForEachItem(obj->child_objects,
                        mark_feature_headers_dirty,
                        NULL,
                        TRUE);
        }
    }

    LOG_PROC_EXIT_INT(DLIST_SUCCESS);
    return DLIST_SUCCESS;
}


/*
 * Find the volumes that are marked dirty and call the plug-in for the
 * volume's object to commit its changes to disk.
 */
static void commit_volumes(uint phase) {
    int rc = 0;
    logical_volume_t * vol;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(VolumeList);

    if (rc == DLIST_SUCCESS) {
        rc = GetObject(VolumeList,
                       sizeof(logical_volume_t),
                       VOLUME_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &vol);

        while ((rc == 0) & (vol != NULL)) {
            if (vol->flags & VOLFLAG_DIRTY) {

                /*
                 * Mark all the feature headers for all the EVMS objects
                 * in the stack dirty in phase one so that the new volume
                 * information will be recorded in all the feature headers.
                 */
                if (phase == FIRST_METADATA_WRITE) {
                    mark_feature_headers_dirty(vol->object,
                                               get_tag_for_object(vol->object),
                                               sizeof(storage_object_t),
                                               NULL,
                                               NULL);
                }

                /*
                 * If the volume's object has a feature header, that means
                 * the volume has no EVMS features.  It just has a feature
                 * header for setting the EVMS volume information.
                 */
                if (vol->object->feature_header != NULL) {
                    commit_parms_t commit_parms = {depth: 1,
                                                   phase: phase};

                    write_feature_header(vol->object,
                                         get_tag_for_object(vol->object),
                                         sizeof(storage_object_t),
                                         NULL,
                                         &commit_parms);

                } else {

                    /*
                     * Use the commit_object() function, which is normally
                     * called by ForEachItem(), to start the commit of the
                     * objects under the volume.  The last parameter is used to
                     * pass the phase of the commit.
                     */
                    commit_parms_t commit_parms = {depth: 1,
                                                   phase: phase};

                    commit_object(vol->object,
                                  get_tag_for_object(vol->object),
                                  sizeof(storage_object_t),
                                  NULL,
                                  &commit_parms);
                }

                /*
                 * Commit of volumes is finished after phase 2.
                 * VOLFLAG_NEW is left to its current setting so that the
                 * code at the end of commit will probe for file systems
                 * on the new volumes.
                 */
                if (phase >= SECOND_METADATA_WRITE) {
                    vol->flags &= ~VOLFLAG_DIRTY;
                }
            }

            if (rc == 0) {
                rc = GetNextObject(VolumeList,
                                   sizeof(logical_volume_t),
                                   VOLUME_TAG,
                                   (ADDRESS *) &vol);
            }
        }

        /* If we got an error it was from GetObject() or GetNextObject(). */
        if ((rc != DLIST_END_OF_LIST) &&
            (rc != DLIST_EMPTY)) {
            LOG_SERIOUS("Error code %d from when getting an object off the VolumeList.\n", rc);
            set_commit_error(SERIOUS, rc);
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the VolumeList.\n", rc);
        set_commit_error(WARNING, rc);
    }

    LOG_PROC_EXIT_VOID();
}


static evms_feature_header_t stop_data = {

    signature:      EVMS_FEATURE_HEADER_SIGNATURE,
    version:        {major:      EVMS_FEATURE_HEADER_MAJOR,
                     minor:      EVMS_FEATURE_HEADER_MINOR,
                     patchlevel: EVMS_FEATURE_HEADER_PATCHLEVEL
                    },
    engine_version: {major:      EVMS_MAJOR_VERSION,
                     minor:      EVMS_MINOR_VERSION,
                     patchlevel: EVMS_PATCHLEVEL_VERSION
                    },
    flags:          EVMS_VOLUME_DATA_STOP,
    object_depth:   1
};


/*
 * Given an object, check to see if it should have stop data written on it.
 * If so, call the function to write stop data.  This function is called
 * only with segments and regions that are topmost and writeable.
 * The parameters of this function are structured such that it can be called
 * by ForEachItem().
 */
static int write_stop_data_on_object(ADDRESS object,
                                     TAG     object_tag,
                                     uint    object_size,
                                     ADDRESS object_handle,
                                     ADDRESS parameters) {
    int rc = 0;
    storage_object_t * obj = (storage_object_t *) object;
    uint * phase = (uint *) parameters;
    sector_count_t  fh_lsn;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Request to write phase %d stop data on object %s.\n", *phase, obj->name);

    if (!(obj->flags & SOFLAG_CORRUPT)) {
        /*
         * Convert to disk endian format and calculate the stop data CRC
         * if we haven't done so already.
         */
        if (stop_data.crc == 0) {
            feature_header_cpu_to_disk(&stop_data);
            stop_data.crc = CPU_TO_DISK32(calculate_CRC(0xFFFFFFFFL, &stop_data, sizeof(evms_feature_header_t)));
        }

        LOG_DEBUG("Write phase %d stop data on object %s.\n", *phase, obj->name);

        status_message("Phase %d:  Writing stop data on object %s...\n", *phase, obj->name);
        if (*phase == FIRST_METADATA_WRITE) {
            fh_lsn = obj->size - FEATURE_HEADER_SECTORS;
        } else {
            fh_lsn = obj->size - (FEATURE_HEADER_SECTORS * 2);
        }
        rc = obj->plugin->functions.plugin->write(obj, fh_lsn, FEATURE_HEADER_SECTORS, &stop_data);
        if (rc != 0) {
            engine_user_message(NULL, NULL, "Error code %d from write of stop data on object %s.\n", rc, obj->name);
            set_commit_error(WARNING, rc);

            /* Don't let errors stop the commit process. */
            rc = 0;
        }

    } else {
        LOG_DEBUG("Object is corrupt.  I'm not writing stop data on it.\n");
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Put stop data on all the top level segments and regions.
 */
static void commit_stop_data(uint phase) {

    int rc = 0;
    dlist_t top_object_list;

    LOG_PROC_ENTRY();

    if ((phase == FIRST_METADATA_WRITE) ||
        (phase == SECOND_METADATA_WRITE)) {

        /* Get a list of disk, segments, and regions that are top objects. */
        rc = engine_get_object_list(DISK | SEGMENT | REGION,
                                    DATA_TYPE,
                                    NULL,
                                    TOPMOST | WRITEABLE,
                                    &top_object_list);

        if (rc == 0) {
            /*
             * Write stop data on the topmost, writeable storage objects that
             * need it.
             */
            ForEachItem(top_object_list,
                        write_stop_data_on_object,
                        &phase,
                        TRUE);

            DestroyList(&top_object_list, FALSE);

        } else {
            set_commit_error(WARNING, rc);
        }
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Find any dirty EVMS objects and call their plug-ins to commit
 * the changes to disk.
 */
static void commit_objects(uint phase) {

    int rc = 0;
    commit_parms_t commit_parms = {depth: 1,
                                   phase: phase};
    dlist_t top_object_list;


    LOG_PROC_ENTRY();

    /* Get a list of EVMS objects that are topmost. */
    rc = engine_get_object_list(EVMS_OBJECT,
                                DATA_TYPE,
                                NULL,
                                TOPMOST,
                                &top_object_list);

    if (rc == 0) {
        /*
         * Call commit_object() for each object in the top_object_list.  The
         * "Parameter" (third parameter in ForEachItem() ) is used to carry
         * the phase of the commit.
         */
        ForEachItem(top_object_list,
                    commit_object,
                    &commit_parms,
                    TRUE);

        DestroyList(&top_object_list, FALSE);

    } else {
        set_commit_error(WARNING, rc);
    }

    LOG_PROC_EXIT_VOID();
}


static int add_disk_handle(ADDRESS object,
                           TAG     object_tag,
                           uint    object_size,
                           ADDRESS object_handle,
                           ADDRESS parameters) {

    evms_rediscover_t * rediscover_parms = (evms_rediscover_t *) parameters;

    LOG_PROC_ENTRY();

    /* Safety check. */
    if (object_tag == DISK_TAG) {
        storage_object_t * disk = (storage_object_t *) object;

        // BUGBUG HACKHACK.  This is a layering violation.  The Engine doesn't
        // really know what the Device Manager has put into the private_data
        // field of the storage_object_t.  For now this gets us up and
        // running, but it should be changed to a layered implementation.

        rediscover_parms->drive_array[rediscover_parms->drive_count] = *((unsigned long *) disk->private_data);
        rediscover_parms->drive_count++;
    }

    LOG_PROC_EXIT_INT(DLIST_SUCCESS);
    return DLIST_SUCCESS;
}

/*
 * Call the kernel ioctl to have the EVMS runtime rediscover the new disk
 * configuration.
 * Don't call set_commit_error() in this function since it can be called
 * by other functions outside of the commit process.
 */
int kernel_rediscover() {
    int rc = 0;

    int status;
    uint disk_count = 0;
    evms_rediscover_t kernel_rediscover;

    LOG_PROC_ENTRY();

    // BUGBUG need to add code here to determine which drives actually have
    // changes on them and only rediscover those drives.  For now just pass
    // in the handles for all drives.

    GetListSize(DiskList, &disk_count);

    if (disk_count > 0) {
        kernel_rediscover.drive_array = malloc(sizeof(unsigned long) * disk_count);
        kernel_rediscover.drive_count = 0;

        /* Put each disk's kernel handle into kernel_rediscover. */
        ForEachItem(DiskList,
                    add_disk_handle,
                    &kernel_rediscover,
                    TRUE);

        kernel_rediscover.status = 0;

        status = ioctl(evms_block_dev_handle, EVMS_REDISCOVER_VOLUMES, &kernel_rediscover);

        if (status) {
            rc = errno;
        } else {
            rc = kernel_rediscover.status;
        }

        if (rc == 0) {
            /*
             * Get a new list of the kernel's view of volumes in the
             * system.
             */
            rc = get_kernel_volume_data();

            if (rc == 0) {
                /*
                 * Make sure our volume minors are in sync with the kernel's
                 * view of volume minors.
                 */
                rc = sync_volume_minors_with_kernel();

                if (rc != 0) {
                    engine_user_message(NULL,NULL,"Error code %d when syncing volume minor numbers with the EVMS kernel.\n", rc);
                }

            } else {
                engine_user_message(NULL,NULL,"Error code %d when getting the volumes exported by the EVMS kernel.\n", rc);
            }

        } else {
            engine_user_message(NULL, NULL, "Error code %d from ioctl to rediscover EVMS kernel volumes.\n", rc);
        }

    } else {
        LOG_DEBUG("The Engine did not discover any disks.  There is nothing to rediscover.\n");
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Find any volumes that are marked for mkfs and run mkfs on them.
 */
static void mkfs_volumes() {

    int rc = 0;
    logical_volume_t * vol;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(VolumeList);

    if (rc == DLIST_SUCCESS) {

        rc = GetObject(VolumeList,
                       sizeof(logical_volume_t),
                       VOLUME_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &vol);

        while ((rc == 0) && (vol != NULL)) {
            if (vol->flags & VOLFLAG_MKFS) {

                /* Make sure there is an FSIM to do the mkfs. */
                if (vol->file_system_manager != NULL) {
                    status_message("Running %s mkfs on volume %s...\n", vol->file_system_manager->short_name, vol->name);
                    rc = vol->file_system_manager->functions.fsim->mkfs(vol, vol->mkfs_options);

                    if (rc == 0) {
                        vol->original_fsim = vol->file_system_manager;
                    } else {
                        set_commit_error(WARNING, rc);
                        engine_user_message(NULL, NULL, "FSIM plug-in %s returned error code %d when called to mkfs on volume %s to %lld sectors.\n",vol->file_system_manager->short_name, rc, vol->name);

                        /* Don't let errors stop the commit process. */
                        rc = 0;

                        /*
                         * mkfs failed.  Reset the FSIM that manages this
                         * volume.
                         */
                        vol->file_system_manager = vol->original_fsim;
                    }
                }
                vol->flags &= ~VOLFLAG_MKFS;

                free_option_array(vol->mkfs_options);
                vol->mkfs_options = NULL;
            }

            /*
             * Ignore error codes from the FSIM's mkfs().  Continue with the
             * rest of the volumes.
             */
            rc = GetNextObject(VolumeList,
                               sizeof(logical_volume_t),
                               VOLUME_TAG,
                               (ADDRESS *) &vol);
        }

        /* Clear up acceptable return codes */
        if ((rc == DLIST_END_OF_LIST) ||
            (rc == DLIST_EMPTY)) {
            rc = DLIST_SUCCESS;
        }

        if (rc != 0) {
            LOG_SERIOUS("Error code %d from when getting an object off the VolumeList.\n", rc);
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the VolumeList.\n", rc);
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Find any volumes that have expanded in size and call the FSIM for
 * the volume to do the expand.
 */
static void expand_volumes() {
    int rc = 0;
    logical_volume_t * vol;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(VolumeList);

    if (rc == DLIST_SUCCESS) {
        rc = GetObject(VolumeList,
                       sizeof(logical_volume_t),
                       VOLUME_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &vol);

        while ((rc == 0) && (vol != NULL)) {
            if (vol->vol_size > vol->original_vol_size) {

                if (vol->file_system_manager != NULL) {
                    u_int64_t new_size = vol->vol_size;

                    status_message("Expanding volume %s...\n", vol->name);
                    rc = vol->file_system_manager->functions.fsim->expand(vol, &new_size);

                    if (rc == 0) {
                        /* Save the new file system size. */
                        vol->fs_size = new_size;

                    } else {
                        set_commit_error(WARNING, rc);
                        engine_user_message(NULL, NULL, "FSIM plug-in %s returned error code %d when called to expand volume %s to %lld sectors.\n",vol->file_system_manager->short_name, rc, vol->name, new_size);

                        /* Don't let errors stop the commit process. */
                        rc = 0;
                    }
                }

                /* Set the volume's new original size. */
                vol->original_vol_size = vol->vol_size;
            }

            if (rc == 0) {
                rc = GetNextObject(VolumeList,
                                   sizeof(logical_volume_t),
                                   VOLUME_TAG,
                                   (ADDRESS *) &vol);
            }
        }

        if ((rc != DLIST_END_OF_LIST) &&
            (rc != DLIST_EMPTY)) {
            LOG_SERIOUS("Error code %d from when getting an object off the VolumeList.\n", rc);
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the VolumeList.\n", rc);
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Find any volumes that are marked for fsck and run fsck on them.
 */
static void fsck_volumes() {

    int rc = 0;
    logical_volume_t * vol;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(VolumeList);

    if (rc == DLIST_SUCCESS) {

        rc = GetObject(VolumeList,
                       sizeof(logical_volume_t),
                       VOLUME_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &vol);

        while ((rc == 0) && (vol != NULL)) {
            if (vol->flags & VOLFLAG_FSCK) {

                /* Make sure there is an FSIM to do the fsck. */
                if (vol->file_system_manager != NULL) {
                    status_message("Running fsck on volume %s...\n", vol->name);
                    rc = vol->file_system_manager->functions.fsim->fsck(vol, vol->fsck_options);

                    if (rc != 0) {
                        set_commit_error(WARNING, rc);
                        engine_user_message(NULL, NULL, "FSIM plug-in %s returned error code %d when called to fsck volume %s.\n",vol->file_system_manager->short_name, rc, vol->name);

                        /* Don't let errors stop the commit process. */
                        rc = 0;
                    }
                }
                vol->flags &= ~VOLFLAG_FSCK;

                free_option_array(vol->fsck_options);
                vol->fsck_options = NULL;
            }

            /*
             * Ignore error codes from the FSIM's fsck().  Continue with the
             * rest of the volumes.
             */
            rc = GetNextObject(VolumeList,
                               sizeof(logical_volume_t),
                               VOLUME_TAG,
                               (ADDRESS *) &vol);
        }

        /* Clear up acceptable return codes */
        if ((rc == DLIST_END_OF_LIST) ||
            (rc == DLIST_EMPTY)) {
            rc = DLIST_SUCCESS;
        }

        if (rc != 0) {
            LOG_SERIOUS("Error code %d from when getting an object off the VolumeList.\n", rc);
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the VolumeList.\n", rc);
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Find any volumes that are marked for defrag and run defrag on them.
 */
static void defrag_volumes() {

    int rc = 0;
    logical_volume_t * vol;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(VolumeList);

    if (rc == DLIST_SUCCESS) {

        rc = GetObject(VolumeList,
                       sizeof(logical_volume_t),
                       VOLUME_TAG,
                       NULL,
                       TRUE,
                       (ADDRESS *) &vol);

        while ((rc == 0) && (vol != NULL)) {
            if (vol->flags & VOLFLAG_DEFRAG) {

                /* Make sure there is an FSIM to do the defrag. */
                if (vol->file_system_manager != NULL) {
                    status_message("Running defrag on volume %s...\n", vol->name);
                    rc = vol->file_system_manager->functions.fsim->defrag(vol, vol->defrag_options);

                    if (rc != 0) {
                        set_commit_error(WARNING, rc);
                        engine_user_message(NULL, NULL, "FSIM plug-in %s returned error code %d when called to defrag volume %s.\n",vol->file_system_manager->short_name, rc, vol->name);

                        /* Don't let errors stop the commit process. */
                        rc = 0;
                    }
                    vol->flags &= ~VOLFLAG_DEFRAG;

                    free_option_array(vol->defrag_options);
                    vol->defrag_options = NULL;
                }
            }

            /*
             * Ignore error codes from the FSIM's defrag().  Continue with the
             * rest of the volumes.
             */
            rc = GetNextObject(VolumeList,
                               sizeof(logical_volume_t),
                               VOLUME_TAG,
                               (ADDRESS *) &vol);
        }

        /* Clear up acceptable return codes */
        if ((rc == DLIST_END_OF_LIST) ||
            (rc == DLIST_EMPTY)) {
            rc = DLIST_SUCCESS;
        }

        if (rc != 0) {
            LOG_SERIOUS("Error code %d from when getting an object off the VolumeList.\n", rc);
        }

    } else {
        /* GoToStartOfList shouldn't fail.  Log an error if it does. */
        LOG_WARNING("Error code %d when going to the start of the VolumeList.\n", rc);
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Check if a given volume has a dev node and the dev node minor is correct.
 * The parameters are structured so that this function can be called by the
 * ForEachItem() dlist processor.
 */
static int vol_has_dev_node(ADDRESS object,
                            TAG     object_tag,
                            uint    object_size,
                            ADDRESS object_handle,
                            ADDRESS parameters) {
    int rc = 0;

    LOG_PROC_ENTRY();

    /* Safety check */
    if (object_tag == VOLUME_TAG) {

        logical_volume_t * vol = (logical_volume_t *) object;

        rc = hasa_dev_node(vol->name, vol->minor_number);
        if ((rc == ENOENT) || (rc == EEXIST)) {
            vol->flags |= VOLFLAG_NEEDS_DEV_NODE;
        } else {
            vol->flags &= ~VOLFLAG_NEEDS_DEV_NODE;
        }
    }

    LOG_PROC_EXIT_INT(DLIST_SUCCESS);
    return DLIST_SUCCESS;
}


/*
 * Check to make sure all volumes have a dev node.
 */
void check_volume_dev_nodes(void) {

    /*
     * If devfs is installed it will keep the /dev tree up to date.  However,
     * volumes that are not exported by the kernel will not be handled by devfs.
     * So we scan the VolumeList and make sure the VOLFLAG_NEEDS_DEV_NODE flag
     * is set correctly for each volume.
     */

    ForEachItem(VolumeList,
                vol_has_dev_node,
                NULL,
                TRUE);
}


/*
 * Check if this volume is new.  If so call find_fsim_for_volume() to see if
 * any FSIM will claim the volume.
 */
static int find_fsim_for_new_volume(ADDRESS object,
                                    TAG     object_tag,
                                    uint    object_size,
                                    ADDRESS object_handle,
                                    ADDRESS parameters) {
    LOG_PROC_ENTRY();

    /* Safety check */
    if (object_tag == VOLUME_TAG) {
        logical_volume_t * vol = (logical_volume_t *) object;

        if (vol->flags & VOLFLAG_NEW) {
            if (!(vol->flags & VOLFLAG_NEEDS_DEV_NODE)) {
                status_message("Checking for a file system on new volume %s...\n", vol->name);
                find_fsim_for_volume(vol);
            }
            vol->flags &= ~VOLFLAG_NEW;
        }
    }

    LOG_PROC_EXIT_INT(DLIST_SUCCESS);
    return DLIST_SUCCESS;
}


/*
 * Find he new volumes and let the FSIMs see if they can find a file system
 * on them.
 */
static void find_fsim_for_new_volumes(void) {

    LOG_PROC_ENTRY();

    ForEachItem(VolumeList,
                find_fsim_for_new_volume,
                NULL,
                TRUE);

    LOG_PROC_EXIT_VOID();
}


/*
 * Check if a given object is dirty.
 * This is a service function of is_something_dirty() below.
 * The parameters are structured so that this function can be called by the
 * ForEachItem() dlist processor.
 */
static int is_dirty(ADDRESS object,
                    TAG     object_tag,
                    uint    object_size,
                    ADDRESS object_handle,
                    ADDRESS parameters) {
    int rc = 0;

    LOG_PROC_ENTRY();

    switch (object_tag) {
        case VOLUME_TAG:
            {
                logical_volume_t * vol = (logical_volume_t *) object;

                if (vol->flags & (VOLFLAG_DIRTY | VOLFLAG_NEEDS_DEV_NODE)) {
                    /* Set a non-zero error to stop the searching. */
                    rc = EBUSY;
                }
            }
            break;

        case CONTAINER_TAG:
            {
                storage_container_t * con = (storage_container_t *) object;

                if (con->flags & SCFLAG_DIRTY) {
                    /* Set a non-zero error to stop the searching. */
                    rc = EBUSY;
                }
            }
            break;

        case EVMS_OBJECT_TAG:
        case REGION_TAG:
        case SEGMENT_TAG:
        case DISK_TAG:
            {
                storage_object_t * obj = (storage_object_t *) object;

                if (obj->flags & SOFLAG_DIRTY) {
                    /* Set a non-zero error to stop the searching. */
                    rc = EBUSY;
                }
            }
            break;

        default:
            LOG_DEBUG("Don't know how to check if an object with tag %d is dirty.\n", object_tag);
            break;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Check if anything in the Engine is dirty.
 */
static BOOLEAN is_anything_dirty() {

    BOOLEAN result = TRUE;
    int rc = 0;

    LOG_PROC_ENTRY();

    rc = ForEachItem(VolumeList, is_dirty, NULL, TRUE);

    if (rc == 0) {
        rc = ForEachItem(EVMSObjectList, is_dirty, NULL, TRUE);

        if (rc == 0) {
            rc = ForEachItem(ContainerList, is_dirty, NULL, TRUE);

            if (rc == 0) {
                rc = ForEachItem(RegionList, is_dirty, NULL, TRUE);

                if (rc == 0) {
                    rc = ForEachItem(SegmentList, is_dirty, NULL, TRUE);

                    if (rc == 0) {
                        rc = ForEachItem(DiskList, is_dirty, NULL, TRUE);
                    }
                }
            }
        }
    }

    result = (rc != 0);

    LOG_PROC_EXIT_BOOLEAN(result);
    return result;
}


/*
 * Commit to disk the configuration changes that are in memory.
 */
int evms_commit_changes(commit_status_func_t callback) {
    int rc = 0;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {

        /* Make sure we have the EVMS block device open */
        if (evms_block_dev_handle == 0) {
            open_evms_block_dev();
        }

        if (evms_block_dev_handle > 0) {
            commit_in_progress = TRUE;
            status_callback = callback;

            /*
             * Make sure the /dev/evms tree is in sync with the kernel
             * so that FSIMs that access the volume through the dev tree
             * get to the correct volume.
             */
            status_message("Updating the /dev/evms tree...\n");
            evms_update_evms_dev_tree(NO_MESSAGES);

            need_reboot = FALSE;
            rc = remove_volumes();

            if (rc == 0) {
                rc = unmkfs_volumes();

                if (rc == 0) {
                    rc = shrink_volumes();

                    if (rc == 0) {
                        rc = delete_volumes();

                        if (rc == 0) {
                            uint phase;

                            clear_commit_errors();

                            status_message("Processing the Kill Sectors List...\n");
                            kill_sectors();

                            for (phase = SETUP; phase <= POST_REDISCOVER; phase++) {
                                slide_segments(phase);
                                commit_segments(phase);
                                move_segments(phase);
                                commit_containers(phase);
                                commit_regions(phase);
                                commit_stop_data(phase);
                                commit_volumes(phase);
                                commit_objects(phase);

                                if (phase == SECOND_METADATA_WRITE) {
                                    status_message("Rediscovering volumes in the kernel...\n");
                                    rc = kernel_rediscover();
                                    if (rc != 0) {
                                        set_commit_error(SERIOUS, rc);
                                    }
                                }
                            }

                            /*
                             * Make sure the /dev/evms tree is in sync with the
                             * kernel so that FSIMs that access the volume through
                             * the dev tree get to the correct volume.
                             */
                            status_message("Updating the /dev/evms tree...\n");
                            evms_update_evms_dev_tree(NO_MESSAGES);

                            mkfs_volumes();

                            expand_volumes();

                            fsck_volumes();

                            defrag_volumes();

                            /* Check to see if all volumes have a dev node. */
                            status_message("Checking to make sure all volumes have a dev node...\n");
                            check_volume_dev_nodes();

                            /*
                             * Check if any of the new volumes has a file system
                             * on it.
                             */
                            find_fsim_for_new_volumes();

                            status_message("Checking if anything in the system is still dirty...\n");
                            changes_pending = is_anything_dirty();

                            rc = get_commit_error();
                        }
                    }
                }

                if (need_reboot) {
                    char * choices[] = {"Reboot now", "Reboot later", NULL};
                    int answer = 0;     /* Default is "Reboot now". */

                    engine_user_message(&answer, choices,
                                        "The system must be rebooted for all the changes to take effect.  "
                                        "If you do not reboot now, some of the device nodes for the mounted volumes will be in an "
                                        "inconsistent state and may affect the operation of the system.  "
                                        "Do you want to reboot now or reboot at a later time?\n");
                    switch (answer) {
                        case 0:
                            {
                                /* Reboot now */
                                status_message("Rebooting the system...\n");
                                fsync(log_file);

                                /* Switch to runlevel 6. */
                                execl("/sbin/init", "init", "6", NULL);
                                execl("/etc/init",  "init", "6", NULL);
                                execl("/bin/init",  "init", "6", NULL);

                                engine_user_message(NULL, NULL,
                                                    "Unable to switch to runlevel 6.  "
                                                    "You must reboot the system manually.\n");
                            }
                            break;

                        case 1:
                        default:
                            break;
                    }
                }
            }

            status_message("Finished committing changes.\n");
            commit_in_progress = FALSE;
            status_callback = NULL;

        } else {
            /*
             * There was an error opening the EVMS block device.
             * evms_block_dev_handle contains the error code.
             */
            rc = evms_block_dev_handle;
            LOG_SERIOUS("Error code %d when opening the EVMS block device.\n", rc);
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}
