/*
 * Everybuddy
 * Copyright (C) 1999, Torrey Searle <tsearle@uci.edu>
 *
 * code orriginally derrived from 
 *
 * gaim
 *
 * Copyright (C) 1998-1999, Mark Spencer <markster@marko.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/signal.h>
#include <errno.h>
#include <math.h>
#include "sound.h"
#include "globals.h"
#include "util.h"
#include "dialog.h"

#ifdef ESD_SOUND
#include <esd.h>
#endif

#ifdef ARTS_SOUND
#include <artsc.h>
#include <audiofile.h>
#define BUFFERED_FRAME_COUNT 20000
#endif


static void play_audio(gchar *soundfile)
{
    int fd;
    struct stat info;	
    char * buf;
    char * audio_device;
    
    fd = open(soundfile,O_RDONLY);
    if (fd <= 0)
    {
#ifdef DEBUG
		fprintf(stderr, "Cannot open file %s\n", soundfile);
#endif
		return;
    }
    fstat(fd, &info);
    buf = alloca(info.st_size);
    read(fd,buf,24);
    read(fd,buf,info.st_size-24);
    close(fd);
#ifdef DEBUG
    fprintf(stderr, "File is %ld bytes\n", info.st_size);
#endif

    audio_device = getenv("AUDIODEV");
    if (audio_device == NULL) {
      /* OK, we're not running on a SunRay */
      audio_device = "/dev/audio";
    }

#ifdef DEBUG
    fprintf(stderr, "sending to %s\n", audio_device);
#endif
    
    fd = open(audio_device, O_WRONLY | O_EXCL);
    if (fd < 0)
		return;
    write(fd, buf, info.st_size-24);
    close(fd);
}

static int can_play_audio()
{
        /* FIXME check for write access and such. */
        return 1;

}

/*
** This routine converts from linear to ulaw
**
** Craig Reese: IDA/Supercomputing Research Center
** Joe Campbell: Department of Defense
** 29 September 1989
**
** References:
** 1) CCITT Recommendation G.711  (very difficult to follow)
** 2) "A New Digital Technique for Implementation of Any
**     Continuous PCM Companding Law," Villeret, Michel,
**     et al. 1973 IEEE Int. Conf. on Communications, Vol 1,
**     1973, pg. 11.12-11.17
** 3) MIL-STD-188-113,"Interoperability and Performance Standards
**     for Analog-to_Digital Conversion Techniques,"
**     17 February 1987
**
** Input: Signed 16 bit linear sample
** Output: 8 bit ulaw sample
*/

#define ZEROTRAP    /* turn on the trap as per the MIL-STD */
#define BIAS 0x84   /* define the add-in bias for 16 bit samples */
#define CLIP 32635

unsigned char linear2ulaw(int sample)
{
  static int exp_lut[256] = {0,0,1,1,2,2,2,2,3,3,3,3,3,3,3,3,
                             4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
                             5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
                             5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
                             6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
                             6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
                             6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
                             6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
                             7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                             7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                             7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                             7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                             7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                             7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                             7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                             7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7};
  int sign, exponent, mantissa;
  unsigned char ulawbyte;

  /* Get the sample into sign-magnitude. */
  sign = (sample >> 8) & 0x80;		/* set aside the sign */
  if (sign != 0) sample = -sample;		/* get magnitude */
  if (sample > CLIP) sample = CLIP;		/* clip the magnitude */

  /* Convert from 16 bit linear to ulaw. */
  sample = sample + BIAS;
  exponent = exp_lut[(sample >> 7) & 0xFF];
  mantissa = (sample >> (exponent + 3)) & 0x0F;
  ulawbyte = ~(sign | (exponent << 4) | mantissa);
#ifdef ZEROTRAP
  if (ulawbyte == 0) ulawbyte = 0x02;	/* optional CCITT trap */
#endif

  return(ulawbyte);
}

/*
** This routine converts from ulaw to 16 bit linear.
**
** Craig Reese: IDA/Supercomputing Research Center
** 29 September 1989
**
** References:
** 1) CCITT Recommendation G.711  (very difficult to follow)
** 2) MIL-STD-188-113,"Interoperability and Performance Standards
**     for Analog-to_Digital Conversion Techniques,"
**     17 February 1987
**
** Input: 8 bit ulaw sample
** Output: signed 16 bit linear sample
** Z-note -- this is from libaudiofile.  Thanks guys!
*/
int _af_ulaw2linear (unsigned char ulawbyte)
{
  static int exp_lut[8] = {0,132,396,924,1980,4092,8316,16764};
  int sign, exponent, mantissa, sample;

  ulawbyte = ~ulawbyte;
  sign = (ulawbyte & 0x80);
  exponent = (ulawbyte >> 4) & 0x07;
  mantissa = ulawbyte & 0x0F;
  sample = exp_lut[exponent] + (mantissa << (exponent + 3));
  if (sign != 0) sample = -sample;
  {
  	return(sample);
  }
}

#ifdef ESD_SOUND

static int can_play_esd()
{
	return TRUE;
}

int play_esd_file(gchar *soundfile)
{
    int esd_stat;
	int fd;
    struct stat info;	
    char * buf;
	
    
    fd = open(soundfile,O_RDONLY);
    if (fd <= 0)
    {
#ifdef DEBUG
		fprintf(stderr, "Cannot open file %s\n", soundfile);
#endif
		return -1;
    }
    fstat(fd, &info);
    buf = alloca(info.st_size);
    read(fd,buf,info.st_size);
    close(fd);
    esd_stat = esd_play_file(NULL, soundfile, 1);
    if (!esd_stat)
    {
		if (errno == ENOENT)
		{
#ifdef DEBUG
	    	fprintf(stderr, "Everybuddy: File not found - file = \"%s\"\n", soundfile);
#endif
		}
		else
		{
#ifdef DEBUG
	    	fprintf(stderr, "Everybuddy: Esd play file failed with code %d\n", errno);
#endif
		}
    }
    return esd_stat;
}

#else

int can_play_esd()
{
    return FALSE;
}

int play_esd_file(gchar *soundfile)
{
    return FALSE;
}

#endif

// *** ARTS Support ***
// Comments/Questions -> sebastian.held@gmx.de
#ifdef ARTS_SOUND
int can_play_arts()
{
    int err;
    
    if (err = arts_init() != 0)
    {
#ifdef DEBUG
	fprintf(stderr, "WARNING (NO ERROR!):\ncan_play_arts(): arts_init() failed: %s\nIs artsd running?\n", arts_error_text( err ));
#endif
	return FALSE;
    }
    arts_free();
    return TRUE;
}

int play_arts_file(gchar *soundfile)
{
    AFfilehandle fd;
    AFframecount frameCount, count;
    arts_stream_t artsStream;
    int frameSize, channelCount, sampleFormat, sampleWidth, err;
    double sampleRate;
    char * buf;
    
    if ((err = arts_init()) < 0)
    {
#ifdef DEBUG
	fprintf(stderr, "play_arts_file(): arts_init() failed: %s\n", arts_error_text( err ));
#endif
	return FALSE;
    }

    fd = afOpenFile( soundfile, "r", NULL );
    if (fd == AF_NULL_FILEHANDLE)
    {
#ifdef DEBUG
		fprintf(stderr, "play_arts_file(): Cannot open file %s\n", soundfile);
#endif
		arts_free();
		return FALSE;
    }
    frameCount = afGetFrameCount( fd, AF_DEFAULT_TRACK );
    frameSize = afGetFrameSize( fd, AF_DEFAULT_TRACK, 1 );
    channelCount = afGetChannels( fd, AF_DEFAULT_TRACK );
    sampleRate = afGetRate( fd, AF_DEFAULT_TRACK );
    afGetSampleFormat( fd, AF_DEFAULT_TRACK, &sampleFormat, &sampleWidth );
#ifdef DEBUG
    fprintf(stderr, "play_arts_file(): frameCount: %d\nframeSize: %i\nchannelCount: %i\nsampleRate: %.2f\n", frameCount, frameSize, channelCount, sampleRate );
#endif

    artsStream = arts_play_stream( sampleRate, sampleWidth, channelCount, "EVERYBUDDY" );
    buf = (char *) malloc( BUFFERED_FRAME_COUNT * frameSize );
    count = afReadFrames( fd, AF_DEFAULT_TRACK, buf, BUFFERED_FRAME_COUNT );
    do
    {
        err = arts_write( artsStream, buf, count * frameSize );
#ifdef DEBUG
        fprintf( stderr, "count = %d\n", count);
        fprintf( stderr, "play_arts_file(): written bytes in artsStream: %i\n", err );
        if(err < 0)
	{
	    fprintf( stderr, "arts_write error: %s\n", arts_error_text( err ) );
	}
#endif
    }
    while ((count = afReadFrames( fd, AF_DEFAULT_TRACK, buf, BUFFERED_FRAME_COUNT)) && (err >= 0));
    free( buf );
    arts_close_stream( artsStream );

    afCloseFile( fd );
    arts_free();
    return (err<0?FALSE:TRUE);
}
#else
int can_play_arts()
{
    return FALSE;
}

int play_arts_file(gchar *soundfile)
{
    return FALSE;
}
#endif

char* auscale(char* infile, char*outfile, float scaling)
{
  int fd_in, fd_out; 
  int bytes_read, x ;         
  unsigned char buf,   buf2[24]; /* 24-byte .au header */
  unsigned char map[256]; /* 8-bit mapping */

  if (scaling == 0.0) return(infile);  /* save time */

  if ((fd_in = open(infile, O_RDONLY, 0600)) < 0)
    {
      fprintf(stderr,"Failed opening %s\n",infile);
      return(infile);
    }

  if ((fd_out = open(outfile, O_WRONLY | O_CREAT)) < 0)
    {
      fprintf(stderr,"Failed opening %s\n",outfile);
      return(infile);
    }

  fprintf(stderr, "Scaling = %f dB\n",scaling);
  scaling = pow(10.0,scaling/20.0);

  /* Build mapping */
  for (x=0; x<256 ; x++) map[x]=linear2ulaw(scaling*_af_ulaw2linear(x));
  
  /* Shift the .au header */
  write(fd_out, buf2, read(fd_in, buf2, sizeof(buf2)));
  
  /* todo - check that we're using 8-bit .au data! */

  /* Write the mapped data out */
  while((bytes_read = read(fd_in, &buf, 1)) > 0) 
    {
      write(fd_out,map+buf,1);
    }
  close(fd_in);
  close(fd_out);
  
  return outfile;
}


/* file_ok 
 *
 * This routine test to see if file can be found and accessed
 * It issues an error message if the file open fails
 * Returns:
 * FALSE - If file open fails
 * TRUE - If file open succeeds
 *
 * MIZHI 04162001
 * I modified this function to use fstat, much cleaner...
 */

gboolean file_ok(gchar *soundfile)
{
  struct stat s;
  gchar message[1024];
    
  if (stat(soundfile, &s) == -1)
    {
      /* find and report the error */
      switch (errno)
	{
	case ENOENT:
	  g_snprintf(message, 1024, "Sound file \'%s\', Not Found", soundfile);
	  break;
	  
	case EACCES:
	  g_snprintf(message, 1024, "Sound file \'%s\', Access Not Allowed", soundfile);
	  break;
	  
	default:
	  g_snprintf(message, 1024, "Error in opening sound file \'%s\', error code = %d", soundfile, errno);
	  break;
	}
      
      g_warning(message);
      return FALSE;
    }
  return TRUE;
}


void playsoundfile(gchar *soundfile)
{
    
#ifdef _WIN32
    return;
#endif

    int pid;
    char tmp_soundfile[strlen(config_dir)+6];
    strcpy(tmp_soundfile, config_dir);
    strcat(tmp_soundfile,"tmp.au");

    if (!file_ok(soundfile))
		return;


    pid = fork();
    if (pid < 0)
	{
		return;
	}
    else if (pid == 0)
    {
                soundfile = auscale(soundfile,tmp_soundfile,volume_sound);

		if(can_play_arts())
		{
			if (play_arts_file(soundfile))
			{
	   		 	_exit(0);
			}
		}
		if(can_play_esd())
		{
			if (play_esd_file(soundfile))
			{
	   		 	_exit(0);
			}
		}
		/* handle sound direct to /dev/audio */
		if (can_play_audio())
		{
	   		play_audio(soundfile);
		}
		_exit(0);
	    
    }
    else
	{
		gtk_timeout_add(100, (GtkFunction)clean_pid, NULL);
	}

}

void play_sound(int sound)
{
	if (do_no_sound_when_away && is_away)
	    return;
	switch(sound) 
	{
		case BUDDY_AWAY:
		{
			playsoundfile(BuddyAwayFilename);
		}
		break;
		case BUDDY_ARRIVE:
		{
			playsoundfile(BuddyArriveFilename);
		}
		break;
		case BUDDY_LEAVE:
		{
			playsoundfile(BuddyLeaveFilename);
		}
		break;
		case SEND:
		{
			playsoundfile(SendFilename);
		}
		break;
		case RECEIVE:
		{
			playsoundfile(RecieveFilename);
			break;
		}
		case FIRSTMSG:
		{
		    playsoundfile(FirstMsgFilename);
		    break;
		}
		default:
			return;
    }
}





