/////////////////////////////////////////////////////////////
//                                                         //
// Copyright (c) 2003-2011 by The University of Queensland //
// Earth Systems Science Computational Centre (ESSCC)      //
// http://www.uq.edu.au/esscc                              //
//                                                         //
// Primary Business: Brisbane, Queensland, Australia       //
// Licensed under the Open Software License version 3.0    //
// http://www.opensource.org/licenses/osl-3.0.php          //
//                                                         //
/////////////////////////////////////////////////////////////

#include <boost/version.hpp>
#include <boost/python.hpp>
#include "Python/esys/lsm/ParticlePy.h"

namespace esys
{
  namespace lsm
  {
    ParticlePy::ParticlePy() : CParticle()
    {
    }

    ParticlePy::ParticlePy(int id, const Vec3Py &posn, double radius, double mass)
      : CParticle(radius, mass, posn, Vec3(), Vec3(), id, true)
    {
    }

    ParticlePy::ParticlePy(const CParticle &p) : CParticle(p)
    {
    }

    ParticlePy::ParticlePy(const ParticlePy &p) : CParticle(p)
    {
    }

    Vec3Py ParticlePy::getPosn() const
    {
      return Vec3Py(getPos());
    }

    void ParticlePy::setPosn(const Vec3Py &posn)
    {
      setPos(posn);
    }

    Vec3Py ParticlePy::getVelocity() const
    {
      return Vec3Py(getVel());
    }

    void ParticlePy::setVelocity(const Vec3Py &vel)
    {
      setVel(vel);
    }

    Vec3Py ParticlePy::getInitialPosn() const
    {
      return Vec3Py(getInitPos());
    }

    Vec3Py ParticlePy::getAcceleration() const
    {
      return Vec3Py(getForce()*getInvMass());
    }

    void ParticlePy::setAcceleration(const Vec3Py &accel)
    {
      setForce(accel*getMass());
    }

    Vec3Py ParticlePy::getForce() const
    {
      return Vec3Py(getForce());
    }

    void ParticlePy::setForce(const Vec3Py &force)
    {
      setForce(force);
    }

    using boost::python::arg;
    void exportParticle()
    {
      // Check that Boost 1.34.0 or higher is being used.
      // If so, disable auto-generation of C++ signatures for Epydoc
      // (which stumbles over indentation in the auto-generated strings).
      #if ((BOOST_VERSION / 100000 >= 1) \
          && (BOOST_VERSION / 100 % 1000 >= 34)) \
          || (BOOST_VERSION / 100000 >= 2)
        boost::python::docstring_options no_autogen(true,false);
      #endif

      boost::python::class_<ParticlePy>("NRotSphere")
        .def(boost::python::init<>())
        .def(boost::python::init<const ParticlePy &>())
        .def(
          boost::python::init<int,const Vec3Py &, double, double>(
            (
              arg("id"),
              arg("posn"),
              arg("radius"),
              arg("mass")
            ),
            "Construct a rotational spherical particle.\n"
            "@type id: int\n"
            "@kwarg id: Unique identifier for particle.\n"
            "@type posn: L{Vec3<esys.lsm.util.FoundationPy.Vec3>}\n"
            "@kwarg posn: Initial position of particle, centre-point of sphere."
            "@type radius: float\n"
            "@kwarg radius: The radius of the sphere.\n"
            "@type mass: float\n"
            "@kwarg mass: Mass of particle."
          )
        )
        .def("getId",           			&ParticlePy::getID)
        .def("getTag",                &ParticlePy::getTag)
        .def("setTag",                &ParticlePy::setTag)
        .def("getPosn",         			&ParticlePy::getPosn)
        .def("getPosition",         	&ParticlePy::getPosn)
        .def("setPosn",         			&ParticlePy::setPosn)
        .def("setPosition",         	&ParticlePy::setPosn)
        .def("getInitialPosn",  			&ParticlePy::getInitialPosn)
        .def("getInitialPosition",		&ParticlePy::getInitialPosn)
        .def("getVelocity",     			&ParticlePy::getVelocity)
        .def("getLinearVelocity",     &ParticlePy::getVelocity)
        .def("setVelocity",     			&ParticlePy::setVelocity)
        .def("setLinearVelocity",     &ParticlePy::setVelocity)
        .def("getAcceleration", 			&ParticlePy::getAcceleration)
        .def("getLinearAcceleration",	&ParticlePy::getAcceleration)
        .def("setAcceleration",			 	&ParticlePy::setAcceleration)
        .def("setLinearAcceleration",	&ParticlePy::setAcceleration)
        .def("getRadius",       			&ParticlePy::getRad)
        .def("getRad",          			&ParticlePy::getRad)
        .def("getCenter",       			&ParticlePy::getPosn)
        .def("getCentre",       			&ParticlePy::getPosn)
        .def("getMass",         			&ParticlePy::getMass)
        .def("getForce",        			&ParticlePy::getForce)
        .def("getLinearForce",        &ParticlePy::getForce)
        .def("setForce",        			&ParticlePy::setForce)
        .def("setLinearForce",        &ParticlePy::setForce)
        ;
    }
  }
}
