#include <glib.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>

#include "entity.h"
#include "modulegen.h"
#include "entity-exec-class.h"

#include "rargv.h"


static void
print_usage (void)
{
    g_print ("Usage: %s  [OPTION]... [FILE]\n"
	     "  -V           Print version and exit.\n"
	     "  -g           Turn on debug messages. "
	     "Accepts domain as argument, or 'all'\n"
	     "  -a           Load all files on command line into Entity.\n"
	     "  -c           Execution class to bind this file to.\n"
	     "  -m           Generate module config files.\n"
	     "  -l           Specify alternate location for libraries and configuration directories.\n"
	     "  -h           Print this and die.\n"
	     "%s %s, compiled: %s %s\n"
	     "Report bugs to <entity-devel@lists.sourceforge.net>\n",
	     PACKAGE, PACKAGE, VERSION, __DATE__, __TIME__);
}

static gchar *
full_path (gchar * filename)
{
    gchar *full_path;
    gchar buf[PATH_MAX];

    if (filename[0] == '/') {
	full_path = g_strdup (filename);
    } else {
	getcwd (buf, PATH_MAX);
	full_path = g_strconcat (buf, "/", filename, NULL);
    }

    return (full_path);
}

int
main (int argc, gchar * argv[])
{
    gchar *wd;
    GSList *files = NULL;
    gint load_all_files = FALSE;

    entity_init (&argc, &argv);

    RARGV_SWITCH {
	case 'g':		/* Debugging flag: -g <domains>. */
	    edebug_enable (RARGV_NEXT);
	    break;

	case 'a':		/* Load all files on command line into entity. */
	    load_all_files = TRUE;
	    break;

	case 'h':		/* Help: -h or -\?. */
	    /* Fall through. */
	case '?':
	    print_usage ();
	    exit (0);
	    /* NOTREACHED */
	    break;

	case 'V':		/* Print the version. */
	    g_print ("%s %s\n", PACKAGE, VERSION);
	    exit (0);
	    /* NOTREACHED */
	    break;
            
#ifndef WIN32
	case 'm':		/* Create a config file from loaded libs. */
	    modulegen (NULL);
	    break;
            
	case 't':		/* Create a config file from loaded libs. */
	    modulegen (NULL);
	    break;
#endif				/* ifndef WIN32 */
	
        case 'l':		/* Set alternative directory for config tree. 
				 */
            optarg = RARGV_NEXT;
            if (!optarg) {
                g_print ("Option explected for -l\n");
                print_usage ();
                exit (0);
            }
	    econfig_set_attr ("config-location", g_strdup (optarg));
	    break;
	
        case 'c':		/* The execution class to start file on:
				 * -c <class>. */
            optarg = RARGV_NEXT;
            if (!optarg) {
                g_print ("Option explected for -c\n");
                print_usage ();
                exit (0);
            }
	    econfig_set_attr ("exec-class", g_strstrip (optarg));
	    break;

        case '*':

            if (!files || load_all_files) { 
#ifdef WIN32
                /* On win32 we just beleive the path they gave us.. no magic with
                 * changing working dirs etc.  Also, determining if it's really a
                 * full path is a lot harder. */
                files = g_slist_append (files, RARGV_CURRENT);
#else
                files = g_slist_append (files, full_path (RARGV_CURRENT));
#endif
            }
            break;

	default:	
            g_print ("Unknown flag: -%c\n", RARGV_CURRENT_FLAG);
            print_usage ();
            exit (0);
            break;

    } RARGV_CLOSE;


#ifndef WIN32
    /* Set working directory */
    wd = g_strconcat (g_get_home_dir (), NULL);
    chdir (wd);
    g_free (wd);
#endif				/* WIN32 */

    /* Load this file up in the execution class specified.  This also ends up
     * calling the mainloop. */
    if (files) {
	entity_exec_class_load (files);
    } else {
	print_usage ();
	exit (0);
    }

    g_slist_free (files);
   
    if (!entity_mainloop_is_registered ()) {
	g_warning ("No mainloop was registered for use in Entity.");
	g_warning ("This usually indicates that no tag was used which has a renderer that supplies a mainloop.");
	exit (1);
    }

    entity_mainloop_run ();	/* Loop waiting on the tree. */

    return (0);
}


