/*
 * Copyright (C) 2000-2004 Carsten Haitzler, Geoff Harrison and various contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "E.h"

struct _pager
{
   char               *name;
   Window              win;
   Pixmap              pmap;
   PmapMask            bgpmap;
   int                 desktop;
   int                 w, h;
   int                 dw, dh;
   char                visible;
   int                 update_phase;
   EWin               *ewin;
   char               *border_name;
   Window              sel_win;
   char                hi_visible;
   Window              hi_win;
   EWin               *hi_ewin;
   int                 hi_win_w, hi_win_h;
};

#define PAGER_EVENT_MOUSE_OUT -1
#define PAGER_EVENT_MOTION     0
#define PAGER_EVENT_MOUSE_IN   1

static void         PagerUpdateTimeout(int val, void *data);
static void         PagerEwinUpdateMini(Pager * p, EWin * ewin);
static void         PagerEwinUpdateFromPager(Pager * p, EWin * ewin);

static Pager       *mode_context_pager = NULL;

#define HIQ Conf.pagers.hiq

static void
PagerUpdateTimeout(int val __UNUSED__, void *data)
{
   Pager              *p;
   char                s[4096];
   static double       last_time = 0.0;
   double              cur_time, in;
   static int          calls = 0;
   int                 y, y2, phase, ax, ay, cx, cy, ww, hh, xx, yy;
   static int          offsets[8] = { 0, 4, 2, 6, 1, 5, 3, 7 };

   p = (Pager *) data;
   Esnprintf(s, sizeof(s), "__.%x", (unsigned)p->win);
   /* prevent runaway pager timeouts - dont knwo how it happens - but hack */
   /* around to stop it */
   cur_time = GetTime();
   if ((cur_time - last_time) < 0.05)
      calls++;
   last_time = cur_time;
   in = 1 / ((double)Conf.pagers.scanspeed);
   if (calls > 50)
     {
	calls = 0;
	in = 0.25;
     }
   if (Conf.pagers.scanspeed > 0)
      DoIn(s, in, PagerUpdateTimeout, 0, p);
   if (!Conf.pagers.snap)
      return;
   if (!p->visible)
      return;
   if (p->ewin && p->ewin->visibility == VisibilityFullyObscured)
      return;
   if (p->desktop != desks.current)
      return;
   if (Mode.mode != MODE_NONE)
      return;

   GetAreaSize(&ax, &ay);
   GetCurrentArea(&cx, &cy);
   ww = p->w / ax;
   hh = p->h / ay;
   xx = cx * ww;
   yy = cy * hh;
   phase = p->update_phase;
   y = ((phase & 0xfffffff8) + offsets[phase % 8]) % hh;
   y2 = (y * VRoot.h) / hh;

   ScaleLine(p->pmap, VRoot.win, xx, yy + y, VRoot.w, ww, y2, (VRoot.h / hh));
   XClearArea(disp, p->win, xx, yy + y, ww, 1, False);

   p->update_phase++;
   if (p->update_phase >= p->h)
     {
	int                 i, num;
	EWin               *const *lst;

	lst = EwinListGetForDesktop(p->desktop, &num);
	for (i = 0; i < num; i++)
	   PagerEwinUpdateFromPager(p, lst[i]);

	p->update_phase = 0;
     }
}

Pager              *
PagerCreate(void)
{
   Pager              *p;
   int                 ax, ay;
   char                pq;
   ImageClass         *ic;
   XSetWindowAttributes attr;
   static char         did_dialog = 0;

   if (!Conf.pagers.enable)
      return NULL;

   if ((!did_dialog) && (Conf.pagers.snap))
      did_dialog = 1;

   GetAreaSize(&ax, &ay);
   p = Ecalloc(1, sizeof(Pager));
   p->name = NULL;
   attr.colormap = VRoot.cmap;
   attr.border_pixel = 0;
   attr.background_pixel = 0;
   attr.save_under = False;
   p->w = ((48 * VRoot.w) / VRoot.h) * ax;
   p->h = 48 * ay;
   p->dw = ((48 * VRoot.w) / VRoot.h);
   p->dh = 48;
   p->win = ECreateWindow(VRoot.win, 0, 0, p->w, p->h, 0);
   p->pmap = ECreatePixmap(disp, p->win, p->w, p->h, VRoot.depth);
   ESetWindowBackgroundPixmap(disp, p->win, p->pmap);
   p->hi_win = ECreateWindow(VRoot.win, 0, 0, 3, 3, 0);
   p->hi_visible = 0;
   p->hi_ewin = NULL;
   XSelectInput(disp, p->hi_win,
		ButtonPressMask | ButtonReleaseMask | PointerMotionMask |
		EnterWindowMask | LeaveWindowMask);
   p->desktop = 0;
   p->visible = 0;
   p->update_phase = 0;
   p->ewin = NULL;
   p->border_name = NULL;
   p->sel_win = ECreateWindow(p->win, 0, 0, p->w / ax, p->h / ay, 0);
   pq = Mode.queue_up;
   Mode.queue_up = 0;
   ic = FindItem("PAGER_SEL", 0, LIST_FINDBY_NAME, LIST_TYPE_ICLASS);
   if (ic)
      IclassApply(ic, p->sel_win, p->w / ax, p->h / ay, 0, 0, STATE_NORMAL, 0,
		  ST_PAGER);
   Mode.queue_up = pq;
   return p;
}

static void
PagerEwinMoveResize(EWin * ewin, int resize)
{
   Pager              *p = ewin->pager;
   int                 w, h;
   int                 ax, ay, cx, cy;
   char                pq;
   ImageClass         *ic;
   EWin               *const *lst;
   int                 i, num;

   if (!Conf.pagers.enable || !p)
      return;

   RememberImportantInfoForEwin(ewin);

   w = ewin->client.w;
   h = ewin->client.h;
   if ((w == p->w) && (h == p->h))
      return;

   GetAreaSize(&ax, &ay);
   EFreePixmap(disp, p->pmap);
   FreePmapMask(&p->bgpmap);
   EResizeWindow(disp, p->win, w, h);
   p->w = w;
   p->h = h;
   p->dw = w / ax;
   p->dh = h / ay;
   p->pmap = ECreatePixmap(disp, p->win, p->w, p->h, VRoot.depth);
   if (p->visible)
      PagerRedraw(p, 1);
   ESetWindowBackgroundPixmap(disp, p->win, p->pmap);
   XClearWindow(disp, p->win);
   if (p->ewin)
     {
	double              aspect;

	aspect = ((double)VRoot.w) / ((double)VRoot.h);
	p->ewin->client.w_inc = ax * 4;
	p->ewin->client.h_inc = ay * 8;
	p->ewin->client.aspect_min = aspect * ((double)ax / (double)ay);
	p->ewin->client.aspect_max = aspect * ((double)ax / (double)ay);
     }
   pq = Mode.queue_up;
   Mode.queue_up = 0;
   ic = FindItem("PAGER_SEL", 0, LIST_FINDBY_NAME, LIST_TYPE_ICLASS);
   if (ic)
     {
	cx = desks.desk[p->desktop].current_area_x;
	cy = desks.desk[p->desktop].current_area_y;
	EMoveResizeWindow(disp, p->sel_win, cx * p->dw, cy * p->dh, p->dw,
			  p->dh);
	IclassApply(ic, p->sel_win, p->dw, p->dh, 0, 0, STATE_NORMAL, 0,
		    ST_PAGER);
     }
   Mode.queue_up = pq;

   lst = EwinListGetForDesktop(p->desktop, &num);
   for (i = 0; i < num; i++)
      PagerEwinUpdateMini(p, lst[i]);

   return;
   resize = 0;
}

static void
PagerEwinRefresh(EWin * ewin)
{
   /* This doesn't do anything anymore apparently
    * --Mandrake
    * This is new code.
    * It can be removed but I have kept it around, just in case /Kim */
   return;
   ewin = NULL;
}

static void
PagerEwinClose(EWin * ewin)
{
   PagerDestroy(ewin->pager);
   ewin->pager = NULL;
}

static void
PagerEwinInit(EWin * ewin, void *ptr)
{
   ewin->pager = (Pager *) ptr;
   ewin->MoveResize = PagerEwinMoveResize;
   ewin->Refresh = PagerEwinRefresh;
   ewin->Close = PagerEwinClose;
}

void
PagerShow(Pager * p)
{
   EWin               *ewin = NULL;
   char                s[4096];
   char                pq;

   if (!Conf.pagers.enable)
      return;

   if (p->ewin)
     {
	ShowEwin(p->ewin);
	return;
     }

   Esnprintf(s, sizeof(s), "%i", p->desktop);
   HintsSetWindowClass(p->win, s, "Enlightenment_Pager");

   pq = Mode.queue_up;
   Mode.queue_up = 0;

   ewin = AddInternalToFamily(p->win, (p->border_name) ? p->border_name :
			      "PAGER", EWIN_TYPE_PAGER, p, PagerEwinInit);
   if (ewin)
     {
	int                 ax, ay, w, h;
	Snapshot           *sn;
	double              aspect;

	ewin->client.event_mask |=
	   ButtonPressMask | ButtonReleaseMask | PointerMotionMask;
	XSelectInput(disp, p->win, ewin->client.event_mask);

	aspect = ((double)VRoot.w) / ((double)VRoot.h);
	GetAreaSize(&ax, &ay);
	ewin->client.aspect_min = aspect * ((double)ax / (double)ay);
	ewin->client.aspect_max = aspect * ((double)ax / (double)ay);
	ewin->client.w_inc = ax * 4;
	ewin->client.h_inc = ay * 8;
	ewin->client.width.min = 10 * ax;
	ewin->client.height.min = 8 * ay;
	ewin->client.width.max = 320 * ax;
	ewin->client.height.max = 240 * ay;

	p->ewin = ewin;
	p->visible = 1;
	sn = FindSnapshot(ewin);

	/* get the size right damnit! */
	w = ewin->client.w;
	h = ewin->client.h;
	ewin->client.w = 1;
	ewin->client.h = 1;
	if (sn)
	  {
	     MoveResizeEwin(ewin, ewin->x, ewin->y, w, h);
	  }
	else
	  {
	     /* no snapshots ? first time ? make a row on the bottom left up */
	     MoveResizeEwin(ewin, 0, VRoot.h - (Conf.desks.num -
						p->desktop) * ewin->h, w, h);
	  }
	PagerRedraw(p, 1);

	/* show the pager ewin */
	ShowEwin(ewin);
	if (Conf.pagers.snap)
	  {
	     Esnprintf(s, sizeof(s), "__.%x", (unsigned)p->win);
	     if (Conf.pagers.scanspeed > 0)
		DoIn(s, 1 / ((double)Conf.pagers.scanspeed), PagerUpdateTimeout,
		     0, p);
	  }
	AddItem(p, "PAGER", p->win, LIST_TYPE_PAGER);
     }

   Mode.queue_up = pq;
}

void
PagerDestroy(Pager * p)
{
   char                s[4096];

   RemoveItem("PAGER", p->win, LIST_FINDBY_ID, LIST_TYPE_PAGER);
   Esnprintf(s, sizeof(s), "__.%x", (unsigned)p->win);
   RemoveTimerEvent(s);
   if (p->name)
      Efree(p->name);
   EDestroyWindow(disp, p->win);
   if (p->hi_win)
      EDestroyWindow(disp, p->hi_win);
   if (p->pmap)
      EFreePixmap(disp, p->pmap);
   FreePmapMask(&p->bgpmap);
   if (p->border_name)
      Efree(p->border_name);
   Efree(p);
}

Pager             **
PagersForDesktop(int d, int *num)
{
   Pager             **pp = NULL;
   Pager             **pl = NULL;
   int                 i, pnum;

   if (!Conf.pagers.enable)
      return NULL;

   *num = 0;
   pl = (Pager **) ListItemType(&pnum, LIST_TYPE_PAGER);
   if (pl)
     {
	for (i = 0; i < pnum; i++)
	  {
	     if (pl[i]->desktop == d)
	       {
		  (*num)++;
		  pp = Erealloc(pp, sizeof(Pager *) * (*num));
		  pp[(*num) - 1] = pl[i];
	       }
	  }
	Efree(pl);
     }
   return pp;
}

void
RedrawPagersForDesktop(int d, char newbg)
{
   Pager             **pl;
   int                 i, num;

   if (!Conf.pagers.enable)
      return;

   pl = PagersForDesktop(d, &num);
   if (pl)
     {
	for (i = 0; i < num; i++)
	   PagerRedraw(pl[i], newbg);
	Efree(pl);
     }
}

void
ForceUpdatePagersForDesktop(int d)
{
   Pager             **pl;
   int                 i, num;

   if (!Conf.pagers.enable)
      return;

   pl = PagersForDesktop(d, &num);
   if (pl)
     {
	for (i = 0; i < num; i++)
	   PagerForceUpdate(pl[i]);
	Efree(pl);
     }
}

static void
PagerUpdateHiWin(Pager * p, EWin * ewin)
{
   Imlib_Image        *im;

   if (!p->hi_visible || !ewin->mini_pmm.pmap)
      return;

   imlib_context_set_drawable(ewin->mini_pmm.pmap);
   im = imlib_create_image_from_drawable(0, 0, 0,
					 ewin->mini_w, ewin->mini_h, 0);
   imlib_context_set_image(im);
   imlib_context_set_drawable(p->hi_win);
   imlib_render_image_on_drawable_at_size(0, 0, p->hi_win_w, p->hi_win_h);
   imlib_free_image_and_decache();
}

static void
PagerEwinUpdateMini(Pager * p, EWin * ewin)
{
   int                 w, h, ax, ay, cx, cy;

   if (!Conf.pagers.enable)
      return;

   GetAreaSize(&ax, &ay);
   cx = desks.desk[p->desktop].current_area_x;
   cy = desks.desk[p->desktop].current_area_y;

   w = ((ewin->w) * (p->w / ax)) / VRoot.w;
   h = ((ewin->h) * (p->h / ay)) / VRoot.h;

   if (w < 1)
      w = 1;
   if (h < 1)
      h = 1;
   if ((ewin->mini_w != w) || (ewin->mini_h != h))
     {
	FreePmapMask(&ewin->mini_pmm);

	ewin->mini_w = w;
	ewin->mini_h = h;

	if ((ewin->desktop != desks.current) || (ewin->area_x != cx)
	    || (ewin->area_y != cy) || (!Conf.pagers.snap))
	  {
	     ImageClass         *ic = NULL;

	     ic = FindItem("PAGER_WIN", 0, LIST_FINDBY_NAME, LIST_TYPE_ICLASS);
	     if (ic)
		IclassApplyCopy(ic, ewin->win, w, h, 0, 0, STATE_NORMAL,
				&ewin->mini_pmm, 1, ST_PAGER);
	  }
	else
	  {
	     ewin->mini_pmm.type = 0;
	     ewin->mini_pmm.pmap =
		ECreatePixmap(disp, p->win, w, h, VRoot.depth);
	     ewin->mini_pmm.mask = None;
	     ScaleRect(ewin->mini_pmm.pmap, ewin->win, 0, 0, 0, 0, ewin->w,
		       ewin->h, w, h);
	  }
     }

   if (ewin == p->hi_ewin)
      PagerUpdateHiWin(p, ewin);
}

static void
PagerEwinUpdateFromPager(Pager * p, EWin * ewin)
{
   int                 x, y, w, h, ax, ay, cx, cy;
   static GC           gc = 0;
   XGCValues           gcv;

   if (!Conf.pagers.snap)
     {
	PagerEwinUpdateMini(p, ewin);
	return;
     }
   if (!Conf.pagers.enable)
      return;

   GetAreaSize(&ax, &ay);
   cx = desks.desk[p->desktop].current_area_x;
   cy = desks.desk[p->desktop].current_area_y;
   x = ((ewin->x + (cx * VRoot.w)) * (p->w / ax)) / VRoot.w;
   y = ((ewin->y + (cy * VRoot.h)) * (p->h / ay)) / VRoot.h;
   w = ((ewin->w) * (p->w / ax)) / VRoot.w;
   h = ((ewin->h) * (p->h / ay)) / VRoot.h;
   if (!gc)
      gc = XCreateGC(disp, p->pmap, 0, &gcv);

   /* NB! If the pixmap/mask was created by imlib, free it. Due to imlibs */
   /*     image/pixmap cache it may be in use elsewhere. */
   if (ewin->mini_pmm.pmap &&
       ((ewin->mini_pmm.type) || (ewin->mini_w != w) || (ewin->mini_h != h)))
      FreePmapMask(&ewin->mini_pmm);

   if (!ewin->mini_pmm.pmap)
     {
	ewin->mini_w = w;
	ewin->mini_h = h;
	ewin->mini_pmm.type = 0;
	ewin->mini_pmm.pmap = ECreatePixmap(disp, p->win, w, h, VRoot.depth);
	ewin->mini_pmm.mask = None;
     }
   XCopyArea(disp, p->pmap, ewin->mini_pmm.pmap, gc, x, y, w, h, 0, 0);

   if (ewin == p->hi_ewin)
      PagerUpdateHiWin(p, ewin);
}

void
PagerRedraw(Pager * p, char newbg)
{
   int                 x, y, ax, ay, cx, cy;
   GC                  gc;
   XGCValues           gcv;
   EWin               *const *lst;
   int                 i, num;

   if (!Conf.pagers.enable || Mode.mode == MODE_DESKSWITCH)
      return;

   if (Mode.queue_up)
     {
	DrawQueue          *dq;

	dq = Emalloc(sizeof(DrawQueue));
	dq->win = p->win;
	dq->iclass = NULL;
	dq->w = p->w;
	dq->h = p->h;
	dq->active = 0;
	dq->sticky = 0;
	dq->state = 0;
	dq->expose = 0;
	dq->tclass = NULL;
	dq->text = NULL;
	dq->shape_propagate = 0;
	dq->pager = NULL;
	dq->redraw_pager = p;
	dq->newbg = newbg;
	dq->d = NULL;
	dq->di = NULL;
	dq->x = 0;
	dq->y = 0;
	dq->image_type = ST_PAGER;
	AddItem(dq, "DRAW", dq->win, LIST_TYPE_DRAW);
	return;
     }

   p->update_phase = 0;
   GetAreaSize(&ax, &ay);
   cx = desks.desk[p->desktop].current_area_x;
   cy = desks.desk[p->desktop].current_area_y;
   gc = XCreateGC(disp, p->pmap, 0, &gcv);
   if (gc)
     {
	if ((newbg > 0) && (newbg < 3))
	  {
	     FreePmapMask(&p->bgpmap);

	     if (!Conf.pagers.snap)
	       {
		  ImageClass         *ic = NULL;

		  ic = FindItem("PAGER_BACKGROUND", 0, LIST_FINDBY_NAME,
				LIST_TYPE_ICLASS);
		  if (ic)
		     IclassApplyCopy(ic, p->win, p->w / ax, p->h / ay, 0, 0,
				     STATE_NORMAL, &p->bgpmap, 0, ST_PAGER);
	       }
	     else
	       {
		  if (desks.desk[p->desktop].bg)
		    {
		       char                s[4096];
		       char               *uniq;
		       Imlib_Image        *im;

		       uniq =
			  BackgroundGetUniqueString(desks.desk[p->desktop].bg);
		       Esnprintf(s, sizeof(s), "%s/cached/pager/%s.%i.%i.%s",
				 EDirUserCache(),
				 BackgroundGetName(desks.desk[p->desktop].bg),
				 (p->w / ax), (p->h / ay), uniq);
		       Efree(uniq);

		       im = imlib_load_image(s);
		       if (im)
			 {
			    imlib_context_set_image(im);
			    p->bgpmap.type = 1;
			    imlib_render_pixmaps_for_whole_image_at_size(&p->
									 bgpmap.
									 pmap,
									 &p->
									 bgpmap.
									 mask,
									 (p->w /
									  ax),
									 (p->h /
									  ay));
			    imlib_free_image_and_decache();
			 }
		       else
			 {
			    p->bgpmap.type = 0;
			    p->bgpmap.pmap =
			       ECreatePixmap(disp, p->win, p->w / ax, p->h / ay,
					     VRoot.depth);
			    p->bgpmap.mask = None;
			    BackgroundApply(desks.desk[p->desktop].bg,
					    p->bgpmap.pmap, 0);
			    imlib_context_set_drawable(p->bgpmap.pmap);
			    im =
			       imlib_create_image_from_drawable(0, 0, 0,
								(p->w / ax),
								(p->h / ay), 0);
			    imlib_context_set_image(im);
			    imlib_image_set_format("png");
			    imlib_save_image(s);
			    imlib_free_image_and_decache();
			 }
		    }
		  else
		    {
		       p->bgpmap.type = 0;
		       p->bgpmap.pmap =
			  ECreatePixmap(disp, p->win, p->w / ax, p->h / ay,
					VRoot.depth);
		       p->bgpmap.mask = None;
		       XSetForeground(disp, gc, BlackPixel(disp, VRoot.scr));
		       XDrawRectangle(disp, p->bgpmap.pmap, gc, 0, 0, p->dw,
				      p->dh);
		       XSetForeground(disp, gc, WhitePixel(disp, VRoot.scr));
		       XFillRectangle(disp, p->bgpmap.pmap, gc, 1, 1, p->dw - 2,
				      p->dh - 2);
		    }
	       }
	  }

	for (y = 0; y < ay; y++)
	  {
	     for (x = 0; x < ax; x++)
		XCopyArea(disp, p->bgpmap.pmap, p->pmap, gc, 0, 0, p->w / ax,
			  p->h / ay, x * (p->w / ax), y * (p->h / ay));
	  }

	lst = EwinListGetForDesktop(p->desktop, &num);
	for (i = num - 1; i >= 0; i--)
	  {
	     EWin               *ewin;
	     int                 wx, wy, ww, wh;

	     ewin = lst[i];
	     if (!ewin->iconified && ewin->shown)
	       {
		  wx = ((ewin->x + (cx * VRoot.w)) * (p->w / ax)) / VRoot.w;
		  wy = ((ewin->y + (cy * VRoot.h)) * (p->h / ay)) / VRoot.h;
		  ww = ((ewin->w) * (p->w / ax)) / VRoot.w;
		  wh = ((ewin->h) * (p->h / ay)) / VRoot.h;
		  PagerEwinUpdateMini(p, ewin);
		  if (ewin->mini_pmm.pmap)
		    {
		       if (ewin->mini_pmm.mask)
			 {
			    XSetClipMask(disp, gc, ewin->mini_pmm.mask);
			    XSetClipOrigin(disp, gc, wx, wy);
			 }
		       XCopyArea(disp, ewin->mini_pmm.pmap, p->pmap, gc, 0, 0,
				 ww, wh, wx, wy);
		       if (ewin->mini_pmm.mask)
			  XSetClipMask(disp, gc, None);
		    }
		  else
		    {
		       XSetForeground(disp, gc, BlackPixel(disp, VRoot.scr));
		       XDrawRectangle(disp, p->pmap, gc, wx - 1, wy - 1, ww + 1,
				      wh + 1);
		       XSetForeground(disp, gc, WhitePixel(disp, VRoot.scr));
		       XFillRectangle(disp, p->pmap, gc, wx, wy, ww, wh);
		    }
	       }
	  }

	if (newbg < 2)
	  {
	     ESetWindowBackgroundPixmap(disp, p->win, p->pmap);
	     XClearWindow(disp, p->win);
	  }

	XFreeGC(disp, gc);
     }
}

void
PagerForceUpdate(Pager * p)
{
   int                 ww, hh, xx, yy, ax, ay, cx, cy;
   EWin               *const *lst;
   int                 i, num;

   if (!Conf.pagers.enable || Mode.mode == MODE_DESKSWITCH)
      return;

   if (Mode.queue_up)
     {
	DrawQueue          *dq;

	dq = Emalloc(sizeof(DrawQueue));
	dq->win = p->win;
	dq->iclass = NULL;
	dq->w = p->w;
	dq->h = p->h;
	dq->active = 0;
	dq->sticky = 0;
	dq->state = 0;
	dq->expose = 0;
	dq->tclass = NULL;
	dq->text = NULL;
	dq->shape_propagate = 0;
	dq->pager = p;
	dq->redraw_pager = NULL;
	dq->d = NULL;
	dq->di = NULL;
	dq->x = 0;
	dq->y = 0;
	AddItem(dq, "DRAW", dq->win, LIST_TYPE_DRAW);
	return;
     }

   if ((p->desktop != desks.current) || (!Conf.pagers.snap))
     {
	PagerRedraw(p, 0);
	return;
     }

   p->update_phase = 0;
   GetAreaSize(&ax, &ay);
   cx = desks.desk[p->desktop].current_area_x;
   cy = desks.desk[p->desktop].current_area_y;
   ww = p->w / ax;
   hh = p->h / ay;
   xx = cx * ww;
   yy = cy * hh;

   ScaleRect(p->pmap, VRoot.win, 0, 0, xx, yy, VRoot.w, VRoot.h, ww, hh);
   XClearWindow(disp, p->win);

   lst = EwinListGetForDesktop(p->desktop, &num);
   for (i = 0; i < num; i++)
      PagerEwinUpdateFromPager(p, lst[i]);
}

void
PagerReArea(void)
{
   Pager             **pl = NULL;
   int                 i, pnum, w, h, ax, ay;

   if (!Conf.pagers.enable)
      return;

   pl = (Pager **) ListItemType(&pnum, LIST_TYPE_PAGER);
   GetAreaSize(&ax, &ay);
   if (pl)
     {
	for (i = 0; i < pnum; i++)
	  {
	     w = pl[i]->dw * ax;
	     h = pl[i]->dh * ay;
	     if (pl[i]->ewin)
	       {
		  double              aspect;

		  aspect = ((double)VRoot.w) / ((double)VRoot.h);
		  pl[i]->ewin->client.w_inc = ax * 4;
		  pl[i]->ewin->client.h_inc = ay * 8;
		  pl[i]->ewin->client.aspect_min =
		     aspect * ((double)ax / (double)ay);
		  pl[i]->ewin->client.aspect_max =
		     aspect * ((double)ax / (double)ay);
		  MoveResizeEwin(pl[i]->ewin, pl[i]->ewin->x, pl[i]->ewin->y, w,
				 h);
	       }
	  }
	Efree(pl);
     }
}

void
PagerEwinOutsideAreaUpdate(EWin * ewin)
{
   if (!Conf.pagers.enable)
      return;

   if (ewin->sticky)
     {
	int                 i;

	for (i = 0; i < Conf.desks.num; i++)
	   RedrawPagersForDesktop(i, 0);
	ForceUpdatePagersForDesktop(ewin->desktop);
	return;
     }
   else if (ewin->desktop != desks.current)
     {
	RedrawPagersForDesktop(ewin->desktop, 0);
	ForceUpdatePagersForDesktop(ewin->desktop);
	return;
     }

   if ((ewin->x < 0) || (ewin->y < 0) || ((ewin->x + ewin->w) > VRoot.w)
       || ((ewin->y + ewin->h) > VRoot.h))
      RedrawPagersForDesktop(ewin->desktop, 3);
   ForceUpdatePagersForDesktop(ewin->desktop);
}

static EWin        *
EwinInPagerAt(Pager * p, int x, int y)
{
   int                 wx, wy, ww, wh, ax, ay, cx, cy;
   EWin               *const *lst;
   int                 i, num;

   if (!Conf.pagers.enable)
      return NULL;

   GetAreaSize(&ax, &ay);
   cx = desks.desk[p->desktop].current_area_x;
   cy = desks.desk[p->desktop].current_area_y;
   lst = EwinListGetForDesktop(p->desktop, &num);
   for (i = 0; i < num; i++)
     {
	EWin               *ewin;

	ewin = lst[i];
	if (!ewin->iconified && ewin->shown)
	  {
	     wx = ((ewin->x + (cx * VRoot.w)) * (p->w / ax)) / VRoot.w;
	     wy = ((ewin->y + (cy * VRoot.h)) * (p->h / ay)) / VRoot.h;
	     ww = ((ewin->w) * (p->w / ax)) / VRoot.w;
	     wh = ((ewin->h) * (p->h / ay)) / VRoot.h;
	     if ((x >= wx) && (y >= wy) && (x < (wx + ww)) && (y < (wy + wh)))
		return ewin;
	  }
     }

   return NULL;
}

static void
PagerAreaAt(Pager * p, int x, int y, int *ax, int *ay)
{
   int                 asx, asy;

   if (!Conf.pagers.enable)
      return;

   GetAreaSize(&asx, &asy);
   *ax = x / (p->w / asx);
   *ay = y / (p->h / asy);
}

static void
PagerMenuShow(Pager * p, int x, int y)
{
   static Menu        *p_menu = NULL, *pw_menu = NULL;
   MenuItem           *mi;
   EWin               *ewin;
   char                s[1024];
   int                 ax, ay;

   if (!Conf.pagers.enable)
      return;

   ewin = EwinInPagerAt(p, x, y);
   if (ewin)
     {
	if (pw_menu)
	   MenuDestroy(pw_menu);

	pw_menu = MenuCreate("__DESK_WIN_MENU");
	MenuAddTitle(pw_menu, _("Window Options"));
	MenuAddStyle(pw_menu, "DEFAULT");

	Esnprintf(s, sizeof(s), "%i", (unsigned)ewin->client.win);
	mi = MenuItemCreate(_("Iconify"), NULL, ACTION_ICONIFY, s, NULL);
	MenuAddItem(pw_menu, mi);

	mi = MenuItemCreate(_("Close"), NULL, ACTION_KILL, s, NULL);
	MenuAddItem(pw_menu, mi);

	mi = MenuItemCreate(_("Annihilate"), NULL, ACTION_KILL_NASTY, s, NULL);
	MenuAddItem(pw_menu, mi);

	mi = MenuItemCreate(_("Stick / Unstick"), NULL, ACTION_STICK, s, NULL);
	MenuAddItem(pw_menu, mi);

	ActionsCall(ACTION_SHOW_MENU, ewin, "named __DESK_WIN_MENU");
	return;
     }

   PagerAreaAt(p, x, y, &ax, &ay);
   if (p_menu)
      MenuDestroy(p_menu);
   p_menu = MenuCreate("__DESK_MENU");
   MenuAddTitle(p_menu, _("Desktop Options"));
   MenuAddStyle(p_menu, "DEFAULT");

   mi = MenuItemCreate(_("Pager Settings..."), NULL, ACTION_CONFIG, "pager",
		       NULL);
   MenuAddItem(p_menu, mi);

   mi = MenuItemCreate(_("Snapshotting On"), NULL, ACTION_SET_PAGER_SNAP, "1",
		       NULL);
   MenuAddItem(p_menu, mi);

   mi = MenuItemCreate(_("Snapshotting Off"), NULL, ACTION_SET_PAGER_SNAP, "0",
		       NULL);
   MenuAddItem(p_menu, mi);

   if (Conf.pagers.snap)
     {
	mi = MenuItemCreate(_("High Quality On"), NULL, ACTION_SET_PAGER_HIQ,
			    "1", NULL);
	MenuAddItem(p_menu, mi);

	mi = MenuItemCreate(_("High Quality Off"), NULL, ACTION_SET_PAGER_HIQ,
			    "0", NULL);
	MenuAddItem(p_menu, mi);
     }

   ActionsCall(ACTION_SHOW_MENU, NULL, "named __DESK_MENU");
}

void
PagerHide(Pager * p)
{
   if (p->ewin)
      HideEwin(p->ewin);
}

void
UpdatePagerSel(void)
{
   Pager             **pl;
   Pager              *p;
   int                 i, pnum, cx, cy;
   ImageClass         *ic;

   if (!Conf.pagers.enable)
      return;

   pl = (Pager **) ListItemType(&pnum, LIST_TYPE_PAGER);
   if (pl)
     {
	for (i = 0; i < pnum; i++)
	  {
	     p = pl[i];
	     if (p->desktop != desks.current)
		EUnmapWindow(disp, p->sel_win);
	     else
	       {
		  cx = desks.desk[p->desktop].current_area_x;
		  cy = desks.desk[p->desktop].current_area_y;
		  EMoveWindow(disp, p->sel_win, cx * p->dw, cy * p->dh);
		  EMapWindow(disp, p->sel_win);
		  ic = FindItem("PAGER_SEL", 0, LIST_FINDBY_NAME,
				LIST_TYPE_ICLASS);
		  if (ic)
		     IclassApply(ic, p->sel_win, p->dw, p->dh, 0, 0,
				 STATE_NORMAL, 0, ST_PAGER);
	       }
	  }
	Efree(pl);
     }
}

static void
PagerShowTt(EWin * ewin)
{
   static EWin        *tt_ewin = NULL;
   ToolTip            *tt;

   if (!Conf.pagers.title || (ewin == tt_ewin))
      return;

   if (MenusActive())		/* Don't show Tooltip when menu is up */
      return;

   tt = FindItem("PAGER", 0, LIST_FINDBY_NAME, LIST_TYPE_TOOLTIP);
   if (tt)
     {
	if (ewin)
	   ShowToolTip(tt, EwinGetIconName(ewin), NULL, Mode.x, Mode.y);
	else
	   HideToolTip(tt);
     }

   tt_ewin = ewin;
}

void
PagerHideAllHi(void)
{
   Pager             **pl = NULL;
   int                 i, pnum;

   if (!Conf.pagers.enable)
      return;

   pl = (Pager **) ListItemType(&pnum, LIST_TYPE_PAGER);
   if (pl)
     {
	for (i = 0; i < pnum; i++)
	   PagerHideHi(pl[i]);
	Efree(pl);
     }
}

void
PagerHideHi(Pager * p)
{
   if (p->hi_visible)
     {
	p->hi_visible = 0;
	EUnmapWindow(disp, p->hi_win);
     }
   p->hi_ewin = NULL;

   PagerShowTt(NULL);
}

static void
PagerShowHi(Pager * p, EWin * ewin, int x, int y, int w, int h)
{
   char                pq;
   ImageClass         *ic = NULL;

   if (MenusActive())		/* Don't show HiWin when menu is up */
      return;

   pq = Mode.queue_up;

   p->hi_win_w = 2 * w;
   p->hi_win_h = 2 * h;

   ic = FindItem("PAGER_WIN", 0, LIST_FINDBY_NAME, LIST_TYPE_ICLASS);
   EMoveResizeWindow(disp, p->hi_win, x, y, w, h);
   EMapRaised(disp, p->hi_win);
   if (ewin->mini_pmm.pmap)
     {
	Imlib_Image        *im;
	Pixmap              pmap, mask;
	int                 xx, yy, ww, hh, i;

	imlib_context_set_drawable(ewin->mini_pmm.pmap);
	im = imlib_create_image_from_drawable(0, 0, 0, ewin->mini_w,
					      ewin->mini_h, 0);
	imlib_context_set_image(im);
	if (w > h)
	  {
	     for (i = w; i < (w * 2); i++)
	       {
		  ww = i;
		  hh = (i * h) / w;
		  xx = x + ((w - ww) / 2);
		  yy = y + ((h - hh) / 2);
		  imlib_render_pixmaps_for_whole_image_at_size(&pmap,
							       &mask, ww, hh);
		  ESetWindowBackgroundPixmap(disp, p->hi_win, pmap);
		  imlib_free_pixmap_and_mask(pmap);
		  EMoveResizeWindow(disp, p->hi_win, xx, yy, ww, hh);
		  XClearWindow(disp, p->hi_win);
		  {
		     int                 px, py;

		     PointerAt(&px, &py);
		     if ((px < x) || (py < y) || (px >= (x + w))
			 || (py >= (y + h)))
		       {
			  imlib_free_image_and_decache();
			  EUnmapWindow(disp, p->hi_win);
			  goto done;
		       }
		  }
	       }
	  }
	else
	  {
	     for (i = h; i < (h * 2); i++)
	       {
		  ww = (i * w) / h;
		  hh = i;
		  xx = x + ((w - ww) / 2);
		  yy = y + ((h - hh) / 2);
		  imlib_render_pixmaps_for_whole_image_at_size(&pmap,
							       &mask, ww, hh);
		  ESetWindowBackgroundPixmap(disp, p->hi_win, pmap);
		  imlib_free_pixmap_and_mask(pmap);
		  EMoveResizeWindow(disp, p->hi_win, xx, yy, ww, hh);
		  XClearWindow(disp, p->hi_win);
		  {
		     int                 px, py;

		     PointerAt(&px, &py);
		     if ((px < x) || (py < y) || (px >= (x + w))
			 || (py >= (y + h)))
		       {
			  imlib_free_image_and_decache();
			  EUnmapWindow(disp, p->hi_win);
			  goto done;
		       }
		  }
	       }
	  }
	EMoveResizeWindow(disp, p->hi_win, x - (w / 2), y - (h / 2),
			  w * 2, h * 2);
	imlib_context_set_image(im);
	imlib_context_set_drawable(p->hi_win);
	imlib_render_image_on_drawable_at_size(0, 0, p->hi_win_w, p->hi_win_h);
	imlib_free_image_and_decache();
     }
   else if (ic)
     {
	int                 xx, yy, ww, hh, i;

	Mode.queue_up = 0;
	if (w > h)
	  {
	     for (i = w; i < (w * 2); i++)
	       {
		  ww = i;
		  hh = (i * h) / w;
		  xx = x + ((w - ww) / 2);
		  yy = y + ((h - hh) / 2);
		  IclassApply(ic, p->hi_win, ww, hh, 0, 0, STATE_NORMAL, 0,
			      ST_PAGER);
		  EMoveResizeWindow(disp, p->hi_win, xx, yy, ww, hh);
		  XClearWindow(disp, p->hi_win);
		  {
		     int                 px, py;

		     PointerAt(&px, &py);
		     if ((px < x) || (py < y) || (px >= (x + w))
			 || (py >= (y + h)))
		       {
			  EUnmapWindow(disp, p->hi_win);
			  goto done;
		       }
		  }
	       }
	  }
	else
	  {
	     for (i = h; i < (h * 2); i++)
	       {
		  ww = (i * w) / h;
		  hh = i;
		  xx = x + ((w - ww) / 2);
		  yy = y + ((h - hh) / 2);
		  IclassApply(ic, p->hi_win, ww, hh, 0, 0, STATE_NORMAL, 0,
			      ST_PAGER);
		  EMoveResizeWindow(disp, p->hi_win, xx, yy, ww, hh);
		  XClearWindow(disp, p->hi_win);
		  {
		     int                 px, py;

		     PointerAt(&px, &py);
		     if ((px < x) || (py < y) || (px >= (x + w))
			 || (py >= (y + h)))
		       {
			  EUnmapWindow(disp, p->hi_win);
			  goto done;
		       }
		  }
	       }
	  }
	EMoveResizeWindow(disp, p->hi_win, x - (w / 2), y - (h / 2), w * 2,
			  h * 2);
     }
   else
     {
	Pixmap              pmap;
	GC                  gc = 0;
	XGCValues           gcv;
	int                 xx, yy, ww, hh, i;

	pmap = ECreatePixmap(disp, p->hi_win, w * 2, h * 2, VRoot.depth);
	ESetWindowBackgroundPixmap(disp, p->hi_win, pmap);
	if (!gc)
	   gc = XCreateGC(disp, pmap, 0, &gcv);
	if (w > h)
	  {
	     for (i = w; i < (w * 2); i++)
	       {
		  ww = i;
		  hh = (i * h) / w;
		  xx = x + ((w - ww) / 2);
		  yy = y + ((h - hh) / 2);
		  XSetForeground(disp, gc, BlackPixel(disp, VRoot.scr));
		  XFillRectangle(disp, pmap, gc, 0, 0, ww, hh);
		  XSetForeground(disp, gc, WhitePixel(disp, VRoot.scr));
		  XFillRectangle(disp, pmap, gc, 1, 1, ww - 2, hh - 2);
		  EMoveResizeWindow(disp, p->hi_win, xx, yy, ww, hh);
		  XClearWindow(disp, p->hi_win);
		  {
		     int                 px, py;

		     PointerAt(&px, &py);
		     if ((px < x) || (py < y) || (px >= (x + w))
			 || (py >= (y + h)))
		       {
			  EFreePixmap(disp, pmap);
			  EUnmapWindow(disp, p->hi_win);
			  goto done;
		       }
		  }
	       }
	  }
	else
	  {
	     for (i = h; i < (h * 2); i++)
	       {
		  ww = (i * w) / h;
		  hh = i;
		  xx = x + ((w - ww) / 2);
		  yy = y + ((h - hh) / 2);
		  XSetForeground(disp, gc, BlackPixel(disp, VRoot.scr));
		  XFillRectangle(disp, pmap, gc, 0, 0, ww, hh);
		  XSetForeground(disp, gc, WhitePixel(disp, VRoot.scr));
		  XFillRectangle(disp, pmap, gc, 1, 1, ww - 2, hh - 2);
		  EMoveResizeWindow(disp, p->hi_win, xx, yy, ww, hh);
		  XClearWindow(disp, p->hi_win);
		  {
		     int                 px, py;

		     PointerAt(&px, &py);
		     if ((px < x) || (py < y) || (px >= (x + w))
			 || (py >= (y + h)))
		       {
			  EFreePixmap(disp, pmap);
			  EUnmapWindow(disp, p->hi_win);
			  goto done;
		       }
		  }
	       }
	  }
	EFreePixmap(disp, pmap);
	EMoveResizeWindow(disp, p->hi_win, x - (w / 2), y - (h / 2), w * 2,
			  h * 2);
     }
   p->hi_visible = 1;
   p->hi_ewin = ewin;

 done:
   Mode.queue_up = pq;
}

static void
PagerHandleMotion(Pager * p, Window win, int x, int y, int in)
{
   int                 hx, hy;
   Window              rw, cw;
   EWin               *ewin = NULL;

   if (!Conf.pagers.enable)
      return;

   XQueryPointer(disp, p->win, &rw, &cw, &hx, &hy, &x, &y, &hx);

   if (x >= 0 && x < p->w && y >= 0 && y < p->h)
      ewin = EwinInPagerAt(p, x, y);
   else
      ewin = NULL;

   if (!Conf.pagers.zoom)
     {
	if (in == PAGER_EVENT_MOUSE_OUT)
	   PagerShowTt(NULL);
	else
	   PagerShowTt(ewin);
	return;
     }

   if (ewin == NULL)
     {
	PagerHideHi(p);
	return;
     }

   if (in == PAGER_EVENT_MOUSE_OUT)
     {
	PagerShowTt(NULL);
     }
   else if ((in == PAGER_EVENT_MOTION) && ewin->layer <= 0)
     {
	if (p->hi_ewin)
	   PagerHideHi(p);
	PagerShowTt(ewin);
     }
   else if ((in == PAGER_EVENT_MOTION) && ewin != p->hi_ewin)
     {
	int                 wx, wy, ww, wh, ax, ay, cx, cy, px, py;

	PagerHideHi(p);
	GetAreaSize(&ax, &ay);
	cx = desks.desk[p->desktop].current_area_x;
	cy = desks.desk[p->desktop].current_area_y;
	wx = ((ewin->x + (cx * VRoot.w)) * (p->w / ax)) / VRoot.w;
	wy = ((ewin->y + (cy * VRoot.h)) * (p->h / ay)) / VRoot.h;
	ww = ((ewin->w) * (p->w / ax)) / VRoot.w;
	wh = ((ewin->h) * (p->h / ay)) / VRoot.h;
	XTranslateCoordinates(disp, p->win, VRoot.win, 0, 0, &px, &py, &cw);
	PagerShowHi(p, ewin, px + wx, py + wy, ww, wh);
	PagerShowTt(ewin);
     }
   else if (in == PAGER_EVENT_MOTION)
     {
	PagerShowTt(ewin);
     }

   return;
   win = 0;

}

void
NewPagerForDesktop(int desk)
{

   Pager              *p;
   char                s[1024];

   p = PagerCreate();
   if (p)
     {
	p->desktop = desk;
	Esnprintf(s, sizeof(s), "%i", desk);
	HintsSetWindowName(p->win, s);
	PagerShow(p);
     }
}

void
EnableSinglePagerForDesktop(int desk)
{
   Pager             **pl;
   int                 num;

   pl = PagersForDesktop(desk, &num);
   if (!pl)
      NewPagerForDesktop(desk);
   else
      Efree(pl);
}

void
EnableAllPagers(void)
{
   int                 i;

   if (!Conf.pagers.enable)
     {
	Conf.pagers.enable = 1;
	for (i = 0; i < Conf.desks.num; i++)
	   EnableSinglePagerForDesktop(i);
	UpdatePagerSel();
     }
   return;
}

int
PagerForDesktop(int desk)
{
   Pager             **pl;
   int                 num;

   pl = PagersForDesktop(desk, &num);
   if (pl)
      Efree(pl);
   return num;
}

void
DisablePagersForDesktop(int desk)
{
   Pager             **pl;

   int                 i, num;

   pl = PagersForDesktop(desk, &num);
   if (pl)
     {
	for (i = 0; i < num; i++)
	  {
	     if (pl[i]->ewin)
		ICCCM_Delete(pl[i]->ewin);
	  }
	Efree(pl);
     }
}

void
DisableAllPagers(void)
{
   int                 i;

   if (Conf.pagers.enable)
     {
	for (i = 0; i < Conf.desks.num; i++)
	   DisablePagersForDesktop(i);
	Conf.pagers.enable = 0;
     }
   return;
}

void
PagerSetHiQ(char onoff)
{
   Pager             **pl;
   EWin               *const *lst;
   int                 i, num;

   HIQ = onoff;

   lst = EwinListGetAll(&num);
   for (i = 0; i < num; i++)
     {
	lst[i]->mini_w = 0;
	lst[i]->mini_h = 0;
     }

   pl = (Pager **) ListItemType(&num, LIST_TYPE_PAGER);
   if (pl)
     {
	for (i = 0; i < num; i++)
	  {
	     PagerHideHi(pl[i]);
	     PagerRedraw(pl[i], 2);
	     PagerForceUpdate(pl[i]);
	  }
	Efree(pl);
     }
}

void
PagerSetSnap(char onoff)
{
   Pager             **pl;
   EWin               *const *lst;
   int                 i, num;
   char                s[256];

   Conf.pagers.snap = onoff;

   lst = EwinListGetAll(&num);
   for (i = 0; i < num; i++)
     {
	lst[i]->mini_w = 0;
	lst[i]->mini_h = 0;
     }

   pl = (Pager **) ListItemType(&num, LIST_TYPE_PAGER);
   if (pl)
     {
	for (i = 0; i < num; i++)
	  {
	     PagerHideHi(pl[i]);
	     PagerRedraw(pl[i], 2);
	     PagerForceUpdate(pl[i]);
	     if (Conf.pagers.snap)
	       {
		  Esnprintf(s, sizeof(s), "__.%x", (unsigned)pl[i]->win);
		  if (Conf.pagers.scanspeed > 0)
		     DoIn(s, 1 / ((double)Conf.pagers.scanspeed),
			  PagerUpdateTimeout, 0, pl[i]);
	       }
	  }
	Efree(pl);
     }
}

Window
PagerGetWin(Pager * p)
{
   return (p) ? p->win : 0;
}

Window
PagerGetHiWin(Pager * p)
{
   return (p) ? p->hi_win : 0;
}

Pager              *
FindPager(Window win)
{
   Pager              *p, *pr = NULL;
   Pager             **ps;
   int                 i, num;

   EDBUG(6, "FindDialog");

   ps = (Pager **) ListItemType(&num, LIST_TYPE_PAGER);
   for (i = 0; i < num; i++)
     {
	p = ps[i];
	if ((p->win == win) || (p->hi_win == win))
	  {
	     pr = p;
	     break;
	  }
     }
   if (ps)
      Efree(ps);

   EDBUG_RETURN(pr);
}

/*
 * Pager event handlers
 */

static int         *gwin_px, *gwin_py;

void
PagerEventUnmap(Pager * p)
{
   PagerHideHi(p);
   if (p == mode_context_pager)
     {
	mode_context_pager = NULL;
	Mode.mode = MODE_NONE;
     }
}

static void
EwinGroupMove(EWin * ewin, int desk, int x, int y)
{
   int                 i, num, dx, dy, newdesk;
   EWin              **gwins;

   if (!ewin)
      return;

   /* Move all group members */
   newdesk = desk != ewin->desktop;
   dx = x - ewin->x;
   dy = y - ewin->y;
   gwins = ListWinGroupMembersForEwin(ewin, ACTION_MOVE, Mode.nogroup, &num);
   for (i = 0; i < num; i++)
     {
	if (gwins[i]->pager || gwins[i]->fixedpos)
	   continue;

	if (newdesk)
	   MoveEwinToDesktopAt(gwins[i], desk, gwins[i]->x + dx,
			       gwins[i]->y + dy);
	else
	   MoveEwin(gwins[i], gwins[i]->x + dx, gwins[i]->y + dy);
     }
   if (gwins)
      Efree(gwins);
}

static void
PagerEwinMove(Pager * p, Pager * pd)
{
   int                 x, y, dx, dy, px, py;
   int                 ax, ay, cx, cy;
   Window              child;

   cx = desks.desk[pd->desktop].current_area_x;
   cy = desks.desk[pd->desktop].current_area_y;
   GetAreaSize(&ax, &ay);

   /* Delta in pager coords */
   dx = Mode.x - Mode.px;
   dy = Mode.y - Mode.py;

   if (dx || dy)
     {
	/* Move mini window */
	GetWinXY(p->hi_win, &x, &y);
	x += dx;
	y += dy;
	XRaiseWindow(disp, p->hi_win);
	EMoveWindow(disp, p->hi_win, x, y);
     }

   /* Find real window position */
   XTranslateCoordinates(disp, p->hi_win, pd->win, 0, 0, &px, &py, &child);
   x = (px * ax * VRoot.w) / pd->w - cx * VRoot.w;
   y = (py * ay * VRoot.h) / pd->h - cy * VRoot.h;

   /* Move all group members */
   EwinGroupMove(p->hi_ewin, pd->desktop, x, y);
}

int
PagersEventMotion(XEvent * ev)
{
   int                 used = 0;
   Pager              *p;

   switch (Mode.mode)
     {
     case MODE_NONE:
	p = FindPager(ev->xmotion.window);
	if (p == NULL)
	   break;
	used = 1;
	PagerHandleMotion(p, ev->xmotion.window, ev->xmotion.x,
			  ev->xmotion.y, PAGER_EVENT_MOTION);
	break;

     case MODE_PAGER_DRAG_PENDING:
     case MODE_PAGER_DRAG:
	Mode.mode = MODE_PAGER_DRAG;
	p = mode_context_pager;
	if (p == NULL)
	   break;
	used = 1;

	if (!FindItem((char *)p->hi_ewin, 0, LIST_FINDBY_POINTER,
		      LIST_TYPE_EWIN))
	   p->hi_ewin = NULL;
	if ((!p->hi_ewin) || (p->hi_ewin->pager) || (p->hi_ewin->fixedpos))
	   break;

	PagerEwinMove(p, p);
	break;
     }

   return used;
}

int
PagersEventMouseDown(XEvent * ev)
{
   Window              win = ev->xbutton.window, child;
   int                 i, num, px, py, in_pager;
   Pager              *p;
   EWin               *ewin, **gwins;

   p = FindPager(win);
   if (!p)
      return 0;

   gwins =
      ListWinGroupMembersForEwin(p->hi_ewin, ACTION_MOVE, Mode.nogroup, &num);
   gwin_px = calloc(num, sizeof(int));
   gwin_py = calloc(num, sizeof(int));

   for (i = 0; i < num; i++)
     {
	gwin_px[i] = gwins[i]->x;
	gwin_py[i] = gwins[i]->y;
     }

   if (gwins)
      Efree(gwins);

   px = ev->xbutton.x;
   py = ev->xbutton.y;
   /* If hi-win, translate x,y to pager window coordinates */
   if (win == p->hi_win)
      XTranslateCoordinates(disp, win, p->win, px, py, &px, &py, &child);
   in_pager = (px >= 0 && py >= 0 && px < p->w && py < p->h);

   if ((int)ev->xbutton.button == Conf.pagers.menu_button)
     {
	if (in_pager)
	  {
	     PagerHideHi(p);
	     PagerMenuShow(p, px, py);
	  }
     }
   else if ((int)ev->xbutton.button == Conf.pagers.win_button)
     {
	ewin = EwinInPagerAt(p, px, py);
	if ((ewin) && (!ewin->pager))
	  {
	     Window              dw;
	     int                 wx, wy, ww, wh, ax, ay, cx, cy;

	     PagerHideHi(p);
	     GetAreaSize(&ax, &ay);
	     cx = desks.desk[p->desktop].current_area_x;
	     cy = desks.desk[p->desktop].current_area_y;
	     wx = ((ewin->x + (cx * VRoot.w)) * (p->w / ax)) / VRoot.w;
	     wy = ((ewin->y + (cy * VRoot.h)) * (p->h / ay)) / VRoot.h;
	     ww = ((ewin->w) * (p->w / ax)) / VRoot.w;
	     wh = ((ewin->h) * (p->h / ay)) / VRoot.h;
	     XTranslateCoordinates(disp, p->win, VRoot.win, 0, 0, &px, &py,
				   &dw);
	     EMoveResizeWindow(disp, p->hi_win, px + wx, py + wy, ww, wh);
	     ESetWindowBackgroundPixmap(disp, p->hi_win, ewin->mini_pmm.pmap);
	     EMapRaised(disp, p->hi_win);
	     GrabPointerSet(p->hi_win, None, 0);
	     p->hi_visible = 1;
	     p->hi_ewin = ewin;
	     p->hi_win_w = ww;
	     p->hi_win_h = wh;
	     Mode.mode = MODE_PAGER_DRAG_PENDING;
	     mode_context_pager = p;
	  }
     }

   return 1;
}

int
PagersEventMouseUp(XEvent * ev)
{
   Window              win = ev->xbutton.window, child;
   int                 used = 0;
   int                 i, num, px, py, in_pager, in_vroot;
   Pager              *p;
   EWin               *ewin, **gwins;
   int                 x, y, pax, pay;
   int                 mode_was;

   p = FindPager(win);
   if (!p)
      goto done;

   used = 1;
   mode_was = Mode.mode;
   Mode.mode = MODE_NONE;

   px = ev->xbutton.x;
   py = ev->xbutton.y;
   /* If hi-win, translate x,y to pager window coordinates */
   if (win == p->hi_win)
      XTranslateCoordinates(disp, win, p->win, px, py, &px, &py, &child);
   in_pager = (px >= 0 && py >= 0 && px < p->w && py < p->h);

   in_vroot = (Mode.x >= 0 && Mode.x < VRoot.w &&
	       Mode.y >= 0 && Mode.y < VRoot.h);

   if (((int)ev->xbutton.button == Conf.pagers.sel_button))
     {
	if (win != Mode.last_bpress || !in_pager)
	   goto done;
	PagerAreaAt(p, px, py, &pax, &pay);
	GotoDesktop(p->desktop);
	if (p->desktop != desks.current)
	   SoundPlay("SOUND_DESKTOP_SHUT");
	SetCurrentArea(pax, pay);
     }
   else if (((int)ev->xbutton.button == Conf.pagers.win_button))
     {
	int                 prev_desk = -1;

	if (!FindItem((char *)p->hi_ewin, 0,
		      LIST_FINDBY_POINTER, LIST_TYPE_EWIN))
	   p->hi_ewin = NULL;

	switch (mode_was)
	  {
	  case MODE_PAGER_DRAG:
	     if (!p->hi_ewin)
		break;

	     /* Remember old desk for the dragged window */
	     prev_desk = p->hi_ewin->desktop;

	     /* Find which pager or iconbox we are in (if any) */
	     ewin = GetEwinPointerInClient();
	     if ((ewin) && (ewin->pager))
	       {
		  PagerEwinMove(p, ewin->pager);
	       }
	     else if ((ewin) && (ewin->ibox))
	       {
		  /* Pointer is in iconbox */

		  /* Don't iconify an iconbox by dragging */
		  if (p->hi_ewin->ibox)
		     break;

		  /* Iconify after moving back to pre-drag position */
		  gwins = ListWinGroupMembersForEwin(p->hi_ewin, ACTION_MOVE,
						     Mode.nogroup, &num);
		  for (i = 0; i < num; i++)
		    {
		       if (!gwins[i]->pager)
			 {
			    MoveEwin(gwins[i], gwin_px[i], gwin_py[i]);
			    IconboxIconifyEwin(ewin->ibox, gwins[i]);
			 }
		    }
		  if (gwins)
		     Efree(gwins);
	       }
	     else if (ewin && ewin->props.vroot)
	       {
		  /* Dropping onto virtual root */
		  EwinReparent(p->hi_ewin, ewin->client.win);
	       }
	     else if (!in_vroot)
	       {
		  /* Move back to real root */
		  EwinReparent(p->hi_ewin, RRoot.win);
	       }
	     else
	       {
		  /* Pointer is not in pager or iconbox */
		  /* Move window(s) to pointer location */
		  x = Mode.x - p->hi_ewin->w / 2;
		  y = Mode.y - p->hi_ewin->h / 2;
		  EwinGroupMove(p->hi_ewin, desks.current, x, y);
	       }
	     break;

	  default:
	     if (!in_pager)
		break;
	     PagerAreaAt(p, px, py, &pax, &pay);
	     GotoDesktop(p->desktop);
	     SetCurrentArea(pax, pay);
	     ewin = EwinInPagerAt(p, px, py);
	     if (ewin)
	       {
		  RaiseEwin(ewin);
		  FocusToEWin(ewin, FOCUS_SET);
	       }
	     break;
	  }

	if (p->hi_ewin)
	  {
	     RedrawPagersForDesktop(p->hi_ewin->desktop, 3);
	     ForceUpdatePagersForDesktop(p->hi_ewin->desktop);
	     if (prev_desk >= 0 && prev_desk != p->hi_ewin->desktop)
	       {
		  RedrawPagersForDesktop(prev_desk, 3);
		  ForceUpdatePagersForDesktop(prev_desk);
	       }
	     PagerHideHi(p);
	  }

	mode_context_pager = NULL;
	GrabPointerRelease();
     }

 done:
   /* unallocate the space that was holding the old positions of the */
   /* windows */
   if (gwin_px)
     {
	Efree(gwin_px);
	gwin_px = NULL;
	Efree(gwin_py);
	gwin_py = NULL;
     }

   return used;
}

int
PagersEventMouseIn(XEvent * ev)
{
   Pager              *p;
   Window              win = ev->xcrossing.window;

   p = FindPager(win);
   if (p)
     {
#if 0				/* Nothing done here */
	PagerHandleMotion(p, win, ev->xcrossing.x, ev->xcrossing.y,
			  PAGER_EVENT_MOUSE_IN);
#endif
	return 1;
     }

   return 0;
}

int
PagersEventMouseOut(XEvent * ev)
{
   Pager              *p;
   Window              win = ev->xcrossing.window;

   p = FindPager(win);
   if (p)
     {
	PagerHandleMotion(p, win, ev->xcrossing.x, ev->xcrossing.y,
			  PAGER_EVENT_MOUSE_OUT);
	return 1;
     }

   return 0;
}
