/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2003, 2004 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   ----------------------------------------------------------------------

   mail fetching window
   
*/
/****************************************************************************
 *    IMPLEMENTATION HEADERS
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/mman.h>

#include "ecurses.h"
#include "pop.h"
#include "error.h"
#include "select.h"
#include "xmalloc.h"
#include "wrapbox.h"
#include "misc.h"
#include "cmd.h"
#include "file.h"
#include "ask.h"
#include "fetch.h"
#include "gettext.h"
#include "mybox.h"
#include "eprintf.h"
#include "interface.h"
#include "color.h"
#include "label.h"
#include "mlex.h"
#include "procmail.h"
#include "exec.h"
#include "hook.h"

/****************************************************************************
 *    IMPLEMENTATION PRIVATE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/

#define WINDOW_WIDTH COLS - 6

#define PREAMBLE do { if (fetch_select == NULL) return; } while (0)

/****************************************************************************
 *    IMPLEMENTATION PRIVATE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION REQUIRED EXTERNAL REFERENCES (AVOID)
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE DATA
 ****************************************************************************/

/* Fetch window consists of select_t object, and an optional label. */
static elabel_t *label        = NULL;
static select_t *fetch_select = NULL;

/* Format used when displaing messages in fetch window. */
static char *fetch_fmt = "%?%$  %D %016f (%-06S)  %s";

/* Color used in fetch window. */
static chtype text_color;

/* Used in draw_line. */
static str_t *line_str = NULL;

/* Used to determine, whether the window should actually be refreshed
   because some actions may still be taken after the window is closed. */
static int visible = 0;

/****************************************************************************
 *    INTERFACE DATA
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTION PROTOTYPES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTIONS
 ****************************************************************************/

static void
draw_line (WINDOW *win, int maxlen, int index, search_t *search)
{
        mail_t *mail = pop_mail_info (index);

        if (line_str == NULL)
                line_str = str_create ();

        if (mail == NULL)
                str_clear (line_str);
        else
                eprintf_mail_str (fetch_fmt, mail, line_str);

        maxlen -= window_addnstr (win, line_str->str, maxlen);
        while (maxlen-- > 0)
                window_addch (win, ' ');
}



static int
count (select_t *nothing)
{
        return pop_mail_count ();
}

/* This file is generated by interface.pl script from interface.desc,
   and inc.in. */
static WINDOW *interface_init (void);
#include "fetch.inc"


/****************************************************************************
 *    INTERFACE FUNCTIONS
 ****************************************************************************/


void
fetch_init (void)
{
        WINDOW *window;

        window       = interface_init ();
        fetch_select = select_open (window, 0, draw_line, count);

        window_set_functions (window, fetch_refresh, fetch_redraw,
                              fetch_set_focus, fetch_unset_focus);
}



void
fetch_free_resources (void)
{
        if (fetch_select)
                select_close (fetch_select);
        fetch_select = NULL;

        if (label)
                label_destroy (label);
        label = NULL;

        if (line_str)
                str_destroy (line_str);
        line_str = NULL;
}



void
fetch_refresh (void)
{
        select_show (fetch_select);
        label_show (label);
        fetch_redraw ();
}



void
fetch_redraw (void)
{
        if (! visible)
                return;
        
        select_redraw (fetch_select);
        label_redraw (label);
}



void
fetch_set_focus (void)
{
        label_set_focus (label);
        cmd_state_push (CMD_FETCH);

        fetch_redraw ();
}



void
fetch_unset_focus (void)
{
        label_unset_focus (label);
        label_redraw (label);

        cmd_state_pop ();
}


void
fetch_show (void)
{
        visible = 1;
        
        window_show (fetch_select->win);
        fetch_redraw ();
}



void
fetch_hide (void)
{
        visible = 0;

        pop_close_interactive ();
        window_hide (fetch_select->win);
}



void
fetch_open (void)
{
        pop_open_interactive ();
}



void
fetch_close (void)
{
        fetch_before_close ();
        pop_close_interactive ();
        fetch_hide ();
}


void
fetch_before_close (void)
{
        exec_t *exec;

        exec = exec_lookup_fun (fetch_before_close);

        hook_execute (exec->hook);
}


void
fetch_next (void)
{
        select_next (fetch_select);
}



void
fetch_prev (void)
{
        select_prev (fetch_select);
}



void
fetch_next_page (void)
{
        select_next_page (fetch_select);
}



void
fetch_prev_page (void)
{
        select_prev_page (fetch_select);
}



void
fetch_first (void)
{
        select_first (fetch_select);
}



void
fetch_last (void)
{
        select_last (fetch_select);
}



void
fetch_rset (void)
{
        pop_reset_interactive ();
}



void
fetch_del_mail (void)
{
        pop_delete_interactive (fetch_select->bar_pos + 1);
}


void
fetch_get_mail (void)
{
        pop_fetch_interactive (fetch_select->bar_pos + 1);
}


void
fetch_del_all (void)
{
        pop_delete_all_interactive ();
}


void
fetch_get_all (void)
{
        pop_fetch_all_interactive ();
}


/****************************************************************************
 *    INTERFACE CLASS BODIES
 ****************************************************************************/
/****************************************************************************
 *
 *    END MODULE fetch.c
 *
 ****************************************************************************/
