/* Copyright (C) 2001 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/


#include <cstdlib>
#include <iostream>

#include <gtk--/main.h>
#include <gtk--/scrollbar.h>
#include <gtk--/adjustment.h>
#include <gdk/gdkkeysyms.h> // the key codes are here

#include "dialogs.h"
#include "gpl.h"
#include "helpfile.h"
#include "info_icons.h"


FileReadSelectDialog::FileReadSelectDialog(int size, Gtk::Window& window):
                                           Gtk::FileSelection("File to fax"), in_run_loop(false),
					   standard_size(size), parent(window) {
  get_ok_button()->clicked.connect(SigC::bind(SigC::slot(this, &FileReadSelectDialog::selected), true));
  get_cancel_button()->clicked.connect(SigC::bind(SigC::slot(this, &FileReadSelectDialog::selected), false));

  hide_fileop_buttons();

  string filename;
  if (!prog_config.homedir.empty()) {
    filename = prog_config.homedir;
    filename += "/faxout/";
    set_filename(filename);
  }
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);
  set_position(GTK_WIN_POS_CENTER);

  show_all();
}

string FileReadSelectDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
  return result;
}

void FileReadSelectDialog::selected(bool accept) {
  if (accept) result = get_filename();
  finish();
}

void FileReadSelectDialog::finish(void) {
  parent.set_sensitive(true);
  hide_all();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint FileReadSelectDialog::delete_event_impl(GdkEventAny*) {
  finish();
  return true; // returning true prevents destroy sig being emitted
}

GplDialog::GplDialog(int standard_size): Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
			     result(rejected), accept_button("Accept"), reject_button("Reject"),
		             label("Do you accept the Conditions, Notices and Disclaimers shown above?"),
			     table(3, 3, false) {

  editbox.set_editable(false);
  if (!prog_config.fixed_font.empty()) editbox.set_context(Gdk_Font(prog_config.fixed_font));
  editbox.freeze();
  editbox.insert(copyright_msg);
  editbox.thaw();
  Gtk::Scrollbar* scrollbar_p = manage(new Gtk::VScrollbar(*(editbox.get_vadjustment())));

  table.attach(editbox, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	       GTK_FILL | GTK_EXPAND, 0, 0);
  table.attach(*scrollbar_p, 2, 3, 0, 1, GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);
  table.attach(label, 0, 2, 1, 2, GTK_FILL | GTK_EXPAND,
	       0, 0, standard_size/3);
  table.attach(accept_button, 0, 1, 2, 3,
	       0, 0, 0, standard_size/3);
  table.attach(reject_button, 1, 2, 2, 3,
	       0, 0, 0, standard_size/3);

  accept_button.clicked.connect(SigC::bind(SigC::slot(this, &GplDialog::selected), accepted));
  reject_button.clicked.connect(SigC::bind(SigC::slot(this, &GplDialog::selected), rejected));
  
  add(table);
  
  set_title("efax-gtk: Conditions, Notices and Disclaimers");
  set_modal(true);

  accept_button.set_usize(standard_size * 3, standard_size);
  reject_button.set_usize(standard_size * 3, standard_size);
  set_usize(standard_size * 25, standard_size * 16);
  
  set_border_width(standard_size/4);
  set_position(GTK_WIN_POS_CENTER);
  
  grab_focus();
  scrollbar_p->unset_flags(GTK_CAN_FOCUS);
  accept_button.unset_flags(GTK_CAN_FOCUS);
  reject_button.unset_flags(GTK_CAN_FOCUS);
  editbox.unset_flags(GTK_CAN_FOCUS);
  
  show_all();
}

int GplDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
  return result;
}

void GplDialog::selected(Result selection) {
  hide_all();
  result = selection;
  if (in_run_loop) Gtk::Main::quit(); // this will cause the run() method to return, with result as its return value
}

gint GplDialog::delete_event_impl(GdkEventAny*) {
  selected(rejected);
  return true; // returning true prevents destroy sig being emitted
}

gint GplDialog::key_press_event_impl(GdkEventKey* event_p) {

  int keycode = event_p->keyval;
  
  if (keycode == GDK_Escape) selected(rejected);
  
  else if (keycode == GDK_Home || keycode == GDK_End
	   || keycode == GDK_Up || keycode == GDK_Down
	   || keycode == GDK_Page_Up || keycode == GDK_Page_Down) {
    editbox.key_press_event_impl(event_p);
    return false;
  }
  return true;
}

InfoDialog::InfoDialog(const char* text, const char* caption, int standard_size, InfoType info_type,
		       Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
			     ok_button("OK"), label(text),
			     table(2, 2, false), parent(window) {
  if (info_type == information) pixmap_p = new Gtk::Pixmap(information_xpm);
  else if (info_type == warning) pixmap_p = new Gtk::Pixmap(warning_xpm);
  else pixmap_p = new Gtk::Pixmap(critical_xpm);
  if (!pixmap_p) {
    cerr << "Memory allocation error in InfoDialog::InfoDialog()" << endl;
    exit(MEM_ERROR);
  }

  table.attach(*pixmap_p, 0, 1, 0, 1, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(label, 1, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::slot(this, &InfoDialog::selected));

  add(table);
  
  set_title(caption);
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize(standard_size * 3, standard_size);
  set_border_width(standard_size/2);

  ok_button.grab_focus();

  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void InfoDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void InfoDialog::selected(void) {
  parent.set_sensitive(true);
  hide_all();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint InfoDialog::delete_event_impl(GdkEventAny*) {
  selected();
  return true; // returning true prevents destroy sig being emitted
}

gint InfoDialog::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Escape) selected();
  return false;
}

InfoDialog::~InfoDialog(void) {
  delete pixmap_p;
}

PromptDialog::PromptDialog(const char* text, const char* caption, int standard_size, Gtk::Window& window,
			   const char* accept_name, const char* reject_name):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
			     accept_button(accept_name), reject_button(reject_name),
			     label(text), table(2, 2, false), parent(window) {
  table.attach(label, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(accept_button, 0, 1, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(reject_button, 1, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  accept_button.clicked.connect(SigC::bind(SigC::slot(this, &PromptDialog::selected), true));
  reject_button.clicked.connect(SigC::bind(SigC::slot(this, &PromptDialog::selected), false));

  add(table);
  
  set_title(caption);
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  accept_button.set_usize(standard_size * 3, standard_size);
  reject_button.set_usize(standard_size * 3, standard_size);
  set_border_width(standard_size/2);
  accept_button.grab_focus();
  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void PromptDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void PromptDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted();
  else rejected();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint PromptDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

gint PromptDialog::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Escape) selected(false);
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Tab) Gtk::Window::key_press_event_impl(event_p);
  return false;
}

HelpDialog::HelpDialog(const int standard_size): Gtk::Window(GTK_WINDOW_DIALOG),
			     cancel_button("Cancel"),
			     table(2, 2, false) {

  editbox.set_editable(false);
  if (!prog_config.fixed_font.empty()) editbox.set_context(Gdk_Font(prog_config.fixed_font));
  editbox.freeze();
  editbox.insert(helpfile_msg);
  editbox.thaw();
  Gtk::Scrollbar* scrollbar_p = manage(new Gtk::VScrollbar(*(editbox.get_vadjustment())));
  
  table.attach(editbox, 0, 1, 0, 1, GTK_FILL | GTK_EXPAND,
	       GTK_FILL | GTK_EXPAND, 0, 0);
  table.attach(*scrollbar_p, 1, 2, 0, 1, GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);
  table.attach(cancel_button, 0, 2, 1, 2,
	       0, 0, 0, standard_size/3);
  
  cancel_button.clicked.connect(SigC::slot(this, &HelpDialog::kill_dialog));
  
  add(table);
  
  set_title("efax-gtk: Helpfile");
  cancel_button.set_usize(standard_size * 3, standard_size);
  set_usize(standard_size * 25, standard_size * 14);
  
  set_border_width(standard_size/4);
  set_position(GTK_WIN_POS_NONE);
  
  grab_focus();
  scrollbar_p->unset_flags(GTK_CAN_FOCUS);
  cancel_button.unset_flags(GTK_CAN_FOCUS);
  editbox.unset_flags(GTK_CAN_FOCUS);
  
  show_all();
}

void HelpDialog::kill_dialog(void) {
  hide_all();
  delete this;  // this a modeless dialog, and so safe
}

gint HelpDialog::delete_event_impl(GdkEventAny*) {
  kill_dialog();
  return true; // returning true prevents destroy sig being emitted
}

gint HelpDialog::key_press_event_impl(GdkEventKey* event_p) {

  int keycode = event_p->keyval;
  
  if (keycode == GDK_Escape) kill_dialog();
  
  else if (keycode == GDK_Home || keycode == GDK_End
	   || keycode == GDK_Up || keycode == GDK_Down
	   || keycode == GDK_Page_Up || keycode == GDK_Page_Down) {
    editbox.key_press_event_impl(event_p);
    return false;
  }
  return true;
}
