/* This file is part of EdiTeX, an editor of mathematical
 * expressions based on TeX syntax.
 * 
 * Copyright (C) 2002-2003 Luca Padovani <lpadovan@cs.unibo.it>,
 *                    2003 Paolo Marinelli <pmarinel@cs.unibo.it>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, please visit the project's home page
 * http://helm.cs.unibo.it/editex/
 * or send an email to <lpadovan@cs.unibo.it>
 */

#ifndef __TDictionary_hh__
#define __TDictionary_hh__

#include <config.h>

#include <string>
#include <vector>
#if defined(HAVE_EXT_HASH_MAP)
#include <ext/hash_map>
#elif defined(HAVE_HASH_MAP)
#include <hash_map>
#else
#error "no hash_map could be found"
#endif
#include <list>

#include "dom.hh"
#include "TToken.hh"

class TDictionary
{
public:
  TDictionary(class ALogger& l) : logger(l) { };
  ~TDictionary() { };

  enum Form
    {
      INFIX,
      PREFIX,
      POSTFIX
    };

  enum EntryClass
    {
      UNDEFINED,
      MACRO,
      OPERATOR,
      IDENTIFIER,
      NUMBER
    };

  struct Entry
  {
    Entry(void)
    { 
      cls = UNDEFINED;
      table = delimiter = limits = embellishment = leftOpen = rightOpen = 0;
    };

    std::vector<TToken> pattern;
    std::string value;

    bool defined(void) const { return cls != UNDEFINED; };
    bool hasArguments(void) const { return !pattern.empty(); };
    bool paramDelimited(unsigned) const;
    bool lastDelimiter(unsigned) const;
    unsigned previousParam(unsigned) const;

    EntryClass cls;
    unsigned delimiter : 1;
    unsigned limits : 1;
    unsigned embellishment : 1;
    unsigned leftOpen : 1;
    unsigned rightOpen : 1;
    unsigned table : 1;
  };

  static std::string getDefaultDictionaryPath(void);

  void load(const std::string&);
  void load(const std::string&, const std::string&);
  void load(const DOM::Document&, const std::string& = "");
  const Entry& find(const std::string&) const;
  std::string complete(const std::string, std::list<std::string>&) const;

private:
#if defined(HAVE_EXT_HASH_MAP)
  struct StringHash : public std::unary_function< std::string, size_t >
  { size_t operator()(const std::string& s) const { return __gnu_cxx::hash<char*>()(s.c_str()); } };
#elif defined(HAVE_HASH_MAP)
  struct StringHash : public std::unary_function< std::string, size_t >
  { size_t operator()(const std::string& s) const { return hash<char*>()(s.c_str()); } };
#else
#error "no hash_map could be found"
#endif

#if 0
  struct StringEq : public std::binary_function< std::string, std::string, bool >
  { bool operator()(const std::string&, const class String*) const; };
#endif

  class ALogger& logger;
#if defined(HAVE_EXT_HASH_MAP)
  typedef __gnu_cxx::hash_map< std::string, Entry, StringHash > Dictionary;
#elif defined(HAVE_HASH_MAP)
  typedef std::hash_map< std::string, Entry, StringHash > Dictionary;
#else
#error "no hash_map could be found"
#endif
  Dictionary entries;
};

#endif // __TDictionary_hh__

