/*
***************************************************************************
*
* Author: Teunis van Beelen
*
* Copyright (C) 2007, 2008, 2009 Teunis van Beelen
*
* teuniz@gmail.com
*
***************************************************************************
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation version 2 of the License.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
***************************************************************************
*
* This version of GPL is at http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt
*
***************************************************************************
*/



#include "ascii_export.h"



UI_AsciiExportwindow::UI_AsciiExportwindow(QWidget *parent)
{
  int i;

  mainwindow = (UI_Mainwindow *)parent;

  AsciiExportDialog = new QDialog;

  AsciiExportDialog->setMinimumSize(QSize(800, 180));
  AsciiExportDialog->setMaximumSize(QSize(800, 180));
  AsciiExportDialog->setWindowTitle("Export to ASCII");
  AsciiExportDialog->setModal(TRUE);
  AsciiExportDialog->setAttribute(Qt::WA_DeleteOnClose, TRUE);

  filelist = new QListWidget(AsciiExportDialog);
  filelist->setGeometry(QRect(10, 10, 780, 75));
  filelist->setSelectionBehavior(QAbstractItemView::SelectRows);
  filelist->setSelectionMode(QAbstractItemView::SingleSelection);
  for(i=0; i<mainwindow->files_open; i++)
  {
    if((mainwindow->edfheaderlist[i]->edf)||(mainwindow->edfheaderlist[i]->bdf))
    {
      new QListWidgetItem(mainwindow->edfheaderlist[i]->filename, filelist);
    }
  }

  progressbar = new QProgressBar(AsciiExportDialog);
  progressbar->setGeometry(QRect(10, 100, 780, 25));

  ExportButton = new QPushButton(AsciiExportDialog);
  ExportButton->setGeometry(QRect(10, 140, 100, 25));
  ExportButton->setText("Export");

  CloseButton = new QPushButton(AsciiExportDialog);
  CloseButton->setGeometry(QRect(690, 140, 100, 25));
  CloseButton->setText("Close");

  QObject::connect(CloseButton,  SIGNAL(clicked()), AsciiExportDialog, SLOT(close()));
  QObject::connect(ExportButton, SIGNAL(clicked()), this,              SLOT(ExportButtonClicked()));

  filelist->setCurrentRow(mainwindow->files_open - 1);

  AsciiExportDialog->exec();
}



void UI_AsciiExportwindow::ExportButtonClicked()
{
  int i, j, k, l, m, p, r, n,
      edfsignals,
      datarecords,
      datarecordswritten,
      recordsize,
      recordfull,
      edfplus,
      bdf,
      bdfplus=0,
      samplesize,
      annot_ch[256],
      nr_annot_chns=0,
      skip,
      max,
      onset,
      duration,
      zero,
      max_tal_ln;

  char path[1024],
       ascii_path[1024],
       *edf_hdr,
       *scratchpad,
       *cnv_buf,
       *time_in_txt,
       *duration_in_txt,
       str[1024];

  FILE *inputfile,
       *outputfile,
       *annotationfile;

  double data_record_duration,
         elapsedtime,
         time_tmp,
         d_tmp,
         value_tmp;

  QFileDialog fchooser;


  if(!mainwindow->files_open)
  {
    AsciiExportDialog->close();
    return;
  }

  ExportButton->setEnabled(FALSE);
  CloseButton->setEnabled(FALSE);

  for(i=0; i<mainwindow->files_open; i++)
  {
    if(!strcmp(mainwindow->edfheaderlist[i]->filename, filelist->item(filelist->currentRow())->text().toLatin1().data()))
    {
      break;
    }
  }

  if(i==mainwindow->files_open)  return;

  n = i;

  inputfile = mainwindow->edfheaderlist[n]->file_hdl;

  fchooser.setFileMode(QFileDialog::AnyFile);
  fchooser.setAcceptMode(QFileDialog::AcceptSave);
  fchooser.setWindowTitle("Export to ASCII");
  fchooser.setLabelText(QFileDialog::FileName, "File");
  fchooser.setDefaultSuffix("txt");
  fchooser.setFilter("Text files (*.txt *.TXT)");

  get_filename_from_path(path, mainwindow->edfheaderlist[n]->filename, 1024);
  get_filename_from_path(str, mainwindow->edfheaderlist[n]->filename, 1024);
  remove_extension_from_filename(str);
  strcat(str, "_data.txt");
  fchooser.selectFile(str);
  if(mainwindow->recent_savedir[0] != 0)
  {
    fchooser.setDirectory(mainwindow->recent_savedir);
  }

  if(!(fchooser.exec() == QDialog::Accepted))
  {
    AsciiExportDialog->close();
    return;
  }

  strcpy(mainwindow->recent_savedir, fchooser.directory().absolutePath().toLatin1().data());

  strcpy(path, fchooser.selectedFiles().at(0).toLatin1().data());

  if(!(strcmp(path + strlen(path) - 9, "_data.txt")))
  {
    path[strlen(path) - 9] = 0;
  }

  scratchpad = (char *)calloc(1, 128);
  if(scratchpad==NULL)
  {
    UI_Messagewindow popuperror("Error", "Malloc error! (scratchpad).");
    AsciiExportDialog->close();
    return;
  }

  if(fseeko64(inputfile, 0xfc, SEEK_SET))
  {
    UI_Messagewindow popuperror("Error", "Error, reading file.");
    AsciiExportDialog->close();
    return;
  }

  if(fread(scratchpad, 4, 1, inputfile)!=1)
  {
    UI_Messagewindow popuperror("Error", "Error, reading file.");
    AsciiExportDialog->close();
    return;
  }

  scratchpad[4] = 0;
  edfsignals = atoi(scratchpad);

  edf_hdr = (char *)calloc(1, (edfsignals + 1) * 256);
  if(edf_hdr==NULL)
  {
    UI_Messagewindow popuperror("Error", "Malloc error! (edf_hdr)\n");
    AsciiExportDialog->close();
    return;
  }

  rewind(inputfile);
  if(fread(edf_hdr, (edfsignals + 1) * 256, 1, inputfile)!=1)
  {
    UI_Messagewindow popuperror("Error", "Error, reading file.");
    free(edf_hdr);
    AsciiExportDialog->close();
    return;
  }

  for(i=0; i<((edfsignals+1)*256); i++)
  {
    if(edf_hdr[i]==',') edf_hdr[i] = '\'';  /* replace all comma's in header by single quotes because they */
  }                                         /* interfere with the comma-separated txt-files                */

  strncpy(scratchpad, edf_hdr + 0xec, 8);
  scratchpad[8] = 0;
  datarecords = atoi(scratchpad);

  strncpy(scratchpad, edf_hdr + 0xf4, 8);
  scratchpad[8] = 0;
  data_record_duration = atof(scratchpad);

  nr_annot_chns = 0;

  if(edf_hdr[0]==-1)
  {
    bdf = 1;
    samplesize = 3;
    edfplus = 0;
    if((strncmp(edf_hdr + 0xc0, "BDF+C     ", 10))&&(strncmp(edf_hdr + 0xc0, "BDF+D     ", 10)))
    {
      bdfplus = 0;
    }
    else
    {
      bdfplus = 1;
      for(i=0; i<edfsignals; i++)
      {
        if(!(strncmp(edf_hdr + 256 + i * 16, "BDF Annotations ", 16)))
        {
          annot_ch[nr_annot_chns] = i;
          nr_annot_chns++;
          if(nr_annot_chns>255)  break;
        }
      }
    }
  }
  else
  {
    bdf = 0;
    samplesize = 2;

    if((strncmp(edf_hdr + 0xc0, "EDF+C     ", 10))&&(strncmp(edf_hdr + 0xc0, "EDF+D     ", 10)))
    {
      edfplus = 0;
    }
    else
    {
      edfplus = 1;
      for(i=0; i<edfsignals; i++)
      {
        if(!(strncmp(edf_hdr + 256 + i * 16, "EDF Annotations ", 16)))
        {
          annot_ch[nr_annot_chns] = i;
          nr_annot_chns++;
          if(nr_annot_chns>255)  break;
        }
      }
    }
  }

  edfparamascii = (struct asciiedfparamblock *)calloc(1, sizeof(struct asciiedfparamblock[edfsignals]));
  if(edfparamascii==NULL)
  {
    UI_Messagewindow popuperror("Error", "Malloc error! (edfparam)\n");
    free(edf_hdr);
    AsciiExportDialog->close();
    return;
  }

  recordsize = 0;

  for(i=0; i<edfsignals; i++)
  {
    strncpy(scratchpad, edf_hdr + 256 + edfsignals * 216 + i * 8, 8);
    scratchpad[8] = 0;
    edfparamascii[i].smp_per_record = atoi(scratchpad);
    edfparamascii[i].buf_offset = recordsize;
    recordsize += edfparamascii[i].smp_per_record;
    strncpy(scratchpad, edf_hdr + 256 + edfsignals * 104 + i * 8, 8);
    scratchpad[8] = 0;
    edfparamascii[i].phys_min = atof(scratchpad);
    strncpy(scratchpad, edf_hdr + 256 + edfsignals * 112 + i * 8, 8);
    scratchpad[8] = 0;
    edfparamascii[i].phys_max = atof(scratchpad);
    strncpy(scratchpad, edf_hdr + 256 + edfsignals * 120 + i * 8, 8);
    scratchpad[8] = 0;
    edfparamascii[i].dig_min = atoi(scratchpad);
    strncpy(scratchpad, edf_hdr + 256 + edfsignals * 128 + i * 8, 8);
    scratchpad[8] = 0;
    edfparamascii[i].dig_max = atoi(scratchpad);
    edfparamascii[i].time_step = data_record_duration / edfparamascii[i].smp_per_record;
    edfparamascii[i].sense = (edfparamascii[i].phys_max - edfparamascii[i].phys_min) / (edfparamascii[i].dig_max - edfparamascii[i].dig_min);
    edfparamascii[i].offset = (int)(edfparamascii[i].phys_max / edfparamascii[i].sense - edfparamascii[i].dig_max);
  }

  cnv_buf = (char *)calloc(1, recordsize * samplesize);
  if(cnv_buf==NULL)
  {
    UI_Messagewindow popuperror("Error", "Malloc error! (cnv_buf)");
    free(edf_hdr);
    free(edfparamascii);
    AsciiExportDialog->close();
    return;
  }

  free(scratchpad);

  max_tal_ln = 0;

  for(r=0; r<nr_annot_chns; r++)
  {
    if(max_tal_ln<edfparamascii[annot_ch[r]].smp_per_record * samplesize)  max_tal_ln = edfparamascii[annot_ch[r]].smp_per_record * samplesize;
  }

  if(max_tal_ln<128)  max_tal_ln = 128;

  scratchpad = (char *)calloc(1, max_tal_ln + 2);
  if(scratchpad==NULL)
  {
    UI_Messagewindow popuperror("Error", "Malloc error! (scratchpad)");
    free(cnv_buf);
    free(edf_hdr);
    free(edfparamascii);
    AsciiExportDialog->close();
    return;
  }

  duration_in_txt = (char *)calloc(1, max_tal_ln + 2);
  if(duration_in_txt==NULL)
  {
    UI_Messagewindow popuperror("Error", "Malloc error! (duration_in_txt)");
    free(scratchpad);
    free(cnv_buf);
    free(edf_hdr);
    free(edfparamascii);
    AsciiExportDialog->close();
    return;
  }

  time_in_txt = (char *)calloc(1, max_tal_ln + 2);
  if(time_in_txt==NULL)
  {
    UI_Messagewindow popuperror("Error", "Malloc error! (time_in_txt)");
    free(duration_in_txt);
    free(scratchpad);
    free(cnv_buf);
    free(edf_hdr);
    free(edfparamascii);
    AsciiExportDialog->close();
    return;
  }

/***************** write header ******************************/

  strcpy(ascii_path, path);
  remove_extension_from_filename(ascii_path);
  strcat(ascii_path, "_header.txt");
  outputfile = fopen64(ascii_path, "wb");
  if(outputfile==NULL)
  {
    UI_Messagewindow popuperror("Error", "Error, can not open headerfile for writing.");
    free(edf_hdr);
    free(edfparamascii);
    free(cnv_buf);
    free(time_in_txt);
    free(duration_in_txt);
    free(scratchpad);
    AsciiExportDialog->close();
    return;
  }

  fprintf(outputfile, "Version,Patient,Recording,Startdate,Startime,Bytes,Reserved,NumRec,Duration,NumSig\n");

  if(bdf)
  {
    fputc('.', outputfile);
    fprintf(outputfile, "%.7s,", edf_hdr + 1);
  }
  else
  {
    fprintf(outputfile, "%.8s,", edf_hdr);
  }
  fprintf(outputfile, "%.80s,", edf_hdr + 8);
  fprintf(outputfile, "%.80s,", edf_hdr + 88);
  fprintf(outputfile, "%.8s,", edf_hdr + 168);
  fprintf(outputfile, "%.8s,", edf_hdr + 176);
  fprintf(outputfile, "%.8s,", edf_hdr + 184);
  fprintf(outputfile, "%.44s,", edf_hdr + 192);
  fprintf(outputfile, "%i,", datarecords);
  fprintf(outputfile, "%.8s,", edf_hdr + 244);
  snprintf(scratchpad, 128, "%.4s", edf_hdr + 252);
  fprintf(outputfile, "%i\n", atoi(scratchpad) - nr_annot_chns);

  fclose(outputfile);

/***************** write signals ******************************/

  strcpy(ascii_path, path);
  remove_extension_from_filename(ascii_path);
  strcat(ascii_path, "_signals.txt");
  outputfile = fopen64(ascii_path, "wb");
  if(outputfile==NULL)
  {
    UI_Messagewindow popuperror("Error", "Error, can not open signalfile for writing.");
    free(edf_hdr);
    free(edfparamascii);
    free(cnv_buf);
    free(time_in_txt);
    free(duration_in_txt);
    free(scratchpad);
    AsciiExportDialog->close();
    return;
  }

  fprintf(outputfile, "Signal,Label,Transducer,Units,Min,Max,Dmin,Dmax,PreFilter,Smp/Rec,Reserved\n");

  for(i=0; i<edfsignals; i++)
  {
    if(edfplus || bdfplus)
    {
      skip = 0;

      for(j=0; j<nr_annot_chns; j++)
      {
        if(i==annot_ch[j])  skip = 1;
      }

      if(skip) continue;
    }

    fprintf(outputfile, "%i,", i + 1);
    fprintf(outputfile, "%.16s,", edf_hdr + 256 + i * 16);
    fprintf(outputfile, "%.80s,", edf_hdr + 256 + edfsignals * 16 + i * 80);
    fprintf(outputfile, "%.8s,", edf_hdr + 256 + edfsignals * 96 + i * 8);
    fprintf(outputfile, "%f,", edfparamascii[i].phys_min);
    fprintf(outputfile, "%f,", edfparamascii[i].phys_max);
    fprintf(outputfile, "%i,", edfparamascii[i].dig_min);
    fprintf(outputfile, "%i,", edfparamascii[i].dig_max);
    fprintf(outputfile, "%.80s,", edf_hdr + 256 + edfsignals * 136 + i * 80);
    fprintf(outputfile, "%i,", edfparamascii[i].smp_per_record);
    fprintf(outputfile, "%.32s\n", edf_hdr + 256 + edfsignals * 224 + i * 32);
  }

  fclose(outputfile);

/***************** open annotation file ******************************/

  strcpy(ascii_path, path);
  remove_extension_from_filename(ascii_path);
  strcat(ascii_path, "_annotations.txt");
  annotationfile = fopen64(ascii_path, "wb");
  if(annotationfile==NULL)
  {
    UI_Messagewindow popuperror("Error", "Error, can not open annotationfile for writing.");
    free(edf_hdr);
    free(edfparamascii);
    free(cnv_buf);
    free(time_in_txt);
    free(duration_in_txt);
    free(scratchpad);
    AsciiExportDialog->close();
    return;
  }

  fprintf(annotationfile, "Onset,Duration,Annotation\n");

/***************** write data ******************************/

  strcpy(ascii_path, path);
  remove_extension_from_filename(ascii_path);
  strcat(ascii_path, "_data.txt");
  outputfile = fopen64(ascii_path, "wb");
  if(outputfile==NULL)
  {
    UI_Messagewindow popuperror("Error", "Error, can not open datafile for writing.");
    fclose(annotationfile);
    free(edf_hdr);
    free(edfparamascii);
    free(cnv_buf);
    free(time_in_txt);
    free(duration_in_txt);
    free(scratchpad);
    AsciiExportDialog->close();
    return;
  }

  fprintf(outputfile, "Time");

  for(i=0; i<(edfsignals-nr_annot_chns); i++)
  {
    fprintf(outputfile, ",%i", i + 1);
  }

  if(fputc('\n', outputfile)==EOF)
  {
    UI_Messagewindow popuperror("Error", "Error when writing to outputfile.");
    fclose(annotationfile);
    fclose(outputfile);
    free(edf_hdr);
    free(edfparamascii);
    free(cnv_buf);
    free(time_in_txt);
    free(duration_in_txt);
    free(scratchpad);
    AsciiExportDialog->close();
    return;
  }

  if(fseeko64(inputfile, (edfsignals + 1) * 256, SEEK_SET))
  {
    UI_Messagewindow popuperror("Error", "Error when reading inputfile.");
    fclose(annotationfile);
    fclose(outputfile);
    free(edf_hdr);
    free(edfparamascii);
    free(cnv_buf);
    free(time_in_txt);
    free(duration_in_txt);
    free(scratchpad);
    AsciiExportDialog->close();
    return;
  }

/***************** start data conversion ******************************/

  QApplication::setOverrideCursor(Qt::WaitCursor);

  datarecordswritten = 0;

  progressbar->setRange(0, datarecords);

  for(i=0; i<datarecords; i++)
  {
    if(!(i%100))
    {
      progressbar->setValue(i);

      for(l=0; l<10; l++)  qApp->processEvents();
    }

    for(j=0; j<edfsignals; j++)  edfparamascii[j].smp_written = 0;

    if(fread(cnv_buf, recordsize * samplesize, 1, inputfile)!=1)
    {
      QApplication::restoreOverrideCursor();
      UI_Messagewindow popuperror("Error", "Error when reading inputfile during export.");
      fclose(annotationfile);
      fclose(outputfile);
      free(edf_hdr);
      free(edfparamascii);
      free(cnv_buf);
      free(time_in_txt);
      free(duration_in_txt);
      free(scratchpad);
      AsciiExportDialog->close();
      return;
    }

    if(edfplus || bdfplus)
    {
      max = edfparamascii[annot_ch[0]].smp_per_record * samplesize;
      p = edfparamascii[annot_ch[0]].buf_offset * samplesize;

/* extract time from datarecord */

      for(k=0; k<max; k++)
      {
        if(k>max_tal_ln)
        {
          QApplication::restoreOverrideCursor();
          snprintf(str, 256, "Error, TAL in record %i exceeds my buffer.", datarecordswritten + 1);
          UI_Messagewindow popuperror("Error", str);
          fclose(annotationfile);
          fclose(outputfile);
          free(edf_hdr);
          free(edfparamascii);
          free(cnv_buf);
          free(time_in_txt);
          free(duration_in_txt);
          free(scratchpad);
          AsciiExportDialog->close();
          return;
        }
        scratchpad[k] = cnv_buf[p + k];
        if(scratchpad[k]==20) break;
      }
      scratchpad[k] = 0;
      elapsedtime = atof(scratchpad);

/* process annotations */

      for(r=0; r<nr_annot_chns; r++)
      {
        p = edfparamascii[annot_ch[r]].buf_offset * samplesize;
        max = edfparamascii[annot_ch[r]].smp_per_record * samplesize;
        n = 0;
        zero = 0;
        onset = 0;
        duration = 0;
        time_in_txt[0] = 0;
        duration_in_txt[0] = 0;
        scratchpad[0] = 0;

        for(k=0; k<max; k++)
        {
          if(k>max_tal_ln)
          {
            QApplication::restoreOverrideCursor();
            snprintf(str, 256, "Error, TAL in record %i exceeds my buffer.", datarecordswritten + 1);
            UI_Messagewindow popuperror("Error", str);
            fclose(annotationfile);
            fclose(outputfile);
            free(edf_hdr);
            free(edfparamascii);
            free(cnv_buf);
            free(time_in_txt);
            free(duration_in_txt);
            free(scratchpad);
            AsciiExportDialog->close();
            return;
          }

          scratchpad[n] = cnv_buf[p + k];

          if(scratchpad[n]==0)
          {
            n = 0;
            onset = 0;
            duration = 0;
            time_in_txt[0] = 0;
            duration_in_txt[0] = 0;
            scratchpad[0] = 0;
            zero++;
            continue;
          }
          else  zero = 0;

          if(zero>1)  break;

          if(scratchpad[n]==20)
          {
            if(duration)
            {
              scratchpad[n] = 0;
              strcpy(duration_in_txt, scratchpad);
              n = 0;
              duration = 0;
              scratchpad[0] = 0;
              continue;
            }
            else if(onset)
                 {
                   scratchpad[n] = 0;
                   if(n)
                   {
                     utf8_to_latin1(scratchpad);
                     for(m=0; m<n; m++)
                     {
                       if(scratchpad[m] == 0)
                       {
                         break;
                       }

                       if((((unsigned char *)scratchpad)[m] < 32) || (((unsigned char *)scratchpad)[m] == ','))
                       {
                         scratchpad[m] = '.';
                       }
                     }
                     fprintf(annotationfile, "%s,%s,%s\n", time_in_txt, duration_in_txt, scratchpad);
                   }
                   n = 0;
                   duration = 0;
                   duration_in_txt[0] = 0;
                   scratchpad[0] = 0;
                   continue;
                 }
                 else
                 {
                   scratchpad[n] = 0;
                   strcpy(time_in_txt, scratchpad);
                   n = 0;
                   onset = 1;
                   duration = 0;
                   duration_in_txt[0] = 0;
                   scratchpad[0] = 0;
                   continue;
                 }
          }

          if(scratchpad[n]==21)
          {
            if(!onset)
            {
              scratchpad[n] = 0;
              strcpy(time_in_txt, scratchpad);
              onset = 1;
            }
            n = 0;
            duration = 1;
            duration_in_txt[0] = 0;
            scratchpad[0] = 0;
            continue;
          }

          if(++n>max_tal_ln)
          {
            QApplication::restoreOverrideCursor();
            snprintf(str, 256, "Error, TAL in record %i exceeds my buffer.", datarecordswritten + 1);
            UI_Messagewindow popuperror("Error", str);
            fclose(annotationfile);
            fclose(outputfile);
            free(edf_hdr);
            free(edfparamascii);
            free(cnv_buf);
            free(time_in_txt);
            free(duration_in_txt);
            free(scratchpad);
            AsciiExportDialog->close();
            return;
          }
        }
      }
    }
    else elapsedtime = datarecordswritten * data_record_duration;

/* done with timekeeping and annotations, continue with the data */

    do
    {
      time_tmp = 10000000000.0;
      for(j=0; j<edfsignals; j++)
      {
        if((edfplus)||(bdfplus))
        {
          skip = 0;

          for(p=0; p<nr_annot_chns; p++)
          {
            if(j==annot_ch[p])
            {
              skip = 1;
              break;
            }
          }

          if(skip) continue;
        }

        d_tmp = edfparamascii[j].smp_written * edfparamascii[j].time_step;
        if(d_tmp<time_tmp) time_tmp = d_tmp;
      }
      fprintf(outputfile, "%f", elapsedtime + time_tmp);

      for(j=0; j<edfsignals; j++)
      {
        if((edfplus)||(bdfplus))
        {
          skip = 0;

          for(p=0; p<nr_annot_chns; p++)
          {
            if(j==annot_ch[p])
            {
              skip = 1;
              break;
            }
          }

          if(skip) continue;
        }

        d_tmp = edfparamascii[j].smp_written * edfparamascii[j].time_step;

        if((d_tmp<(time_tmp+0.00000000000001))&&(d_tmp>(time_tmp-0.00000000000001))&&(edfparamascii[j].smp_written<edfparamascii[j].smp_per_record))
        {
          if(bdf)
          {
            value_tmp = *((unsigned short *)(cnv_buf + ((edfparamascii[j].buf_offset + edfparamascii[j].smp_written) * samplesize)));
            value_tmp += (*(cnv_buf + ((edfparamascii[j].buf_offset + edfparamascii[j].smp_written) * samplesize) + 2) * 65536);
            value_tmp += edfparamascii[j].offset;
            value_tmp *= edfparamascii[j].sense;
          }
          else
          {
            value_tmp = ((*(((signed short *)cnv_buf)+edfparamascii[j].buf_offset+edfparamascii[j].smp_written)) + edfparamascii[j].offset) * edfparamascii[j].sense;
          }
          fprintf(outputfile, ",%f", value_tmp);
          edfparamascii[j].smp_written++;
        }
        else fputc(',', outputfile);
      }

      if(fputc('\n', outputfile)==EOF)
      {
        QApplication::restoreOverrideCursor();
        UI_Messagewindow popuperror("Error", "Error when writing to outputfile during conversion.");
        fclose(annotationfile);
        fclose(outputfile);
        free(edf_hdr);
        free(edfparamascii);
        free(cnv_buf);
        free(time_in_txt);
        free(duration_in_txt);
        free(scratchpad);
        AsciiExportDialog->close();
        return;
      }

      recordfull = 1;
      for(j=0; j<edfsignals; j++)
      {
        if(edfparamascii[j].smp_written<edfparamascii[j].smp_per_record)
        {
          if((edfplus)||(bdfplus))
          {
            skip = 0;

            for(p=0; p<nr_annot_chns; p++)
            {
              if(j==annot_ch[p])
              {
                skip = 1;
                break;
              }
            }

            if(skip) continue;
          }

          recordfull = 0;
          break;
        }
      }
    }
    while(!recordfull);

    datarecordswritten++;
  }

  QApplication::restoreOverrideCursor();

  fclose(annotationfile);
  fclose(outputfile);
  free(edf_hdr);
  free(edfparamascii);
  free(cnv_buf);
  free(time_in_txt);
  free(duration_in_txt);
  free(scratchpad);

  AsciiExportDialog->close();
}







