/* ------------------------------------------------------------------------
 * DynamicLibrary.hh
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-07-25 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _DYNAMICLIBRARY_HH_
#define _DYNAMICLIBRARY_HH_

// -- System Includes
#include <string>

// -- Class Declarations

/**
 * Simple dynamic link library (DLL) abstraction.
 **/
class DynamicLibrary {
    
public:

    /**
     * Default constructor.
     **/
    DynamicLibrary() : _handle(0) { }
    

    /**
     * Constructor.
     *
     * @param library filename (must be in path or in local dir).
     * @param now resolve symbols immediately if true, else on-demand.
     **/
    DynamicLibrary(const std::string &library, bool now = true) {
	open(library, now);
    }

    
    /**
     * Destructor.
     **/
    ~DynamicLibrary() { close(); }

    
    /**
     * Open a dynamic link library (DLL) given a filename. 
     * 
     * @param library filename (must be in path or in local dir).
     * @param now resolve symbols immediately if true, else on-demand.
     **/
    void open(const std::string &library, bool now = true);
    
    
    /**
     * Close the library.
     **/
    void close();

    
    /**
     * Retrieve the library name.
     *
     * @return the library name as a string.
     **/
    const std::string &getName() { return _name; }

    
    /**
     * Resolve a symbol in the library and return it.
     * 
     * @param symbol the symbol to resolve.
     * @return a pointer to the entity. 
     **/
    void *resolve(const std::string &symbol);

    
    /**
     * Is the library open and ready to be used?
     * 
     * @return true if it is, false otherwise.
     **/
    bool isOpen() { return _handle; }
    
protected:
    
    /// Library handle
    void *_handle;

    /// Library name
    std::string _name;

};

#endif /* DynamicLibrary.hh */
