/*
	**
	** packet-parse.c
	**
	** Routines to extract data from network packets
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: packet-parse.c,v 1.3 1999/02/11 02:45:09 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <unistd.h>
#include <net/bpf.h>

#include <glib.h>

#include "packet-headers.h"
#include "packet-parse.h"

#define ERRORBUF_LEN 512

static char rcsid[]="$Id: packet-parse.c,v 1.3 1999/02/11 02:45:09 dmiller Exp $";
static char error_buf[ERRORBUF_LEN];

/* Routines to parse a packet, filling structure pinfo */
/* Returns 1 on success, 0 on error */
int parse_eth(const u_int8_t *eth_pkt, unsigned int len, packet_info_t *pinfo);
int parse_fddi(const u_int8_t *fddi_pkt, unsigned int len, packet_info_t *pinfo);
int parse_ppp(const u_int8_t *ppp_pkt, unsigned int len, packet_info_t *pinfo);
int parse_null(const u_int8_t *null_pkt, unsigned int len, packet_info_t *pinfo);
int parse_ipv4(const u_int8_t *ipv4_pkt, unsigned int len, packet_info_t *pinfo);
int parse_udp(const u_int8_t *udp_pkt, unsigned int len, packet_info_t *pinfo);
int parse_tcp(const u_int8_t *tcp_pkt, unsigned int len, packet_info_t *pinfo);

int parse_packet(int link_type, const u_int8_t *pkt, 
                 unsigned int len, packet_info_t *pinfo)
{
	/* Clear error message buffer */
	error_buf[0] = '\0';

	/* Clear packet information */
	memset(pinfo, 0, sizeof(*pinfo));

	pinfo->datalink_type = link_type;
	
	switch(link_type)
	{
		case DLT_EN10MB:
			/* Ethernet 10Mbps */
			return(parse_eth(pkt, len, pinfo));
			break;
			
		case DLT_FDDI:
			/* FDDI Networks */
			return(parse_fddi(pkt, len, pinfo));
			break;
			
		case DLT_PPP:
			/* PPP */
			return(parse_ppp(pkt, len, pinfo));
			break;
			
		case DLT_RAW:
			/* Raw IPv4 datagrams */
			return(parse_ipv4(pkt, len, pinfo));
			break;
			
		case DLT_NULL:
			/* Loopback interfaces */
			return(parse_null(pkt, len, pinfo));
			break;

		default:
			g_snprintf(error_buf, ERRORBUF_LEN, 
			         "Unsupported link type: %04x", link_type);
			return(0);
			break;
	}
}

const char *parse_error(void)
{
	return(error_buf);
}

int parse_eth(const u_int8_t *eth_pkt, unsigned int len, packet_info_t *pinfo)
{
	struct eth_mac_hdr *eth_hdr = (struct eth_mac_hdr *)eth_pkt;

	/* Check for short packets */
	if (len <= sizeof(*eth_hdr))
	{
		g_snprintf(error_buf, ERRORBUF_LEN, "Ethernet packet is too short.");
		return(0);
	}

	/* Copy hardware addresses */
	memcpy(pinfo->src_hw_addr, eth_hdr->src_addr, sizeof(eth_hdr->src_addr));
	memcpy(pinfo->dst_hw_addr, eth_hdr->dst_addr, sizeof(eth_hdr->dst_addr));

	/* Determine frame type and dispatch packet to appropriate handler */
	switch (ntohs(eth_hdr->type))
	{
		case 0x0800:
			/* IPv4 */
			return(parse_ipv4(eth_pkt + sizeof(*eth_hdr), len - sizeof(*eth_hdr), pinfo));
			break;
		default:
			g_snprintf(error_buf, ERRORBUF_LEN, 
			         "Unsupported ethernet frame type: %04x", ntohs(eth_hdr->type));
			return(0);
			break;
	}
}

int parse_fddi(const u_int8_t *fddi_pkt, unsigned int len, packet_info_t *pinfo)
{
	struct fddi_hdr *fdh = (struct fddi_hdr *)fddi_pkt;

	/* Check for short packets */
	if (len <= sizeof(*fdh))
	{
		g_snprintf(error_buf, ERRORBUF_LEN, "FDDI packet is too short.");
		return(0);
	}

	/* Copy hardware addresses */
	memcpy(pinfo->src_hw_addr, fdh->src_addr, sizeof(fdh->src_addr));
	memcpy(pinfo->dst_hw_addr, fdh->dst_addr, sizeof(fdh->dst_addr));

	/* Ensure that this is an Asynch LLC FDDI packet */
	if ((fdh->frame_control & FDDIFC_CLFF) == FDDIFC_LLC_ASYNC)
	{
		/* Determine frame type and dispatch packet to appropriate handler */
		switch (ntohs(fdh->type))
		{
			case 0x0800:
				/* IPv4 */
				return(parse_ipv4(fddi_pkt + sizeof(*fdh), len - sizeof(*fdh), pinfo));
				break;
			default:
				g_snprintf(error_buf, ERRORBUF_LEN, 
			         	"Unsupported FDDI frame type: %04x", ntohs(fdh->type));
				return(0);
				break;
		}
	} else
	{
		g_snprintf(error_buf, ERRORBUF_LEN, 
		         "Unsupported FDDI frame control: %hu", fdh->frame_control);
		return(0);
	}
}

int parse_ppp(const u_int8_t *ppp_pkt, unsigned int len, packet_info_t *pinfo)
{
	int protocol;

	/* Check for short packets */
	if (len <= 5)
	{
		g_snprintf(error_buf, ERRORBUF_LEN, "PPP packet is too short.");
		return(0);
	}

	/* PPP does not have hardware addresses */
	memset(pinfo->src_hw_addr, 0, sizeof(pinfo->src_hw_addr));
	memset(pinfo->dst_hw_addr, 0, sizeof(pinfo->dst_hw_addr));

	/* Determine protocol and dispatch packet to appropriate handler */
	protocol = ntohs((int)*(ppp_pkt + 3));
	switch (protocol)
	{
		case 0x21:
			/* IPv4 */
			return(parse_ipv4(ppp_pkt + 5, len - 5, pinfo));
			break;
		default:
			g_snprintf(error_buf, ERRORBUF_LEN, 
			         "Unsupported PPP protocol: %04x", protocol);
			return(0);
			break;
	}
}

int parse_null(const u_int8_t *null_pkt, unsigned int len, packet_info_t *pinfo)
{
	unsigned long family;

	/* Check for short packets */
	if (len <= 4)
	{
		g_snprintf(error_buf, ERRORBUF_LEN, "NULL packet is too short.");
		return(0);
	}

	/* Determine protocol and dispatch packet to appropriate handler */
	family = ntohl(*(unsigned long*)null_pkt);
	switch (family)
	{
		case 0x0800: /* FIXME: is this correct? */
			/* IPv4 */
			return(parse_ipv4(null_pkt + sizeof(family), len - sizeof(family), pinfo));
			break;

		default:
			g_snprintf(error_buf, ERRORBUF_LEN, 
			         "Unsupported NULL address family: %04lx", family);
			return(0);
			break;
	}
}

int parse_ipv4(const u_int8_t *ipv4_pkt, unsigned int len, packet_info_t *pinfo)
{
	struct ip_hdr *ipv4h = (struct ip_hdr *)ipv4_pkt;
	int ipv4_hdr_length;
	
	/* Check for short packets */
	if (len < sizeof(*ipv4h))
	{
		g_snprintf(error_buf, ERRORBUF_LEN, "IPv4 packet is too short.");
		return(0);
	}

	if (ipv4h->version != 4)
	{
		g_snprintf(error_buf, ERRORBUF_LEN, 
		         "Unsupported IP version: 0x%02x", ipv4h->version);
		return(0);
	}

	ipv4_hdr_length = ipv4h->ihl * 4;
	pinfo->header_length += ipv4_hdr_length;
	pinfo->is_fragment = ((ntohs(ipv4h->frag_off) & 0x1FFF) != 0);
	pinfo->total_length = ntohs(ipv4h->tot_len);
	pinfo->src_ip_addr = ntohl(ipv4h->saddr);
	pinfo->dst_ip_addr = ntohl(ipv4h->daddr);
	pinfo->ip_protocol = ipv4h->protocol;
	
	if (!pinfo->is_fragment)
	{
		switch (ipv4h->protocol)
		{
			case IPPROTO_ICMP:
				pinfo->src_port = 0;
				pinfo->dst_port = 0;
				pinfo->is_connection_request = 0;
				return(1);
				break;

			case IPPROTO_TCP:
				return(parse_tcp(ipv4_pkt + ipv4_hdr_length, len - ipv4_hdr_length, pinfo));
				break;

			case IPPROTO_UDP:
				return(parse_udp(ipv4_pkt + ipv4_hdr_length, len - ipv4_hdr_length, pinfo));
				break;

			default:
				g_snprintf(error_buf, ERRORBUF_LEN, 
				         "Unsupported IPv4 protocol: %04x", ipv4h->protocol);
				return(0);
				break;
		}
	} else
	{
		pinfo->src_port = 0;
		pinfo->dst_port = 0;
		pinfo->is_connection_request = 0;
		return(1);
	}
}

int parse_udp(const u_int8_t *udp_pkt, unsigned int len, packet_info_t *pinfo)
{
	struct udp_hdr *udph = (struct udp_hdr *)udp_pkt;
		
	/* Check for short packets */
	if (len < sizeof(*udph))
	{
		g_snprintf(error_buf, ERRORBUF_LEN, "UDP packet is too short.");
		return(0);
	}

	pinfo->src_port = ntohs(udph->source);
	pinfo->dst_port = ntohs(udph->dest);
	pinfo->is_connection_request = 0;
	pinfo->header_length += ntohs(udph->len);
	
	return(1);
}

int parse_tcp(const u_int8_t *tcp_pkt, unsigned int len, packet_info_t *pinfo)
{
	struct tcp_hdr *tcph = (struct tcp_hdr *)tcp_pkt;
	
	/* Check for short packets */
	if (len < sizeof(*tcph))
	{
		g_snprintf(error_buf, ERRORBUF_LEN, "TCP packet is too short.");
		return(0);
	}

	pinfo->src_port = ntohs(tcph->source);
	pinfo->dst_port = ntohs(tcph->dest);
	pinfo->is_connection_request = tcph->syn && !tcph->ack;
	pinfo->header_length += (tcph->doff * 4);
	
	return(1);
}

