-- `Topal': GPG/Pine integration
--
-- Copyright (C) 2001,2002  Phillip J. Brooke
--
--     This program is free software; you can redistribute it and/or modify
--     it under the terms of the GNU General Public License as published by
--     the Free Software Foundation; either version 2 of the License, or
--     (at your option) any later version.
--
--     This program is distributed in the hope that it will be useful,
--     but WITHOUT ANY WARRANTY; without even the implied warranty of
--     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--     GNU General Public License for more details.
--
--     You should have received a copy of the GNU General Public License
--     along with this program; if not, write to the Free Software
--     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

with Ada.Strings.Fixed;
with Ada.Strings.Maps.Constants;
with Ada.Text_IO;
with Globals;                    use Globals;
with Misc;                       use Misc;

package body Externals.Mail is

   -- We need to figure out what the boundary is.  We can't trust the
   -- Pine and the shell to get it for us, so we'll use this heuristic.
   -- Open Infile, read it in a line at a time.  Save the previous line.
   -- As soon as one line starts `Content-Type: ' and the previous line
   -- starts `--', we'll assume that the previous line was a MIME
   -- boundary, and slice off characters 3-end.
   function Find_Mime_Boundary (Infile : String) return UBS is
      BF : Ada.Text_IO.File_Type;
      L1 : UBS;
      L2 : UBS;
   begin
      Ada.Text_IO.Open(File => BF,
                       Mode => Ada.Text_IO.In_File,
                       Name => Infile);
      L1 := ToUBS("");
  Boundary_Loop:
      loop
         L2 := Unbounded_Get_Line(BF);
         declare
            L1S : String := ToStr(L1);
            L2S : String := Ada.Strings.Fixed.Translate(ToStr(L2),
                                                        Ada.Strings.Maps.Constants.Lower_Case_Map);
         begin
            exit Boundary_Loop when (L1S'Length >= 2 and L2S'Length >= 14)
              and then (L1S(L1S'First..L1S'First+1) = "--"
                        and L2S(L2S'First..L2S'First+13) = "content-type: ");
         end;
         L1 := L2;
      end loop Boundary_Loop;
      Ada.Text_IO.Close(BF);
      -- If we're here, then L1 is the boundary.
      declare
         L1S : String := ToStr(L1);
      begin
         return ToUBS(L1S(3..L1S'Last));
      end;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Find_Mime_Boundary");
         raise;
   end Find_Mime_Boundary;

   -- I should really generalise this to n part MIME....
   procedure Split_Two_Parts (Infile   : in String;
                              Part_One : in String;
                              Part_Two : in String;
                              Boundary : in UBS) is
      Infile_F   : Ada.Text_IO.File_Type;
      Part_One_F : Character_IO.File_Type;
      Part_Two_F : Character_IO.File_Type;
      L1         : UBS;
      L2         : UBS;
      BS         : String := ToStr(Boundary);
   begin
      -- Open files....
      begin
         Ada.Text_IO.Open(File => Infile_F,
                          Mode => Ada.Text_IO.In_File,
                          Name => Infile);
         Character_IO.Create(File => Part_One_F,
                            Mode => Character_IO.Out_File,
                            Name => Part_One);
         Character_IO.Create(File => Part_Two_F,
                            Mode => Character_IO.Out_File,
                            Name => Part_Two);
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Externals.Mail.Split_Two_Parts (Opening files)");
            raise;
      end;
      -- Get the two parts.  The first part, we want to include the MIME
      -- headers, but not the boundaries nor the blank line before the next
      -- part.
      -- So, first, we walk through infile looking for the boundary.
      begin
     Find_First_Boundary:
         loop
            L2 := Unbounded_Get_Line(Infile_F);
            declare
               L2S : String := ToStr(L2);
            begin
               exit Find_First_Boundary when L2S = "--" & BS;
            end;
         end loop Find_First_Boundary;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Externals.Mail.Split_Two_Parts (Find_First_Boundary)");
            raise;
      end;
      -- Now, walk through infile, copying complete lines to Part_One
      -- until we find a blank line - boundary pair.
      begin
         L1 := Unbounded_Get_Line(Infile_F);
     Find_Second_Boundary:
         loop
            L2 := Unbounded_Get_Line(Infile_F);
            declare
               L1S : String := ToStr(L1);
               L2S : String := ToStr(L2);
            begin
               exit Find_Second_Boundary when (L1S'Length = 0
                                               and L2S = "--" & BS);
               -- Trap case where data ends without newline.
               if (L2S = "--" & BS) then
                  Character_IO_Put(Part_One_F, L1S);
                  exit Find_Second_Boundary;
               end if;
               Character_IO_Put_Line(Part_One_F, L1S);
            end;
            L1 := L2;
         end loop Find_Second_Boundary;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Externals.Mail.Split_Two_Parts (Find_Second_Boundary)");
            raise;
      end;
      -- Continue through infile, copying complete lines to Part_Two,
      -- until we find another blank_line - boundary pair.
      begin
         L1 := Unbounded_Get_Line(Infile_F);
     Find_Third_Boundary:
         loop
            L2 := Unbounded_Get_Line(Infile_F);
            declare
               L1S : String := ToStr(L1);
               L2S : String := ToStr(L2);
            begin
               exit Find_Third_Boundary when (L1S'Length = 0
                                              and L2S = "--" & BS & "--");
               -- Trap case where data ends without newline.
               if (L2S = "--" & BS & "--") then
                  Character_IO_Put(Part_Two_F, L1S);
                  exit Find_Third_Boundary;
               end if;
               Character_IO_Put_Line(Part_Two_F, L1S);
            end;
            L1 := L2;
         end loop Find_Third_Boundary;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Externals.Mail.Split_Two_Parts (Find_Third_Boundary)");
            raise;
      end;
      -- Close files.
      begin
         Ada.Text_IO.Close(Infile_F);
         Character_IO.Close(Part_One_F);
         Character_IO.Close(Part_Two_F);
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Externals.Mail.Split_Two_Parts (Closing files)");
            raise;
      end;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Split_Two_Parts");
         raise;
   end Split_Two_Parts;

   procedure Mimeconstruct2 (Part1_Filename  : in String;
                             Part2_Filename  : in String;
                             Output_Filename : in String;
                             Content_Type    : in String) is
   begin
      if ForkExec_Out(Value_Nonempty(Config.Mimeconstruct_Binary),
                      UBS_Array'(0 => ToUBS("mime-construct"),
                                 1 => ToUBS("--output"),
                                 2 => ToUBS("--multipart"),
                                 3 => ToUBS(Content_Type),
                                 4 => ToUBS("--subpart-file"),
                                 5 => ToUBS(Part1_Filename),
                                 6 => ToUBS("--subpart-file"),
                                 7 => ToUBS(Part2_Filename)),
                      Output_Filename) /= 0 then
         Error("Mimeconstruct failed. (ff1)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Mimeconstruct2");
         raise;
   end Mimeconstruct2;

   -- Construct a top-level mulitpart/alternative MIME email.
   procedure Mimeconstruct_Alt (Part1_Filename  : in String;
                                Part2_Filename  : in String;
                                Output_Filename : in String;
                                Header          : in String) is
   begin
      if ForkExec_Out(Value_Nonempty(Config.Mimeconstruct_Binary),
                      UBS_Array'(0 => ToUBS("mime-construct"),
                                 1 => ToUBS("--output"),
                                 2 => ToUBS("--header"),
                                 3 => ToUBS(Header),
                                 4 => ToUBS("--multipart"),
                                 5 => ToUBS("multipart/alternative"),
                                 6 => ToUBS("--subpart-file"),
                                 7 => ToUBS(Part1_Filename),
                                 8 => ToUBS("--subpart-file"),
                                 9 => ToUBS(Part2_Filename)),
                      Output_Filename) /= 0 then
         Error("Mimeconstruct failed. (ff12)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Mimeconstruct_Alt");
         raise;
   end Mimeconstruct_Alt;

   procedure Mimeconstruct_Subpart (Infile       : in String;
                                    Outfile      : in String;
                                    Content_Type : in String;
                                    Dos2UnixU    : in Boolean;
                                    Use_Header   : in Boolean;
                                    Use_Encoding : in Boolean;
                                    Header       : in String := "";
                                    Encoding     : in String := "") is
      A      : UBS_Array(0..9);
      N      : Integer;
      E1, E2 : Integer;
   begin
      A(0) := ToUBS("mime-construct");
      A(1) := ToUBS("--subpart");
      N := 2;
      if Use_Header then
         A(N) := ToUBS("--header");
         A(N+1) := ToUBS(Header);
         N := N + 2;
      end if;
      A(N) := ToUBS("--type");
      A(N+1) := ToUBS(Content_Type);
      N := N + 2;
      if Use_Encoding then
         A(N) := ToUBS("--encoding");
         A(N+1) := ToUBS(Encoding);
         N := N + 2;
      end if;
      A(N) := ToUBS("--file");
      A(N+1) := ToUBS(Infile);
      declare
         B : UBS_Array(0..N+1);
      begin
         B := A(0..N+1);
         if Dos2UnixU then
            ForkExec2_Out(Value_Nonempty(Config.Mimeconstruct_Binary),
                          B,
                          E1,
                          Value_Nonempty(Config.Dos2Unix_Binary),
                          UBS_Array'(0 => ToUBS("dos2unix"),
                                     1 => ToUBS("-u")),
                          E2,
                          Target => Outfile);
            if E1 /= 0 then
               Error("Problem with mime-construct! (ff8a)");
            elsif E2 /= 0 then
               Error("Problem with dos2unix! (ff8b)");
            end if;
         else
            E1 := ForkExec_Out(Value_Nonempty(Config.Mimeconstruct_Binary),
                               B,
                               Target => Outfile);
            if E1 /= 0 then
               Error("Problem with mime-construct! (ff8a)");
            end if;
         end if;
      end;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Mimeconstruct_Subpart");
         raise;
   end Mimeconstruct_Subpart;

   -- Construct an entire multipart/mixed email.
   procedure Mimeconstruct_Mixed (Filenames : in UBS_Array;
                                  Outfile   : in String) is
      -- Arg array size is ( 2*num-filenames ) +2
      A : UBS_Array(0 .. (2 * Filenames'Length+1));
   begin
      A(0) := ToUBS("mime-construct");
      A(1) := ToUBS("--output");
      for I in 1..Filenames'Length loop
         A((2*I)) := ToUBS("--file");
         A(2*I+1) := Filenames(I);
      end loop;
      if ForkExec_Out(Value_Nonempty(Config.Mimeconstruct_Binary),
                      A,
                      Outfile) /= 0 then
         Error("Mimeconstruct failed. (ff14)");
      end if;
   end Mimeconstruct_Mixed;

   procedure Extract_Content_Type_From_Header (Email_Filename  : in String;
                                               Target_Filename : in String) is
      E1, E2 : Integer;
   begin
      ForkExec2_InOut(Value_Nonempty(Config.Sed_Binary),
                      UBS_Array'(0 => ToUBS("sed"),
                                 1 => ToUBS("1,/^[:space:]*$/ ! d")),
                      E1,
                      Value_Nonempty(Config.Grep_Binary),
                      UBS_Array'(0 => ToUBS("grep"),
                                 1 => ToUBS("Content-Type")),
                      E2,
                      Source => Email_Filename,
                      Target => Target_Filename);
      if E1 /= 0 then
         Error("Problem with sed! (ff2a)");
      elsif E2 /= 0 then
         Error("Problem with grep! (ff2b)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Extract_Content_Type_From_Header");
         raise;
   end Extract_Content_Type_From_Header;

   -- Get the header from a header-body mail into a file.
   procedure Extract_Header (Email_Filename  : in String;
                             Target_Filename : in String) is
   begin
      if ForkExec_InOut(Value_Nonempty(Config.Sed_Binary),
                        UBS_Array'(0 => ToUBS("sed"),
                                   1 => ToUBS("-e"),
                                   2 => ToUBS("1,/^[:space:]*$/ ! d"),
                                   3 => ToUBS("-e"),
                                   4 => ToUBS("/^[:space:]*$/ d")),
                        Source => Email_Filename,
                        Target => Target_Filename) /= 0 then
         Error("sed failed! (ff11)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Extract_Header");
         raise;
   end Extract_Header;

   -- Get the body from a header-body mail into a file.
   procedure Extract_Body (Email_Filename  : in String;
                           Target_Filename : in String) is
      E1, E2 : Integer;
   begin
      ForkExec2_InOut(Value_Nonempty(Config.Sed_Binary),
                      UBS_Array'(0 => ToUBS("sed"),
                                 1 => ToUBS("-e"),
                                 2 => ToUBS("/^[:space:]*$/,$ ! d")),
                      E1,
                      Value_Nonempty(Config.Sed_Binary),
                      UBS_Array'(0 => ToUBS("sed"),
                                 1 => ToUBS("-e"),
                                 2 => ToUBS("1,1 { /^[:space:]*$/ d ; }")),
                      E2,
                      Source => Email_Filename,
                      Target => Target_Filename);
      if E1 /= 0 then
         Error("sed failed! (ff3)");
      elsif E2 /= 0 then
         Error("sed failed! (ff13)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Extract_Body");
         raise;
   end Extract_Body;

   procedure Delete_Trailing_Blank_Lines (Infile  : in String;
                                          Outfile : in String) is
   begin
      if ForkExec_InOut(Value_Nonempty(Config.Sed_Binary),
                        UBS_Array'(0 => ToUBS("sed"),
                                   1 => ToUBS("-e"),
                                   2 => ToUBS(":a"),
                                   3 => ToUBS("-e"),
                                   4 => ToUBS("/^\n*$/{$d;N;};/\n$/ba")),
                        Source => Infile,
                        Target => Outfile) /= 0 then
         Error("sed failed! (ff5)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Delete_Trailing_Blank_Lines");
         raise;
   end Delete_Trailing_Blank_Lines;

   procedure Formail_Concat_Extract_InOut (Header : in String;
                                           Source : in String;
                                           Target : in String) is
   begin
      if ForkExec_InOut(Value_Nonempty(Config.Formail_Binary),
                        UBS_Array'(0 => ToUBS("formail"),
                                   1 => ToUBS("-c"),
                                   2 => ToUBS("-x"),
                                   3 => ToUBS(Header)),
                        Source => Source,
                        Target => Target) /= 0 then
         Error("formail failed! (ff9)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Formail_Concat_Extract_InOut");
         raise;
   end Formail_Concat_Extract_InOut;

   procedure Formail_Drop_InOut (Header : in String;
                                  Source : in String;
                                  Target : in String) is
   begin
      if ForkExec_InOut(Value_Nonempty(Config.Formail_Binary),
                        UBS_Array'(0 => ToUBS("formail"),
                                   1 => ToUBS("-I"),
                                   2 => ToUBS(Header)),
                        Source => Source,
                        Target => Target) /= 0 then
         Error("formail failed! (ff10)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Formail_Drop_InOut");
         raise;
   end Formail_Drop_InOut;

   procedure Formail_Action_InOut (Source : in String;
                                   Target : in String;
                                   Action : in String) is
      E : Integer;
   begin
      E := ForkExec_InOut(Value_Nonempty(Config.Formail_Binary),
                          ToUBS("formail -s " & Action),
                          Source => Source,
                          Target => Target);
      -- Ignore the return code of formail.  It seems to return 1 a
      -- lot, and I can't find any documentation telling me what it
      -- _should_ return.
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Formail_Action_InOut");
         raise;
   end Formail_Action_InOut;

end Externals.Mail;
