/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <stdio.h>
#include "stars.h"
#include "glade/glade.h"
#include "coordutils.h"
#include "objectwindow.h"
#include "catinfo.h"

GList *selectedobjects = 0;
struct viewer *recentviewer = 0;
struct catalog *highlightcat = 0;

struct selectedobject
{
	struct catalog *cat;
	int objnum;
	int highlighted;
	char *lstr[10], *rstr[10];
	double ra, dec;
	long x, y, z;
};

struct objectwindow
{
	GtkWidget *objectwin;
	GtkWidget *distlabel;
	GtkWidget *leftlabel[8];
	GtkWidget *rightlabel[8];
	GtkWidget *objlist;
	GtkWidget *ralabel, *declabel, *objtypelabel, *highlighttoggle;
	int isvisible;
	struct selectedobject *curselobj, *lastselobj;
} *objwin = 0;

/* Protos */

void addtohighlights(struct selectedobject *so);
void removefromhighlights(struct selectedobject *so);

/*  Callbacks  */

void dismiss_click(GtkWidget *widget, struct objectwindow *objwin)
{
	hideobjectwindow();
}

void objrowselected(GtkWidget *widget, gint row, gint column,
	GdkEventButton *event, struct objectwindow *objwin)
{
	struct selectedobject *so;
	so = (struct selectedobject *)(g_list_nth(selectedobjects, row)->data);
	if(objwin->curselobj != so) objwin->lastselobj = objwin->curselobj;
	objwin->curselobj = so;
	updateobjectwindow();
}

void centeron_click(GtkWidget *widget, struct objectwindow *objwin)
{
	if(!isvalidviewer(recentviewer)) 
	{
		printf("Invalid window\n");
		return;	/* FIXME == Error window */
	}
	if(objwin == 0) return;
	if(objwin->curselobj == 0) return;
	setcenter(recentviewer, objwin->curselobj->ra, objwin->curselobj->dec);
	paint(recentviewer);
}

void highlight_click(GtkWidget *widget, struct objectwindow *objwin)
{
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)))
		addtohighlights(objwin->curselobj);
	else removefromhighlights(objwin->curselobj);
}

void clearhigh_click(GtkWidget *widget, struct objectwindow *objwin)
{
	removeallhighlights();
}

/*  Private Funcs  */

void inithighlightcat()
{
	int i;

	if(highlightcat) return;
	highlightcat = newcatalog();
	highlightcat->objects = g_new(struct object, MAXNUMSELECTEDOBJECTS);
	for(i = 0; i < MAXNUMSELECTEDOBJECTS; i++)
	{
		highlightcat->objects[i].mask = 0xF0;
		highlightcat->objects[i].type = 230;
		highlightcat->objects[i].icon = 0;
	}
	highlightcat->numobjects = MAXNUMSELECTEDOBJECTS;
	highlightcat->name = g_strdup("Highlights");
	strcpy(highlightcat->type, "Label");
	highlightcat->description = g_strdup("User selected highlights");
	addcatalog(highlightcat);
}

void addtohighlights(struct selectedobject *so)
{
	int i;
	
	if(!highlightcat) inithighlightcat();
	highlightcat->catmask = 0x0F;
	if(so->highlighted >= 0) return;
	for(i = 0; i < MAXNUMSELECTEDOBJECTS; i++) 
		if(highlightcat->objects[i].mask == 0xF0) break;
	if(i == MAXNUMSELECTEDOBJECTS)
	{
		printf("Weird -- no highlight room!\n");
		return;
	}
	so->highlighted = i;
	highlightcat->objects[so->highlighted].mask = 0x0F;	// Change to viewer
	highlightcat->objects[so->highlighted].x = so->x;
	highlightcat->objects[so->highlighted].y = so->y;
	highlightcat->objects[so->highlighted].z = so->z;

	catalog_render_pixmap(highlightcat, icons, recentviewer);
	refreshimage(recentviewer);
}

void removefromhighlights(struct selectedobject *so)
{
	if(!so || !highlightcat) return;
	if(so->highlighted < 0) return;
	if(so->highlighted >= MAXNUMSELECTEDOBJECTS)
	{
		printf("BOGUS == so->highlighted >= MAXNUMSELECTEDOBJECTS\n");
		return;
	}
	highlightcat->objects[so->highlighted].mask = 0xF0;
	so->highlighted = -1;

	paintall();
}

void removeallhighlights()
{
	int i;
	GList *lis;
	
	if(!highlightcat) return;
	for(i = 0; i < MAXNUMSELECTEDOBJECTS; i++)
		highlightcat->objects[i].mask = 0xF0;
	
	for(lis = selectedobjects; lis != 0; lis = lis->next)
		((struct selectedobject *)(lis->data))->highlighted = -1;
	paintall();
}

void freeselectedobject(struct selectedobject *so)
{
	int i;

	if(!so)
	{
		printf("Trying to nuke empty selected object\n");
		return;
	}
	for(i = 0; i < 10; i++) 
	{
		if(so->lstr[i]) g_free(so->lstr[i]);
		if(so->rstr[i]) g_free(so->rstr[i]);
	}
	g_free(so);
}
		
struct objectwindow *createobjectwindow()
{
	GtkWidget *clearhighbut, *centeronbut, *dismissbut;
	static GladeXML *objectxml = 0;
	struct objectwindow *objwin;
	char key[3] = "L0";
	int i;
	
	if(objectxml == 0) objectxml = glade_xml_new(PREFIXDIR "/share/stars/glade/object.glade", 0);
	if(objectxml == 0)
	{
		printf("Bad XML!\n");
		return 0;
	}
	
	objwin = g_new(struct objectwindow, 1);
	objwin->isvisible = 0;
	objwin->curselobj = 0;
	objwin->lastselobj = 0;
	objwin->objectwin = glade_xml_get_widget(objectxml, "objectwin");
	objwin->distlabel = glade_xml_get_widget(objectxml, "distlabel");
	objwin->objlist   = glade_xml_get_widget(objectxml, "objlist");
	objwin->ralabel   = glade_xml_get_widget(objectxml, "ralabel");
	objwin->declabel  = glade_xml_get_widget(objectxml, "declabel");
	objwin->objtypelabel = glade_xml_get_widget(objectxml, "objtypelabel");
	
	objwin->highlighttoggle = glade_xml_get_widget(objectxml, "highlight");
	clearhighbut    = glade_xml_get_widget(objectxml, "clearhigh");
	centeronbut     = glade_xml_get_widget(objectxml, "centeron");
	dismissbut      = glade_xml_get_widget(objectxml, "dismiss");
	
	key[0] = 'L';
	for(i = 0; i < 8; i++)
	{
		key[1] = '0' + i;
		objwin->leftlabel[i] = glade_xml_get_widget(objectxml, key);
	}

	key[0] = 'R';
	for(i = 0; i < 8; i++)
	{
		key[1] = '0' + i;
		objwin->rightlabel[i] = glade_xml_get_widget(objectxml, key);
	}

	gtk_signal_connect(GTK_OBJECT(objwin->objectwin), "delete_event",
		GTK_SIGNAL_FUNC(dismiss_click), objwin);
	gtk_signal_connect(GTK_OBJECT(dismissbut), "clicked",
		GTK_SIGNAL_FUNC(dismiss_click), objwin);
	gtk_signal_connect(GTK_OBJECT(objwin->objlist), "select_row",
		GTK_SIGNAL_FUNC(objrowselected), objwin);
	gtk_signal_connect(GTK_OBJECT(centeronbut), "clicked",
		GTK_SIGNAL_FUNC(centeron_click), objwin);
	gtk_signal_connect(GTK_OBJECT(clearhighbut), "clicked",
		GTK_SIGNAL_FUNC(clearhigh_click), objwin);
	gtk_signal_connect(GTK_OBJECT(objwin->highlighttoggle), "clicked",
		GTK_SIGNAL_FUNC(highlight_click), objwin);

	return objwin;
}

/*  Public Funcs  */

void showobjectwindow()
{
	if(objwin == 0)
	{
		objwin = createobjectwindow();
		if(objwin == 0) return;
	}
	if(objwin->isvisible) return;
	
	gtk_widget_show(objwin->objectwin);
	objwin->isvisible = 1;
}

void hideobjectwindow()
{
	if(objwin->isvisible) gtk_widget_hide(objwin->objectwin);
	objwin->isvisible = 0;
}

void updateobjectwindow()
{
	int i, num;
	struct selectedobject *so = objwin->curselobj;
	char tmp[100];
	double tmpra;

	if(so) 
	{
		if(so->highlighted >= 0)
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(
				objwin->highlighttoggle), TRUE);
		else gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(
				objwin->highlighttoggle), FALSE);
		for(i = 0; i < 8; i++)
		{
			gtk_label_set_text(GTK_LABEL(objwin->leftlabel[i]), so->lstr[i]);
			gtk_label_set_text(GTK_LABEL(objwin->rightlabel[i]), so->rstr[i]);
		}
		if(so->cat->ch)
			gtk_label_set_text(GTK_LABEL(objwin->objtypelabel), so->cat->ch->fullname);
		else gtk_label_set_text(GTK_LABEL(objwin->objtypelabel), so->cat->name);
		tmp[0] = 'R'; tmp[1] = 'A' ; tmp[2] = '=' ;
		tmpra = so->ra/PI_12;
		if(tmpra < 0.0) tmpra += 24.0;
		double2text(tmpra, tmp+3, ':');
		tmp[14] = 0;
		gtk_label_set_text(GTK_LABEL(objwin->ralabel), tmp);
		tmp[0] = 'D'; tmp[1] = 'e' ; tmp[2] = 'c' ; tmp[3] = '=' ; 
		double2text(so->dec/PI_180, tmp+4, ':');
		tmp[15] = 0;
		gtk_label_set_text(GTK_LABEL(objwin->declabel), tmp);
		if(objwin->lastselobj)
		{
			g_snprintf(tmp, 99, "Dist to %s: %5.3f degrees", 
			 	objwin->lastselobj->rstr[0], catdist(
					objwin->lastselobj->cat,
					objwin->lastselobj->objnum,
					so->cat,
					so->objnum)/PI_180);
			gtk_label_set_text(GTK_LABEL(objwin->distlabel), tmp);
		}
		else gtk_label_set_text(GTK_LABEL(objwin->distlabel), 0);
	}
	
	num = -1;
	i = 0;
	
	if(objwin->isvisible == 0) showobjectwindow();
}

void addselectedobject(struct catalog *cat, int objnum, struct viewer *v)
{
	FILE *in;
        char line[1000], tmp[100];
        int i = 0, j, k, num, hasname;
	struct catinfo *info;
	struct selectedobject *so = 0;
	struct selectedobject *obj;
	struct object *theobj;
	GList *lis = 0;
	gchar *fullfilename;

	if(objwin == 0)
	{
		objwin = createobjectwindow();
		if(objwin == 0) return;
	}

	recentviewer = v;
	info = cat->info;

	if(selectedobjects) for(lis = selectedobjects; lis != 0; lis=lis->next)
	{
		so = (struct selectedobject *)(lis->data);
		if(so->cat == cat && so->objnum == objnum) break;
	}

	if(lis) 
	{
		selectedobjects = g_list_remove_link(selectedobjects, lis);
		lis->data = 0;
		g_list_free(lis);
	}
	else
	{
		so = g_new(struct selectedobject, 1);

		so->cat = cat;
		so->objnum = objnum;
		so->highlighted = -1;
		theobj = cat->objects+objnum;
		so->x = theobj->x;
		so->y = theobj->y;
		so->z = theobj->z;
		getobjradec(theobj, &so->ra, &so->dec);
		for(i = 0; i < 10; i++) so->lstr[i] = so->rstr[i] = 0;
		if(objwin->curselobj != so) objwin->lastselobj = objwin->curselobj;
		objwin->curselobj = so;

		if(cat->info == 0)
		{
			so->lstr[0] = g_strdup("Name");
			so->rstr[0] = g_strdup_printf("%s %d", cat->name, objnum+1);
		}
		else
		{
			fullfilename = g_strconcat(PREFIXDIR, "/share/stars/data/", info->rawfile, 0);
			in = fopen(fullfilename, "r");
			if(!in)
			{
				fprintf(stderr, "Can't load raw data : %s\n", fullfilename);
				g_free(fullfilename);
				return;
			}
			g_free(fullfilename);

			if(objnum > 0)
			{
				if(info->reclen > 0) fseek(in, objnum*info->reclen, SEEK_SET);
				else for(i = 0; i < objnum; i++) fgets(line, 999, in);
			}
			fgets(line, 999, in);
			num = info->num;
			if(num > 10) num = 10;
			for(i = 0; i < num; i++)
			{
				k = 0;
				for(j = info->items[i].startcol; j<=info->items[i].stopcol; j++)
					tmp[k++] = line[j];
				tmp[k] = 0;
				so->lstr[i] = g_strdup(info->items[i].name);
				so->rstr[i] = g_strdup(tmp);
			}
			fclose(in);

			hasname = 0;
			for(i = 0; i < num; i++) if(strcmp(so->lstr[i], "Name") == 0) hasname = 1;
			if(!hasname)
			{
				for(i = num; i > 0; i--)
				{
					so->lstr[i] = so->lstr[i-1];
					so->rstr[i] = so->rstr[i-1];
				}
				so->lstr[0] = g_strdup("Name");
				so->rstr[0] = g_strdup_printf("%s %d", cat->name, objnum+1);
			}
		}
	}

	if(!so)
	{
		printf("BOGUS!  so = 0!\n");
	}
	
	selectedobjects = g_list_prepend(selectedobjects, so);

	while(g_list_length(selectedobjects) > MAXNUMSELECTEDOBJECTS)
	{
		lis = g_list_last(selectedobjects);
		removefromhighlights((struct selectedobject *)(lis->data));
		freeselectedobject((struct selectedobject *)(lis->data));
		lis->data = 0;
		selectedobjects = g_list_remove_link(selectedobjects, lis);
		g_list_free(lis);
	}

	gtk_clist_freeze(GTK_CLIST(objwin->objlist));
	gtk_clist_clear(GTK_CLIST(objwin->objlist));
	
	for(lis = selectedobjects; lis != 0; lis = lis->next)
	{
		obj = (struct selectedobject *)(lis->data);
		gtk_clist_append(GTK_CLIST(objwin->objlist), &obj->rstr[0]);
	}

	gtk_clist_thaw(GTK_CLIST(objwin->objlist));
	gtk_clist_select_row(GTK_CLIST(objwin->objlist), 0, 0);
	
	updateobjectwindow();
}

