/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.treedisplay;

/**
 * Represents a node in the layout of a tree. Objects of this class
 * maintain information about the physical location of a node, its
 * height, width, bounding height, bounding width, and so on.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */
public interface TreeLayoutNode {
	/**
	 * Get the TreeLayoutModel that this node belongs to.
	 *
	 * @return a <code>TreeLayoutModel</code> value
	 */
	public TreeLayoutModel getTreeLayoutModel();

	/**
	 * Set the TreeLayoutModel that this node belongs to.
	 *
	 * @param m a <code>TreeLayoutModel</code> value
	 */
	public void setTreeLayoutModel(TreeLayoutModel m);

	/**
	 * Get the value of width.
	 *
	 * @return value of width.
	 */
	public int getWidth();
	
	/**
	 * Set the value of width.
	 *
	 * @param v  Value to assign to width.
	 */
	public void setWidth(int v);
	
	/**
	 * Get the value of height.
	 *
	 * @return value of height.
	 */
	public int getHeight();
	
	/**
	 * Set the value of height.
	 *
	 * @param v  Value to assign to height.
	 */
	public void setHeight(int v);

	/**
	 * Get the bounding width.
	 *
	 * @return bounding width
	 */
	public int getBoundingWidth();

	/**
	 * Get the bounding height.
	 *
	 * @return an <code>int</code> value
	 */
	public int getBoundingHeight();

	/**
	 * Get the X coordinate of top left hand corner of bounding box.
	 *
	 * @return an <code>int</code> value
	 */
	public int getBoundingX();

	/**
	 * Get the Y coordinate of top left hand corner of bounding box.
	 *
	 * @return an <code>int</code> value
	 */
	public int getBoundingY();
	
	/**
	 * Does the node contain the point (x, y)?
	 *
	 * @param x an <code>int</code> value
	 * @param y an <code>int</code> value
	 * @return a <code>boolean</code> value
	 */
	public boolean contains(int x, int y);
	
	/**
	 * Get the value of x.
	 *
	 * @return value of x.
	 */
	public int getX();
	
	/**
	 * Set the value of x.
	 *
	 * @param v  Value to assign to x.
	 */
	public void setX(int v);
	
	/**
	 * Get the value of y.
	 *
	 * @return value of y.
	 */
	public int getY();
	
	/**
	 * Set the value of y.
	 *
	 * @param v  Value to assign to y.
	 */
	public void setY(int v);

	/**
	 * Get the value of model.
	 *
	 * @return value of model.
	 */
	public Object getData();
	
	/**
	 * Set the value of model.
	 *
	 * @param v  Value to assign to model.
	 */
	public void setData(Object v);
	
	/**
	 * Get the value of children.
	 *
	 * @return value of children.
	 */
	public TreeLayoutNode[] getChildren();
	
	/**
	 * Set the value of children.
	 *
	 * @param v  Value to assign to children.
	 */
	public void setChildren(TreeLayoutNode[] v);

	/**
	 * Add a child to the node.
	 *
	 * @param v a <code>TreeLayoutNode</code> value
	 */
	public void addChild(TreeLayoutNode v);

	/**
	 * Removes the child node specified by the given index.
	 * Indexes start at 0.
	 *
	 * @param n an <code>int</code> value
	 */
	public void removeChild(int n);

	/**
	 * Removes the given child node.
	 *
	 * @param child an <code>ExplainTreeNode</code> value
	 */
	public void removeChild(TreeLayoutNode child);

	/**
	 * Returns the number of children that this node has.
	 *
	 * @return an <code>int</code> value
	 */
	public int getChildCount();
	
	/**
	 * Returns the n'th child of this node. Indexes starts at 0.
	 *
	 * @param n an <code>int</code> value
	 * @return an <code>TreeLayoutNode</code> value or null if
	 * the given child doesn't exist.
	 */
	public TreeLayoutNode getChild(int n);

	/**
	 * For the specified child, return its index. Indexes begin at 0.
	 *
	 * @param child an <code>TreeLayoutNode</code> value
	 * @return an <code>int</code> value
	 */
	public int getIndexOfChild(TreeLayoutNode child);
	
	/**
	 * Gets the parent of the current node.
	 *
	 * @return an <code>TreeLayoutNode</code> value or null
	 * if the node has no parent.
	 */
	public TreeLayoutNode getParent();

	/**
	 * Set the parent of the current node.
	 *
	 * @param n an <code>TreeLayoutNode</code> value
	 *
	 * @exception IllegalArgumentException if the parent is the node itself.
	 */
	public void setParent(TreeLayoutNode n);
}
