/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.treedisplay;

import java.beans.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;

/**
 * Dialog that provides a birds-eye view of a given TreeModel using a
 * TreeDisplay object.
 * <p>
 * If a sister TreeDisplay has been set, a pane is drawn that represents
 * the viewable area of the sister. Panning in the overview dialog can
 * be done and any change in pane location is reflected in the sister
 * tree display view. Similarly, if the sister is scrolled, then the pane
 * in the overview dialog will be updated.
 * <i>Note: panning is not yet implemented.</i>
 *
 * @author Liam Stewart
 * @version 1.2.0
 */

public class TreeDisplayOverviewWindow extends JDialog implements PropertyChangeListener {
	private TreeDisplay display;
	private TreeDisplay sister;
	private TreeModel model;
	private Class tdclass;
	private JLayeredPane layeredpane;
	private MyJLabel box;

	/**
	 * Creates a new <code>TreeDisplayOverviewWindow</code> instance.
	 */
	public TreeDisplayOverviewWindow()
	{
		this(null, null, "Tree Overview");
	}
	
	/**
	 * Creates a new <code>TreeDisplayOverviewWindow</code> instance.
	 *
	 * @param tm a <code>TreeModel</code> value
	 */
	public TreeDisplayOverviewWindow(TreeModel tm, String title)
	{
		this(null, tm, title);
	}

	/**
	 * Creates a new <code>TreeDisplayOverviewWindow</code> instance.
	 *
	 * @param owner a <code>Component</code> value
	 */
	public TreeDisplayOverviewWindow(Component owner, String title)
	{
		this(owner, null, title);
	}
	
	/**
	 * Creates a new <code>TreeDisplayOverviewWindow</code> instance.
	 *
	 * @param owner a <code>Component</code> value
	 * @param tm a <code>TreeModel</code> value
	 */
	public TreeDisplayOverviewWindow(Component owner, TreeModel tm,  String title)
	{
		tdclass = TreeDisplay.class;
		model = tm;

		layeredpane = new JLayeredPane();
		layeredpane.setLayout(new SimpleLayout());
		getContentPane().add(layeredpane);
		box = new MyJLabel();

		update();

		setModal(false);
		setTitle(title);
		setLocationRelativeTo(owner);
		setSize(200, 200);
		setVisible(false);
	}

	/**
	 * Get the model that the dialog is currently visualizing.
	 *
	 * @return a <code>TreeModel</code> value
	 */
	public TreeModel getModel()
	{
		return display.getModel();
	}

	/**
	 * Get the sister tree display.
	 *
	 * @return a <code>TreeDisplay</code> value
	 */
	public TreeDisplay getSisterTreeDisplay()
	{
		return sister;
	}

	/**
	 * Set the sister tree display.
	 *
	 * @param s a <code>TreeDisplay</code> value
	 */
	public void setSisterTreeDisplay(TreeDisplay s)
	{
		if (sister == s)
			return;
		if (sister != null)
		{
			sister.removePropertyChangeListener(this);
			if (sister.getParent() instanceof JViewport)
			{
				layeredpane.remove(box);
				JViewport p = (JViewport) sister.getParent();
				p.removeChangeListener(box);
			}
		}

		sister = s;

		if (sister != null)
		{
			sister.addPropertyChangeListener(this);
			if (sister.getParent() instanceof JViewport)
			{
				layeredpane.add(box, new Integer(10));
				JViewport p = (JViewport) sister.getParent();
				p.addChangeListener(box);
			}
		}

		update();
	}

	/**
	 * Get the tree display component.
	 *
	 * @return a <code>TreeDisplay</code> value.
	 */
	public TreeDisplay getTreeDisplay()
	{
		return display;
	}

	public void setTreeDisplayClass(Class c)
	{
		boolean ok = false;
		Class cl = c;

		while (cl != null && ! ok)
		{
			if (cl.getName().equals(TreeDisplay.class.getName()))
				ok = true;
			else
				cl = cl.getSuperclass();
		}

		if (ok)
		{
			tdclass = c;
			update();
		}
	}

	public Class getTreeDisplayClass()
	{
		return tdclass;
	}

	public void propertyChange(PropertyChangeEvent evt)
	{
		if (evt.getSource() != sister)
			return;

		String prop = evt.getPropertyName();

		// we only care about orientation, model, layout model, and
		// renderer. zoom, mode, padding, anchoring, and everything
		// else is ignored.

		if (prop.equals(TreeDisplay.ORIENTATION_PROPERTY))
		{
			try {
				display.setOrientation(((Integer)evt.getNewValue()).intValue());
			} catch (TreeDisplayException ex) { }
		}
		if (prop.equals(TreeDisplay.TREE_MODEL_PROPERTY))
		{
			display.setModel((TreeModel)evt.getNewValue());
		}
		else if (prop.equals(TreeDisplay.TREE_RENDERER_PROPERTY))
		{
			display.setTreeRenderer((TreeRenderer)evt.getNewValue());
		}
		else if (prop.equals(TreeDisplay.TREE_LAYOUT_MODEL_PROPERTY))
		{
			display.setTreeLayoutModel((TreeLayoutModel)evt.getNewValue());
		}
		else if (prop.equals(TreeDisplay.SELECTION_MODEL_PROPERTY))
		{
			display.setSelectionModel((TreeSelectionModel)evt.getNewValue());
		}
	}

	//
	// private methods
	//

	private void update()
	{
		TreeDisplay old = display;

		try {
			display = (TreeDisplay) tdclass.newInstance();
		} catch (Exception ex) {
			display = old;
		}

		display.setZoomMode(TreeDisplay.FIT_WINDOW);
		display.setMode(TreeDisplay.PAN);
		display.setAnchor(TreeDisplay.CENTER);
		display.setAntiAlias(true);

		if (sister != null)
		{
			display.setTreeRenderer(sister.getTreeRenderer());
			display.setTreeLayoutModel(sister.getTreeLayoutModel());
			display.setModel(sister.getModel());
			display.setSelectionModel(sister.getSelectionModel());
		}
		else
		{
			display.setModel(model);
		}

		try {
			layeredpane.remove(old);
		} catch (Exception ex) { }

		layeredpane.add(display, new Integer(1));
	}

	private class MyJLabel extends JLabel implements ChangeListener, MouseListener
	{
		private Rectangle port, viewable;

		public MyJLabel()
		{
			setOpaque(false);
			addMouseListener(this);
		}

		public void stateChanged(ChangeEvent e)
		{
			// request a repaint
			repaint();
		}

		public void mouseClicked(MouseEvent e)
		{
			Point p = e.getPoint();

			if (viewable.contains(p))
			{
				// p should be center point of new viewport
				int x = p.x - viewable.x - (int) (port.width / 2);
				if (x < 0) x = 0;
				if (x + port.width > viewable.width) x = viewable.width - port.width;

				int y = p.y - viewable.y - (int) (port.height / 2);
				if (y < 0) y = 0;
				if (y + port.height > viewable.height) y = viewable.height - port.height;

				double r_x = (double) viewable.width / (double) sister.getWidth();
				double r_y = (double) viewable.height / (double) sister.getHeight();

				int nx = (int) (x / r_x);
				int ny = (int) (y / r_y);

				JViewport jvp = (JViewport) sister.getParent();
				jvp.setViewPosition(new Point(nx, ny));
			}
		}

		public void mouseEntered(MouseEvent e) { }
		public void mouseExited(MouseEvent e) { }
		public void mousePressed(MouseEvent e) { }
		public void mouseReleased(MouseEvent e) { }

		public void paintComponent(Graphics g)
		{
			super.paintComponent(g);

			if (sister == null)
				return;
			if (! (sister.getParent() instanceof JViewport))
				return;

			computeViewport();

			// shade irrelevant parts of window
			Graphics2D g2d = (Graphics2D) g;
			Area big   = new Area(new Rectangle(0, 0, getWidth(), getHeight()));
			Area small = new Area(viewable);
			big.exclusiveOr(small);
			g2d.setColor(Color.lightGray);
			g2d.fill(big);

			// draw the viewport
			g.setColor(Color.blue);
			g.drawRect(port.x, port.y, port.width, port.height);
		}

		private void computeViewport()
		{
			// define the rectangle that corresponds to sister
			int new_x = (int) (display.getTranslateX());
			int new_y = (int) (display.getTranslateY());
			int eff_w = getWidth() - 2 * (int) (display.getTranslateX());
			int eff_h = getHeight() - 2 * (int) (display.getTranslateY());

			viewable = new Rectangle(new_x, new_y, eff_w, eff_h);

			// viewport
			JViewport jvp = (JViewport) sister.getParent();
			Rectangle r = jvp.getViewRect();
			int w = (int)((r.getWidth() * eff_w) / sister.getWidth());
			int h = (int)((r.getHeight() * eff_h) / sister.getHeight());
			double r_x = (double) eff_w / (double) sister.getWidth();
			double r_y = (double) eff_h / (double) sister.getHeight();
			int x = (int) (r_x * r.x);
			int y = (int) (r_y * r.y);

			port = new Rectangle(new_x + x, new_y + y, w, h);
		}
	}

	private class SimpleLayout implements LayoutManager
	{
		public void addLayoutComponent(String name, Component comp) { }

		public void removeLayoutComponent(Component comp) { }

		public void layoutContainer(Container parent)
		{
			int i, h, w;
			h = parent.getHeight();
			w = parent.getWidth();
			for (i = 0; i < parent.getComponentCount(); i++)
			{
				Component c = parent.getComponent(i);
				c.setBounds(0, 0, w, h);
			}
		}

		public Dimension preferredLayoutSize(Container parent)
		{
			return new Dimension(parent.getWidth(), parent.getHeight());
		}

		public Dimension minimumLayoutSize(Container parent)
		{
			return new Dimension(parent.getWidth(), parent.getHeight());
		}
	}
}// TreeDisplayOverviewWindow
