// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc_v2/mrl_gc_v1.cpp,v 1.20 2002/01/11 15:47:38 weldon Exp $
//


#include "gc_for_orp.h"
#include "finalize.h"

#include "mrl_gc_v1.h"
#include "nursery_step_gen.h"
#include "train_generation.h"

#include "los.h"
#include "gc_hooks.h"
#include "gc_consts.h"
#include "gc_plan.h"
#include "gc_globals.h"
#include "gc_debug.h"
#include "verify_heap.h"
#include "cheney_fetcher.h"

#ifdef ORP_POSIX
#include "linux_api.h"
#endif

#ifdef ORP_NT
#include "win32_api.h"
#endif

extern bool garbage_collector_is_initialized;
extern LARGE_INTEGER performance_frequency;
extern LARGE_INTEGER gc_start_time;
extern LARGE_INTEGER gc_end_time;
extern double elapsed_time;
//
// This global is defined in gc_interface.cpp, and is used to
// determine if an object is large, in routine gc_malloc_3.
// This may need to be cleaned up when we speed up allocation.
//
extern unsigned los_threshold_bytes;

extern Step_Plus_Nursery_Generation *p_young_gen;

class ORP_Control;

extern Block_Store *p_global_bs;

//
// This is a fast verification guard to ensure that the
// JIT is not enumerating at an inappropriate time.
// This could result in stale references.
//
bool _collection_in_progress;

//
// The remembered set used to store all the live
// references enumerated by the ORP.
//
Root_List *p_root_set;
Root_List *p_verify_root_set;

// There are two Remembered Sets that need to be maintained for
// the train algorithm. Since the other algorithms are just subsets
// of the train algorithm these sets also work for them.
Remembered_Set *_p_refs_to_young_object_space;
Remembered_Set *_p_refs_to_focus_car;

// If there is a reference to the train it is noted here. This will
// Keep the train alive.
bool ref_to_focus_train_exists;

Car *_p_focus_car;
Train *_p_focus_train;

//
// The young generation has nurseries and steps.
//
Step_Plus_Nursery_Generation *_p_young_generation;

void notify_out_of_space (unsigned int size) 
{
    p_gc->reclaim_heap(size, false);
}

Mrl_Gc_V1::Mrl_Gc_V1(ORP_Control *p_orp_control,
                     char *p_plan_file_name)
                     : Gc_Interface(p_orp_control)
{
    
#if (GC_DEBUG>2)
    //
    // At debug time (fast verification), this goes on when
    // the GC stop-the-world kicks in.
    //
    _collection_in_progress = false;
#endif
    
    //
    // This is the interface object for communicating back to
    // the ORP.
    //
    _p_orp_control = p_orp_control;
    
    //
    // Set up the plan object which defines GC parameters,
    // and set up the right level of hooks, depending on
    // the debug level specified by the build.
    //
    _initialize_plan_and_hooks(p_plan_file_name);
    
    //
    // Make this a global for use by gc_malloc_3 in gc_interface.cpp
    // till we figure out the optimal code path for allocation.
    //
    // UNNECESSARY: INIT_GC HAS DONE THIS ALREADY. <<<<CHECK>>>>
#ifdef GC_COPY_V2
    los_threshold_bytes = _p_gc_plan->los_threshold_bytes();
#endif // GC_COPY_V2
#ifdef GC_GEN_V3
    los_threshold_bytes = _p_gc_plan->los_threshold_bytes();
#endif // GC_GEN_V3
    
    // Initialization (perhaps later the ORP should set defaults):
    if (initial_heap_size_bytes == 0) {
        initial_heap_size_bytes = _p_gc_plan->default_initial_heap_size_bytes();
    }
    
    if (final_heap_size_bytes == 0) {
        final_heap_size_bytes = _p_gc_plan->default_final_heap_size_bytes();
    }
    
    current_heap_size_bytes = initial_heap_size_bytes;
    
    
    unsigned int sub_block_size_bytes = _p_gc_plan->sub_block_size_bytes();
    
    unsigned long card_size_bytes  = _p_gc_plan->card_size_bytes(); 
    
    POINTER_SIZE_INT heap_base;
    
    _p_block_store  = 
        new Block_Store(_p_gc_hooks,
        _p_gc_plan,
        sub_block_size_bytes);    
    
    heap_base = (POINTER_SIZE_INT)(_p_block_store->get_heap_base());
    //
    // Give the write barrier code global access to the heap base,
    // so that it can use this for card marking without needing
    // to make a function call.
    //
    global_heap_base = heap_base;
    
    p_global_bs     = _p_block_store;
    
    p_root_set = new Root_List();
    p_root_set->reset();
    p_verify_root_set = new Root_List();
    p_verify_root_set->reset();
    
    //
    // This is the remembered set used to store all the
    // weak references enumerated by the ORP.
    //
    _p_weak_references = new Remembered_Set();
    
    // These are the root sets that are required for each and every
    // collection. They are empty at the start of the collection, filled
    // by the orp the MOS remembered sets, or the collection of YOS.
    // They are emptied at the end of the collection.
    
    // A combination of the orp roots and MOS to YOS pointers.
    _p_refs_to_young_object_space = new Remembered_Set();
    // Starting with the _p_container_write_barrier for this car this
    // is augmented by the orp roots as well as the YOS to MOS pointers.
    // It is scanned for entries from outside the train is none are found
    // from YOS to MOS or from ORP to the focus train.
    // This can hold refs from both inside and outside of the train.
    _p_refs_to_focus_car = new Remembered_Set();
    
    // This is created as we go through the _p_refs_to_focus_car set
    // It is empty at the start of a collection prior to enumerating
    // the roots. It is augmented by looping through the non-focus cars in
    // the focus train.
    //    _p_refs_to_focus_car_from_focus_train = new Remembered_Set();
    ref_to_focus_train_exists = false;
    
    // For the time being, create two generations:
    
    _number_of_generations = 2;
    
    // For GC_COPY_V2   we use a single younger generation with
    //                  there is a train but we do not tenure into it.
    //                  assert are set to ensure that no objects ever reside
    //                  in the train. 
    //                  There is also a fixed collector
    
    // For GC_GEN_V3    we have a nursery step generation and two trains.
    //                  Each train contains a single car that is large enough
    //                  to hold all object in the MOS generation. The youngest 
    //                  train spends it life empty except during a collection when
    //                  it exists only to receive object from the oldest
    //                  train. Since the oldest train will evacuate all its objects
    //                  during a GC_GEN_V3 collection it will be eliminated and
    //                  the youngest train will become the oldest train. 
    //
    //                  INVARIANT - we have only 2 trains and at the start and
    //                  end of each collection the youngest train is empty.
    //                  INVARIANT - a single car can hold all objects in MOS.
    //                  INVARIANT - no train has 2 cars.
    //                  
    //                  We tenure objects into the oldest train. 
    //                  For now the strategy is to have N steps and collect
    //                  MOS every N collections. This means that every live object
    //                  is moved at most N + M/N times where N is the number of steps
    //                  and M is the number of collection since the birth of the object.
    //
    //                  We also have a Fixed collector associated with the older space.
    //
    // For GC_TRAIN_V5
    //                  TBD - once I have done some tuning...
    //
    
    // Use the train algorithm here.
    
    _p_young_generation = 
        new Step_Plus_Nursery_Generation(YOS_GEN_NUMBER,
        _p_gc_hooks,
        _p_gc_plan,
        this,
#ifdef GC_COPY_V2 
        // There is not superior generation in the 
        // simple copy situation.
        NULL,
#else
        NULL,
        // p_mature_generation,
#endif // GC_COPY_V2.
        _p_card_table,
        _p_block_store);
    // This car and train are the focus of this collection.
    
    _p_focus_car = NULL;
    _p_focus_train = NULL;
    
    //
    // For the time being, keep all large objects in the youngest
    // generation and avoid tenuring them. The down side of this 
    // approach is that we will always scan these objects.
    //
    _p_los_container = new LOS_Container(this,
        _p_gc_hooks,
        _p_gc_plan,
        _p_young_generation,
        _p_card_table,
        _p_block_store);
    //
    // This global is made available for simplicity during
    // allocation, since there are a number of confusing allocation
    // paths due to different available options during development.
    // Later on this needs to be removed, and the container that
    // allocates the finalizable object should send a message
    // directly to its containing generation.
    //
    p_young_gen = _p_young_generation;
    
    //
    // Finally register the GC as being fully initialized.
    //
    garbage_collector_is_initialized = true;
}

Mrl_Gc_V1::~Mrl_Gc_V1()
{
    return;
    // If we are on our way out whay do we bother to delete this stuff.
    //
    // Make sure the block store is deleted after all the
    // generations and containers (which still need use of
    // this block store for releasing blocks.)
    //
    delete p_root_set;
    delete p_verify_root_set;
    delete _p_weak_references;
    delete _p_refs_to_young_object_space;
    delete _p_refs_to_focus_car;
    //    delete _p_refs_to_focus_car_from_focus_train;
    
    delete _p_orp_control;
    delete _p_young_generation;
    delete _p_los_container;
    delete _p_block_store;
}

//
// This is the last thing done by reclaim_heap, i.e. at
// the end of an incremental reclamation. This gives all
// generations a chance to clean up, then signals the 
// ORP to restart.
//
void
Mrl_Gc_V1::resume_orp()
{
    _p_orp_control->resume_mutators();
    
    return;
}
//
// Scan a card and if you discover an object that requires that the card stay marked then return true.
// otherwise return false.
void scan_card (Java_java_lang_Object *first_obj, Java_java_lang_Object *last_obj)
{
    
    Java_java_lang_Object *this_obj = first_obj;
    while (this_obj <= last_obj) {
        gc_trace (this_obj, "scan_card is scanning this object.");
        scan_object (this_obj);
        this_obj = (Java_java_lang_Object *)((POINTER_SIZE_INT)get_object_size_bytes(this_obj) + (POINTER_SIZE_INT)this_obj);
    }
    return;
}

//
// Each block has a card table and a table that gives the last object in a card.
// As we advance through the cards we keep track of the last object in the previous card
// and the last object in the card that is marked. 
// Note that we depend on the fact that an object is marked at its base so any 
// card that is marked will have an entry in the last object table.
//
void scan_cards_in_block (block_info *this_block)
{
    gc_trace_block (this_block, "Scanning cards in this block");
    
    int card_index = 0;
    void *first_obj_start = (void *)GC_BLOCK_ALLOC_START(this_block);
    Java_java_lang_Object *first_obj = P_OBJ_FROM_START(first_obj_start);
    void *last_obj_start = (void *)GC_BLOCK_ALLOC_START(this_block);
    Java_java_lang_Object *last_obj = P_OBJ_FROM_START (last_obj_start);
    // First card (0 index) will always be clear since it is the header.
    for (card_index = 1; card_index < GC_CARDS_IN_BLOCK; card_index++) {
        last_obj = this_block->card_last_object_table[card_index];
        if (last_obj) {
            if (this_block->card_table[card_index]) {
                // Clear the mark, it will be reset if needed.
                this_block->card_table[card_index] = false;
                // Scan the card and if it does not have an interesting pointer adjust the card table.
                scan_card(first_obj, last_obj); 
            }
            // We scan the last object in the previous card but
            
            first_obj = last_obj;
        } else {
            assert (!this_block->card_table[card_index]); // only cards with an object in them can be marked.
        }
    }
}

int get_free_block_count () 
{
    block_group_link *the_group_list = p_global_bs->block_group_list;
    int free_count = 0;
    while (the_group_list) {
        block_info *the_blocks = the_group_list->block_list;
        while (the_blocks) {
            if (the_blocks->in_free_p) {
                free_count = free_count + the_blocks->number_of_blocks;
            } else if (the_blocks->in_nursery_p) {
                if ((the_blocks->nursery_status == free_nursery) || 
                    (the_blocks->nursery_status == free_uncleared_nursery) ||
                    (the_blocks->nursery_status == thread_clearing_nursery)) {
                    free_count += the_blocks->number_of_blocks;
                }
            }
            the_blocks = the_blocks->all_blocks_next;
        }
        the_group_list = the_group_list->next;
    } 
    return free_count;
}

int get_free_non_nursery_block_count ()
{
    
    block_group_link *the_group_list = p_global_bs->block_group_list;
    int free_count = 0;
    while (the_group_list) {
        block_info *the_blocks = the_group_list->block_list;
        while (the_blocks) {
            if (the_blocks->in_free_p) {
                free_count = free_count + the_blocks->number_of_blocks;
            }
            the_blocks = the_blocks->all_blocks_next;
        }
        the_group_list = the_group_list->next;
    } 
    return free_count;
}

int get_los_block_count () 
{
    block_group_link *the_group_list = p_global_bs->block_group_list;
    int los_count = 0;
    while (the_group_list) {
        block_info *the_blocks = the_group_list->block_list;
        while (the_blocks) {
            if (the_blocks->in_los_p) {
                assert(!the_blocks->in_free_p);
                los_count = los_count + the_blocks->number_of_blocks;
            }
            the_blocks = the_blocks->all_blocks_next;
        }
        the_group_list = the_group_list->next;
    } 
    return los_count;
}

int get_total_block_count ()
{   
    block_group_link *the_group_list = p_global_bs->block_group_list;
    int total_count = 0;
    while (the_group_list) {
        block_info *the_blocks = the_group_list->block_list;
        while (the_blocks) {
            total_count += the_blocks->number_of_blocks;
            the_blocks = the_blocks->all_blocks_next;
        }
        the_group_list = the_group_list->next;
    }
    return total_count;
}


void report_train_stats()
{    
    train_info *the_train = trains;
    int car_block_count = 0;
    while (the_train) {
        if (stats_gc) {
            orp_cout << "Train " << (int)the_train->birthday << endl;
        }
        car_info *the_cars = the_train->cars;
        while (the_cars) {
            car_block_count = 0;
            block_info * the_blocks = the_cars->blocks;
            while (the_blocks) {
                car_block_count++;
                the_blocks = the_blocks->next;
            }
            
            if (stats_gc) {
                orp_cout << "   Car " << (int)the_cars->birthday << " has " << car_block_count << " blocks." << endl;
            }
            the_cars = the_cars->next;
        }
        the_train = the_train->next;
    }
}

void report_block_stats()
{    
    block_group_link *the_group_list = p_global_bs->block_group_list;
    int free_count = 0;
    int used_count = 0;
    int step_count = 0;
    int los_count = 0;
    int train_count = 0;
    int nursery_count = 0;
    int total_count = 0;
    while (the_group_list) {
        block_info *the_blocks = the_group_list->block_list;
        while (the_blocks) {
            total_count += the_blocks->number_of_blocks;
            if (the_blocks->in_free_p) {
                free_count = free_count + the_blocks->number_of_blocks;
            } else if (the_blocks->in_los_p) {
                los_count = los_count + the_blocks->number_of_blocks;
            } else if (the_blocks->in_step_p) {
                step_count = step_count + the_blocks->number_of_blocks;
            } else if (the_blocks->train_birthday) {
                train_count = train_count + the_blocks->number_of_blocks;
            } else if (the_blocks->in_nursery_p) {
                nursery_count = nursery_count + the_blocks->number_of_blocks;
            } else {
                assert (0);
            }
            the_blocks = the_blocks->all_blocks_next;
        }
        the_group_list = the_group_list->next;
    }
    used_count = step_count + los_count + train_count + nursery_count;
    
    if (stats_gc) {
        orp_cout << "Free blocks = " << free_count << endl;
        orp_cout << " FLOS count = " << los_count << endl;
        orp_cout << " Step count = " << step_count << endl;
        orp_cout << " Train count = " << train_count << endl;
        orp_cout << " Nursery count = " << nursery_count << endl;
        orp_cout << " Total used count = " << used_count << endl;
        orp_cout << " Total count = " << total_count << endl;
        report_train_stats();
    }
}

car_info *get_focus_car ()
{
    assert (trains);
    return trains->cars;
}

void add_car(train_info *a_train)
{
    car_info *new_car = (car_info *)malloc(sizeof (car_info)); 
    new_car->alloc_block = p_get_new_block(true, false, true);
    gc_trace_block (new_car->alloc_block, "This block is the first in a new car.");
    new_car->alloc_block->list_info = (block_list_info *)new_car;
    new_car->alloc_block->train_birthday = a_train->birthday;
    gc_block_status_table[new_car->alloc_block->block_status_table_index] = block_in_mos;
    new_car->blocks = new_car->alloc_block;
    new_car->my_train = a_train;
    new_car->next = NULL;
    new_car->scan_block = new_car->alloc_block;
    if (a_train->cars == NULL) {
        new_car->birthday = 0;
        new_car->alloc_block->car_birthday = new_car->birthday;
        a_train->cars = new_car;
        a_train->last_car = new_car;
    } else {
        new_car->birthday = a_train->last_car->birthday + 1;
        new_car->alloc_block->car_birthday = new_car->birthday;
        a_train->last_car->next = new_car;
        a_train->last_car = new_car;
    }
    new_car->alloc_block->in_free_p = false;
}

void add_train()
{
    train_info *this_train = (train_info *)malloc(sizeof (train_info));
    this_train->birthday = current_train_creation_date++;
    this_train->next = NULL;
    this_train->cars = NULL;
    this_train->last_car = NULL;
    this_train->previous = NULL;
    add_car (this_train);
    if (!trains) {
        // This is the first train;
        trains = this_train;
    } else {
        train_info *last_train = get_youngest_train();
        assert (last_train->next == NULL);
        last_train->next = this_train;
        this_train->previous = last_train;
    }
}

int adjustment_count = 0;
// Do a train adjustment and return true cause the caller to try to collect a single car.
boolean train_adjustment()
{
    if (fixed_gc) {
        // We are doing only fixed GC so we should always indicate that we need to do a GC.
        return true;
    }
    boolean do_gc = false;
    // Make sure we have a train.
    if (!trains) {
        add_train();
    }
    
    // Make sure we always have two trains.
    if (!trains->next) {
        add_train();
    }
    
    // We now have the required 2 trains, the next thing to do is to see if we have filled up 
    // the cars in any of the trains, and if we have then we need to add a car to that train.
    boolean added_car = false;
    train_info *a_train = trains;
    while (a_train) {
        train_info *this_train = a_train;
        car_info *last_car = this_train->last_car;
        block_info *a_block = last_car->blocks;
        unsigned int block_count = 0;
        while (a_block) {
            block_count++;
            a_block = a_block->next;
        }
        if (block_count > GC_MAX_BLOCKS_PER_CAR) {
            add_car(this_train);
            do_gc = true;
        }
        a_train = a_train->next;
    }
    if (adjustment_count++ > 8) {
        do_gc = true;
        adjustment_count = 0;
    }
    return do_gc;
}


//
// If the collection was forced the entire MOS was collected.
//
void cleanup_mos (boolean forced) {
    if (forced) { // We are collecting more than one car, we collected all the cars and all the trains.
        train_info *train_list = trains;
        train_info *this_train;
        car_info *car_list;
        car_info *this_car;
        while (train_list) {
            this_train = train_list;
            train_list = train_list->next;
            car_list = this_train->cars;
            block_info *block_list;
            while (car_list) {
                this_car = car_list;
                car_list = car_list->next;
                if (this_car->blocks->c_area_p) { 
                    // We collected this car (and the entire train for that matter)
                    block_list = this_car->blocks;
                    block_info *this_block;
                    while (block_list) {
                        this_block = block_list;
                        block_list = block_list->next;
                        // relink the blocks in this car.
                        gc_trace_block (this_block, "This block is being freed from MOS. line 939");
                        assert (this_block->number_of_blocks == 1);
                        if (this_block->c_area_p) {
                            p_global_bs->link_free_blocks (this_block, this_block->number_of_blocks);
                        } else {
                            assert (0); // If one is in c_area_p they had all better be.
                        }
                    } // All the blocks in this car have been made available for reuse.
                    if (this_car->next) {
                        this_train->cars = this_car->next;
                        free (this_car); // Return this car.
                    } else {
                        // The train has no cars left so free the train.
                        trains = trains->next;
                        trains->previous = NULL;
                        assert (trains);
                        free (this_car);
                        free (this_train);
                    }
                }
            }
        }
        // If we collected all the trains then we should only have left the one we moved all the objects into.
        assert (fixed_gc?true:!(trains->next));
    } else {
        if (trains->cars->blocks->c_area_p) {
            train_info *this_train = trains;
            car_info *this_car = trains->cars;
            block_info *a_block  = trains->cars->blocks;
            while (a_block) {
                block_info *this_block = a_block;
                // relink the blocks in this car.
                gc_trace_block (a_block, "This block is being freed from MOS. line 900");
                assert (this_block->number_of_blocks == 1);
                
                a_block = a_block->next;
                p_global_bs->link_free_blocks (this_block, this_block->number_of_blocks);
            }
            // There are more cars in this train so adjust the train structure.
            if (this_car->next) {
                this_train->cars = this_car->next;
                free (this_car);
            } else {
                trains = trains->next;
                trains->previous = NULL;
                assert (trains);
                /**************** We need to do these deletes someday *****************/
                //                free(this_car);
                //                free(this_train);
            }
        }
    }
}

void do_cheney_scan ()
{
    if (use_cheney_clock) {
        start_cheney_clock();
    }

    bool object_scanned = true;
    // Cheney scan the step.
    while (object_scanned) {
        Object_Gc_Header *scan_ptr;
        
        object_scanned = false;
        // First cheney scan the step.
        block_list_info *the_step = step;
        block_info *the_scan_block = the_step->scan_block;
        assert (the_scan_block);
        block_info *the_alloc_block;
        while (the_scan_block) {
            the_alloc_block = the_step->alloc_block;
            scan_ptr = (Object_Gc_Header *)the_scan_block->scan;
            if (use_fetcher_thread) {
                // This is a bit of a tuning issue. It makes little sense to scan
                // in the same block we are fetching in since the scanner will 
                // quickly catch up with the fetcher since the scanner does
                // very little work a lot of the time. So we scan a block ahead.
                // Notice that if we do catch up then we will skip a blocks
                // worth of scanning.
                if (the_scan_block->next) {
                    start_fetcher_thread(the_scan_block->next);
                } else {
                    start_fetcher_thread(the_scan_block);
                }
                // vvvvvv
            }
            while (scan_ptr < the_scan_block->free) {
                gc_trace (P_OBJ_FROM_START(scan_ptr), "Cheney scanning this object line 987.");
                unsigned int real_object_size_bytes =
                    get_real_object_size_bytes(P_OBJ_FROM_START(scan_ptr));
                scan_object(P_OBJ_FROM_START(scan_ptr));
                scan_ptr = (Object_Gc_Header *)((char *)scan_ptr + real_object_size_bytes);
                object_scanned = true;
            }
            the_scan_block->scan = scan_ptr;
            the_step->scan_block = the_scan_block;
            the_scan_block = the_scan_block->next;
        }
        assert (the_step->scan_block == the_alloc_block);
        assert (the_step->scan_block->scan == the_step->alloc_block->free);
        
        // Now cheney scan the trains.
        
        train_info *the_train;
        the_train = trains;
        while (the_train) { // Loop through all the trains.
            // Idea idea RLH **** 
            // Assuming that we can at this point ask for all the pages that have been dirtied
            // Since the last time this block was scanned then we will get a collection of the
            // following pages. First all the pages that have been written to by the mutator, these
            // are the pages that are scanned by inspecting the card table.
            // In addition the pages that have been written to by the GC need to be scanned. Typically
            // this is done using a cheney scan where we manage the scan pointer and the free pointer.
            // If we use the dirty bits we will get these pages. We still have the problem of figuring out
            // where to start the scan....
            // ** end patent idea **
            car_info *the_car = the_train->cars;
            while (the_car) { // Loop through all the cars in the train.
                the_scan_block = the_car->scan_block;
                gc_trace_block (the_scan_block, "Cheney scanning this block in a train");
                while (the_scan_block) {
                    
                    if (use_fetcher_thread) {
                        start_fetcher_thread(the_scan_block);             // vvvvvvv
                    }

                    the_alloc_block = the_car->alloc_block;
                    scan_ptr =  (Object_Gc_Header *)the_scan_block->scan;
                    while (scan_ptr < the_scan_block->free) {
                        unsigned int real_object_size_bytes =
                            get_real_object_size_bytes(P_OBJ_FROM_START(scan_ptr));                        
                        gc_trace (P_OBJ_FROM_START(scan_ptr), "Cheney scanning this object line 1025.");
                        scan_object(P_OBJ_FROM_START(scan_ptr));
                        scan_ptr = (Object_Gc_Header *)((char *)scan_ptr + real_object_size_bytes);
                        object_scanned = true;
                    }
                    the_scan_block->scan = scan_ptr;
                    the_car->scan_block = the_scan_block;
                    the_scan_block = the_scan_block->next;
                }
                assert (the_car->alloc_block == the_alloc_block);
                assert (the_car->scan_block->scan == the_alloc_block->free);
                the_car = the_car->next;
            }
            the_train = the_train->next;
        }
        // We keep repeating until we do not need to scan any objects.
    }
    
    if (use_fetcher_thread) {
        stop_fetcher_thread();  
    }
    if (use_cheney_clock) {
        stop_cheney_clock();
    }
}

void
Mrl_Gc_V1::_execute_collection()
{
    
    // p_root_set holds all the slots not in the heap that have pointers into 
    // areas that are being collected during this collection.
    
    // Loop through these slots moving objects.
    
    p_root_set->rewind();
    train_info *youngest_train = NULL;
    if (!fixed_gc) {
        youngest_train = get_youngest_train(); // Make sure we have a youngest train and return it.
    }
    int temp_root_count = 0;
    for (Java_java_lang_Object **pp_slot = p_root_set->pp_next(); pp_slot; pp_slot = p_root_set->pp_next()) {
        Java_java_lang_Object *p_obj = *pp_slot;
//        assert (GC_BLOCK_INFO(p_obj)->c_area_p);
        block_info *target_block;
        // We have a object referenced by a root which target block do we move it to.
        if (GC_BLOCK_INFO(p_obj)->in_nursery_p) {
            target_block = step->alloc_block; // Move nursery objects into the step
        } else if (GC_BLOCK_INFO(p_obj)->in_los_p)  {
            target_block = NULL;
        } else {
            target_block = youngest_train->last_car->alloc_block; // objects anywhere else (steps, focus car) go to younges train.
            
        }
        gc_trace_slot ((void **)pp_slot, *pp_slot, "Slot being updated (if needed)");
        // If it is a nursery we will move to the step otherwise it goes to the youngest train.
        *pp_slot = p_get_forwarded_object(target_block, p_obj); // Moves an object (or discovers where it has been moved) and
        // returns "to" version of the object. 
        // Marks and scans unmarkedLOS objects.
        
        gc_trace_slot ((void **)pp_slot, *pp_slot, "Slot was updated (if needed)");
        temp_root_count++;
    }
    
    if (verbose_gc) {
        //        orp_cout << "number of roots is " << temp_root_count << endl;
    }
    
    // For simple generational GC scan the cards of all the trains.
    
#ifdef GC_GEN_V3
    // Scan the cards.
    // Train requires we do this from youngest train to oldest (focus) train
    train_info *this_train = get_youngest_train(); // Start with the youngest train an move to the oldest.
    while (this_train) { // Loop through all the trains.
        car_info *the_car = this_train->cars;
        while (the_car) { // Loop through all the cars in the train.
            // Start by scanning the card table of each block.
            block_info *this_block = the_car->blocks;
            while (this_block) {
                if (this_block->c_area_p) { // block is in focus car, do not scan.
#ifdef _DEBUG
                    while (this_block) {
                        assert (this_block->c_area_p);
                        this_block = this_block->next;
                    }
#endif 
                    this_block = NULL; // end the loop.
                } else {
                    scan_cards_in_block (this_block);
                    this_block = this_block->next;
                }
            }
            the_car = the_car->next;
        }
        
        do_cheney_scan();   // cheney scan - try to move as much to the younger trains as possible.
        this_train = this_train->previous; // Move from youngest to oldest train
    }
    
#endif // GC_GEN_V3
    
}

Car *
Mrl_Gc_V1::p_get_focus_car() {
    return _p_focus_car;
}

Train *
Mrl_Gc_V1::p_get_focus_train() {
    return _p_focus_train;
}

void
Mrl_Gc_V1::free(Java_java_lang_Object *p_obj)
{
    // hint from the mutator: p_obj is toast.
}

void
Mrl_Gc_V1::enumerate_weak_reference_queues ()
{
#ifdef _DEBUG
    if (verbose_gc) {
        orp_cout << "Not enumerating weak references (yet)." << endl;
    }
#endif
#if 0
    _p_young_generation->enumerate_reference_queues ();
    _p_mature_generation->enumerate_reference_queues ();
#endif
}

//
// Just before doing a stop-the-world collection, we go to
// the ORP asking for all live references. In the current 
// scheme, rather than return the root set, the ORP makes
// a call back into the GC for each live reference. This is
// the interface routine gc_add_root_set_entry in gc\gc_interface.cpp
//
void
Mrl_Gc_V1::_get_orp_live_references()
{
#if (GC_DEBUG>2)
    //
    // Open a window during which the JIT can legitimately
    // enumerate live references. Fast verification for
    // debugging the JIT.
    //
    _collection_in_progress = true;
#endif
    
    
    enumeration_start_hook(_p_refs_to_young_object_space,
        _p_refs_to_focus_car, 
        _p_weak_references);
    
    //
    // First ask the mutator for the live references:
    //
#ifdef GC_TRAIN_TRACE
    orp_cout << "Train trace - Empty _p_refs_to_young_object_space and _p_refs_to_focus_car." << endl;
#endif
    
    // Clear the p_root_set from previous GCs.
    p_root_set->reset();
    p_verify_root_set->reset();
    
    _p_refs_to_young_object_space->empty_all();
    _p_weak_references->empty_all();
    _p_refs_to_focus_car->empty_all();
    
    // Perhaps this is not where this should be done...
    //    _p_refs_to_focus_car_from_focus_train->empty_all(); 
    
    // This fills the _p_refs_to_young_object_space 
    // and _p_refs_to_focus_car.
    _p_orp_control->enumerate_live_references();
    
    // There are a reference queues associated with the step_generation and the
    // cars that need to be enumerated.
    enumerate_weak_reference_queues();
    
    enumeration_end_hook(_p_refs_to_young_object_space,
        _p_refs_to_focus_car,
        _p_weak_references);
    
#if (GC_DEBUG>2)
    //
    // Close the window during which the JIT can enumerate.
    //
    _collection_in_progress = false;
#endif
}

//
// This private method is called during the creation of the
// Mrl_Gc_V1 object in order to set up the hooks and 
// create the plan object.
//
void
Mrl_Gc_V1::_initialize_plan_and_hooks(char *p_plan_file_name)
{
    //
    // Create a plan object. This process will check if the
    // user has a .plan file in the right place. If so, the
    // contents of that file override the default settings
    // in the plan.
    //  
    if (p_plan_file_name) {
        _p_gc_plan = new Gc_Plan(p_plan_file_name);
    } else {
        _p_gc_plan = new Gc_Plan();
    }
}
// GC_INTERIOR_POINTERS
//
// This code is called only at the end of the GC, and by software convention, no more calls
// are made to gc_add_root_set_entry_interior_pointer during this GC. 
// This code loops through the interior pointers updating each of them, one at a time.
// It then resets the interior pointer table so that it is empty in preperation for the 
// next GC.
// In the case of SAPPHIRE this routine is called for each thread whenever we update the
// root set for that particular thread. 
void fixup_interior_pointers ()
{
    void **slot;
    interior_pointer_table->rewind();
    while (interior_pointer_table->available()) {
        slot = interior_pointer_table->get_slot();
        *slot = (void *)((POINTER_SIZE_INT)interior_pointer_table->get_base() + 
            (POINTER_SIZE_INT)interior_pointer_table->get_offset());
        interior_pointer_table->next();
    }
    // We are done with this round reset it for the next GC.
    interior_pointer_table->reset();
}

//
//  reclaim heap is called when there are no longer any nurseries left to
//  allocate objects into or when there is no space in LOS to place an object.
//
//  One of several things can happen at this point. Traditionally this would
// indicate a garbage collection of at least the YOS and the LOS. It could mean that
// a single car is collected. Finally we are adding a new twist called a "declined
// collection."
//
//  A "declined collection" is a "logical" collection of the nurseries where all the objects
//  in the nurseries that are filled are moved en masse into the step generation. The gain
//  is that since the objects are not actually copied to new addresses and no objects are
//  being evicted then we do not have to stop threads or scan the cards. The downside is that
//  we do not collect any objects. All we have succeeded in doing is moving objects that are
//  live at this time from the nurseries to the step.
//
//  Motivation: Several programs create large data structures at the beginning of
//      execution and these objects live for a long time. If we would copy them from
//      the nurseries into a step and then into MOS they would have been copied twice.
//      If however we use the empty space in the steps to stash the objects into so they
//      can age a little before they are moved into MOS then we end up only copying the
//      objects once and having to scan stacks and heaps less.
//      Of course this has the cost of doubling the space needed for nurseries but if the
//      space is not needed for anything else it is a win. Obviously the space isn't
//      being used during the first collection.
// 
// Argument size:
//  The size argument is of interest only in the case of the LOS container wanting more
//  space. If the size is 0 then we just need nurseries for small objects. If the size
//  is >0 then it indicates how much space we need.
//

// The convention here is that the caller holds and will relinquish the gc_lock.

int _yos_collections_since_mos_collection = 0;

int _calls_since_yos_collection = 0;

volatile bool gc_just_completed = false;
volatile bool mos_collection_was_needed = true;
volatile int number_of_blocks_freed_last_yos = 0;
volatile int number_of_blocks_freed_next_to_last_yos = 0;
volatile int number_of_blocks_freed_third_to_last_yos = 0;
volatile int ave_blocks_recovered_by_yos = 0;

// Set the areas in this block to be in the "c" (collected) area.
void set_c_area_for_nurseries () {
    block_info *this_nursery;
    int i;
    for (i = 0; i < allocated_nurseries; i++) {
        this_nursery = nurseries[i];
        assert (this_nursery->in_nursery_p);
//        if ((this_nursery->nursery_status == spent_nursery) || (this_nursery->nursery_status == active_nursery)) {
            this_nursery->c_area_p = true; // Both active and spent nurseries are collected. all nurseries are collected.
//        }
    }
    assert (nurseries[allocated_nurseries] == NULL); // Make sure we have all the nurseries.
}

// Set the areas in this block to be in the "c" (collected) area.
void set_c_area (block_info *the_blocks) {
    for (block_info *this_block=the_blocks; this_block; this_block=this_block->next) {
        this_block->c_area_p = true;
    }
}


void set_up_c_areas_in_all_trains ()
{
    train_info *train_list = trains;
    // Now set all the blocks in all the trains to become collect (c) areas.
    while (train_list) {
        car_info *car_list = train_list->cars;
        while (car_list) {
            block_info *block_list = car_list->blocks;
            set_c_area (block_list);
            car_list = car_list->next;
        }
        train_list = train_list->next;
    }
    // Now add a train to move objects into.
    add_train();
}
//
//
//
void set_up_c_areas (car_info *focus_car, block_info *the_from_step_blocks, bool forced) {
    set_c_area_for_nurseries ();    // all nurseries are always in the c area.
    set_c_area (the_from_step_blocks);
    if (forced) {
        set_up_c_areas_in_all_trains ();
    } else {
        if (focus_car) {
            set_c_area (focus_car->blocks);
        }
    }
}

int collection_defered_count = 0;

void
Mrl_Gc_V1::reclaim_heap(unsigned int size, bool forced)   // This guy needs to know why he was called was it
// an the LOS that ran out of space or was it a nursery.
// a true forced indicates a forced gc.
{
#ifdef GC_SAPPHIRE
    p_sapphire_lock->_lock();
#endif // GC_SAPPHIRE
    gc_start_hook(this);
#ifdef GC_SAPPHIRE
    extern void sapphire_driver();
    
    global_gc_count++; // Also set at end of reclaim_heap.
    unsigned long sapphire_number_of_spent_nurseries = _p_young_generation->get_number_of_spent_nurseries();
    unsigned long sapphire_number_of_nurseries = _p_young_generation->get_number_of_nurseries();
    //    orp_cout << "Number_of_spent_nurseries " << sapphire_number_of_spent_nurseries << endl;
    //    orp_cout << "Number_of_nurseries " << sapphire_number_of_nurseries << endl;
    //    orp_cout << "s";
    sapphire_driver();
    p_sapphire_lock->_unlock();
    //    orp_cout << "Heap reclaimed";
    gc_end_hook(this);
    return;
#endif // GC_SAPPHIRE
    
    // If the stack scans or the scanning of the non-focus trains produce a pointer to the focus train
    // then this will be set to true.
    
    ref_to_focus_train_exists = false;
    
    if (forced) {
        if (verbose_gc) {
            orp_cout << "     -- GC requested by Java program." << endl;
        }
    }
    // There might be race conditions here with the asserts since threads aren't all stopped.
    // Analyze the heap.
    
    unsigned long number_of_threads = thread_gc_number_of_threads() + 1;    
    int final_heap_size_blocks = final_heap_size_bytes / GC_BLOCK_SIZE_BYTES;

#if 0
    block_group_link *the_group_list = p_global_bs->block_group_list;
    int i_free_count = 0;
    int i_used_count = 0;
    int i_step_count = 0;
    int i_los_count = 0;
    int i_train_count = 0;
    int i_nursery_count = 0;
    int i_total_count = 0;
    while (the_group_list) {
        block_info *the_blocks = the_group_list->block_list;
        while (the_blocks) {
            i_total_count += the_blocks->number_of_blocks;
            if (the_blocks->in_free_p) {
                assert (gc_block_status_table[the_blocks->block_status_table_index] == block_in_free);
                i_free_count = i_free_count + the_blocks->number_of_blocks;
            } else if (the_blocks->in_los_p) {
                unsigned int i;
                for (i = the_blocks->block_status_table_index; i < the_blocks->block_status_table_index + the_blocks->number_of_blocks; i++) {
                    assert (gc_block_status_table[i] == block_in_los);
                }
                assert (gc_block_status_table[the_blocks->block_status_table_index] == block_in_los);
                i_los_count = i_los_count + the_blocks->number_of_blocks;
            } else if (the_blocks->in_step_p) {
                assert (gc_block_status_table[the_blocks->block_status_table_index] == block_in_step);
                i_step_count = i_step_count + the_blocks->number_of_blocks;
            } else if (the_blocks->train_birthday) {
                assert (gc_block_status_table[the_blocks->block_status_table_index] == block_in_mos);
                i_train_count = i_train_count + the_blocks->number_of_blocks;
            } else if (the_blocks->in_nursery_p) {
                assert (gc_block_status_table[the_blocks->block_status_table_index] == block_in_nursery);
                i_nursery_count = i_nursery_count + the_blocks->number_of_blocks;
            } else {
                // Since other threads continue to run there can be a race condition here that might
                // cause this block to be in a transition state. For example after we check for in_los_p
                // after ->in_free_p is set to false but before in_los_p is set to true we have a race.
                // Since such races are rare we will just ignore the accounting for such blocks.
                // assert (0);
            }
            the_blocks = the_blocks->all_blocks_next;
        }
        the_group_list = the_group_list->next;
    }
    i_used_count = i_step_count + i_los_count + i_train_count + i_nursery_count;
    assert(i_used_count+i_free_count == i_total_count);
    // Decide if we should do a GC or just create some more nurseries.
    
    // use the block_status_table to discover the state of the heap.
#endif 


    int free_count = 0;
    int used_count = 0;
    int step_count = 0;
    int los_count = 0;
    int train_count = 0;
    int nursery_count = 0;
    int total_count = 0;
    int block_status_table_index_max = current_heap_size_bytes / GC_BLOCK_SIZE_BYTES;
    int block_status_index = 0;
    for (block_status_index = 0; block_status_index < block_status_table_index_max; block_status_index++ ) {
            total_count++;
            if (gc_block_status_table[block_status_index] == block_in_free) {
                free_count++;
            } else if (gc_block_status_table[block_status_index] == block_in_los) {
                los_count++;
            } else if (gc_block_status_table[block_status_index] == block_in_step) {
                step_count++;
            } else if (gc_block_status_table[block_status_index] == block_in_mos) {
                train_count++;
            } else if (gc_block_status_table[block_status_index] == block_in_nursery) {
                nursery_count++;
            } else {
                // Since other threads continue to run there can be a race condition here that might
                // cause this block to be in a transition state. For example after we check for in_los_p
                // after ->in_free_p is set to false but before in_los_p is set to true we have a race.
                // Since such races are rare we will just ignore the accounting for such blocks.
                // assert (0);
            }
    }
    used_count = step_count + los_count + train_count + nursery_count;
    assert(used_count+free_count == total_count);
#if 0
    // race conditions make these not exactly true.
    assert(i_free_count == free_count);
    assert(i_used_count == used_count) ;
    assert(i_step_count == step_count);
    assert(i_los_count == los_count);
    assert(i_train_count == train_count);
    assert(i_nursery_count == nursery_count);
    assert(i_total_count == total_count);
#endif
    if ((int)nursery_count < (int)(number_of_threads * 4)) {
        if (!forced && !size) {
            
            if (verbose_gc) {
                orp_cout << "<-- Adding nurseries to accomodate additional threads >" << endl;
            }
            add_nurseries(number_of_threads);
            collection_defered_count++;
            if (verbose_gc || stats_gc) {
                int free_block_count = get_free_block_count();
                int total_block_count = get_total_block_count();
                orp_cout << "<gc delayed " << global_gc_count << ", free blocks = " << free_block_count << ", total_blocks = " << total_block_count << ">" << endl;
            }
            return;
        }
    }
    // define the amount of overdraft you are going to allow.
    // Each time you extend the heap get more paranoid.
    static int free_count_multiple = 2; // We have 2 used for every free.
    if (heap_was_extended) {
        if (free_count_multiple > 1) {
            heap_was_extended = false;    
            free_count_multiple--; // We have extended so go for as much free as used.
        }
    }
    // Let free space drop to 1/3. If the heap gets extended cut back to 1/2
    // If we move the nurseries into a step we still need to account for nurseries used during the next cycle so add that
    // to used count.
    if (((used_count+nursery_count)*2) < final_heap_size_blocks ) {
        // Magic number should be places elsewhere.
        if (!forced && !size) {
            
            if (move_spent_nurseries_into_step()) {
                collection_defered_count++;
                
                if (stats_gc) {
                    int free_block_count = get_free_block_count();
                    int total_block_count = get_total_block_count();
                    orp_cout << "<gc delayed " << global_gc_count << ", free blocks = " << free_block_count << ", total_blocks = " << total_block_count << ">" << endl;
                }
                return;
            }
        }
    }

#ifdef CONCURRENCY_ANALYSIS
        uint64 start_time = readTimeStampCounter();
#ifdef ORP_POSIX
        fprintf(f_concur, 
            "GC_START( tid = %d, gc_thread_id = %d, time = %llu )\n",
            p_TLS_orpthread->thread_index, p_TLS_orpthread->thread_index, start_time );
#else
        fprintf(f_concur, 
            "GC_START( tid = %d, gc_thread_id = %d, time = %I64u )\n",
            p_TLS_orpthread->thread_index, p_TLS_orpthread->thread_index, start_time );
#endif //#ifdef ORP_POSIX
#endif //#ifdef CONCURRENCY_ANALYSIS

    
    if (stats_gc) {
        report_block_stats();
    }
    
    global_gc_count++; // Also set at end of reclaim_heap.
    // It would appear that doing a complete GC is always a good idea, much to my embarassment.
    // This is mostly due to the lack of any benchmark that cares about latency. (oh well)
    // orp_cout << "Not Forcing GC" << endl;
    forced = true;
    
    if (verbose_gc || stats_gc) {
        int free_block_count = get_free_block_count();
        int total_block_count = get_total_block_count();

        orp_cout << "<gc enter " << global_gc_count << ", free blocks = " << free_block_count << ", total_blocks = " << total_block_count << ">" << endl;
    }
    
    // Make sure we do a collection at least every other time.
    collection_defered_count = 0;
    
    //
    // Ask the ORP for all live references. The ORP will
    // make a series of callbacks into the GC for each
    // live reference. (See gc\gc_interface.cpp, function
    // gc_add_root_set_entry). This results in _p_refs_to_young_object_space and
    // _p_refs_to_focus_car being filled in. 
    //                          
    //
    
    car_info *focus_car = NULL;
    if (!fixed_gc) {
        boolean car_added = train_adjustment();
        if (car_added) {
            // we added a car so collect a car.
            focus_car = trains->cars;
        }
    }    
    // Check that step has valid free and scan pointers and all objects were scanned during last GC.
    assert (step->alloc_block == step->scan_block);
    assert (step->alloc_block->free == step->scan_block->scan);
    
    if (!fixed_gc) {
        // Fixed GC doesn't use cheney spaces and los always has c areas set.
        // Replace blocks the step is using.
        set_up_step_cheney_spaces(step, &step_from_blocks);
        
        // Set up the blocks we are going to collect.
        set_up_c_areas(focus_car, step_from_blocks, forced);
    }
    _get_orp_live_references(); 
    
    verify_heap(p_verify_root_set);
    
    _execute_collection();
    // GC_INTERIOR_POINTERS
    fixup_interior_pointers();
    
#if 0 
    // This code is not turned on yet....
    
    //
    // Update all weak references to objects that have moved.
    //
    _update_soft_references(false, false);
    _update_weak_references(false);
    _update_phantom_references(false);
    
    //
    // We have finished a minor collection. Beforp we return we need to
    // scan the spent nurseries, from spaces and LOS for unreachable 
    // objects that have just become candidates for finalization.
    //
    _locate_and_process_finalizable_objects(false);
#endif 
    //
    // Give the generations a chance to clean up.
    //
    _p_young_generation->cleanup();
    
    cleanup_mos(forced);
    //
    // Give the Large Object Store a chance to clean up.
    // The sweep is done here... or we could do it while allocating if we
    // want better (hopefully) scalability and locality.
    //
    _p_los_container->cleanup();
    
    // These from blocks are free make them available.
    block_info *temp_from_block = step_from_blocks;
    while (temp_from_block) {
        block_info *this_block = temp_from_block;
        temp_from_block = temp_from_block->next;
        this_block->next = NULL;
        p_global_bs->link_free_blocks (this_block, this_block->number_of_blocks);
        assert (this_block->number_of_blocks == 1);
    }
    
    
    verify_heap(p_verify_root_set);
    
#ifdef GC_PT_WB
    // Clear the page dirty bits
    PVOID heap_base = p_global_bs->get_heap_base();
    SIZE_T region_size = current_heap_size_bytes
    UINT foo = dll_ResetWriteWatch (heap_base, region_size);
#endif // GC_PT_WB
    
    if (size > 0) {
        p_global_bs->coalesce_free_blocks();
        if (! _p_los_container->is_block_available(size)) {
            _p_los_container->extend_los (size);
        } else {
            verify_heap(p_verify_root_set);
        }
    }
    
    //
    // _after_stop_the_world_hook is called in the above 
    // cleanup routine *before* re-starting the ORP.
    //
    
    if (verbose_gc || stats_gc) {
        gc_end_hook(this);
    }
    
    resume_orp();
    
    if (verbose_gc || stats_gc) {
        int free_block_count = get_free_block_count();
        int total_block_count = get_total_block_count();
        orp_cout << "<gc exit " << global_gc_count << ", free blocks = " << free_block_count << ", total_blocks = " << total_block_count << ">" << endl;
    }
    
    if (stats_gc) {
        report_block_stats();
    }

#ifdef CONCURRENCY_ANALYSIS
        uint64 end_time = readTimeStampCounter();
#ifdef ORP_POSIX
        fprintf(f_concur, 
            "GC_END( tid = %d, gc_thread_id=%d, time = %llu )\n",
            p_TLS_orpthread->thread_index, p_TLS_orpthread->thread_index, end_time );
#else
        fprintf(f_concur, 
            "GC_END( tid = %d, gc_thread_id = %d, time = %I64u )\n",
            p_TLS_orpthread->thread_index, p_TLS_orpthread->thread_index, end_time );
#endif //#ifdef ORP_POSIX
#endif //#ifdef CONCURRENCY_ANALYSIS
    
    return;
}

void
Mrl_Gc_V1::run_all_finalizers()
{
#ifdef _DEBUG
    if (verbose_gc) {
        orp_cout << "Not running all finalizers in run_all_finalizers" << endl;
    }
#endif
#if 0
    _p_young_generation->run_all_finalizers();
    _p_mature_generation->run_all_finalizers();
#endif // GC_REWORK
}

void
Mrl_Gc_V1::set_verbose_level(unsigned int level)
{
    _p_gc_plan->verbose_gc = level;
}

//
// ORP notification to GC to clean up.
//
void 
Mrl_Gc_V1::wrapup_gc() 
{
    //
    // Currently does nothing.
    //
}

//
// The ORP is providing us with yet another live reference.
// We add it to our root set. We do some quick filtering her
// to determine if the reference is interesting. If it is it
// is placed in an SSB style root_set.
//
// It appears that all the thread are not yet stopped and this
// is why we don't immediately deal with these roots.
//
void Mrl_Gc_V1::gc_add_root_set_entry(Java_java_lang_Object **ref) 
{
    gc_trace ((void *)*ref, "gc_add_root_set_entry enumerates this pointer"); 
#ifdef GC_SAPPHIRE
    extern void sapphire_add_root_set_entry(Java_java_lang_Object **ref);
    // let sapphire deal with this.
    sapphire_add_root_set_entry(ref);
    return;
#endif // GC_SAPPHIRE
    
    // Is is NULL.  The NULL reference is not interesting.
    if (*ref == NULL) {
        return;
    }
#ifdef _DEBUG
    Java_java_lang_Object *the_obj = *ref;
    assert (the_obj->vt);
    assert (GC_BLOCK_INFO(the_obj->vt->clss)->in_los_p);
    assert (GC_BLOCK_INFO(the_obj->vt->clss)->number_of_blocks == 1);
#endif
    // Is it a pointer into an area being collected.
    if (GC_BLOCK_INFO(*ref)->c_area_p) {
        // Yes so put it in the root_set.
        p_root_set->add_entry(ref);    
        p_verify_root_set->add_entry(ref);
    } else {
        assert(0);
    }
}

// end file gc\mrl_gc_v1.cpp
