// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/generation.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//


#ifndef _generation_H_
#define _generation_H_


//
// The MRL GC may have a variable number of generations, from zero
// up to a maximum limit. In the current implementation (Mrl_Gc_V1)
// there are three types of generations, related by inheritance as
// follows.
//
//                Generation
//                    |
//                    |
//       ---------------------------
//       |                         |
//       |                         |
//    Step_Generation              |
//       |                         |
//       |                  Train_Generation
//       |
//  Step_Plus_Nursery_Generation
//


#include "platform.h"
#include "gc_space.h"
#include "Block_Store.h"
#include "remembered_set.h"
#include "scan_set.h"
#include "gc_component.h"
#include "gc_globals.h"

#define MAXIMUM_STEPS 64

class Card_Table;
class Nursery;
class Gc_Interface;
class Step;
class Gc_Fast_Hooks;
class Gc_Plan;
class Generation : public Gc_Component {
public:
	Generation(unsigned long generation_number,
               Gc_Fast_Hooks *p_gc_hooks,
               Gc_Plan       *p_gc_plan,
		       Gc_Interface  *p_container,
			   Generation    *p_superior,
               Card_Table    *p_card_table,
		       Block_Store   *p_block_store);
	
	virtual ~Generation() {
        delete _p_pending_scans;
		delete _p_young_to_old_rs;
	}
    //
    // Add an entry to the remembered set recording mature to young
    // references.
    //
    void add_young_to_old_ref(Java_java_lang_Object **pp_ref) {
        assert (*pp_ref != NULL);
        _p_young_to_old_rs->add_entry(pp_ref);
    }

	//
	// Every generation needs to define a policy for handling objects
	// found during cheney scans.
	//
	virtual void cheney_scan_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection) = 0;
	
// #ifdef GC_OFFSET_LOGIC
    //
	// Every generation needs to define a policy for handling objects
	// found during cheney scans.
	//
	virtual void cheney_scan_array_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection) = 0;
//#endif  // GC_OFFSET_LOGIC

	//
	// Our container GC is asking us if there are any cheney scans pending.
	//

    bool cheney_scan_pending() {
        assert (0); // This should be overwritten....
		return false;
	}

    //
    // A reachable object needs to be promoted. Possibly tenured.
    //
	virtual Java_java_lang_Object *p_evict_object(Java_java_lang_Object **pp_obj_ref, bool doing_mos_collection) {
		if (is_object_forwarded(*pp_obj_ref)) {
            
			update_reference_forwarded(pp_obj_ref);
			return *pp_obj_ref;
		} else {
			Gc_Space *p_container =
                    _p_block_store->
                        p_get_object_container(*pp_obj_ref);
            // RLH-TRAIN go ahead and update the pointer.
			*pp_obj_ref =  p_container->p_evict_object(pp_obj_ref, doing_mos_collection);
            return *pp_obj_ref;
		}
	}
  
	//
	// Our container GC is telling us to dispatch all pending cheney scans.
	//
	virtual void execute_pending_cheney_scans(bool doing_mos_collection);

	//
	// The GC interface gives each generation an opportunity to
	// clean up after a collection of that generation.
	//
	virtual void cleanup() {
		cout << "Error - this generation does not need cleanup." << endl;
		orp_exit(1);
	}

	// 
	// A contained space encountered a request for allocation
	// of a large object. 
	//
	virtual Java_java_lang_Object *gc_pinned_malloc(unsigned size, VTable *p_vt,
		                           bool return_null_on_fail,
                                   bool double_align
                                   ) {
		cerr << "Error - this generation doesn't support pinning" << endl;
		assert(0);
		orp_exit(1);
		return NULL;
	}

	virtual Car *p_nominate_focus_car() {
		cout << "Error: illegal for this generation." << endl;
		return NULL;
	}

	virtual Train *p_nominate_focus_train() {
		cout << "Error: illegal for this generation." << endl;
		return NULL;
	}
    //
    // Get the id of this generation
    //
    unsigned long get_gen_num() {
        return _generation_number;
    }
	//
	// Provided with a remembered set of live references into the GC
	// (for all generations), collect this generation and then return
	// the young-to-old references:
	virtual Remembered_Set *p_reclaim_generation(Remembered_Set *p_rs,
                                                 Remembered_Set *p_weak_refs,
		                                         bool is_train_alive) {
		//
		// This will get specialized if legal.
		//
		cout << "Error: illegal for this generation." << endl;
		return NULL;
	}

	//
	// This version is used by younger generations that are adjacent
	// to an elder train generation. Then they get pointers to oldest
	// cars and trains, so that they can focus their collection of
	// young-to-old references to just those appropriate for the from
	// car.
	//
	virtual Remembered_Set *p_reclaim_generation(Remembered_Set *p_rs,
                                                 Remembered_Set *p_weak_refs,
												 bool *p_train_is_alive) {
		//
		// This will be specialized if legal.
		//
		cout << "Error: illegal for this generation." << endl;
		return NULL;
	}

    //
    // When shutting down the ORP, all generations need to
    // be prepared to run all finalizers.
    //
    virtual void run_all_finalizers() = 0;

    // 
    // A space in a contained space is telling us
    // to tenure the object into my superior generation.
	// Generations will specialize as appropriate.
    //
	virtual
    Java_java_lang_Object *p_tenure_object(Java_java_lang_Object **pp_obj) {
		cout << "Error: tenuring not supported by this generation." << endl;
		orp_exit(1);
		return NULL;
	}

	//
	// My subordinate generation is telling me to tenure-in an object.
	//
    virtual
		Java_java_lang_Object *p_import_object_from_yos(Java_java_lang_Object **pp_obj) {
		cout << "Error: importing not supported by this generation." << endl;
		orp_exit(1); // illegal for Step_Plus_Nursery generations
		return NULL;
	}
	//
	// A contained space is telling us that it has a pending cheney scan
	// that needs to be done at some point.Also may need to update the
	// write barrier to reflect an object move from FROM to TO space.
	//
	virtual void 
	reg_pending_cheney_scan(Gc_Space *p_gc_space,
	                        Java_java_lang_Object *pp_obj) {

		if (p_gc_space != _p_space_being_scanned) {
			//
			// If the space is already in the process of being
			// scanned, don't re-register it for scanning.
			//
			_p_pending_scans->add_entry(p_gc_space);
		}
	}
    //
    // Confirm that I'm indeed a generation.
    //
    virtual bool is_generation() {
        return true;
    }
    //
    // Is specified address in a space that this generation contains?
    //
    inline bool is_object_in_my_generation(Java_java_lang_Object *p_obj) {
        return is_address_in_my_generation((void *)p_obj);
    }
	inline bool is_object_not_in_my_generation(Java_java_lang_Object *p_obj) {
		if (is_object_in_my_generation(p_obj)) {
			return false;
		}
		return true;
	}

    //
    // Is specified reference in a space that this generation contains?
    //
    inline bool is_reference_in_my_generation(Java_java_lang_Object **pp_obj_ref) {
        return is_address_in_my_generation((void *)pp_obj_ref);
    }
	//
	// Is specified object in a space that this generation contains.
	//
#if (GC_DEBUG>0) // inline? <<<<<<<<<THIS NEEDS TO BE SPED UP A LOT>>>>>>>>>>
    bool is_address_in_my_generation(void *p_addr);
#else
    inline bool is_address_in_my_generation(void *p_addr){
#if 1 // speedup
    int gen_encoding = _p_block_store->get_address_generation (p_addr);
    bool result = ( _generation_number == (unsigned long)gen_encoding );

    return result;
       // otherwise calculate it the slow way and make sure we have the correct
       // answer.
#endif // speedup.

    if (_p_block_store->is_address_outside_heap(p_addr)) {
//        assert (0); // how is this possible.
        return false;
	}

    Generation *p_gen = 
            _p_block_store->p_get_address_generation(p_addr);

	unsigned long gen_num =
		p_gen->get_generation_number();

    if (gen_num == _generation_number) {
		return true;
    }

    return false;
}

#endif // GC_DEBUG>0 inline

	//
	// Return my unique generation number.
	//
	unsigned long get_generation_number(void) { 
        return _generation_number; 
    }
	//
	// Overload operator== to compare generations of objects.
	//
	bool operator==(Generation &generation2) {
		if (_generation_number == generation2.get_generation_number())
        	return true;

		return false;
	}

	bool operator<(Generation &generation2) {
		if (_generation_number < generation2.get_generation_number())
        	return true;

		return false;
	}

	bool operator>(Generation &generation2) {
		if (_generation_number > generation2.get_generation_number())
        	return true;

		return false;
	}

	//
	// Routines for introspection.
	//

    virtual bool is_train_generation() = 0;

    virtual bool is_step_generation() = 0;

    virtual bool is_step_plus_nursery_generation() = 0;

#if (GC_DEBUG>3)
	//
	// Routine to display the contents of the generation.
	//
	virtual void inspect(unsigned int level) = 0;
#endif // _DEBUG

	//
	// Some generations have the ability to supply nurseries.
	//
	virtual Nursery *p_cycle_nursery(Nursery *p_nursery,
		                             bool returnNullOnFail) = 0;
#ifndef PER_CAR_BARRIERS
	//
	// The GC Interface is asking us to selectively purge some write
	// barrier entries, most probably because some car just got reclaimed.
	//
	virtual void selective_flush_write_barrier(Car *p_car) {
		//
		// This better be specialized.
		//
		orp_exit(1);
	}
#endif
    //
    // The GC interface is processing (hybrid) write barriers and
    // has found an entry that may be of interest to this generation.
    //
    virtual void 
		add_entry_to_generation_write_barriers(Java_java_lang_Object **pp_obj_ref,
                                               Java_java_lang_Object *p_obj_ref) = 0;

    //
    // make sure this write barrier is consistent.
    //
    virtual void 
        verify_old_to_young_write_barrier () {
#if (GC_DEBUG>3)
        _p_old_to_young_write_barrier->verify_no_obsolete_references ();
#endif
    }
    //
	// An entry was found to be stale. Purge it. Needs to be specialized by
    // generations which need it.
	//
	virtual void
		purge_write_barrier_remembered_set(Java_java_lang_Object **pp_obj_ref) {
		cout << "Error: purge illegal for this generation" << endl;
		orp_exit(1);
	}

#if (GC_DEBUG>3)
	//
	// Routine to verify the consistency of all the spaces of this generation.
	//
	virtual bool verify_generation() = 0;
#endif // _DEBUG
    
    Remembered_Set *get_p_old_to_young_remembered_set () {
        return _p_old_to_young_remembered_set;
    }

	virtual void enqueue_soft_ref (java_lang_ref_Reference *soft_reference) = 0;
	virtual void enqueue_weak_ref (java_lang_ref_Reference *weak_reference) = 0;
	virtual void enqueue_phantom_ref (java_lang_ref_Reference *phantom_reference) = 0;

protected:

    Card_Table *_p_card_table;
	//
	// My container is an object which implements a GC Interface.
	// (A Garbage Collector for some ORP.)
	//
    Gc_Interface *_p_container;
	//
	// My unique generation number.
	//
    unsigned long _generation_number;

#ifndef PER_CAR_BARRIERS
    //
    // Repository for incoming pointers from older generations.
    //
    Remembered_Set *_p_old_to_young_remembered_set;
#endif


	//
	// During collection, this hash table records the steps (or cars)
	// which have pending cheney scan operations.
	//
	Scan_Set *_p_pending_scans;
    //
    // The plan object which specifies the gc parameters.
    //
    Gc_Plan *_p_gc_plan;

	//
	// Location to record GC space that is being cheney scanned,
	// so that duplicate cheney scan requests don't get registered,
	// and recursive cheney scans don't get attempted.
	//
	Gc_Space *_p_space_being_scanned;
	//
	// Pointer to the upper generation I target object tenuring at.
	//
    Generation *_p_superior_generation;
	//
	// Repository for the pointers into older spaces.
	//
	Remembered_Set *_p_young_to_old_rs;

private:

    Generation();
};

#endif // _generation_H_

