// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/class_loader/String_Pool.cpp,v 1.1.1.1 2001/07/23 07:25:38 xli18 Exp $
//


#include "platform.h"
#include <iostream.h>
#include <assert.h>
#include "String_Pool.h"
#include "orp_synch.h"

// In order to make this code thread safe a light weight lock is used.
// The protocol is simple and the lock is local to this string pool.
volatile POINTER_SIZE_INT string_pool_lock; // 1 is locked, 0 is unlocked

#ifdef POINTER64
// InterlockCompareExchange should result in an .acq on IA64.
void String_Pool::lock_pool ()
{
    // Spin until lock is free.
    while (InterlockedCompareExchangePointer ((PVOID *)&string_pool_lock, (PVOID)1, (PVOID)0) != 0) {
        Sleep (0);
    }
}
#else

// InterlockCompareExchange should result in an .acq on IA64.
void String_Pool::lock_pool ()
{
    // Spin until lock is free.
    while (InterlockedCompareExchange ((PVOID *)&string_pool_lock, (PVOID)1, (PVOID)0) != 0) {
        Sleep (0);
    }
}
#endif

// Release lock. string_pool_lock is volatile which results in a st.rel on IA64
void String_Pool::unlock_pool ()
{
    //assert (lock_pool != 0);
    string_pool_lock = 0;
}
//
#ifdef _DEBUG
// #define STRING_POOL_TRACE
#endif

#ifdef STRING_POOL_TRACE
static ostream& trace = cerr;
#endif

String_Pool::String_Pool()
{
    lock_pool();
	//
	// allocate the string table and zero it out
	//
	unsigned table_size = sizeof(_Entry*) * STRING_TABLE_SIZE;
	memset(_table,0,table_size);
	//
	// allocate initial arena for string pool
	//
	_arena = alloc_arena(_arena, INITIAL_STRING_ARENA_SIZE);
    unlock_pool();
} //String_Pool::String_Pool



String *String_Pool::lookup(const char *s)
{
	//
	// compute hash
	//
	unsigned short hash = 0;
	unsigned short len = 0;
	const char *t = s;
	unsigned char c;
    unsigned char h1 = 0, h2 = 0;
    while ((c = *t++) != '\0') {
        h1 += c;
        if((c = *t++) == 0) {
            break;
        }
        h2 += c;
    }
    hash = h1 + (h2 << 8);
    len = (t - s) - 1;
	hash = hash % STRING_TABLE_SIZE;

    lock_pool();
	//
	// search bucket for string
	//
	_Entry *e;
	for (e = _table[hash]; e != NULL; e = e->next) {
		if (e->str.len == len && strcmp(s, e->str.bytes) == 0) {
			//
			// found string in table
			//
#ifdef STRING_POOL_TRACE
			trace << "String_Pool::lookup: found " << s 
				  << " @ index " << hash << endl;
#endif
            unlock_pool();
			return	&e->str;
		}
	}
	//
	// string not in table; insert a new string entry into string pool
	//
	// compute size of _Entry record
	// add one to str_len for '\0'
	// subtract 2 for bytes[2] already in _Entry
	//
	unsigned entry_size = sizeof(_Entry) + len + 1 - 2;
	//
	// round up to word aligned size
	//

	entry_size = (entry_size + 3) & 0xFFFFFFFC; 

	void *mem;
	if ((mem = arena_alloc_space(_arena,entry_size)) == NULL) {
		//
		// not enough arena space; allocate a new arena
		//
		unsigned arena_size = entry_size > STRING_ARENA_SIZE ? entry_size : STRING_ARENA_SIZE;
#ifdef STRING_POOL_TRACE
		trace << "String_Pool::_new_string: allocating arena of size " << size << endl;
#endif
		_arena = alloc_arena(_arena,arena_size);
		mem = arena_alloc_space(_arena,entry_size);
	}
	_table[hash] = e = new(mem) _Entry(s,len,_table[hash]);

#ifdef STRING_POOL_TRACE
	trace << "String_Pool::lookup: inserting " << s
		  << " @ index " << hash << endl;
#endif
    unlock_pool();
	return &e->str;
} //String_Pool::lookup



String *String_Pool::lookup(const char *s, unsigned len)
{
	//
	// compute hash
	//
	unsigned short hash = 0;
	//unsigned short len = 0;
	const char *t = s;
	const char *eos = s + len;
	char c;
    unsigned char h1 = 0, h2 = 0;
    while (t != eos) {
        c = *t++;
        h1 += c;
        if(t == eos) {
            break;
        }
        c = *t++;
        h2 += c;
    }
    hash = h1 + (h2 << 8);
    //len = length;
	hash = hash % STRING_TABLE_SIZE;

    lock_pool();
    //
	// search bucket for string
	//
	_Entry *e;
	for (e = _table[hash]; e != NULL; e = e->next) {
		if (e->str.len == len && strcmp(s, e->str.bytes) == 0) {
			//
			// found string in table
			//
#ifdef STRING_POOL_TRACE
			trace << "String_Pool::lookup: found " << s 
				  << " @ index " << hash << endl;
#endif
            unlock_pool();
			return	&e->str;
		}
	}
	//
	// string not in table; insert a new string entry into string pool
	//
	// compute size of _Entry record
	// add one to str_len for '\0'
	// subtract 2 for bytes[2] already in _Entry
	//
	unsigned entry_size = sizeof(_Entry) + len + 1 - 2;
	//
	// round up to word aligned size
	//
	entry_size = (entry_size + 3) & 0xFFFFFFFC;
	void *mem;
	if ((mem = arena_alloc_space(_arena,entry_size)) == NULL) {
		//
		// not enough arena space; allocate a new arena
		//
		unsigned arena_size = entry_size > STRING_ARENA_SIZE ? entry_size : STRING_ARENA_SIZE;
#ifdef STRING_POOL_TRACE
		trace << "String_Pool::_new_string: allocating arena of size " << size << endl;
#endif
		_arena = alloc_arena(_arena,arena_size);
		mem = arena_alloc_space(_arena,entry_size);
	}
	_table[hash] = e = new(mem) _Entry(s,len,_table[hash]);
#ifdef STRING_POOL_TRACE
	trace << "String_Pool::lookup: inserting " << s
		  << " @ index " << hash << endl;
#endif
    unlock_pool();
	return &e->str;
} //String_Pool::lookup



// A simple iterator.
// It is used for GC at the moment, but it should be replaced with
// something that would allow GC of classes.
String *String_Pool::get_first_string()
{
    for(unsigned i = 0; i < STRING_TABLE_SIZE; i++) {
        if(_table[i] != NULL)
            return &(_table[i]->str);
    }
    return NULL;
} //String_Pool::get_first_string



String *String_Pool::get_next_string(String *prev)
{
    assert(prev);
    // hash on string name address
	unsigned short hash = 0;
	unsigned short len = 0;
	const char *s = prev->bytes;
	const char *t = s;
	char c;
    unsigned char h1 = 0, h2 = 0;
    while ((c = *t++) != '\0') {
        h1 += c;
        if((c = *t++) == 0) {
            break;
        }
        h2 += c;
    }
    hash = h1 + (h2 << 8);
    len = (t - s) - 1;
	hash = hash % STRING_TABLE_SIZE;

	for(_Entry *e = _table[hash]; e != NULL; e = e->next) {
        if((&(e->str)) == prev) {
            if(e->next) {
                return &(e->next->str);
            } else {
                for(unsigned i = hash + 1; i < STRING_TABLE_SIZE; i++) {
                    if(_table[i] != NULL)
                        return &(_table[i]->str);
                }
                return NULL;
            }
        }
    }
    assert(0);
    return NULL;
} //String_Pool::get_next_string

String *String_Pool::get_first_string_intern(unsigned *cookie)
{
    for (unsigned i = 0; i < STRING_TABLE_SIZE; i++) {
        for (_Entry *e = _table[i]; e != NULL; e = e->next) {
            if (e->str.intern) {
                *cookie = i;
                return &(e->str);
            }
        }
    }
    *cookie = 0;
    return NULL;
}


String *String_Pool::get_next_string_intern(String *prev, unsigned *cookie)
{
    assert(prev);
    unsigned short hash = (short)*cookie;
    for (_Entry *e = _table[hash]; e != NULL; e = e->next) {
        if ((&(e->str)) == prev) {
            for (e = e->next; e != NULL; e = e->next) {
                if (e->str.intern) {
                    // same cookie
                    return &(e->str);
                }
            }
            break;
        }
    }
    for (unsigned i = hash +1; i < STRING_TABLE_SIZE; i++ ) {
        for (_Entry *e = _table[i]; e!= NULL; e= e->next) {
            if (e->str.intern) {
                *cookie = i;
                return &(e->str);
            }
        }
    }
    return NULL;
}


