// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/base_natives/gnu_classpath/java_lang_Math.cpp,v 1.6 2001/11/14 02:21:48 gwu2 Exp $
//


#include "platform.h"
#include "jni.h"
#include "math.h"
#include "assert.h"

#include "java_lang_Math.h"


/*
 * Class:     java_lang_Math
 * Method:    sin
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_sin
  (JNIEnv *, jclass, jdouble d)
{
	return sin(d);
} // Java_java_lang_Math_sin


/*
 * Class:     java_lang_Math
 * Method:    cos
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_cos
  (JNIEnv *, jclass, jdouble d)
{
	return cos(d);
} // Java_java_lang_Math_cos


/*
 * Class:     java_lang_Math
 * Method:    tan
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_tan
  (JNIEnv *, jclass, jdouble d)
{
	return tan(d);
} // Java_java_lang_Math_tan

/*
 * Class:     java_lang_Math
 * Method:    asin
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_asin
  (JNIEnv *, jclass, jdouble d)
{
   return asin(d);
} // Java_java_lang_Math_asin


/*
 * Class:     java_lang_Math
 * Method:    acos
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_acos
  (JNIEnv *, jclass, jdouble d)
{
	return acos(d);
} // Java_java_lang_Math_acos

/*
 * Class:     java_lang_Math
 * Method:    atan
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_atan
  (JNIEnv *, jclass, jdouble d)
{
	return atan(d);
} // Java_java_lang_Math_atan

/*
 * Class:     java_lang_Math
 * Method:    atan2
 * Signature: (DD)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_atan2
  (JNIEnv *, jclass, jdouble y, jdouble x)
{
	return atan2(y, x);
} // Java_java_lang_Math_atan2

/*
 * Class:     java_lang_Math
 * Method:    exp
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_exp
  (JNIEnv *, jclass, jdouble d)
{
	return exp(d);
} // Java_java_lang_Math_exp

/*
 * Class:     java_lang_Math
 * Method:    log
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_log
  (JNIEnv *, jclass, jdouble d)
{
	return log(d);
} // Java_java_lang_Math_log

/*
 * Class:     java_lang_Math
 * Method:    sqrt
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_sqrt
  (JNIEnv *, jclass, jdouble d)
{
	return sqrt(d);
} // Java_java_lang_Math_sqrt

/*
 * Class:     java_lang_Math
 * Method:    pow
 * Signature: (DD)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_pow
  (JNIEnv *, jclass, jdouble x, jdouble y)
{
	if(y < -2)
		int i = 0;
	double dd = pow(x, y);
	union U {
		jlong l;
		jdouble d;
	}u;
	u.d = dd;
	if(u.l == __INT64_C(0x7ff0000000000000) ||
	   u.l == __INT64_C(0xfff0000000000000) )
		dd = pow(x, y);
	return pow(x, y);
} // Java_java_lang_Math_pow


inline jlong castDoubleToLong(jdouble d)
{
	union {
		jlong l;
		jdouble d;
	} U;
	U.d = d;
	return U.l;
} // castDoubleToLong

inline jdouble castLongToDouble(jlong l)
{
	union {
		jlong l;
		jdouble d;
	} U;
	U.l = l;
	return U.d;
} // castLongToDouble

inline bool is_nan(jlong long_bits)
{
	jlong e = (long_bits & __INT64_C(0x7ff0000000000000));
	jlong g = (long_bits & __INT64_C(0x000fffffffffffff));
	
	if ((g != 0) && (e == __INT64_C(0x7ff0000000000000)))
		return true;
	return false;
} // is_nan

/* use isnan instead
#ifdef  ORP_POSIX
int  _isnan(double d)
{
	jlong j = castDoubleToLong(d);
	return is_nan(d);
}
#endif
*/

inline bool is_positive_inf(jlong long_bits)
{
	jlong e = (long_bits & __INT64_C(0x7ff0000000000000));
	
	if (e == __INT64_C(0x7ff0000000000000) )
		return true;
	return false;
} // is_positive_inf

inline bool is_negative_inf(jlong long_bits)
{
	jlong e = (long_bits & __INT64_C(0xfff0000000000000));
	
	if (e == __INT64_C(0xfff0000000000000) )
		return true;

	return false;
} // is_negative_inf

inline bool is_inf(jlong long_bits)
{
	return is_positive_inf(long_bits) || is_negative_inf(long_bits);
}

jdouble get_java_lang_Double_NAN(JNIEnv *jenv)
{
	jclass double_class = jenv->FindClass("java/lang/Double");
	assert(double_class);
	jfieldID NaN_id = jenv->GetStaticFieldID(double_class, "NaN", "D");
	assert(NaN_id);
	jdouble NaN = jenv->GetStaticDoubleField(double_class, NaN_id);
	return NaN;
}

/************************************************************************
 * The result of a Java floating-point remainder operation is determined 
 * by the rules of IEEE arithmetic:
 *
 *   1.If either operand is NaN, the result is NaN. 
 *   2.If the result is not NaN, the sign of the result equals the sign of 
 *     the dividend. 
 *   3.If the dividend is an infinity, or the divisor is a zero, or both, 
 *     the result is NaN. 
 *   4.If the dividend is finite and the divisor is an infinity, the result 
 *     equals the dividend. 
 *   5.If the dividend is a zero and the divisor is finite, the result equals
 *     the dividend. 
 *   6.In the remaining cases, where neither an infinity, nor a zero, nor NaN
 *     is involved, the floating-point remainder r from the division of a 
 *     dividend n by a divisor d is defined by the mathematical relation:
 *         r = n - (d * q)
 *	   where q is the mathematical integer closest to the exact mathematical 
 *     value of the quotient n/d; if two mathematical integers are equally 
 *     close to  then n is the integer that is even. If the remainder is zero, 
 *     its sign is the same as the sign of the first argument. 
 *
 **************************************************************************/

 /*
 * Class:     java_lang_Math
 * Method:    IEEEremainder
 * Signature: (DD)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_IEEEremainder
  (JNIEnv *jenv, jclass jclazz, jdouble x, jdouble y)
{
	// It's wrong to use fmod to implement IEEEremainder, 
	// fmod should be used to implement %
	// double q = fmod(x, y);
	
	jlong nl = castDoubleToLong(x);
	jlong dl = castDoubleToLong(y);
	
	// Step 1:
	if(is_nan(nl) || is_nan(dl))
		return get_java_lang_Double_NAN(jenv);

	// Step 2:
	jlong sign = __INT64_C(0x7fffffffffffffff) | (nl & __INT64_C(0x8000000000000000) );

	// Step 3:
	if( y == 0 || is_inf(nl) )
		return get_java_lang_Double_NAN(jenv);

	// Step 4:
	if( !is_inf(nl) && is_inf(dl) )
		return x;

	// Step 5:
	if( x == 0 && is_inf(dl) )
		return 0;

	// Step 6:
	double q = x / y;
	double q1 = floor(q);
	double q2 = ceil(q);
	double dq1 = q - q1;
	double dq2 = q2 - q;
	if(dq1 > dq2)
		q = q2;
	else
		if(dq1 < dq2)
			q = q1;
		else
			if((jlong)q1 & 0x1 == 0)
				q = q1;
			else 
				q = q2;
	double r = x - q * y;
	if(r == 0){
		jlong rl = castDoubleToLong(r);
		rl &= sign;
		r = castLongToDouble(rl);
	}
			
	return r;
} // Java_java_lang_Math_IEEEremainder

/*
 * Class:     java_lang_Math
 * Method:    ceil
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_ceil
  (JNIEnv *, jclass, jdouble d) 
{
	return ceil(d);
} // Java_java_lang_Math_ceil

/*
 * Class:     java_lang_Math
 * Method:    floor
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_floor
  (JNIEnv *, jclass, jdouble d)
{
	return floor(d);
} // Java_java_lang_Math_floor

/*
 * Class:     java_lang_Math
 * Method:    rint
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_rint
  (JNIEnv *, jclass, jdouble d)
{

	assert(0);
	return 0;
} // Java_java_lang_Math_rint

/*
 * Class:     java_lang_Math
 * Method:    round
 * Signature: (F)I
 */
JNIEXPORT jint JNICALL Java_java_lang_Math_round__F
  (JNIEnv *jenv, jclass, jfloat f)
{  
	assert(0);
	return 0;
} // Java_java_lang_Math_round__F


/*
 * Class:     java_lang_Math
 * Method:    round
 * Signature: (D)J
 */
JNIEXPORT jlong JNICALL Java_java_lang_Math_round__D
  (JNIEnv *jenv, jclass, jdouble d)
{
	assert(0);
	return 0;
} // Java_java_lang_Math_round__D


/*
 * Class:     java_lang_Math
 * Method:    abs
 * Signature: (F)F
 */
JNIEXPORT jfloat JNICALL Java_java_lang_Math_abs__F
  (JNIEnv *, jclass, jfloat f)
{
  return (jfloat) fabsf((float)f);
} // Java_java_lang_Math_abs__F


/*
 * Class:     java_lang_Math
 * Method:    abs
 * Signature: (D)D
 */
JNIEXPORT jdouble JNICALL Java_java_lang_Math_abs__D
  (JNIEnv *, jclass, jdouble d)
{
	return (jdouble) fabs((double)d);
} // Java_java_lang_Math_abs__D



inline jint castFloatToInt(jfloat f)
{
	union {
		jint l;
		jfloat d;
	} U;
	U.d = f;
	return U.l;
} // castFloatToInt

inline bool is_nan(jint int_bits)
{
	jint e = (int_bits & 0x7f800000);
	jint f = (int_bits & 0x007fffff);
	
	if ((f != 0) && (e == 0x7f800000))
		return true;
	return false;
} // is_nan


/*
 * Class:     java_lang_Math
 * Method:    min
 * Signature: (FF)F
 */

JNIEXPORT jfloat JNICALL Java_java_lang_Math_min__FF
  (JNIEnv *jenv, jclass, jfloat f1, jfloat f2)
{
	jint int1 = castFloatToInt(f1);
	jint int2 = castFloatToInt(f2);

	if (is_nan(int1) || is_nan(int2)) {
		// Return NaN
		jclass float_class = jenv->FindClass("java/lang/Float");
		assert(float_class);
		jfieldID NaN_id = jenv->GetStaticFieldID(float_class, "NaN", "F");
		jfloat NaN = jenv->GetStaticFloatField(float_class, NaN_id);
		return NaN;
	}

	if (int1 >= 0) {
		// f1 is positive
		return (int2 >= 0) ? ((int2 > int1) ? f1 : f2) : f2;
	} else { 
		// f1 is negative
		return (int2 >= 0) ? f1 : ((int1 > int2) ? f1 : f2);
	}
} // Java_java_lang_Math_min__FF


/*
 * Class:     java_lang_Math
 * Method:    max
 * Signature: (FF)F
 */

JNIEXPORT jfloat JNICALL Java_java_lang_Math_max__FF
  (JNIEnv *jenv, jclass, jfloat f1, jfloat f2)
{
	jint int1 = castFloatToInt(f1);
	jint int2 = castFloatToInt(f2);

	if (is_nan(int1) || is_nan(int2)) {
		// Return NaN
		jclass float_class = jenv->FindClass("java/lang/Float");
		assert(float_class);
		jfieldID NaN_id = jenv->GetStaticFieldID(float_class, "NaN", "F");
		jfloat NaN = jenv->GetStaticFloatField(float_class, NaN_id);
		return NaN;
	}

	if (int1 >= 0) {
		// f1 is positive
		return (int2 < 0) ? f1 : ((int1 > int2) ? f1 : f2);
	} else { 
		// f1 is negative
		return (int2 >= 0) ? f2 : ((int1 < int2) ? f1 : f2);
	}
} // Java_java_lang_Math_max__FF


/*
 * Class:     java_lang_Math
 * Method:    min
 * Signature: (DD)D
 */

JNIEXPORT jdouble JNICALL Java_java_lang_Math_min__DD
  (JNIEnv *jenv, jclass, jdouble d1, jdouble d2)
{
	jlong long1 = castDoubleToLong(d1);
	jlong long2 = castDoubleToLong(d2);

	if (is_nan(long1) || is_nan(long2)) {
		// Return NaN
		jclass double_class = jenv->FindClass("java/lang/Double");
		assert(double_class);
		jfieldID NaN_id = jenv->GetStaticFieldID(double_class, "NaN", "D");
		jdouble NaN = jenv->GetStaticDoubleField(double_class, NaN_id);
		return NaN;
	}

	if (long1 >= 0) {
		// d1 is positive
		return (long2 >= 0) ? ((long2 > long1) ? d1 : d2) : d2;
	} else { 
		// d1 is negative
		return (long2 >= 0) ? d1 : ((long1 > long2) ? d1 : d2);
	}
} // Java_java_lang_Math_min__DD





/*
 * Class:     java_lang_Math
 * Method:    max
 * Signature: (DD)D
 */

JNIEXPORT jdouble JNICALL Java_java_lang_Math_max__DD
  (JNIEnv *jenv, jclass, jdouble d1, jdouble d2)
{
	jlong long1 = castDoubleToLong(d1);
	jlong long2 = castDoubleToLong(d2);

	if (is_nan(long1) || is_nan(long2)) {
		// Return NaN
		jclass double_class = jenv->FindClass("java/lang/Double");
		assert(double_class);
		jfieldID NaN_id = jenv->GetStaticFieldID(double_class, "NaN", "D");
		jdouble NaN = jenv->GetStaticDoubleField(double_class, NaN_id);
		return NaN;
	}

	if (long1 >= 0) {
		// d1 is positive
		return (long2 >= 0) ? ((long2 > long1) ? d2 : d1) : d1;
	} else { 
		// d1 is negative
		return (long2 >= 0) ? d2 : ((long1 < long2) ? d1 : d2);
	}
} // Java_java_lang_Math_max__DD



