// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/x86/x86.h,v 1.4 2001/12/18 07:59:44 zying1 Exp $
//

#ifndef X86_H
#define X86_H
#include <assert.h>
//
// x86 register numbers
//
enum X86_Reg_No {
	eax_reg=0, ecx_reg=1, edx_reg=2, ebx_reg=3,
	esp_reg=4, ebp_reg=5, esi_reg=6, edi_reg=7,
	n_reg
};
//
// x86 instruction operand sizes: 8,16,32, and 64 bits
//
enum X86_Opnd_Size {
	opnd_8=0, opnd_16, opnd_32, opnd_64,
	n_opnd_size
};
//
// opcodes for alu instructions
//
enum X86_ALU_Opcode {
	add_opc=0,  or_opc=1, adc_opc=2, sbb_opc=3,
	and_opc=4, sub_opc=5, xor_opc=6, cmp_opc=7,
	n_alu
};
//
// opcodes for shift instructions
//
enum X86_Shift_Opcode {
        shld_opc, shrd_opc,
	shl_opc=4, shr_opc=5, sar_opc=7, max_shift_opcode=8
};
//
// opcodes for floating-point instructions
//
enum X86_FP_Opcode {
	fadd_opc=0,  fmul_opc=1,  fcom_opc=2, fcomp_opc=3,
	fsub_opc=4, fsubr_opc=5,  fdiv_opc=6, fdivr_opc=7,
	max_fp_opcode=8
};
//
// integer conditions codes
//
enum X86_CC {
	cc_eq=0,cc_ne,cc_lt,cc_le,cc_gt,cc_ge,
	cc_lz,cc_gez,cc_p,cc_np,n_cc
};
extern X86_CC cc_commute_map[n_cc];
//
// prefix code
//
enum X86_Prefix {
	lock_prefix = 0xF0,
    repnz_prefix = 0xF2,
    repz_prefix = 0xF3, rep_prefix = 0xF3,
    cs_prefix = 0x2E,
    ss_prefix = 0x36,
    ds_prefix = 0x3E,
    es_prefix = 0x26,
    fs_prefix = 0x64,
    gs_prefix = 0x65,
    operand_prefix = 0x66,
    address_prefix = 0x67
};
//
// bitvector mask for callee-saved registers
//
const unsigned callee_saved_mask=(1<<ebx_reg)|(1<<esi_reg)|(1<<edi_reg)|(1<<ebp_reg);
const unsigned callee_saved_esi_mask =(1<<esi_reg);
const unsigned callee_saved_edi_mask =(1<<edi_reg);
const unsigned callee_saved_ebx_mask =(1<<ebx_reg);
const unsigned callee_saved_ebp_mask =(1<<ebp_reg);

const unsigned ALL_X86_CALLER_REGS=((1<<eax_reg) | (1<<ecx_reg) | (1<<edx_reg));
const unsigned ALL_X86_CALLEE_REGS=((1<<ebx_reg) | (1<<ebp_reg) | (1<<esi_reg) | (1<<edi_reg));
const unsigned ALL_X86_BYTE_REGS=  ((1<<eax_reg) | (1<<ecx_reg) | (1<<edx_reg) | (1<<ebx_reg));

extern int is_scratch_x86reg(X86_Reg_No r) ; // eax_reg, ecx_reg, or edx_reg
extern int is_callee_x86reg(X86_Reg_No r);	 // esi_reg, edi_reg, ebx_reg, or ebp_reg

//
// Frame structure:
//
//      +--------------------------------+
//      | in_arg[0]       = var[0]	     |
//      | in_arg[1]	      = var[1]	     |
//      |	      . . .			         |
//      | in_arg[n_arg-1] = var[n_arg-1] |
//      +--------------------------------+
//      |       return IP                |
//      +--------------------------------+
//      |       saved EBP                | <-- frame pointer (EBP)
//      +--------------------------------+
//      |            ...                 |  n_extra
//      +--------------------------------+
//      |	    var[n_arg]	             |
//      |	    var[n_arg+1]             |  local variables area
//      |          . . .                 |
//      |	    var[n_var-1]             | 
//      +--------------------------------+
//      |			                     |
//      |			                     |  
//      |		spill area               | area for spilling mimic stack
//      |			                     |
//      +--------------------------------|
//      |          ebx                   |
//      |          ebp [ESP_Frame only]  |
//      |	       esi                   |  0..3 callee-saved regs
//      |          edi                   | <-- stack pointer (ESP)
//      +--------------------------------+
//      |	stk0	                     |
//      |	stk1	                     |  operand stack area/
//      |	. . .	                     |  out args
//      |	stkn-1	                     |
//      +--------------------------------|
//
//
//

class Frame {
public:
	const unsigned n_args;	// number of incoming args
 	const unsigned n_vars;	// number of local variables (not including in args)
	const unsigned n_extra;	// number of extra words between IP and locals
	const unsigned n_spill; // space for spilling 
	const unsigned n_callee;// callee-save regs
	const X86_Reg_No base_reg;	// base reg: ebp or esp

	void push(unsigned n=1) {n_stack += n;}
	void pop(unsigned n=1)	{n_stack -= n;}
	void clear()			{n_stack = 0;}

	virtual int var_offset(unsigned n) = 0;
	virtual int spill_offset(unsigned s) = 0;
	virtual int extra_offset(unsigned x) = 0;
	virtual int callee_offset(X86_Reg_No reg) = 0;
    virtual int outarg_offset(unsigned n) = 0;
    //
    // return true if var_i and var_j are contiguous in memory
    //
    virtual bool contiguous_loc(unsigned i, unsigned j) = 0;
protected:
	Frame(X86_Reg_No base_reg,
          unsigned   args,
          unsigned   vars,
          unsigned   extra,
          unsigned   spill,
          unsigned   callee)
	  :  base_reg(base_reg), n_args(args), n_vars(vars), n_extra(extra),
         n_spill(spill), n_callee(callee), n_stack(0) {}

	unsigned n_stack;	// number of words in the outgoing/scratch area
};

class ESP_Frame : public Frame {
public:
	ESP_Frame(unsigned args,unsigned vars,unsigned extra,unsigned spill,unsigned callee) 
		: Frame(esp_reg,args,vars,extra,spill,callee) {}
	int var_offset(unsigned n) {
		unsigned offset = n_extra + n_stack + n_callee + n_spill + n_vars + n_args - n;
		if (n >= n_args)
			// not an incoming argument variable; subtract out n_extra and saved IP
			offset -= n_extra + 1;
		return (offset << 2);
	}
	int spill_offset(unsigned n) {
		unsigned offset = n_stack + n_callee + n_spill - n - 1;
		return (offset << 2);
	}
	int extra_offset(unsigned x) {
		unsigned offset = n_stack + n_callee + n_spill + n_vars + n_extra - x - 1;
		return (offset << 2);
	}
	int callee_offset(X86_Reg_No reg) {
		unsigned offset = n_stack + n_callee - 1;
		if (reg == ebx_reg)
			return offset << 2;
		if (reg == ebp_reg)
			return (offset - 1) << 2;
		if (reg == esi_reg)
			return (offset - 2) << 2;
		if (reg == edi_reg)
			return (offset - 3) << 2;
		cout << reg << " is not a callee-saved reg" << endl;
		assert(0);
		return 0;
	}
    int outarg_offset(unsigned n) {
        assert(n < n_stack);
        unsigned offset = n_stack - n - 1;
        return (offset << 2);
    }
    bool contiguous_loc(unsigned i, unsigned j) {
        return var_offset(j)-var_offset(i) == sizeof(int);
    }
};

class EBP_Frame : public Frame {
public:
	EBP_Frame(unsigned args,unsigned vars,unsigned extra,unsigned spill,unsigned callee)
		: Frame(ebp_reg,args,vars,extra,spill,callee) {}
	int var_offset(unsigned n)  {
		if (n < n_args) {
			n = ((n_args-n)+1);
		} else {
			n = ((n_args-n)-1-n_extra);
		}
		return (n << 2);
	}
	int spill_offset(unsigned depth) {
		return 0 -((n_vars + n_extra + depth + 1) << 2);
	}
	int extra_offset(unsigned x) {
		return 0 - ((x + 1) << 2);
	}
	int callee_offset(X86_Reg_No reg) {
		unsigned offset = 0 - (n_extra + n_vars  + n_spill + 1);
		if (reg == ebx_reg) 
			return offset << 2;
		if (reg == ebp_reg)
			return 0;
		if (reg == esi_reg)
			return (offset - 1) << 2;
		if (reg == edi_reg)
			return (offset - 2) << 2;
		cout << reg << " is not a callee-saved reg" << endl;
		assert(0);
		return 0;
	}
    int outarg_offset(unsigned n) {
        assert(n < n_stack);
        int offset = 0 - (n_extra + n_vars + n_spill + n_callee + n + 1);
        return (offset << 2);
    }
    bool contiguous_loc(unsigned i, unsigned j) {
        return var_offset(i)-var_offset(j) == sizeof(int);
    }
};

//
// floating-point register stack
//
class FP_Stack {
public:
	FP_Stack() : _top(0) {}
	enum {size=8};
	unsigned offset(unsigned i) const {return _top-i;}
	void push() {_top++;}
	void pop()	{_top--;}
	unsigned top()	const {return _top;}
	int is_empty()  const {return _top==0;}
	int is_full()	const {return _top==size;}
private:
	unsigned _top;
};
//
// an x86 instruction operand
//
class Opnd {
public:
	enum Tag {Imm,Reg,Mem,FP};
	const Tag tag;
	Opnd(Tag t) : tag(t) {}
	virtual void print(ostream& cout,X86_Opnd_Size=opnd_32) const =0;
};
class FP_Opnd : public Opnd {
public:
	FP_Opnd(unsigned n) : Opnd(FP), _reg_no(n) {}
	void print(ostream& cout,X86_Opnd_Size=opnd_32) const {
		cout << "st(" << _reg_no << ")";
	}
	unsigned const reg_no() {return _reg_no;}
private:
	unsigned _reg_no;
};
//
// an immediate operand
//
class Imm_Opnd : public Opnd {
public:
	Imm_Opnd(unsigned val) : Opnd(Imm) {value = val;}
	char *emit32(char *inst) const {
		inst[0] = bytes[0];
		inst[1] = bytes[1];
		inst[2] = bytes[2];
		inst[3] = bytes[3];
		return inst+4;
	}
	char *emit8(char *inst) const {
		*inst = (unsigned char)(value & 0xff);
		return inst+1;
	}
	char *emit16(char *inst) const {
		inst[0] = (unsigned char)(value & 0xff);
		inst[1] = (unsigned char)((value>>8) & 0xff);
		return inst+2;
	}
	int is_imm8() const {
		return ((int)value >= -128 &&
				(int)value <= 127);
	}
    int is_imm16() const {
        return ((int)value >= -(1<<16) &&
                (int)value <= ((1<<16)-1));
    }
	void print(ostream& cout,X86_Opnd_Size=opnd_32) const {
		cout << value;
	}
	union {
		// unsigned value;
		int value;
		unsigned char bytes[4];
	};
};

class RM_Opnd : public Opnd {
public:
	RM_Opnd(Tag t) : Opnd(t) {}
	virtual char *emit(char *,unsigned r) const = 0;
	virtual void print(ostream&,X86_Opnd_Size=opnd_32) const = 0;
	virtual int is_eax_reg()	const {return 0;}
	virtual int is_reg()		const {return 0;}
};

class R_Opnd : public RM_Opnd {
public:
	R_Opnd(X86_Reg_No r) : _reg_no(r), RM_Opnd(Reg) {}
	char *emit(char*,unsigned) const;
	void print(ostream&,X86_Opnd_Size) const;
	int is_eax_reg()	const {return _reg_no == eax_reg;}
	int is_reg()		const {return 1;}
	X86_Reg_No reg_no()	const {return _reg_no;}
private:
//	const X86_Reg_No _reg_no;
	X86_Reg_No _reg_no;
};
//
//  a memory operand with displacement
//
class M_Opnd : public RM_Opnd {
public:
	M_Opnd(unsigned i) : RM_Opnd(Mem), disp(i) {}
	virtual char *emit(char*,unsigned) const;
	virtual void print(ostream&,X86_Opnd_Size) const;
	int off() {return disp.value;}
//protected:
	Imm_Opnd	disp;
};
//
//  a memory operand with base register and displacement
//
class M_Base_Opnd : public M_Opnd {
public:
	M_Base_Opnd(X86_Reg_No r,unsigned i) : M_Opnd(i), base_reg(r) {}
	virtual char *emit(char*,unsigned) const;
	virtual void print(ostream&,X86_Opnd_Size) const;
//	const X86_Reg_No base_reg;
	X86_Reg_No base_reg;
};
//
//  a memory operand with base register, scaled index register
//  and displacement.
//
class M_Index_Opnd : public M_Base_Opnd {
public:
	M_Index_Opnd(X86_Reg_No b,X86_Reg_No i,unsigned d,unsigned s) :
	  M_Base_Opnd(b,d), index_reg(i), shift_amount(s) {}
	char *emit(char*,unsigned) const;
	void print(ostream&,X86_Opnd_Size) const;
//	const X86_Reg_No index_reg;
//	const unsigned shift_amount;
	X86_Reg_No index_reg;
	unsigned shift_amount;
};

class M_Var_Opnd : public M_Base_Opnd {
public:
	M_Var_Opnd(Frame& frame,unsigned var_no) :
	  M_Base_Opnd(frame.base_reg,frame.var_offset(var_no)) {}
};

class M_Spill_Opnd : public M_Base_Opnd {
public:
	M_Spill_Opnd(Frame& frame,unsigned depth) :
	  M_Base_Opnd(frame.base_reg,frame.spill_offset(depth)) {}
};

//
// operand structures for x86 registers
//
extern R_Opnd ax_opnd;
extern R_Opnd al_opnd;
extern R_Opnd eax_opnd;
extern R_Opnd ecx_opnd;
extern R_Opnd edx_opnd;
extern R_Opnd ebx_opnd;
extern R_Opnd esp_opnd;
extern R_Opnd ebp_opnd;
extern R_Opnd esi_opnd;
extern R_Opnd edi_opnd;

//
// stack top operand
//
extern M_Base_Opnd stack_top_opnd;
extern M_Base_Opnd stack_top_opnd_1;
extern M_Base_Opnd stack_top_opnd_2;
extern M_Base_Opnd stack_top_opnd_3;
extern M_Base_Opnd stack_top_opnd_4;

//
// X86 reg string
//
extern char* X86_Reg_Str[n_reg+1];

////////////////////////////////////////////////////////////////////////////////
//
// prefix
//
////////////////////////////////////////////////////////////////////////////////
char *prefix(char *inst,const X86_Prefix p);
////////////////////////////////////////////////////////////////////////////////
//
// cmpxchg or xchg
//
////////////////////////////////////////////////////////////////////////////////
char *cmpxchg(char *inst,const RM_Opnd *rm,const R_Opnd *r);
char *cmpxchg(char *inst,const RM_Opnd *rm,const R_Opnd *r,X86_Opnd_Size sz);
char *xchg(char *inst,const RM_Opnd *rm,const R_Opnd *r,X86_Opnd_Size sz);
////////////////////////////////////////////////////////////////////////////////
//
// inc(rement), dec(rement), not, neg(ate) instructions
//
////////////////////////////////////////////////////////////////////////////////
char *inc(char *inst,const M_Opnd *m);
char *inc(char *inst,const R_Opnd *r);
char *dec(char *inst,const M_Opnd *m);
char *dec(char *inst,const R_Opnd *r);
char *_not(char *inst,const RM_Opnd *rm);
char *neg(char *inst,const RM_Opnd *rm);
char *nop(char *inst);
////////////////////////////////////////////////////////////////////////////////
//
// alu instructions: add, or, adc, sbb, and, sub, xor, cmp
//
////////////////////////////////////////////////////////////////////////////////
char *alu(char *inst,X86_ALU_Opcode opc,const RM_Opnd *rm,const Imm_Opnd *imm,bool is_lo_reg=true);
char *alu(char *inst,X86_ALU_Opcode opc,const M_Opnd *m,const R_Opnd *r);
char *alu(char *inst,X86_ALU_Opcode opc,const R_Opnd *r,const RM_Opnd *rm);
char *alu(char *inst,X86_ALU_Opcode opc,const R_Opnd *r,const RM_Opnd *rm,X86_Opnd_Size sz);
////////////////////////////////////////////////////////////////////////////////
//
// test instruction
//
////////////////////////////////////////////////////////////////////////////////
char *test(char *inst,const RM_Opnd *rm,const Imm_Opnd *imm);
char *test(char *inst,const RM_Opnd *rm,const R_Opnd *r);
////////////////////////////////////////////////////////////////////////////////
//
// shift instructions: shl, shr, sar, shld, shrd
//
////////////////////////////////////////////////////////////////////////////////
char *shift(char *inst,X86_Shift_Opcode opc,const RM_Opnd *rm,const Imm_Opnd *imm);
char *shift(char *inst,X86_Shift_Opcode opc,const RM_Opnd *rm);
char *shift(char *inst,X86_Shift_Opcode opc,const RM_Opnd *rm,const R_Opnd *r,const Imm_Opnd *imm);
char *shift(char *inst,X86_Shift_Opcode opc,const RM_Opnd *rm,const R_Opnd *r);
////////////////////////////////////////////////////////////////////////////////
//
// multiply instructions: mul, imul
//
////////////////////////////////////////////////////////////////////////////////
char *mul(char *inst,const RM_Opnd *rm,int is_signed);
char *imul(char *inst,const R_Opnd *r,const RM_Opnd *rm);
char *imul(char *inst,const R_Opnd *r,const Imm_Opnd *imm);
char *imul(char *inst,const R_Opnd *r,const RM_Opnd *rm,const Imm_Opnd *imm);
////////////////////////////////////////////////////////////////////////////////
//
// divide instructions: div, idiv
//
////////////////////////////////////////////////////////////////////////////////
char *div(char *inst,const RM_Opnd *rm,int is_signed);
////////////////////////////////////////////////////////////////////////////////
//
// data movement: mov
//
////////////////////////////////////////////////////////////////////////////////
char *mov(char *inst,const M_Opnd *m,const R_Opnd *r,X86_Opnd_Size=opnd_32,bool is_lo_reg=true);
char *mov(char *inst,const R_Opnd *r,const RM_Opnd *rm,X86_Opnd_Size=opnd_32);
char *mov(char *inst,const R_Opnd *r,const Imm_Opnd *imm);
char *mov(char *inst,const M_Opnd *m,const Imm_Opnd *imm,X86_Opnd_Size=opnd_32);
char *mov_imm32(char *inst,const R_Opnd *r,const unsigned imm32);
char *cmov(char *inst, X86_CC cc, unsigned is_signed, const R_Opnd *r,const RM_Opnd *rm);
////////////////////////////////////////////////////////////////////////////////
//
// load effective address: lea
//
////////////////////////////////////////////////////////////////////////////////
char *lea(char *inst,const R_Opnd *r,const M_Opnd *m);
char *cdq(char *inst);
char *wait(char *inst);
////////////////////////////////////////////////////////////////////////////////
//
// conversions, i.e., widening instructions
//
////////////////////////////////////////////////////////////////////////////////
char *widen(char *inst,const R_Opnd *r,const RM_Opnd *rm,
			unsigned is_signed,unsigned is_half);
////////////////////////////////////////////////////////////////////////////////
//
// floating-point instructions
//
////////////////////////////////////////////////////////////////////////////////
//
//		st(0) = st(0) fp_op m{32,64}real
//
char *fp_op_mem(char *inst,X86_FP_Opcode opc,const M_Opnd *mem,int is_double);
//
//		st(0) = st(0) fp_op st(i)
//
char *fp_op(char *inst,X86_FP_Opcode opc,unsigned i);
//
//		st(i) = st(i) fp_op st(0)	; optionally pop stack
//
char *fp_op(char *inst,X86_FP_Opcode opc,unsigned i,unsigned pop_stk);

//
// Pops the floating point top of stack after copying
// values from location i to top of stack. Takes 1 clock
char *fstp(char *inst,unsigned i);

//
//		compare st(0),st(1) and pop stack twice
//
char *fcompp(char *inst);
char *fnstsw(char *inst);
char *fnstcw(char *inst,const M_Opnd *mem);
char *fldcw(char *inst,const M_Opnd *mem);
char *fchs(char *inst);
char *frem(char *inst);
char *fxch(char *inst,unsigned i);
char *fcomip(char *inst, unsigned i);
//
// load from memory (as fp) into fp register stack
//
char *fld(char *inst,const M_Opnd *mem,int is_double);
char *fld80(char *inst,const M_Opnd *mem);
//
// load from memory (as int) into fp register stack
//
char *fild(char *inst,const M_Opnd *mem,int is_long);
//
// push st(i) onto fp register stack
//
char *fld(char *inst,unsigned i);
//
// push the constants 0.0 and 1.0 onto the fp register stack
//
char *fldz(char *inst);
char *fld1(char *inst);
//
// store stack to to memory (as fp), optionally popping the stack
//
char *fst(char *inst,const M_Opnd *mem,int is_double,unsigned pop_stk);
//
// store stack to memory (as int), always popping the stack
//
char *fist_pop(char *inst,const M_Opnd *mem,int is_long);
////////////////////////////////////////////////////////////////////////////////
//
// stack push and pop instructions
//
////////////////////////////////////////////////////////////////////////////////
char *push(char *inst,const M_Opnd *rm);
char *push(char *inst,const Imm_Opnd *imm);
char *push(char *inst,const R_Opnd *r);
char *pop(char *inst,const R_Opnd *r);
char *pop(char *inst,const M_Opnd *mem);
char *pushad(char *inst);
char *pushfd(char *inst);
char *popad(char *inst);
char *popfd(char *inst);
////////////////////////////////////////////////////////////////////////////////
//
// control-flow instructions
//
////////////////////////////////////////////////////////////////////////////////
//
// jump with 32-bit relative
//
char *jump32(char *inst,const Imm_Opnd *imm);
//
// jump with 8-bit relative
//
char *jump8(char *inst,const Imm_Opnd *imm);
//
// register indirect jump
//
char *jump(char *inst,const RM_Opnd *rm);
//
// jump to target address
//
char *jump(char *inst,char *target);
//
// jump with displacement
//
char *jump(char *inst,int disp);
//
// conditional branch with 8-bit branch offset
//
char *branch8(char *inst,X86_CC cc,const Imm_Opnd *imm,unsigned is_signed);
//
// conditional branch with 32-bit branch offset
//
char *branch32(char *inst,X86_CC cc,const Imm_Opnd *imm,unsigned is_signed);
//
// conditional branch with target label address
//
char *branch(char *inst,X86_CC cc,char *target,unsigned is_signed);
//
// conditional branch with displacement immediate
//
char *branch(char *inst,X86_CC cc,int disp,unsigned is_signed);
//
// call with displacement
//
char *call(char *inst,const Imm_Opnd *imm);
//
// indirect call through register or memory location
//
char *call(char *inst,const RM_Opnd *rm);
//
// call target address
//
char *call(char *inst,char *target);
//
// return instruction
//
char *ret(char *inst);
char *ret(char *inst,Imm_Opnd *imm);
////////////////////////////////////////////////////////////////////////////////
//
// stack frame allocation instructions: enter & leave
//
////////////////////////////////////////////////////////////////////////////////
//
//	enter frame_size 
//
//	is equivalent to:
//
//	push	ebp
//	mov		ebp,esp
//	sub		esp,frame_size
//
char *enter(char *inst,const Imm_Opnd *imm);
//
// leave
//
// is equivalent to:
//
// mov		esp,ebp
// pop		ebp
//
char *leave(char *inst);
//
// sahf  loads SF, ZF, AF, PF, and CF flags from eax
//
char *sahf(char *inst);

//
// Intrinsic FP math functions
//
char *math_fsin(char *inst);
char *math_fcos(char *inst);
char *math_fabs(char *inst);
char *math_fpatan(char *inst);
char *math_fprem(char *inst);
char *math_fprem1(char *inst);
char *math_frndint(char *inst);
char *math_fsqrt(char *inst);
char *math_fptan(char *inst);

//
// Add 1-7 bytes padding, with as few instructions as possible,
// with no effect on the processor state (e.g., registers, flags)
//
char *padding(char *inst, unsigned num);
////////////////////////////////////////////////////////////////////////////////
//
// prolog and epilog code generation
//
////////////////////////////////////////////////////////////////////////////////
char *prolog(char *inst,unsigned frame_size,unsigned reg_save_mask);
char *epilog(char *inst,unsigned reg_save_mask);

extern R_Opnd reg_operand_array[];

#endif // X86_H
