// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/opt_branch_folding.cpp,v 1.2 2001/08/13 09:52:49 xhshi Exp $
//



#include "defines.h"
#include <iostream.h>
#include <assert.h>
#include "Mem_Manager.h"
#include "ir.h"
#include "expression.h"
#include "flow_graph.h"

//
// return the target that is never reached (dead)
//
static Cfg_Node *check_eq_ne(Inst_Exp *exp, 
                             bool     test_eq,
                             Cfg_Node *fallthrough,
                             Cfg_Node *branchtarget) {
    Exp *left = exp->left_child();
    if (exp->op == Exp::Test) { // compare against zero
        if (!left->is_imm_exp()) return NULL;
        unsigned imm = ((Imm_Operand*)((Operand_Exp*)left)->opnd)->imm();
        if ((test_eq && imm == 0) ||    // branch is always taken
            (!test_eq && imm != 0))
            return fallthrough;         // fall-through is never taken
        else                            // fall-through is always taken
            return branchtarget;        // branch is never taken
    } else if (exp->op == Exp::Compare) {
        if (test_eq && left == exp->rght_child() &&
            !(left->is_opnd_exp() && ((Operand_Exp *)left)->opnd->is_ret()))
            return fallthrough;
    }
    return NULL;
}

class Fold_Branch_Closure : public Closure
{
public:
    Fold_Branch_Closure(bool p, Flow_Graph *f) : need_to_prune(p), fg(f) {}
    bool need_to_prune;
    Flow_Graph *fg;
};

//
// we try to fold branches, e.g  eax = 0;
//                               test eax, eax
//                               je   target
// Those cases may happen after inlining or constant/copy propagation.
//
void fold_branch_of(Cfg_Node *node, Closure *cc) {
    // Big hack -- if there's a single out-edge, but it's a conditional
    // branch, just remove the branch instruction.
    Inst *br = node->IR_instruction_list()->prev(); // last instruction
    if (br == node->IR_instruction_list() || !br->is_branch()) return;
    if (node->out_edge_size() == 1)
    {
        br->unlink();
        return;
    }
    if (node->out_edge_size() != 2) return; // only consider branches
    O3_Jit_Type ty = ((Inst_Exp *)br->exp)->left_child()->type;
    if (IS_FP_DBL_TYPE(ty))
        return;
    Fold_Branch_Closure *fc = (Fold_Branch_Closure *) cc;

    //
    // fall-through is not frequently executed
    // change branch condition so that the target becomes fall-through
    //
    Cfg_Node *fallthrough  = node->get_fallthrough();
    Cfg_Node *branchtarget = node->get_branch_target();

    // If the fallthrough branches out of the loop to the epilog, then
    // mark the fallthrough as cold.
    if (false && node->loop_depth() > 0/* && node->loop_header != node*/)
    {
        Cfg_Node *epilog = fc->fg->epilog();
        if (fallthrough == epilog ||
            (fallthrough->out_edge_size() == 1 && fallthrough->out_edges(0) == epilog))
            fallthrough->set_cold_code();
    }
    
    if (fallthrough->is_cold() && ((Branch_Inst*)br)->can_commute()) {
        ((Branch_Inst*)br)->commute_condition();
        Mem_Manager& mm = node->flowgraph->mem_manager;
        Cfg_Node *target = node->out_edges(1);
        node->swap_edges(); // swap two edges
        //
        // find the root flow graph
        //
        Flow_Graph *root = node->flowgraph;
        while (root->calling_fg != NULL) root = root->calling_fg;
        root->set_need_linearization();
    }

    //
    // determine which succ is not dead
    //
    Inst_Exp *bexp = (Inst_Exp*)br->exp;
    Cfg_Node *target = NULL;
    assert(bexp->left_child()->is_inst_exp());
    Inst_Exp *left = (Inst_Exp*)bexp->left_child();
    assert(br->src(0)->is_status_flags());
    Status_Flags *cond = (Status_Flags *) br->src(0);
    switch (bexp->op) {
    case Exp::Beq:
        target = check_eq_ne(left, true, fallthrough, branchtarget);
        break;
    case Exp::Bne:
        target = check_eq_ne(left, false, fallthrough, branchtarget);
        break;
    case Exp::Blt:
        if (cond->is_known())
            target = (cond->is_lt())? fallthrough : branchtarget;
        break;
    case Exp::Ble:
        if (cond->is_known())
            target = (cond->is_lt() || cond->is_eq())? fallthrough : branchtarget;
        break;
    case Exp::Bgt:
        if (cond->is_known())
            target = (cond->is_gt())? fallthrough : branchtarget;
        break;
    case Exp::Bge:
        if (cond->is_known())
            target = (cond->is_gt() || cond->is_eq())? fallthrough : branchtarget;
        break;
    default:
        assert(0);
    }
    //
    // remove edge <node,target>
    //
    if (target != NULL) {
        node->delete_edge(target);
        if (true || target->in_edge_size() == 0)
            fc->need_to_prune = true;
        Inst *cmp = br->prev();
        br->unlink();  // remove the branch from the instruction list
        if (cmp != node->IR_instruction_list() && cmp->is_compare() &&
            cmp->dst() == br->src(0))
            cmp->unlink();
    }
}

void fold_branch(Flow_Graph *fg) {
    //bool need_to_prune = false;
    Fold_Branch_Closure fc(false, fg);
    fg->apply(fold_branch_of, &fc);
    //
    // check if we need to prune the flow graph
    //
    if (fc.need_to_prune)
        fg->prune_unreachable();
}
