// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/fpspill.h,v 1.2 2001/08/13 09:54:55 xhshi Exp $
//

#ifndef _FPspill_h
#define _FPspill_h

#include "ir.h"
#include "expression.h"
#include "flow_graph.h"  // for Closure
#include "bit_vector_group.h"



class Fp_Mimic_Stack
{
public:
    Fp_Mimic_Stack() : _top(0) {}
    enum { size=8 };  // total number of registers
    void reset() { _top = 0; }
    unsigned num_pushed() { return _top; }
    void push(Reg_Operand *opnd, Operand *src1, Operand *src2, Inst *inst,
              Expressions &exprs, Mem_Manager &mm);
    Reg_Operand *pop() { assert(_top > 0); return _array[--_top]; }
    void swap(unsigned idx) {  // swap stk(0) and stk(idx)
        assert(_top > 0 && idx < _top);
        Reg_Operand *tmp = _array[_top-idx-1];
        _array[_top-idx-1] = _array[_top-1];
        _array[_top-1] = tmp;
        bool d = _dirty[_top-idx-1];
        _dirty[_top-idx-1] = _dirty[_top-1];
        _dirty[_top-1] = d;
#ifdef PRINTABLE_O3
        unsigned id = _ids[_top-idx-1];
        _ids[_top-idx-1] = _ids[_top-1];
        _ids[_top-1] = id;
#endif // PRINTABLE_O3
    }
    unsigned find_operand(Reg_Operand *opnd) {  // returns idx where stk(idx)==opnd; -1 on failure
        assert(opnd != NULL);
        unsigned i;
        for (i=_top; i>0; i--)  // return the copy closest to the top of the stack
            if (_array[i-1] == opnd) return _top-i;
        return (unsigned)-1;
    }
    void set_dirty(bool is_dirty) {
        assert(_top > 0);
        _dirty[_top-1] = is_dirty;
    }
    bool is_dirty(unsigned idx) {
        assert(idx < _top);
        return _dirty[_top-idx-1];
    }
    bool is_empty() { return (_top == 0); }
    bool equals(Fp_Mimic_Stack *stk) {
        if (_top != stk->_top)
            return false;
        unsigned i;
        for (i=0; i<_top; i++)
        {
            if (_array[i] != stk->_array[i])
                return false;
        }
        return true;
    }
    bool contains_only_globals() {
        unsigned i;
        for (i=0; i<_top; i++) {
            if (!_array[i]->is_globally_allocated_fp())
                return false;
        }
        return true;
    }
    unsigned num_free() { return size - _top; }
    Reg_Operand *stack_loc(unsigned idx) {
        assert(idx < _top);
        return _array[_top-idx-1];
    }
    void store_and_pop(unsigned dst)
    {
        unsigned realdst = _top-dst-1;
        assert(_top > 0);
        _array[realdst] = _array[_top-1];
        _dirty[realdst] = _dirty[_top-1];
#ifdef PRINTABLE_O3
        _ids[realdst] = _ids[_top-1];
#endif // PRINTABLE_O3
        pop();
    }
    void overwrite(unsigned idx, Reg_Operand *opnd) {
        assert(_top > 0);
        assert(idx < _top);
        unsigned realdst = _top - idx - 1;
        _array[realdst] = opnd;
        _dirty[realdst] = true;
#ifdef PRINTABLE_O3
        _ids[realdst] = (opnd->is_fp_stk() ? -1 : opnd->id);
#endif // PRINTABLE_O3
    }
    void copy_from(Fp_Mimic_Stack *stk) {
        *this = *stk;
    }
    void copy_to(Fp_Mimic_Stack *stk) {
        *stk = *this;
    }
    void spill_top(Inst *inst, Expressions &exprs, Mem_Manager &mm);
    void spill_one(Operand *src1, Operand *src2, Inst *inst,
                   Expressions &exprs, Mem_Manager &mm);
    void convert_to(Fp_Mimic_Stack *target, Inst *inst,
                    Expressions &exprs, Mem_Manager &mm);
    void convert_to_global(Fp_Mimic_Stack *target, Inst *inst,
                           Expressions &exprs, Mem_Manager &mm);
    void remove_operands(Reg_Operand **opnds, unsigned num, Inst *inst,
                         Expressions &exprs, Mem_Manager &mm);
    void remove_operands(Reg_Operand **opnds, unsigned num);
private:
    void spill_specific(unsigned loc, Inst *inst, Expressions &exprs, Mem_Manager &mm);
    unsigned _top;
    Reg_Operand *_array[size];
    bool _dirty[size];
#ifdef PRINTABLE_O3
    unsigned _ids[size];
#endif // PRINTABLE_O3
};

#define NUM_SPILL_ENTRIES 32

struct Spill_Hash_Entry {
    Operand *opnd;
    unsigned regmask;
};
class Spill_Reg_Manager {
    Expressions& _exprs;
    bool _freeze;
public:
    Spill_Hash_Entry *value_in;
    Spill_Reg_Manager(Expressions& exprs) 
        : _exprs(exprs), prev_spill_reg(n_reg), value_in(NULL), _freeze(false) {}
    void reset_all() {
        int i;
        _freeze = false;
        for (i=0; i<NUM_SPILL_ENTRIES; i++)
        {
            value_in[i].opnd = NULL;
            value_in[i].regmask = 0;
        }
        // XXX- We could do a better job at initializing the equivalences.
        for (i=0; i<n_reg; i++)
            _equivalences[i] = (1u << i);
    }
    void init(int edge, Spill_Hash_Entry *init) {
        _freeze = false;
        int i;
        // XXX- We could do a better job at initializing the equivalences.
        for (i=0; i<n_reg; i++)
            _equivalences[i] = (1u << i);
        if (edge == 0) {
            for (i = 0; i < NUM_SPILL_ENTRIES; i++) 
                value_in[i] = init[i];
        } else {
            for (i = 0; i < NUM_SPILL_ENTRIES; i++)
            {
                if (value_in[i].opnd != init[i].opnd || (value_in[i].regmask & init[i].regmask) == 0) {
                    value_in[i].opnd = NULL;
                    value_in[i].regmask = 0;
                }
                else value_in[i].regmask &= init[i].regmask;
            }
        }
    }
    void freeze_spill_manager() { _freeze = true; }
    Operand *lookup(Operand *o) {
        if (_freeze) return NULL;
#ifdef NO_SPILL_MANAGER
        return NULL;
#endif
        assert(o != NULL);
        if (o->kind != Operand::GCTrack)
            return NULL;
        unsigned hash = _hash(o);
        if (value_in[hash].opnd == o)
        {
            unsigned regno = lowest_bit_set(value_in[hash].regmask);
            return _exprs.lookup_reg_exp(regno, o->type, 0)->opnd;
        }
        return NULL;
    }
    void reset_entry(unsigned regno) {
        assert(regno<n_reg);
        unsigned i;
        for (i=0; i<NUM_SPILL_ENTRIES; i++)
        {
#if 0
            if (value_in[i].regmask & (1u << regno))
            {
                value_in[i].regmask &= ~(1u << regno);
                //value_in[i].regmask = 0;
                if (value_in[i].regmask == 0)
                    value_in[i].opnd = NULL;
            }
#else // 0
            if ((value_in[i].regmask &= ~(1u << regno)) == 0)
                value_in[i].opnd = NULL;
#endif // 0
        }
        for (i=0; i<n_reg; i++)
            _equivalences[i] &= ~(1u << regno);
        _equivalences[regno] = (1u << regno);
    }
    void reset_scratch() {
        unsigned mask = ~ALL_X86_CALLER_REGS;
        unsigned i;
        for (i=0; i<NUM_SPILL_ENTRIES; i++)
        {
            if ((value_in[i].regmask &= mask) == 0)
                value_in[i].opnd = NULL;
        }
        for (i=0; i<n_reg; i++)
        {
            if (ALL_X86_CALLER_REGS & (1<<i))
                _equivalences[i] = (1u << i);
            else
                _equivalences[i] &= mask;
        }
    }
    void reset_entry(Operand *d) {
        unsigned preg = d->assigned_preg();
        if (preg == n_reg)
        {
            if (d->kind == Operand::GCTrack)
            {
                unsigned hash = _hash(d);
                if (value_in[hash].opnd == d)
                {
                    value_in[hash].opnd = NULL;
                    value_in[hash].regmask = 0;
                }
            }
        }
        else
            reset_entry(preg);
    }
    // If the form is "eax=t27", set value_in[eax]=t27.  But if _freeze is true,
    // clear value_in[eax].
    // If the form is "t27=eax", clear t27 from all entries.  Then set
    // value_in[eax]=t27. But if _freeze is true, clear value_in[eax].
    // If the form is "eax=ecx", set value_in[eax]=value_in[ecx].  But if
    // _freeze is true, clear value_in[eax].
    void set_entry(Operand *lhs, Operand *rhs)
    {
        if (lhs->is_fp_stk() || rhs->is_fp_stk())
            return;
        unsigned lhs_reg = lhs->assigned_preg();
        unsigned rhs_reg = rhs->assigned_preg();
        if (lhs_reg == n_reg)
        {
            // t27=eax
            assert(rhs_reg != n_reg);
            if (lhs->kind == Operand::GCTrack)
            {
                unsigned hash = _hash(lhs);
                if (_freeze || !lhs->is_reg())
                {
                    if (value_in[hash].opnd == lhs)
                    {
                        value_in[hash].opnd = NULL;
                        value_in[hash].regmask = 0;
                    }
                }
                else
                {
                    value_in[hash].regmask = 0;
                    value_in[hash].opnd = lhs;
                    value_in[hash].regmask |= _equivalences[rhs_reg];
                }
            }
            if (!_freeze && rhs->is_physical_reg())
                prev_spill_reg = rhs_reg;
        }
        else
        {
            if (rhs_reg == n_reg)
            {
                // eax=t27
                reset_entry(lhs_reg);
                if (rhs->kind == Operand::GCTrack)
                {
                    unsigned hash = _hash(rhs);
                    if (_freeze || !rhs->is_reg())
                    {
                        //value_in[hash].opnd = NULL;
                        //value_in[hash].regno = n_reg;
                    }
                    else
                    {
                        if (value_in[hash].opnd != rhs)
                            value_in[hash].regmask = 0;
                        value_in[hash].opnd = rhs;
                        value_in[hash].regmask |= (1u << lhs_reg);
                    }
                }
                if (!_freeze && lhs->is_physical_reg())
                    prev_spill_reg = lhs_reg;
            }
            else
            {
                // eax=ecx
                if (lhs_reg != rhs_reg)
                {
                    reset_entry(lhs_reg);
                    unsigned i;
                    for (i=0; i<NUM_SPILL_ENTRIES; i++)
                    {
                        if (value_in[i].regmask & (1u << rhs_reg))
                            value_in[i].regmask |= (1u << lhs_reg);
                    }
                    for (i=0; i<n_reg; i++)
                    {
                        if (_equivalences[i] & (1u << rhs_reg))
                            _equivalences[i] |= (1u << lhs_reg);
                    }
                    _equivalences[lhs_reg] = _equivalences[rhs_reg];
                }
                //value_in[lhs_reg] = (_freeze ? NULL : value_in[rhs_reg]);
                // Is this the right heuristic?  Use lhs_reg or rhs_reg?
                if (!_freeze && lhs->is_physical_reg())
                    prev_spill_reg = lhs_reg;
            }
        }
    }
    bool is_redundant_assn(Operand *lhs, Operand *rhs) {
        if (lhs->is_fp_stk() || rhs->is_fp_stk())
            return false;
        unsigned lhs_reg = lhs->assigned_preg();
        unsigned rhs_reg = rhs->assigned_preg();
        // Assume they are physical registers for now.
        assert(lhs_reg < n_reg);
        assert(rhs_reg < n_reg);
        return (_equivalences[lhs_reg] == _equivalences[rhs_reg]);
    }
    void reset_after_call() {
#if 0
        unsigned i;
        for (i = 0; i < n_reg; i++)
            if (ALL_X86_CALLER_REGS & (1<<i))
                reset_entry(i);
#else // 0
        reset_scratch();
#endif // 0
        prev_spill_reg = n_reg;
    }
    unsigned get_register(unsigned avail)
    {
        assert(avail);
        unsigned end = prev_spill_reg;
        if (end == n_reg) end = n_reg - 1;
        unsigned i = (end == 0) ? n_reg - 1 : end - 1;
        // Find the highest bit set in the first part of "avail".
        // If nothing, find the highest bit set in all of "avail".
        if (avail & ((1u << (i+1)) - 1))
            return highest_bit_set(avail & ((1u << (i+1)) - 1));
        else
            return highest_bit_set(avail);
    }
    unsigned find_free_reg(unsigned avail)
    {
        avail &= (ALL_X86_CALLEE_REGS | ALL_X86_CALLER_REGS);
        if (!avail)
            return n_reg;
        unsigned fallback = n_reg;
        unsigned i,j;
        for (i=0; i<n_reg; i++)
        {
            if (avail & (1u << i))
            {
                fallback = i;
                if (_equivalences[i] == (1u << i))
                {
                    for (j=0; j<NUM_SPILL_ENTRIES; j++)
                    {
                        if (value_in[j].regmask & (1u << i))
                            break;
                    }
                    if (j >= NUM_SPILL_ENTRIES)
                        break;
                }
            }
        }
#if 0  // Disable the "fallback" policy.
        if (i == n_reg)
            return fallback;
#endif // 0
        return i;
    }
    unsigned find_avail_reg_containing(Operand *opnd, unsigned avail)
    {
        unsigned hash = _hash(opnd);
        unsigned result = n_reg;
        if (value_in[hash].opnd == opnd)
        {
            result = lowest_bit_set(value_in[hash].regmask & avail);
            if (result == -1)
                result = n_reg;
        }
        return result;
    }
    private:
        unsigned prev_spill_reg;
        unsigned char _equivalences[n_reg];
        static int lowest_bit_set(unsigned val)
        {
            static char table[256] =
            {
                   -1, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    7, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
                    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0
            };
            return table[val];
        }
        static int highest_bit_set(unsigned val)
        {
            static char table[256] =
            {
                   -1, 0, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3,
                    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
                    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
                    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
                    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
                    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
                    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
                    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
                    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7
            };
            return table[val];
        }
        static unsigned _hash(Operand *opnd)
        {
            assert(opnd->kind == Operand::GCTrack);
            return ((GCTrack_Operand *)opnd)->id % NUM_SPILL_ENTRIES;
        }
};

class Spill_Closure : public Closure
{
public:
    Spill_Closure(unsigned regs, unsigned callee,
        Expressions &exprs, bvp_homeloc_mapping *bvpmap, Mem_Manager &mm, 
        Spill_Reg_Manager& srm,
        Spill_Hash_Entry *sr,
        char *def,
        Fp_Mimic_Stack &fpstk,
        Fp_Mimic_Stack *pre_fpstacks,
        Fp_Mimic_Stack *post_fpstacks,
        Cfg_Node *epilog,
        bool has_fp)
        : all_available_registers(regs), exprs(exprs), mm(mm), 
        callee(callee), next_home_location(0), bvpmap(bvpmap), spill_manager(srm), 
        spill_regs(sr), def_cnt(def), fpstk(fpstk),
        pre_fpstacks(pre_fpstacks),
        post_fpstacks(post_fpstacks),
        epilog(epilog),
        has_fp(has_fp), fpstk0(exprs.fp_stk_opnd(0)) {
        unsigned i;
        for (i=0; i<n_reg; i++)
            preg_spill_location[i] = NULL;
    }
    unsigned all_available_registers;
    unsigned callee;
    Expressions &exprs;
    Mem_Manager &mm;
    Spill_Reg_Manager& spill_manager;
    int next_home_location;
    Operand *preg_spill_location[n_reg];
    bvp_homeloc_mapping *bvpmap;
    Spill_Hash_Entry *spill_regs;
    char *def_cnt;
    Fp_Mimic_Stack &fpstk;
    Cfg_Node *epilog;
    Fp_Mimic_Stack *pre_fpstacks;
    Fp_Mimic_Stack *post_fpstacks;
    bool has_fp;
    Fp_Stk *fpstk0;
};

Inst *O3_do_fp_spilling(Inst *inst, Spill_Closure *sc, bool should_discard_fpstk0);

#endif // _FPspill_h
