// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/flow_graph_eliminate.cpp,v 1.9 2001/12/27 03:24:51 xhshi Exp $
//



#include "defines.h"
#include <assert.h>
#include "bit_vector.h"
#include "bit_vector_group.h"
#include "flow_graph.h"
#include "expression.h"
#include "handles.h"
#include "inlining.h"

static void remove_call_and_args(Call_Inst *cinst)
{
    unsigned n;
    for (n=0; n<cinst->n_args(); n++)
    {
        Inst *arg = cinst->get_arg(n);
        if (arg->is_push())
            arg->unlink();
    }
    cinst->unlink();
}

// This function actually eliminates the instruction and its arguments,
// and inserts any new instructions required to maintain semantics.
static bool safe_to_eliminate_call(Inst *inst, Expressions &exprs)
{
    Call_Inst *cinst = (Call_Inst *) inst;
    switch (cinst->kind)
    {
    case Call_Inst::llsh_call:
    case Call_Inst::lrsh_call:
    case Call_Inst::lrsz_call:
    case Call_Inst::f2i_call:
    case Call_Inst::d2i_call:
    case Call_Inst::f2l_call:
    case Call_Inst::d2l_call:
    case Call_Inst::lmul_call:
#ifdef ORP_LONG_OPT
	case Call_Inst::lmul_const_multiplier_call:
#endif
    //case Call_Inst::ldiv_call:
    //case Call_Inst::lrem_call:
    case Call_Inst::getstring_call:
        remove_call_and_args(cinst);
        return true;
        break;
    default:
        break;
    }
    Method_Handle mh = cinst->get_mhandle();
    if (mh == NULL)
        return false;
    if (method_get_class(mh) == cached_class_handles[ch_java_lang_String])
    {
        if (mh == cached_method_handles[mh_java_lang_String_indexOf__1]
         || mh == cached_method_handles[mh_java_lang_String_indexOf__2]
         || mh == cached_method_handles[mh_java_lang_String_indexOf__3]
         || mh == cached_method_handles[mh_java_lang_String_indexOf__4]
         )
        {
            unsigned n;
            for (n=0; n<cinst->n_args(); n++)
            {
                Inst *arg = cinst->get_arg(n);
                if (arg->is_push())
                {
                    O3_Jit_Type ty = arg->type();
                    if (n == 1 && (ty == JIT_TYPE_CLASS || ty == JIT_TYPE_ARRAY))
                    {
                        Operand *opnd = arg->src(0);
                        assert(ty == opnd->type);
                        dereference_object(opnd, arg->exp, exprs, arg);
                    }
                    arg->unlink();
                }
            }
            cinst->unlink();
            return true;
        }
    }
    return false;
}

// This routine takes an operand as input, finds the physical register
// associated with it (if there is one), and marks that operand as live.
// If it's an argument or a return value, it also looks at whether the
// associated hi_opnd is set, and does the same with it.
static void mark_operand_live(Operand *opnd, Bit_Vector_Group *live,
                              int is_arg_or_ret_and_was_expanded,
                              Inst *inst)
{
    unsigned varno = opnd->bv_position();
    if (opnd->type == JIT_TYPE_UNKNOWN)
    {
        assert(inst != NULL);
        assert(inst->is_assignment());
        live->mark_live(varno, ((Assign_Inst *)inst)->real_type);
    }
    else
        live->mark_live(varno, opnd->type);
    //::if we use a hi_opnd dierctly, set live its lo.
    if(varno>0 && opnd->is_temp_reg() && opnd->type==JIT_TYPE_LONG && opnd->is_hi()){
        live->mark_live(varno-1, opnd->type) ;
    }
    //::

    if (is_arg_or_ret_and_was_expanded && opnd->hi_opnd() != NULL)
        mark_operand_live(opnd->hi_opnd(), live, is_arg_or_ret_and_was_expanded,
        NULL);
}

static void mark_operand_dead(Operand *opnd, Bit_Vector_Group *live,
                              int is_arg_or_ret_and_was_expanded)
{
    live->mark_dead(opnd->bv_position());
    if (is_arg_or_ret_and_was_expanded && opnd->hi_opnd() != NULL)
        live->mark_dead(opnd->hi_opnd()->bv_position());
}

// Each instruction can have MAX_SRCS source operands, and each source operand
// could contain up to 2 registers (specifically, an array operand).  Also, the
// destination operand could be an array operand.  This means that the
// instruction could be the end of at most 2*(MAX_SRCS+1) live ranges.
//
// For spill code: if opnd->type==JIT_TYPE_UNKNOWN, then use inst->real_type instead,
// for distinguishing whether it's a live ref, nonref, or unknown.  This should
// only happen for reg1=reg2 assignments.
static void liven(Operand *opnd, Expressions &exprs, Bit_Vector_Group *live,
                  Bit_Vector_Group *region_live,
                  Eh_Node *ehnode,
                  Inst *inst, unsigned bitpos,
                  bool has_fp)
{
    if (opnd == NULL)
        return;
    if (opnd->is_reg() || opnd->is_status_flags())
    {
        unsigned varno = opnd->bv_position();
        if (varno == esp_reg)
            return;
        // The live range cannot end here if the variable is live in the
        // exception handler.
        if (!live->is_live_var(varno) &&
            (ehnode == NULL || ehnode->live == NULL || !ehnode->live->is_live_var(varno)))
            inst->live_ranges_ended |= (1u << bitpos);
        mark_operand_live(opnd, live, 0, inst);
        // For a push instruction of a floating point value, we can't push directly from
        // the floating point stack, so we need to force it to be spilled to memory.  We
        // do this by not allowing a push instruction to be marked as the last use of the
        // variable in the region.
        if (has_fp && !inst->is_push())
        {
            if (!region_live->is_live_var(varno))
                inst->region_live_ranges_ended |= (1u << bitpos);
            mark_operand_live(opnd, region_live, 0, inst);
        }
    }
    else
    {
        liven(opnd->base(),  exprs, live, region_live, ehnode,
              inst, bitpos, has_fp);
        liven(opnd->index(), exprs, live, region_live, ehnode,
              inst, bitpos+1, has_fp);
    }
}
static void determine_liveness(Inst *inst, Expressions &exprs, Bit_Vector_Group *live,
                               Bit_Vector_Group *region_live,
                               Operand *sync_this_ptr,
                               Cfg_Node *node,
                               bool has_fp)
{
    unsigned varno;
    Operand *opnd;
    inst->live_ranges_ended = 0;
    inst->region_live_ranges_ended = 0;
    inst->region_dst_live_range_ends = 0;
    inst->set_changes_ret_addr(false);
    inst->unmark_dead();
    //::Patch. Don't kill txx = edx:eax 
    Operand* phy_reg = inst->n_srcs >0 ? inst->src(0) : NULL ;
    if(phy_reg && phy_reg->is_physical_reg() &&
        phy_reg->hi_opnd() && phy_reg->hi_opnd()->is_physical_reg()){
        mark_operand_live(phy_reg, live, inst->been_expanded(), NULL);
    }
    //::end
    // Special case for a call instruction.
    // Kill return value, liven arguments.
    if (inst->is_call())
    {
        Call_Inst *call = (Call_Inst *)inst;
        // Kill the method's return value for an athrow instruction.
        // However, for a synchronized method, we need to keep the "this" pointer alive.
        if (call->kind == Call_Inst::athrow_call || call->kind == Call_Inst::athrow_lazy_call)
        {
            live->init(false);
            if (has_fp)
                region_live->init(false);
            if (sync_this_ptr != NULL)
            {
                live->mark_live(sync_this_ptr->bv_position(), JIT_TYPE_CLASS);
                if (has_fp)
                    region_live->mark_live(sync_this_ptr->bv_position(), JIT_TYPE_CLASS);
            }
        }
        Inst *call_ret = call->get_ret();
        if (call_ret != NULL)
        {
            // call_ret is actually the statement that follows the call statement,
            // which assigns the return value to a temporary register.  The dst()
            // of the temporary is *not* what we want; we want the src(0), which is
            // the actual Ret_Operand.
            if (!live->is_live_var(call_ret->src(0)->bv_position()) &&
                inst->can_eliminate())
                inst->mark_dead();
            mark_operand_dead(call_ret->src(0), live, call_ret->been_expanded());
        }
        // Kill the scratch registers, before livening the args.
        live->mark_dead((unsigned)eax_reg);
        live->mark_dead((unsigned)ecx_reg);
        live->mark_dead((unsigned)edx_reg);
        unsigned argno;
        for (argno=0; argno<call->n_args(); argno++)
        {
            Operand *arg = call->get_arg_opnd(argno);
            if (arg->kind == Operand::GCTrack && !call->get_arg(argno)->is_push())
                mark_operand_live(arg, live, call->get_arg(argno)->been_expanded(), NULL);
        }
        if (has_fp)
            region_live->init(false);
    }
    // Can't eliminate a statement with no destination operand.
    opnd = inst->dst();
    if (opnd != NULL && (opnd->is_reg() || opnd->is_status_flags()))
    {
        varno = opnd->bv_position();

        if (has_fp &&
            !opnd->is_fp_stk() &&
            !region_live->is_live_var(varno) &&
            (node->eh_out_edge() == NULL ||
            node->eh_out_edge()->live == NULL ||
            !node->eh_out_edge()->live->is_live_var(varno)))
        {
            // not live, so just eliminate the instruction
            inst->region_dst_live_range_ends = 1;
        }
        if (!opnd->is_fp_stk() &&
            !live->is_live_var(varno) &&
            (node->eh_out_edge() == NULL ||
            node->eh_out_edge()->live == NULL ||
            !node->eh_out_edge()->live->is_live_var(varno)) &&
            inst->can_eliminate())
        {
            // not live, so just eliminate the instruction
            inst->mark_dead();
            return;
        }
    }
    // Special case for a "ret" where there's only one outgoing edge.
    if (inst->is_ret_from_jsr() && node->out_edge_size() == 1)
    {
        inst->mark_dead();
        return;
    }

    // If inst is a "ret", mark that the register contains the return address
    // for this BB and for this subroutine.
    if (inst->is_ret_from_jsr())
    {
        assert(inst->src(0)->kind == Operand::GCTrack);
        GCTrack_Operand *reg = (GCTrack_Operand *) inst->src(0);
        reg->which_subr = node->get_enclosing_subr();
        node->set_where_is_ret_addr_bb(reg);
        node->get_enclosing_subr()->set_where_is_ret_addr_subr(reg);
    }

    if (inst->is_assignment() && inst->type() == JIT_TYPE_RETADDR)
    {
        if (node->get_jsr_succ() == NULL || inst->next() != node->IR_instruction_list())
        {
            assert(inst->src(0)->kind == Operand::GCTrack);
            assert(inst->dst()->kind == Operand::GCTrack);
            GCTrack_Operand *src = (GCTrack_Operand *)inst->src(0);
            GCTrack_Operand *dst = (GCTrack_Operand *)inst->dst();
            Cfg_Node *subr = dst->which_subr;
            dst->which_subr = NULL;  // KEN why reset to NULL?
            src->which_subr = subr;
            if (src != dst && subr == node->get_enclosing_subr())
            {
                inst->set_changes_ret_addr(true);
                node->set_where_is_ret_addr_bb(src);
            }
            src->which_subr->set_where_is_ret_addr_subr(src);
        }
    }

    // Kill the destination operand.
    // Liven it if it's a field or array access.
    if (opnd != NULL)
    {
        switch (opnd->kind)
        {
        case Operand::GCTrack:
            if (opnd->type == JIT_TYPE_UNKNOWN)
            {
                assert(inst->is_assignment());
                Assign_Inst *assn = (Assign_Inst *)inst;
                assn->real_type = live->approx_type(opnd->bv_position());
            }
            //::Patch. Don't kill edx:eax = ...
            if(opnd && opnd->is_physical_reg() &&
                opnd->hi_opnd() && opnd->hi_opnd()->is_physical_reg()){
                liven(opnd, exprs, live, region_live, node->eh_out_edge(),
                      inst, 0, has_fp);
                break ;
            }
            //::end
            mark_operand_dead(opnd, live, (opnd->type==JIT_TYPE_DOUBLE));
            if (has_fp)
                mark_operand_dead(opnd, region_live, (opnd->type==JIT_TYPE_DOUBLE));
            break;
        case Operand::Status:
            mark_operand_dead(opnd, live, 0);
            break;
        case Operand::Immediate:
        case Operand::Static:
        case Operand::Const:
            break;
        case Operand::Field:
        case Operand::Array:
            liven(opnd, exprs, live, region_live, node->eh_out_edge(),
                  inst, 0, has_fp);
            break;
        }
    }
    // Make the source operands live.
    unsigned i;
    for (i=0; i<inst->n_srcs; i++)
        liven(inst->src(i), exprs, live, region_live, node->eh_out_edge(),
              inst, 2*i+2, has_fp);
}

static void init_region_from_successors(Bit_Vector_Group *region_tmp, Cfg_Node *node,
                                        Bit_Vector_Group *tmp)
{
    Cfg_Node *succ = NULL;
    // See if node and its successor are essentially a single basic block.
    if (is_artificially_split_edge(node, succ))
    {
        if (succ->region_live == NULL)
            region_tmp->copy_from(tmp);
        else
            region_tmp->copy_from(succ->region_live);
    }
    // See if node is the end of a virtual inlining.
    else if (is_edge_outof_hot_inlined(node, succ) ||
         is_edge_outof_cold_inlined(node, succ))
    {
        if (succ->region_live == NULL)
            region_tmp->copy_from(tmp);
        else
            region_tmp->copy_from(succ->region_live);
    }
    // See if succ is the beginning of a virtual inlining.
    else if (is_edge_into_hot_inlined(node, succ))
    {
        if (succ->region_live == NULL)
            region_tmp->copy_from(tmp);
        else
            region_tmp->copy_from(succ->region_live);
    }
    else
        region_tmp->init(false);
}

static bool should_update_region_at_top(Cfg_Node *node, Bit_Vector_Group *tmp,
                                        Bit_Vector_Group *region_tmp)
{
    Cfg_Node *pred = NULL;
    if (is_artificially_split_edge(pred, node)/* &&
        !tmp->equals(region_tmp, lvh)*/)
        return true;
    if (is_edge_into_hot_inlined(pred, node))
        return true;
    if (is_edge_outof_hot_inlined(pred, node))
        return true;
    return false;
}

//
// return true, if the method is non-static synchronized
//
static void init_epilog(Expressions&     exprs,
                        Bit_Vector_Group *tmp,
                        Operand          *sync_this_ptr,
                        Method_Handle    m_handle)
{
    tmp->init(false);
    // Initialize live_vars to the method's return value, if any.
    O3_Jit_Type ret_type = (O3_Jit_Type)method_get_return_type(m_handle);
    CONVERT_TO_INT_JIT(ret_type);
    if (ret_type != JIT_TYPE_VOID)
    {
        Operand *ret_val = exprs.lookup_ret_exp(ret_type)->opnd;
        mark_operand_live(ret_val, tmp, 1, NULL);
    }
    //
    // mark this ptr live for synchronized methods
    //
    if (sync_this_ptr != NULL)
        mark_operand_live(sync_this_ptr, tmp, 0, NULL);
}

void Flow_Graph::eliminate(Cfg_Node *node, 
                           Expressions &exprs,
                           GCTrack_Operand *where_is_ret_addr,
                           Bit_Vector_Group *&tmp, 
                           Bit_Vector_Group *&region_tmp,
                           Bit_Vector *scratch,
                           Operand *sync_this_ptr,
                           Bit_Vector *node_bv)
{
    Cfg_Int edge;
    Eh_Node *eh;
    bool node_liveness_changed = false;
    bool eh_liveness_changed = false;
    if (node->live == NULL)
        node->live = tmp->clone(mem_manager, false);
    node_bv->reset(node->label);

    // Decide whether we even need to process this node.
    // Determine what type of merge this is.
    Bit_Vector_Group::MergeType mt;
    // It's a merge_ret if the last statement of the BB is a "ret" instruction,
    // and there's more than one outgoing edge.
    if (node->out_edge_size() > 1 && node->IR_instruction_list()->prev()->is_ret_from_jsr())
        mt = Bit_Vector_Group::merge_ret;
    // It's a merge_jsr if enclosing_subr of the node and its successor are different.
    else if (node->out_edge_size() == 1 &&
        node->get_enclosing_subr() != node->out_edges(0)->get_enclosing_subr()
        && node->get_jsr_succ() != NULL  // XXX-JMS: hack; remove later when subroutine info is fixed
        )
        mt = Bit_Vector_Group::merge_jsr;
    else
        mt = Bit_Vector_Group::merge_default;
    // XXX: To do: figure out what to do for merge_jsr and merge_ret in the presence of exception handlers.
    // Initialize outgoing liveness bit vectors.
    if (mt == Bit_Vector_Group::merge_jsr)
    {
        if (node->out_edges(0)->live == NULL)
            tmp->init(false);
        else
            tmp->copy_from(node->out_edges(0)->live);
        tmp->merge_incoming(node->get_jsr_succ()->live, scratch, mt);
        tmp->fixup_final(mt);
    }
    else
    {
        unsigned n_succs = node->out_edge_size();
        if (node == _epilog)
        {
            init_epilog(exprs,tmp,sync_this_ptr,m_handle());
        }
        else if (node->out_edge_size() == 1)
        {
            if (node->out_edges(0)->live == NULL)
                tmp->init(false);
            else
                tmp->copy_from(node->out_edges(0)->live);
        }
        else if (mt == Bit_Vector_Group::merge_default && node->out_edge_size() == 2)
        {
            Bit_Vector_Group *live1 = node->out_edges(0)->live;
            Bit_Vector_Group *live2 = node->out_edges(1)->live;
            if (live1 == NULL && live2 == NULL)
                tmp->init(false);
            else if (live1 == NULL)
                tmp->copy_from(live2);
            else if (live2 == NULL)
                tmp->copy_from(live1);
            else
                tmp->merge_two(live1, live2, mt);
        }
        else
        {
            tmp->init(true);
            for (edge=0; edge<node->out_edge_size(); edge++)
                tmp->merge_incoming(node->out_edges(edge)->live, scratch,mt);
            tmp->fixup_final(mt);
        }
    }

    node->set_where_is_ret_addr_bb(where_is_ret_addr);
    
    // If there's a single edge between this node and its successor,
    // and both nodes share the same exception handler, then treat
    // them as a single basic block; i.e., don't require the region
    // to end at the basic block boundary.  This is to make the floating
    // point register allocation more effective in the presence of
    // inlining.
    //
    // HOWEVER: When this happens, it is not necessarily the case that the
    // region liveness is equal to the global liveness at the end of the
    // basic block.  This is because we special-case the "push" instruction
    // so that it cannot be the last use of the operand in the region.  As
    // a result, if the first use of the operand in the basic block is in
    // the push instruction, then at the beginning of the basic block, the
    // operand is live globally but dead regionally.
    //
    // To fix this, once in a while we have to explicitly store the regional
    // liveness at the top of the basic block.
    if (has_fp)
        init_region_from_successors(region_tmp, node, tmp);


//    eh = node->eh_out_edge();
//    if (eh != NULL && eh->live != NULL)
//        tmp->merge_two(tmp, eh->live, Bit_Vector_Group::merge_default);
    
	//::move to 
//	eh = node->eh_out_edge();
//	if (eh != NULL && eh->live != NULL)
//		tmp->merge_two(tmp, eh->live, Bit_Vector_Group::merge_default);

    // Iterate backwards through the instruction list.
    Inst *first_inst = node->IR_instruction_list();
    Inst *last_inst = first_inst->prev();
    while (last_inst != first_inst)
    {
		//::move to 
		eh = node->eh_out_edge();
		if (eh != NULL && eh->live != NULL)
			tmp->merge_two(tmp, eh->live, Bit_Vector_Group::merge_default);

		determine_liveness(last_inst, exprs, tmp, region_tmp, sync_this_ptr,
            node, has_fp);
        last_inst = last_inst->prev();
    }

    //::commented for testDCE
//    eh = node->eh_out_edge();
//    if (eh != NULL && eh->live != NULL)
//        tmp->merge_two(tmp, eh->live, Bit_Vector_Group::merge_default);

   
    if (has_fp && should_update_region_at_top(node, tmp, region_tmp))
    {
        if (node->region_live == NULL)
            node->region_live = region_tmp->clone(mem_manager,false);
        if (!region_tmp->equals(node->region_live))
        {
            node_liveness_changed = true;
            Bit_Vector_Group *foo = region_tmp;
            region_tmp = node->region_live;
            node->region_live = foo;
        }
    }
    // Decide whether anything has changed.
    if (!tmp->equals(node->live))
    {
		node_liveness_changed = true;
        Bit_Vector_Group *foo = tmp;
        tmp = node->live;
        node->live = foo;
    }

    // Process this node's predecessors.  Start with exception handlers.
    Eh_In_Header *ehin = node->eh_in_edge();
    if (ehin != NULL)
    {
        Cfg_Int i, j;
        Bit_Vector_Group::MergeType mt = Bit_Vector_Group::merge_default;
        for (i=0; i<ehin->eh_in_edge_size(); i++)
        {
            // Process each Eh_Node, ehin->eh_in_edges(i).
            // Initialize the bit vectors.
            Eh_Node *ehnode = ehin->eh_in_edges(i);
            if (ehnode->live == NULL)
                ehnode->live = tmp->clone(mem_manager,false);
            tmp->init(true);
            for (j=0; j<ehnode->out_edge_size(); j++)
                tmp->merge_incoming(ehnode->out_edges(j)->handler->live, scratch, mt);
            tmp->fixup_final(mt); // not really necessary
            // Kill the exception object in eax.
            tmp->mark_dead((unsigned)eax_reg);
            assert(ehnode->in_edge_size() > 0);
            if (!tmp->equals(ehnode->live))
            {
                eh_liveness_changed = true;
                for (j=0; j<ehnode->in_edge_size(); j++)
                    node_bv->set(ehnode->in_edges(j)->label);
            }
            ehnode->live->copy_from(tmp);
        }
    }

    if (node_liveness_changed)
    {
        for (edge=0; edge<node->in_edge_size(); edge++)
        {
            // I saw a problem here.  There's an incoming edge for a node that's
            // actually dead, but hasn't been actually removed from all the edges.
            // As a workaround, we detect this when its label is larger than the
            // bit vector size.
            if (node->in_edges(edge)->label < (int)node_bv->numbits())
                node_bv->set(node->in_edges(edge)->label);
        }
    }
    if (node_liveness_changed || eh_liveness_changed)
        liveness_changed = true;
}

class Unlink_Closure : public Closure
{
public:
    Unlink_Closure(Expressions &exprs): rerun(false), exprs(exprs) {}
    bool rerun;
    Expressions &exprs;
};

static void unlink_all_killed_instructions(Cfg_Node *node, Closure *c)
{
    Unlink_Closure *uc = (Unlink_Closure *) c;
    Inst *head = node->IR_instruction_list();
    Inst *i = head->next(); 
    while (i != head) {
        Inst *next = i->next();
        if (i->is_dead())
        {
            if (i->is_call())
            {
                Call_Inst *cinst = (Call_Inst *) i;
                if (safe_to_eliminate_call(i, uc->exprs))
                {
                    //orp_cout << "remove call instruction" << endl;
                    uc->rerun = true;
#if 0
                    unsigned n;
                    for (n=0; n<cinst->n_args(); n++)
                    {
                        Inst *arg = cinst->get_arg(n);
                        if (arg->is_push())
                            arg->unlink();
                    }
                    i->unlink();
#endif // 0
                }
                else
                {
                    ((Call_Inst *)i)->kill_ret();
                    i->unmark_dead();
                }
            }
            else
                i->unlink();
        }
        i = next;
    }
}

static void clear_all_live_fields(Cfg_Node *node, Closure *c)
{
    node->live = NULL;
    node->region_live = NULL;
    if (node->eh_out_edge() != NULL)
        node->eh_out_edge()->live = NULL;
}

void Flow_Graph::dead_code_eliminate(Expressions &exprs, bool compute_live_refs)
{
    apply(clear_all_live_fields, NULL);
    Eh_Node *eh;
    for (eh = _handlers.next(); eh!=&_handlers; eh=eh->next())
        eh->live = NULL;
    did_dead_elim=true;
    unsigned number_of_bits_required = exprs.reg_map.curr_tmp_reg_id();
    Bit_Vector_Group *tmp;
    Bit_Vector_Group *region_tmp = NULL;
    if (compute_live_refs)
    {
        has_fp = false;
        if (num_jsr > 1)
            tmp = new(mem_manager) Bit_Vector_Group_GC_jsr(number_of_bits_required,mem_manager,1);
        else
            tmp = new(mem_manager) Bit_Vector_Group_GC_nojsr(number_of_bits_required,mem_manager,1);
    }
    else
    {
        tmp = new(mem_manager) Bit_Vector_Group_Elim(number_of_bits_required,mem_manager,1);
        if (has_fp)
            region_tmp = new(mem_manager) Bit_Vector_Group_Elim(number_of_bits_required,mem_manager,1);
    }

    Bit_Vector scratch(number_of_bits_required, mem_manager, false);

    Operand *sync_this_ptr = NULL;
    if (method_is_synchronized(m_handle()) && !method_is_static(m_handle()))
        sync_this_ptr = this_pointer_of_method;

    // Create an array of Cfg_Nodes and initialize it to the "reverse depth-first
    // ordering" to optimize the backwards dataflow analysis.
    int num_nodes;
    Mem_Manager tmem(1000);
    Cfg_Node **nodearray;
    create_dataflow_ordering(tmem, nodearray, num_nodes);
    Bit_Vector node_bv(num_nodes,tmem,true);

    do
    {
        liveness_changed = false;
        int i;
        for (i=0; i<num_nodes; i++)
        {

			//::for debug
			//print the ref, nonref ,dead and unknown(if has)
/*			{
				unsigned j ;
				Cfg_Node* node = nodearray[i] ;
				if(node->live && node->live->is_gc()){
					printf("B %2d",node->unique_label) ;
					const Bit_Vector* b = ((Bit_Vector_Group_GC*)(node->live))->_dead ;
					for(j = 0 ; j<b->_n_words ; j++) printf(" D%08x ", b->_words[j]) ;
					b = ((Bit_Vector_Group_GC*)(node->live))->_refs ;
					for(j = 0 ; j<b->_n_words ; j++) printf(" R%08x ", b->_words[j]) ;
					b = ((Bit_Vector_Group_GC*)(node->live))->_nonrefs ;
					for(j = 0 ; j<b->_n_words ; j++) printf(" N%08x ", b->_words[j]) ;
					if(!((Bit_Vector_Group_GC*)(node->live))->no_unknowns()){
						b = ((Bit_Vector_Group_GC_jsr*)(node->live))->get_unknown() ;
						for(j = 0 ; j<b->_n_words ; j++) printf(" U%08x", b->_words[j]) ;
					}
					printf("\n") ;
				}
			}*/

            if (node_bv.is_set(nodearray[i]->label)){
                eliminate(nodearray[i], exprs, NULL, tmp,  region_tmp,
                          &scratch, sync_this_ptr, &node_bv);
			}
/*
			//::for debug
			//print the ref, nonref ,dead and unknown(if has)
			{
				unsigned j ;
				Cfg_Node* node = nodearray[i] ;
				if(node->live && node->live->is_gc()){
					printf("O %2d",node->unique_label) ;
					const Bit_Vector* b = ((Bit_Vector_Group_GC*)(node->live))->_dead ;
					for(j = 0 ; j<b->_n_words ; j++) printf(" D%08x ", b->_words[j]) ;
					b = ((Bit_Vector_Group_GC*)(node->live))->_refs ;
					for(j = 0 ; j<b->_n_words ; j++) printf(" R%08x ", b->_words[j]) ;
					b = ((Bit_Vector_Group_GC*)(node->live))->_nonrefs ;
					for(j = 0 ; j<b->_n_words ; j++) printf(" N%08x ", b->_words[j]) ;
					if(!((Bit_Vector_Group_GC*)(node->live))->no_unknowns()){
						b = ((Bit_Vector_Group_GC_jsr*)(node->live))->get_unknown() ;
						for(j = 0 ; j<b->_n_words ; j++) printf(" U%08x", b->_words[j]) ;
					}
					printf("\n") ;
				}
			}
*/
        }
    }
    while (liveness_changed);

    Unlink_Closure uc(exprs);
    apply(unlink_all_killed_instructions, &uc);
    if (uc.rerun)
        dead_code_eliminate(exprs, compute_live_refs);
}
