// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/copy_prop.cpp,v 1.15 2002/01/10 03:12:09 xhshi Exp $
//


#include "defines.h"
#include "copy_prop.h"
#include "expression.h"
#include "is_subclass_of.h"
#include "bit_vector.h"
#include "handles.h"
#include "../ia32_o1_jit/profiling.h"
#include "internal_jit_intf.h"

//#define USE_KILL_SET

#define BLAH(inst) (!((inst)->is_compare()||(inst)->is_div_inst()))

Operand *Prop_Closure::PROP_CONFLICT = NULL;
Operand *clone_array_operand(Mem_Manager& mm, Operand *b, Operand *i, Operand *ary);
Operand *clone_field_operand(Mem_Manager& mm, Operand *b, Operand *fld);


// This is currently an n^2 algorithm, which probably could be
// written more efficiently.
static Class_Handle lub_ch(Class_Handle ch1, Class_Handle ch2)
{
    while (!O3_is_subclass_of(ch1, ch2))
        ch2 = class_get_super_class(ch2);
    return ch2;
}

static Class_Handle combine_ch(Class_Handle ch1, Class_Handle ch2)
{
    Class_Handle result;

    if (ch1 == ch2)
        result = ch1;
    else if (ch1 == NULL || ch2 == NULL)
        result = NULL;
    else
        result = lub_ch(ch1, ch2);

    return result;
}

void PC_Hash::combine_ch_hash(Tp_Hash_Entry &dst, Tp_Hash_Entry &src)
{
    if (dst.opnd == NULL)
        return;
    if (dst.opnd != src.opnd)
        dst.opnd = NULL;
    else if (dst.replace == NULL || src.replace == NULL)
        dst.replace = NULL;
    else
        dst.replace = lub_ch(dst.replace, src.replace);
}

void PC_Exact::add_to_kill_set(unsigned lhs_bvp, Operand *rhs)
{
    if (rhs->kind == Operand::GCTrack)
    {
        unsigned rhs_bvp = rhs->bv_position();
        kill_set[rhs_bvp] = new(mem) Kill_Set_Entry(lhs_bvp, kill_set[rhs_bvp]);
    }
}

#ifdef USE_KILL_SET
void PC_Hash::add_to_kill_set(unsigned lhs_hash, Operand *rhs)
{
    if (rhs->kind == Operand::GCTrack)
    {
        unsigned rhs_hash = _hash(rhs->bv_position());
        kill_set[rhs_hash] = new(mem) Kill_Set_Entry(lhs_hash, kill_set[rhs_hash]);
    }
}
#endif // USE_KILL_SET

void PC_Exact::kill_all_in_kill_set_by_bvp(unsigned bvp)
{
    Kill_Set_Entry *search = kill_set[bvp];
    for (; search != NULL; search = search->next)
    {
        working_set[search->bvp] = PROP_CONFLICT;
    }
    kill_set[bvp] = NULL;
}

void PC_Exact::kill_all_in_kill_set(Operand *opnd)
{
    unsigned bvp = opnd->bv_position();
    Kill_Set_Entry *search = kill_set[bvp];
    for (; search != NULL; search = search->next)
    {
        working_set[search->bvp] = PROP_CONFLICT;
    }
    kill_set[bvp] = NULL;
}

#ifdef USE_KILL_SET
void PC_Hash::kill_all_in_kill_set(Operand *opnd)
{
    unsigned bvp = _hash(opnd->bv_position());
    Kill_Set_Entry *search = kill_set[bvp];
    for (; search != NULL; search = search->next)
    {
        working_set[search->bvp].opnd = NULL;
    }
    kill_set[bvp] = NULL;
}
#endif // USE_KILL_SET

void PC_Exact::init_working_set(Cfg_Node *node)
{
    unsigned i;
    Cfg_Node *pred = NULL;
    if (for_real)
        memset(defined_in_this_bb, 0, num_vars * sizeof(defined_in_this_bb[0]));
    if (node->eh_in_edge() != NULL || node->in_edge_size() == 0)
    {
        // If the node is an exception handler, we conservatively clear everything.
        // This is overly conservative.  A better approach would be to clear only
        // those elements that were touched in one of the in-edges to the EH node.
        memset(working_set, 0, num_vars * sizeof(working_set[0]));
        memset(kill_set,    0, num_vars * sizeof(   kill_set[0]));
        if (type_prop && node->label != 0)
        {
            // XXX- should be java/lang/Object or something
            memset(type_working_set, 0, num_vars * sizeof(type_working_set[0]));
        }
    }
    else if (is_artificially_split_edge(pred, node))
    {
        // Just use the working set from the previous node.
    }
    else
    {
        Cfg_Int edge;
        bool first = true;
        for (edge=0; edge<node->in_edge_size(); edge++)
        {
            Cfg_Node *incoming = node->in_edges(edge);
            if (incoming->latest_traversal < traversal_num)
                continue;
            Operand **pv = &prop_values[num_vars * incoming->label];
            if (first)
            {
                memcpy(working_set, pv, num_vars * sizeof(working_set[0]));
                if (type_prop)
                {
                    Class_Handle *tpv = &type_prop_values[num_vars * incoming->label];
                    memcpy(type_working_set, tpv, num_vars * sizeof(working_set[0]));
                }
            }
            else
            {
                for (i=0; i<num_vars; i++)
                    combine(working_set[i], pv[i]);
                if (type_prop)
                {
                    Class_Handle *tpv = &type_prop_values[num_vars * incoming->label];
                    for (i=0; i<num_vars; i++)
                        type_working_set[i] = combine_ch(type_working_set[i], tpv[i]);
                }
            }
            first = false;
        }
        assert(!first);
        memset(kill_set, 0, num_vars * sizeof(kill_set[0]));
        for (i=0; i<num_vars; i++)
        {
            Operand *opnd = working_set[i];
            if (opnd != PROP_CONFLICT)
                add_to_kill_set(i, opnd);
        }
    }
}

void PC_Hash::init_working_set(Cfg_Node *node)
{
    unsigned i;
    Cfg_Node *pred = NULL;
    if (for_real)
        memset(defined_in_this_bb, 0, num_vars);
    if (node->eh_in_edge() != NULL || node->in_edge_size() == 0)
    {
        // If the node is an exception handler, we conservatively clear everything.
        // This is overly conservative.  A better approach would be to clear only
        // those elements that were touched in one of the in-edges to the EH node.
        memset(working_set, 0, CP_HASH_SIZE * sizeof(working_set[0]));
#ifdef USE_KILL_SET
        memset(kill_set,    0, CP_HASH_SIZE * sizeof(   kill_set[0]));
#endif // 0
        if (type_prop && node->label != 0)
        {
            memset(type_working_set, 0, CP_HASH_SIZE * sizeof(type_working_set[0]));
        }
    }
    else if (is_artificially_split_edge(pred, node))
    {
        // Just use the working set from the previous node.
    }
    else
    {
        Cfg_Int edge;
        bool first = true;
        for (edge=0; edge<node->in_edge_size(); edge++)
        {
            Cfg_Node *incoming = node->in_edges(edge);
            if (incoming->latest_traversal < traversal_num)
                continue;
            Cp_Hash_Entry *pv = &prop_values[CP_HASH_SIZE * incoming->label];
            if (first)
            {
                memcpy(working_set, pv, CP_HASH_SIZE * sizeof(working_set[0]));
                if (type_prop)
                {
                    Tp_Hash_Entry *tpv = &type_prop_values[CP_HASH_SIZE * incoming->label];
                    memcpy(type_working_set, tpv, CP_HASH_SIZE * sizeof(type_working_set[0]));
                }
            }
            else
            {
                for (i=0; i<CP_HASH_SIZE; i++)
                    combine_hash(working_set[i], pv[i]);
                if (type_prop)
                {
                    Tp_Hash_Entry *tpv = &type_prop_values[CP_HASH_SIZE * incoming->label];
                    for (i=0; i<CP_HASH_SIZE; i++)
                        combine_ch_hash(type_working_set[i], tpv[i]);
                }
            }
            first = false;
        }
        assert(!first);
#ifdef USE_KILL_SET
        memset(kill_set, 0, CP_HASH_SIZE * sizeof(kill_set[0]));
        for (i=0; i<CP_HASH_SIZE; i++)
        {
            if (working_set[i].opnd != NULL)
            {
                Operand *replace = working_set[i].replace;
                add_to_kill_set(i, replace);
            }
        }
#endif // USE_KILL_SET
    }
}

Operand *Prop_Closure::replace_with_prop_value(Operand *opnd,
                                               bool update_grc, bool imm_ok)
{
    Operand *result = opnd;
    if (opnd->is_arg() || opnd->is_ret()) {}
    else if (opnd->is_reg())
    {
        unsigned bvp = _hash(opnd->bv_position());
        Operand *replace = get_replace_opnd(bvp);
        if (is_valid_replacement(bvp, opnd, replace) &&
            (imm_ok || replace->kind != Operand::Immediate))
        {
            result = replace;
            if (update_grc && for_real && replace->is_temp_reg() &&
                !defined_in_this_bb[replace->bv_position()])
            {
                ((Reg_Operand *)replace)->set_global_reg_cand();
                if (replace->hi_opnd() != NULL)
                    ((Reg_Operand *)replace->hi_opnd())->set_global_reg_cand();
            }
        }
    }
    else if (opnd->kind == Operand::Field)
    {
        Operand *base = opnd->base();
        assert(base != NULL);
        Operand *new_base = replace_with_prop_value(base, update_grc, false);
        if (new_base != base)
            result = clone_field_operand(exprs.mem, new_base, opnd);
    }
    else if (opnd->kind == Operand::Array)
    {
        Operand *base = opnd->base();
        Operand *indx = opnd->index();
        assert(base != NULL);
        assert(indx != NULL);
        Operand *new_base = replace_with_prop_value(base, update_grc, false);
        Operand *new_indx = replace_with_prop_value(indx, update_grc, true);
        if (new_base != base || new_indx != indx)
            result = clone_array_operand(exprs.mem, new_base, new_indx, opnd);
    }
    return result;
}

static Inst *replace_with_prop_values(Inst *inst, Prop_Closure *pc)
{
    unsigned i;
    Operand *opnd, *replace;
    bool all_srcs_constant = (inst->n_srcs > 0);
    for (i=0; i<inst->n_srcs && all_srcs_constant; i++)
    {
        opnd = inst->src(i);
        replace = pc->replace_with_prop_value(opnd, false, /*BLAH(inst)*/!inst->is_div_inst());
        if (replace->kind != Operand::Immediate || replace->type != JIT_TYPE_INT)
            all_srcs_constant = false;
    }
    // We can't allow constant propagation if both operands would end up constant.
    bool enable_imm = true;
    unsigned j;
    for (j=inst->n_srcs; j>0; j--)
    {
        i = j - 1;
        opnd = inst->src(i);
        replace = pc->replace_with_prop_value(opnd, true,
            all_srcs_constant ||
            (enable_imm &&
            (BLAH(inst)||!inst->info()->src_in_reg[i])));
        //
        // We avoid propagting an immediate to the base
        //
        if (replace != opnd && (!inst->is_obj_info() || replace->kind != Operand::Immediate))
            inst->replace_src(i, replace);
        if (replace->kind == Operand::Immediate || opnd->kind == Operand::Immediate)
            enable_imm = false;
    }
    opnd = inst->dst();
    if (opnd != NULL && opnd->kind != Operand::GCTrack)
    {
        replace = pc->replace_with_prop_value(opnd, true, true);
        if (replace != opnd)
            inst->set_dst(replace);
    }
    if (all_srcs_constant && !inst->is_assignment())
        inst = inst->simplify(pc->perm_mem, pc->exprs);
    return inst;
}

static Class_Handle get_class_handle_from_string(const char *str, Method_Handle mh)
{
    Loader_Exception lexc;
    return get_curr_arg_class((Arg_List_Iterator)str, mh, &lexc);
}

static Class_Handle get_class_handle_from_new_inst(Call_Inst *cinst, unsigned argno)
{
    Class_Handle ch;
    Inst *assn1 = cinst->get_arg(argno);
    assert(assn1->is_assignment());
    assert(assn1->dst()->is_arg());
    Operand *opnd1 = assn1->src(0);
    if (opnd1->kind == Operand::Immediate)
    {
        ch = (Class_Handle) ((Imm_Operand *)opnd1)->imm();
    }
    else
    {
        assert(opnd1->is_temp_reg());
        Temp_Reg *treg = (Temp_Reg *)opnd1;
        Inst *assn2 = treg->inst();
        assert(assn2->is_assignment());
        Operand *opnd2 = assn2->src(0);
        assert(opnd2->kind == Operand::Immediate);
        Imm_Operand *imm = (Imm_Operand *)opnd2;
        ch = (Class_Handle) imm->imm();
    }
    return ch;
}

// If it's an instanceof instruction, return NULL.
static Class_Handle get_class_handle_from_checkcast_inst(Inst *inst)
{
    Type_Inst *tinst = (Type_Inst *)inst;
    if (tinst->kind == Type_Inst::instance)
        return NULL;
    Class_Handle ch;
    Operand *opnd1 = inst->src(0);
    if (opnd1->kind == Operand::Immediate)
    {
        ch = (Class_Handle) ((Imm_Operand *)opnd1)->imm();
    }
    else
    {
        assert(opnd1->is_temp_reg());
        Temp_Reg *treg = (Temp_Reg *)opnd1;
        Inst *assn2 = treg->inst();
        assert(assn2->is_assignment());
        Operand *opnd2 = assn2->src(0);
        assert(opnd2->kind == Operand::Immediate);
        Imm_Operand *imm = (Imm_Operand *)opnd2;
        ch = (Class_Handle) imm->imm();
    }
    return ch;
}

// Modify the type_working_set to reflect the changes in the
// instruction.
// t11 = checkcast(handle, t10)
static void prop_type_info(Prop_Closure *pc, Inst *inst)
{
    if (inst->is_call())
    {
        Call_Inst *cinst = (Call_Inst *)inst;
        if (cinst->get_ret() == NULL)
            return;
        O3_Jit_Type cinst_ret = cinst->get_ret()->type();
        if (cinst_ret != JIT_TYPE_CLASS && cinst_ret != JIT_TYPE_ARRAY)
            return;
        Class_Handle ch = NULL;
        Java_Type ret_ty;
        Method_Handle mh;
        switch ((Call_Inst::Kind)cinst->kind)
        {
        case Call_Inst::new_call:        // new
            ch = get_class_handle_from_new_inst(cinst, 0);
            break;
        case Call_Inst::virtual_call:    // invokevirtual
        case Call_Inst::special_call:    // invokespecial
        case Call_Inst::static_call:     // invokestatic
        case Call_Inst::interface_call:  // invokeinterface
            mh = cinst->get_mhandle();
            ret_ty = method_get_return_type(mh);
            if (ret_ty == JAVA_TYPE_CLASS || ret_ty == JAVA_TYPE_ARRAY)
            {
                Loader_Exception lexc;
                ch = method_get_return_type_class(cinst->get_mhandle(), &lexc);
            }
            else
                assert(0);
            break;
        case Call_Inst::anewarray_call:  // anewarray
            ch = get_class_handle_from_new_inst(cinst, 1);
            break;
        case Call_Inst::newarray_call:   // newarray
            {
                Loader_Exception lexc;
                ch = class_load_class_by_name("[I", method_get_class(pc->mh), &lexc);
            }
            break;
        case Call_Inst::multinew_call:   // multianewarray
            ch = get_class_handle_from_new_inst(cinst, cinst->n_args()-1);
            break;
        case Call_Inst::llsh_call:       // long shift left
        case Call_Inst::lrsh_call:       // long arith shift right
        case Call_Inst::lrsz_call:       // long unsigned shift right
        case Call_Inst::f2i_call:        // convert float to int
        case Call_Inst::d2i_call:        // convert double to int
        case Call_Inst::f2l_call:        // convert float to long
        case Call_Inst::d2l_call:        // convert double to long
        case Call_Inst::lmul_call:       // long multiplication
#ifdef ORP_LONG_OPT
		case Call_Inst::lmul_const_multiplier_call:
#endif
        case Call_Inst::ldiv_call:       // long division
        case Call_Inst::lrem_call:       // long remainder
        case Call_Inst::getstring_call:  // string of constant pool
        case Call_Inst::bounds_call:     // throw ArrayIndexOutOfBoundsException
        case Call_Inst::checkcast_call:  // checkcast
        case Call_Inst::instanceof_call: // is instance of
        case Call_Inst::classinit_call:  // initialize class
        case Call_Inst::resintfc_call:   // resolve interface call
        case Call_Inst::aastore_call:
        case Call_Inst::writebarrier_call:
#ifdef JIT_SAPPHIRE
        case Call_Inst::readbarrier_call: //:: read barrier call
#endif
#ifdef STAT_INDIRECT_CALL
		case Call_Inst::stat_indirect_call: // for statistics
#endif
#ifdef O3_VTune_Support
		case Call_Inst::vtune_method_call:
#endif
		case Call_Inst::const_ldiv_call :
		case Call_Inst::const_lrem_call :
        default:
            assert(0);
            break;
        case Call_Inst::athrow_call:     // athrow
        case Call_Inst::athrow_lazy_call:     // athrow
        case Call_Inst::monenter_call:   // monitor enter
        case Call_Inst::monexit_call:    // monitor exit
        case Call_Inst::monenter_static_call:   // monitor enter for static methods
        case Call_Inst::monexit_static_call:    // monitor exit  for static methods
            break;
        }
        if (ch != NULL)
        {
            pc->set_type(cinst->get_ret()->src(0), ch);
#ifdef TRACE_O3xxx
            if (pc->for_real)
            {
                cout << "Instruction " << inst->global_id << ": new type for " << bvp << " is ";
                if (ch == NULL)
                    cout << "(NULL)";
                else
                    cout << class_get_name(ch);
                cout << endl;
            }
#endif // TRACE_O3
        }
        return;
    }
    Operand *dst = inst->dst();
    if (dst == NULL)
        return;
    if (dst->is_status_flags())
    {
        if (pc->for_real)
        {
            assert(inst->is_compare());
            Compare_Inst *cmp = (Compare_Inst *) inst;
            if (cmp->is_vtable_compare())
            {
                // We can rewrite the compare instruction to test the object's
                // vtable against the class that we've actually proven that the
                // object has at this point.
                // First, do a sanity check: make sure the propagated type
                // is actually a subclass of the class handle for the method
                // being invoked.
                // Then, only rewrite the instruction if no class between the
                // derived class and the inlined method's class overrides the
                // method being invoked.
                // XXX- use class_lookup_method_recursive()
                Operand *this_ptr = cmp->object_compared_with_vtable();
                Class_Handle ch = pc->get_type(this_ptr);
                Method_Handle mh = cmp->method_inlined();
                if (!O3_is_subclass_of(ch, method_get_class(mh)))
                {
                    // This is probably because of a previous instanceof instruction.
#if 0
                    cout << "**********bogus type info" << endl;
#endif // 0
                    return;
                }
                Method_Handle mh1 =
                    class_lookup_method_recursively(ch, method_get_name(mh),
                    method_get_descriptor(mh));
                if (mh1 != mh)
                {
#if 0
                    cout << "**********overridden method" << endl;
#endif // 0
                    return;
                }
                void *vt = class_get_vtable(ch);
                cmp->replace_src(1, pc->exprs.lookup_imm_exp((unsigned)vt, JIT_TYPE_ADDR)->opnd);
            }
        }
        return;
    }
    if (dst->type != JIT_TYPE_CLASS && dst->type != JIT_TYPE_ARRAY)
        return;
    assert(
        (inst->n_srcs == 1 && (inst->src(0)->type == JIT_TYPE_CLASS || inst->src(0)->type == JIT_TYPE_ARRAY)) ||
        (inst->n_srcs == 2 && inst->src(0)->type == JIT_TYPE_ADDR && (inst->src(1)->type == JIT_TYPE_CLASS || inst->src(1)->type == JIT_TYPE_ARRAY)) || // checkcast
        (inst->n_srcs == 2 && inst->src(0)->type == JIT_TYPE_INT && inst->src(1)->type == JIT_TYPE_ADDR) || // ldc
        false);
    if (inst->n_srcs == 1) // simple assignment statement
    {
        assert(inst->n_srcs == 1 && (inst->src(0)->type == JIT_TYPE_CLASS || inst->src(0)->type == JIT_TYPE_ARRAY));
        Operand *dst = inst->dst();
        Operand *src = inst->src(0);
        if ((dst->type == JIT_TYPE_CLASS || dst->type == JIT_TYPE_ARRAY) && dst->is_reg())
        {
            assert(src->type == JIT_TYPE_CLASS || src->type == JIT_TYPE_ARRAY);
            Class_Handle base_class;
            Field_Handle fh;
            switch (src->kind)
            {
            case Operand::GCTrack:
                pc->set_type(dst, pc->get_type(src));
#ifdef TRACE_O3xxx
                if (pc->for_real)
                {
                    Class_Handle ch = pc->type_working_set[dst_bvp];
                    cout << "Instruction " << inst->global_id << ": new type for " << dst_bvp << " is ";
                    if (ch == NULL)
                        cout << "(NULL)";
                    else
                        cout << class_get_name(ch);
                    cout << endl;
                }
#endif // TRACE_O3
                break;
            case Operand::Field:
                fh = (Field_Handle) ((Field_Operand *)src)->fid;
                base_class = get_class_handle_from_string(field_get_descriptor(fh), pc->mh);
                pc->set_type(dst, base_class);
#ifdef TRACE_O3xxx
                if (pc->for_real)
                {
                    Class_Handle ch = pc->type_working_set[dst_bvp];
                    cout << "Instruction " << inst->global_id << ": new type for " << dst_bvp << " is ";
                    if (ch == NULL)
                        cout << "(NULL)";
                    else
                        cout << class_get_name(ch);
                    cout << endl;
                }
#endif // TRACE_O3
                break;
            case Operand::Array:
                base_class = pc->get_type(src->base());
                if (base_class != NULL)
                {
                    pc->set_type(dst, class_get_array_element_class(base_class));
#ifdef TRACE_O3xxx
                    if (pc->for_real)
                    {
                        Class_Handle ch = pc->type_working_set[dst_bvp];
                        cout << "Instruction " << inst->global_id << ": new type for " << dst_bvp << " is ";
                        if (ch == NULL)
                            cout << "(NULL)";
                        else
                            cout << class_get_name(ch);
                        cout << endl;
                    }
#endif // TRACE_O3
                }
                break;
            case Operand::Static:
                fh = ((Static_Operand *)src)->get_field_handle();
                if (fh != NULL) {
                    base_class = get_class_handle_from_string(field_get_descriptor(fh), pc->mh);
                    pc->set_type(dst, base_class);
                } else
                    pc->set_type(dst, NULL);
#ifdef TRACE_O3xxx
                if (pc->for_real)
                {
                    Class_Handle ch = pc->type_working_set[dst_bvp];
                    cout << "Instruction " << inst->global_id << ": new type for " << dst_bvp << " is ";
                    if (ch == NULL)
                        cout << "(NULL)";
                    else
                        cout << class_get_name(ch);
                    cout << endl;
                }
#endif // TRACE_O3
                break;
            case Operand::Immediate:
                assert(((Imm_Operand *)src)->imm() == 0);
                pc->set_type(dst, NULL);
#ifdef TRACE_O3xxx
                if (pc->for_real)
                {
                    Class_Handle ch = pc->type_working_set[dst_bvp];
                    cout << "Instruction " << inst->global_id << ": new type for " << dst_bvp << " is ";
                    if (ch == NULL)
                        cout << "(NULL)";
                    else
                        cout << class_get_name(ch);
                    cout << endl;
                }
#endif // TRACE_O3
                break;
            default:
                assert(0);
                break;
            }
        }
    }
    else if (inst->src(1)->type == JIT_TYPE_CLASS ||
             inst->src(1)->type == JIT_TYPE_ARRAY) // checkcast
    {
        assert(inst->n_srcs == 2 &&
               inst->src(0)->type == JIT_TYPE_ADDR &&
              (inst->src(1)->type == JIT_TYPE_CLASS || inst->src(1)->type == JIT_TYPE_ARRAY));
        Class_Handle ch = get_class_handle_from_checkcast_inst(inst);
        if(inst->src(1)->is_reg()){ //::for non-reg type, need not to be record.
			if (ch != NULL)
			{
				pc->set_type(inst->src(1), ch);
			}
			pc->set_type(inst->dst(), pc->get_type(inst->src(1)));
#ifdef TRACE_O3xxx
			if (pc->for_real)
			{
				Class_Handle ch = pc->type_working_set[dst_bvp];
				cout << "Instruction " << inst->global_id << ": new type for " << dst_bvp << " is ";
				if (ch == NULL)
					cout << "(NULL)";
				else
					cout << class_get_name(ch);
				cout << endl;
			}
#endif // TRACE_O3
		}
    }
    else // ldc
    {
        assert(inst->n_srcs == 2 && inst->src(0)->type == JIT_TYPE_INT &&
               inst->src(1)->type == JIT_TYPE_ADDR);
        assert(inst->dst()->is_reg());
        unsigned dst_bvp = inst->dst()->bv_position();
        pc->set_type(inst->dst(), cached_class_handles[ch_java_lang_String]);
#ifdef TRACE_O3xxx
        if (pc->for_real)
        {
            Class_Handle ch = pc->type_working_set[dst_bvp];
            cout << "Instruction " << inst->global_id << ": new type for " << dst_bvp << " is ";
            if (ch == NULL)
                cout << "(NULL)";
            else
                cout << class_get_name(ch);
            cout << endl;
        }
#endif // TRACE_O3
    }
}

void PC_Exact::process_bb(Cfg_Node *node)
{
    Inst *inst;
    Inst *last_inst = node->IR_instruction_list();
    Inst *next_inst;
    for (inst=last_inst->next(); inst!=last_inst; inst=next_inst)
    {
        next_inst = inst->next();
        if (inst->is_dead() || inst->is_pseudo_asgn())
        {
            inst->unlink();
            continue;
        }
        // If for_real, replace all source operands with their prop values.
        if (for_real)
            inst = replace_with_prop_values(inst, this);

        // XXX- if for_real, and this is a special compare instruction,
        // do what is necessary.

        // Propagate type information for the instruction.
        if (type_prop)
            prop_type_info(this, inst);

        // [???] If call, kill scratch registers.

        // If a=b or a=7, set prop(a)=prop(rhs).  If for_real, assert(rhs==prop(rhs)).
        Operand *lhs = inst->dst();
        if (lhs != NULL && lhs->is_reg() && !lhs->is_arg() && !lhs->is_ret())
        {
            bool reset_lhs = true;
            Operand *rhs = NULL;
            Operand *src0 = NULL;
            Operand *src1 = NULL;
            Operand *src2 = NULL;
            if (inst->n_srcs > 0)
                src0 = replace_with_prop_value(inst->src(0), false, true);
            if (inst->n_srcs > 1)
                src1 = replace_with_prop_value(inst->src(1), false, true);
            if (inst->n_srcs > 2)
                src2 = replace_with_prop_value(inst->src(2), false, true);
            if (/*inst->is_reg_assignment() || inst->is_imm_assignment()*/
                inst->is_copy_prop_assignment(exprs, src0, src1, src2, rhs))
            {
//#ifdef INLINE_NATIVE
                if(lhs->is_hi() && lhs->is_temp_reg()){ //:: hI =
                    working_set[lhs->bv_position()-1] = PROP_CONFLICT ;//lo 
                }
				if(rhs && rhs->is_hi() && lhs->is_temp_reg()){ //:: = hi
					working_set[rhs->bv_position()-1] = PROP_CONFLICT ;//lo
				}
//#endif
                if (
//#ifdef INLINE_NATIVE
					((Temp_Reg*)rhs)->inst() &&
//#endif
					!rhs->is_arg() && !rhs->is_ret() &&
                    // Don't allow physical registers to be propagated, for now.
                    // The problem is that in an exception handler, the exception
                    // object is hard-coded to eax, and we don't want it
                    // propagated across call sites, or to cause it to increase
                    // the register pressure for scratch registers.
                    !(rhs->is_reg() && rhs->assigned_preg() != n_reg))
                {
                    // Don't propagate the copy-in of input arguments.
                    if (!(rhs->is_single_def_temp_reg() &&
                         (((Temp_Reg*)rhs)->inst()->n_srcs == 0 ||
                          ((Temp_Reg*)rhs)->inst()->src(0)->is_arg())))
                    {
                        Operand *replace = replace_with_prop_value(rhs, false, true);
                        working_set[lhs->bv_position()] = replace;
                        add_to_kill_set(lhs->bv_position(), replace);
                        reset_lhs = false;
                    }
                }
            }
            if (reset_lhs)
                working_set[lhs->bv_position()] = PROP_CONFLICT;

        }

        // Maintain defined_in_this_bb[].
        if (for_real && lhs != NULL && lhs->is_reg())
            defined_in_this_bb[lhs->bv_position()] = true;

        // If a=<xxx>, kill a everywhere.
        if (lhs != NULL && lhs->is_reg())
        {
            kill_all_in_kill_set(lhs);
        }
    }
}

void PC_Hash::process_bb(Cfg_Node *node)
{
    Inst *inst;
    Inst *last_inst = node->IR_instruction_list();
    Inst *next_inst;
    for (inst=last_inst->next(); inst!=last_inst; inst=next_inst)
    {
        next_inst = inst->next();
        if (inst->is_dead() || inst->is_pseudo_asgn())
        {
            inst->unlink();
            continue;
        }
        // If for_real, replace all source operands with their prop values.
        if (for_real)
            inst = replace_with_prop_values(inst, this);

        // XXX- if for_real, and this is a special compare instruction,
        // do what is necessary.

        // Propagate type information for the instruction.
        if (type_prop)
            prop_type_info(this, inst);

        // [???] If call, kill scratch registers.

        // If a=b or a=7, set prop(a)=prop(rhs).  If for_real, assert(rhs==prop(rhs)).
        Operand *lhs = inst->dst();
        if (lhs != NULL && lhs->is_reg() && !lhs->is_arg() && !lhs->is_ret())
        {
            bool reset_lhs = true;
            Operand *rhs = NULL;
            Operand *src0 = NULL;
            Operand *src1 = NULL;
            Operand *src2 = NULL;
            if (inst->n_srcs > 0)
                src0 = replace_with_prop_value(inst->src(0), false, true);
            if (inst->n_srcs > 1)
                src1 = replace_with_prop_value(inst->src(1), false, true);
            if (inst->n_srcs > 2)
                src2 = replace_with_prop_value(inst->src(2), false, true);
            if (inst->is_copy_prop_assignment(exprs, src0, src1, src2, rhs))
            {
                if (
//#ifdef INLINE_NATIVE
					((Temp_Reg*)rhs)->inst() &&
//#endif
                    !rhs->is_arg() && !rhs->is_ret() &&
                    // Don't allow physical registers to be propagated, for now.
                    // The problem is that in an exception handler, the exception
                    // object is hard-coded to eax, and we don't want it
                    // propagated across call sites, or to cause it to increase
                    // the register pressure for scratch registers.
                    !(rhs->is_reg() && rhs->assigned_preg() != n_reg))
                {
                    // Don't propagate the copy-in of input arguments.
                    if (!(rhs->is_single_def_temp_reg() &&
                         (((Temp_Reg*)rhs)->inst()->n_srcs == 0 ||
                          ((Temp_Reg*)rhs)->inst()->src(0)->is_arg())))
                    {
                        Operand *replace = replace_with_prop_value(rhs, false, true);
                        unsigned bvp = _hash(lhs->bv_position());
                        working_set[bvp].opnd = lhs;
                        working_set[bvp].replace = replace;
#ifdef USE_KILL_SET
                        add_to_kill_set(bvp, replace);
#endif // USE_KILL_SET
                        reset_lhs = false;
                    }
                }
            }
            if (reset_lhs)
                working_set[_hash(lhs->bv_position())].opnd = NULL;
        }

        // Maintain defined_in_this_bb[].
        if (for_real && lhs != NULL && lhs->is_reg())
            defined_in_this_bb[lhs->bv_position()] = true;

        // If a=<xxx>, kill a everywhere.
        if (lhs != NULL && lhs->is_reg())
        {
#ifndef USE_KILL_SET
            unsigned i;
            for (i=0; i<CP_HASH_SIZE; i++)
            {
                if (working_set[i].replace == lhs)
                    working_set[i].opnd = NULL;
            }
#else // USE_KILL_SET
            kill_all_in_kill_set(lhs);
#endif // USE_KILL_SET
        }
    }
}

void PC_Exact::update_outgoing(Cfg_Node *node, Bit_Vector *node_bv)
{
    Operand **pv = &prop_values[num_vars * node->label];
    unsigned i;
    bool node_changed = false;
    Cfg_Node *succ = NULL;
    if (node->latest_traversal < traversal_num)
    {
        node_changed = true;
        memcpy(pv, working_set, num_vars * sizeof(pv[0]));
        if (type_prop)
        {
            Class_Handle *tpv = &type_prop_values[num_vars * node->label];
            memcpy(tpv, type_working_set, num_vars * sizeof(tpv[0]));
        }
    }
    else if (is_artificially_split_edge(node, succ))
    {
        // Let the working set fall through to the next node.
        node_changed = true;
    }
    else
    {
        for (i=0; i<num_vars; i++)
        {
            Operand *old = pv[i];
            combine(pv[i], working_set[i]);
            if (pv[i] != old)
                node_changed = true;
        }
        if (type_prop)
        {
            Class_Handle *tpv = &type_prop_values[num_vars * node->label];
            for (i=0; i<num_vars; i++)
            {
                Class_Handle old = tpv[i];
                tpv[i] = combine_ch(tpv[i], type_working_set[i]);
                if (tpv[i] != old)
                    node_changed = true;
            }
        }
    }
    if (node_changed)
    {
        changed = true;
        Cfg_Int edge;
        for (edge=0; edge<node->out_edge_size(); edge++)
            node_bv->set(node->out_edges(edge)->label);
    }
}

void PC_Hash::update_outgoing(Cfg_Node *node, Bit_Vector *node_bv)
{
    Cp_Hash_Entry *pv = &prop_values[CP_HASH_SIZE * node->label];
    unsigned i;
    bool node_changed = false;
    Cfg_Node *succ = NULL;
    if (node->latest_traversal < traversal_num)
    {
        node_changed = true;
        memcpy(pv, working_set, CP_HASH_SIZE * sizeof(pv[0]));
        if (type_prop)
        {
            Tp_Hash_Entry *tpv = &type_prop_values[CP_HASH_SIZE * node->label];
            memcpy(tpv, type_working_set, CP_HASH_SIZE * sizeof(tpv[0]));
        }
    }
    else if (is_artificially_split_edge(node, succ))
    {
        // Let the working set fall through to the next node.
        node_changed = true;
    }
    else
    {
        for (i=0; i<CP_HASH_SIZE; i++)
        {
            Cp_Hash_Entry old = pv[i];
            combine_hash(pv[i], working_set[i]);
            if (pv[i].opnd != old.opnd || pv[i].replace != old.replace)
                node_changed = true;
        }
        if (type_prop)
        {
            Tp_Hash_Entry *tpv = &type_prop_values[CP_HASH_SIZE * node->label];
            for (i=0; i<CP_HASH_SIZE; i++)
            {
                Tp_Hash_Entry old = tpv[i];
                combine_ch_hash(tpv[i], type_working_set[i]);
                if (tpv[i].opnd != old.opnd || tpv[i].replace != old.replace)
                    node_changed = true;
            }
        }
    }
    if (node_changed)
    {
        changed = true;
        Cfg_Int edge;
        for (edge=0; edge<node->out_edge_size(); edge++)
            node_bv->set(node->out_edges(edge)->label);
    }
}

static void prop_bb(Cfg_Node *node, Prop_Closure *pc, Bit_Vector *node_bv)
{
    node_bv->reset(node->label);
    pc->init_working_set(node);
    pc->process_bb(node);
    pc->update_outgoing(node, node_bv);
}

// Initialize the type information for the incoming arguments.
// This code is modeled after build_incoming_arg_assignment() and
// formal_to_actual() in Build_IR_Routines.cpp.
void PC_Exact::init_types(Flow_Graph *fg)
{
    if (!type_prop)
        return;
    Method_Handle m_handle = fg->m_handle();
    Java_Type type;
    O3_Jit_Type ty;
    unsigned varno;
    unsigned ith = 0;
    unsigned i;
    for (i=0; i<num_vars; i++)
        type_working_set[i] = NULL;
    // process "this" pointer
    if (!method_is_static(m_handle)) {
        ty = JIT_TYPE_CLASS;
        varno = exprs.lookup_arg_exp(ith,ty)->opnd->bv_position();
        type_working_set[varno] = fg->c_handle();
        ith++;
    }

    Arg_List_Iterator iter = method_get_argument_list(m_handle);
    while((type = curr_arg(iter)) != JAVA_TYPE_END) {
        ty = (O3_Jit_Type) type;
        CONVERT_TO_INT_JIT(ty);
        if (ty == JAVA_TYPE_CLASS)
        {
            varno = exprs.lookup_arg_exp(ith,ty)->opnd->bv_position();
            // Convert "iter" into a Class_Handle.
            Loader_Exception lexc;
            type_working_set[varno] = get_curr_arg_class(iter, m_handle, &lexc);
        }
        ith += num_words_of_type(type);
        iter = advance_arg_iterator(iter);
    }
}

void PC_Hash::init_types(Flow_Graph *fg)
{
    if (!type_prop)
        return;
    Method_Handle m_handle = fg->m_handle();
    Java_Type type;
    O3_Jit_Type ty;
    unsigned varno;
    unsigned ith = 0;
    unsigned i;
    for (i=0; i<CP_HASH_SIZE; i++)
        type_working_set[i].opnd = NULL;
    // process "this" pointer
    if (!method_is_static(m_handle)) {
        ty = JIT_TYPE_CLASS;
        varno = _hash(exprs.lookup_arg_exp(ith,ty)->opnd->bv_position());
        type_working_set[varno].opnd = exprs.lookup_arg_exp(ith,ty)->opnd;
        type_working_set[varno].replace = fg->c_handle();
        ith++;
    }

    Arg_List_Iterator iter = method_get_argument_list(m_handle);
    while((type = curr_arg(iter)) != JAVA_TYPE_END) {
        ty = (O3_Jit_Type) type;
        CONVERT_TO_INT_JIT(ty);
        if (ty == JAVA_TYPE_CLASS)
        {
            varno = _hash(exprs.lookup_arg_exp(ith,ty)->opnd->bv_position());
            // Convert "iter" into a Class_Handle.
            Loader_Exception lexc;
            type_working_set[varno].opnd = exprs.lookup_arg_exp(ith,ty)->opnd;
            type_working_set[varno].replace = get_curr_arg_class(iter, m_handle, &lexc);
        }
        ith += num_words_of_type(type);
        iter = advance_arg_iterator(iter);
    }
}

void copy_prop(Flow_Graph *fg, Expressions &exprs)
{
    extern bool O3_no_cp, O3_cp_hash;
    if (O3_no_cp)
        return;
#ifdef TRACE_O3
    fg->print_cfg("0pp");
#endif // TRACE_O3
    unsigned num_bb = fg->reassign_label();
    unsigned num_vars = exprs.reg_map.curr_tmp_reg_id();
    unsigned asize = num_vars;
    unsigned threshold = (instrumenting ? /*50000*/14000 : 14000);
    bool use_hash = false;
    if (num_bb * num_vars > threshold)
    {
        if (false || !O3_cp_hash || num_bb * CP_HASH_SIZE > threshold)
            return;
        use_hash = true;
        asize = CP_HASH_SIZE;
    }
    bool type_prop = fg->has_virtual_inline;
//    type_prop = false; // turn off type prop by enabling this statement
    Mem_Manager mem(num_bb*asize*sizeof(Operand*) + num_vars*(sizeof(Operand*)+sizeof(bool)));
    int num_nodes;
    Mem_Manager tmem(num_bb * sizeof(Cfg_Node *));
    Cfg_Node **nodearray;
    fg->create_dataflow_ordering(tmem, nodearray, num_nodes);
    Bit_Vector node_bv(num_nodes, tmem, true);

    unsigned short traversal_num = fg->traversal_num() + 1;
    fg->set_traversal_num(traversal_num);
    Prop_Closure *pc;
    if (use_hash)
        pc = new(mem) PC_Hash(mem, fg->mem_manager, exprs, fg->m_handle(), fg->cmpl_handle(),
        type_prop, num_bb, traversal_num);
    else
        pc = new(mem) PC_Exact(mem, fg->mem_manager, exprs, fg->m_handle(), fg->cmpl_handle(),
        type_prop, num_bb, traversal_num);

    unsigned pass = 1;
    do {
        pass ++;
        pc->changed = false;
        pc->for_real = false;
        pc->init_types(fg);
        int ii;
        for (ii=num_nodes-1; ii>=0; ii--)
        {
            if (node_bv.is_set(nodearray[ii]->label))
                prop_bb(nodearray[ii], pc, &node_bv);
            nodearray[ii]->latest_traversal = traversal_num;
        }
    } while (pc->changed);
    pc->for_real = true;
    pc->init_types(fg);
    int ii;
    for (ii=num_nodes-1; ii>=0; ii--)
        prop_bb(nodearray[ii], pc, &node_bv);
    
    // Now we need to adjust fg->this_pointer_of_method to its
    // propagation value at the epilog.
    // Assuming there *is* an epilog, of course.
    if (fg->epilog() != NULL && !method_is_static(fg->m_handle()))
    {
        unsigned this_bvp = fg->this_pointer_of_method->bv_position();
        pc->update_this_ptr(fg, this_bvp);
    }
}
