// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/bit_vector_group.h,v 1.2 2001/08/13 09:54:55 xhshi Exp $
//



#ifndef _BIT_VECTOR_GROUP_H
#define _BIT_VECTOR_GROUP_H

#include <iostream.h>
#include "bit_vector.h"
#include "jit_types.h"
#include "../x86/x86.h"

class Bit_Vector_Group // abstract class
{
public:
    enum MergeType { merge_jsr, merge_ret, merge_default };

    void *operator new(size_t sz, Mem_Manager & mm) { return mm.alloc(sz); }
    virtual Bit_Vector_Group *clone(Mem_Manager& mm, bool set_all) = 0;
    virtual void init(bool set_all) = 0;
    virtual void copy_from(Bit_Vector_Group *src) = 0;
    virtual void mark_live(unsigned varno, O3_Jit_Type type) = 0;
    virtual void mark_dead(unsigned varno) = 0;
    virtual unsigned is_live_var(unsigned varno) = 0;
    virtual unsigned is_live_ref(unsigned varno) = 0;
    virtual unsigned is_live_nonref(unsigned varno) = 0;
    virtual unsigned is_unknown(unsigned varno) = 0;
    virtual void merge_incoming(Bit_Vector_Group *incoming, Bit_Vector *scratch,
                                MergeType mt) = 0;
    virtual void merge_two(Bit_Vector_Group *live1, Bit_Vector_Group *live2,
                           MergeType mt) = 0;
    virtual void fixup_final(MergeType mt) = 0;
    virtual bool equals(Bit_Vector_Group *src) = 0;
    virtual unsigned available_physical_registers(unsigned n_reg) = 0;
    virtual unsigned live_refs_regs() = 0;
    virtual O3_Jit_Type approx_type(unsigned varno) = 0;
    virtual unsigned num_live_refs() = 0;
    virtual void fill_in_lr_array(unsigned *array) = 0;
    virtual unsigned num_live() = 0;
    virtual void fill_in_live_array(unsigned *array) = 0;
    virtual unsigned numbits()=0;
};

class Bit_Vector_Group_GC : public Bit_Vector_Group // another abstract class
{
public:
    virtual bool no_unknowns() = 0;
    unsigned num_live() { assert(0); return (unsigned)-1; }
    void fill_in_live_array(unsigned *array) { assert(0); }
protected: 
    Bit_Vector *_refs, *_nonrefs, *_dead;
};

class Bit_Vector_Group_GC_jsr : public Bit_Vector_Group_GC
{
public:
    Bit_Vector_Group *clone(Mem_Manager& mm, bool set_all) {
        return new(mm) Bit_Vector_Group_GC_jsr(_refs->numbits(), mm, set_all); }
    Bit_Vector_Group_GC_jsr(unsigned n_bit, Mem_Manager& mm, bool set_all) {
        _refs =    new(mm) Bit_Vector(n_bit, mm, set_all);
        _nonrefs = new(mm) Bit_Vector(n_bit, mm, set_all);
        _unknown = new(mm) Bit_Vector(n_bit, mm, set_all);
        _dead =    new(mm) Bit_Vector(n_bit, mm, true); // always set all for dead_vars
    }
    void init(bool set_all) {
        if (set_all)
        {
            _refs->set_all();
            _nonrefs->set_all();
            _unknown->set_all();
        }
        else
        {
            _refs->reset_all();
            _nonrefs->reset_all();
            _unknown->reset_all();
        }
        _dead->set_all(); // always set all for dead_vars
    }
    void copy_from(Bit_Vector_Group *src_in) {
        Bit_Vector_Group_GC_jsr *src = (Bit_Vector_Group_GC_jsr *) src_in;
        _refs->   copy_from(src->_refs);
        _nonrefs->copy_from(src->_nonrefs);
        _unknown->copy_from(src->_unknown);
        _dead->   copy_from(src->_dead);
    }
    void mark_live(unsigned varno, O3_Jit_Type type);
    void mark_dead(unsigned varno) {
        _dead->set(varno);
        _unknown->reset(varno);
        _refs->reset(varno);
        _nonrefs->reset(varno); }
    unsigned is_live_var(unsigned varno) { return (!_dead->is_set(varno)); }
    unsigned is_live_ref(unsigned varno) { return (_refs->is_set(varno)); }
    unsigned is_live_nonref(unsigned varno) { return (_nonrefs->is_set(varno)); }
    unsigned is_unknown(unsigned varno) {return (_unknown->is_set(varno)); }
    void merge_incoming(Bit_Vector_Group *incoming, Bit_Vector *scratch,
                        MergeType mt);
    void merge_two(Bit_Vector_Group *live1, Bit_Vector_Group *live2,
                   MergeType mt) {
        assert(mt == merge_default);
        Bit_Vector_Group_GC_jsr *l1 = (Bit_Vector_Group_GC_jsr *) live1;
        Bit_Vector_Group_GC_jsr *l2 = (Bit_Vector_Group_GC_jsr *) live2;
        _dead->intersect_two(l1->_dead, l2->_dead);
        _unknown->union_two(l1->_unknown, l2->_unknown);
        _refs->union_two(l1->_refs, l2->_refs);
        _refs->subtract(_unknown);
        _nonrefs->union_two(l1->_nonrefs, l2->_nonrefs);
        _nonrefs->subtract(_unknown);
    }
    void fixup_final(MergeType mt);
    bool equals(Bit_Vector_Group *src_in) {
        Bit_Vector_Group_GC_jsr *src = (Bit_Vector_Group_GC_jsr *) src_in;
        return (_dead->equals(src->_dead) &&
            /*_refs->equals(src->_refs) &&
            _nonrefs->equals(src->_nonrefs)*/
            _unknown->equals(src->_unknown)); }
    unsigned available_physical_registers(unsigned n_reg) {
        return (_dead->first_word() & ((1u << n_reg) - 1)); }
    unsigned live_refs_regs() { return _refs->first_word() & ((1u << n_reg) - 1); }
    O3_Jit_Type approx_type(unsigned varno) {
        if (is_live_nonref(varno)) return JIT_TYPE_INT;
        else if (is_live_ref(varno)) return JIT_TYPE_CLASS;
        else if (is_unknown(varno)) return JIT_TYPE_UNKNOWN;
        else assert(0);
        return JIT_TYPE_INT; }
    bool no_unknowns() { return _unknown->is_empty(); }
    unsigned num_live_refs() { return _refs->bits_set_without_regs(); }
    void fill_in_lr_array(unsigned *array) { _refs->fill_in_index_array_no_regs(array); }
    unsigned numbits() { return _refs->numbits();}
private:
    Bit_Vector *_unknown;
};

class Bit_Vector_Group_GC_nojsr : public Bit_Vector_Group_GC
{
public:
    Bit_Vector_Group *clone(Mem_Manager& mm, bool set_all) {
        return new(mm) Bit_Vector_Group_GC_nojsr(_refs->numbits(), mm, set_all); }
    Bit_Vector_Group_GC_nojsr(unsigned n_bit, Mem_Manager& mm, bool set_all) {
        _refs =    new(mm) Bit_Vector(n_bit, mm, set_all);
        _nonrefs = new(mm) Bit_Vector(n_bit, mm, set_all);
        _dead =    new(mm) Bit_Vector(n_bit, mm, true); // always set all for dead_vars
    }
    void init(bool set_all) {
        set_all = false;
        if (set_all)
        {
            _refs->set_all();
            _nonrefs->set_all();
        }
        else
        {
            _refs->reset_all();
            _nonrefs->reset_all();
        }
        _dead->set_all(); // always set all for dead_vars
    }
    void copy_from(Bit_Vector_Group *src_in) {
        Bit_Vector_Group_GC_nojsr *src = (Bit_Vector_Group_GC_nojsr *) src_in;
        _refs->   copy_from(src->_refs);
        _nonrefs->copy_from(src->_nonrefs);
        _dead->   copy_from(src->_dead);
    }
    void mark_live(unsigned varno, O3_Jit_Type type);
    void mark_dead(unsigned varno) {
        _dead->set(varno);
        _refs->reset(varno);
        _nonrefs->reset(varno); }
    unsigned is_live_var(unsigned varno) { return (!_dead->is_set(varno)); }
    unsigned is_live_ref(unsigned varno) { return (_refs->is_set(varno)); }
    unsigned is_live_nonref(unsigned varno) { return (_nonrefs->is_set(varno)); }
    unsigned is_unknown(unsigned varno) {return false; }
    void merge_incoming(Bit_Vector_Group *incoming, Bit_Vector *scratch,
                        MergeType mt);
    void merge_two(Bit_Vector_Group *live1, Bit_Vector_Group *live2,
                   MergeType mt) {
        assert(mt == merge_default);
        Bit_Vector_Group_GC_nojsr *l1 = (Bit_Vector_Group_GC_nojsr *) live1;
        Bit_Vector_Group_GC_nojsr *l2 = (Bit_Vector_Group_GC_nojsr *) live2;
        _dead->intersect_two(l1->_dead, l2->_dead);
        _refs->union_two(l1->_refs, l2->_refs);
        _nonrefs->union_two(l1->_nonrefs, l2->_nonrefs);
    }
    void fixup_final(MergeType mt);
    bool equals(Bit_Vector_Group *src_in) {
        Bit_Vector_Group_GC_nojsr *src = (Bit_Vector_Group_GC_nojsr *) src_in;
        return (_dead->equals(src->_dead)); }
    unsigned available_physical_registers(unsigned n_reg) {
        return (_dead->first_word() & ((1u << n_reg) - 1)); }
    unsigned live_refs_regs() { return _refs->first_word() & ((1u << n_reg) - 1); }
    O3_Jit_Type approx_type(unsigned varno) {
        if (is_live_nonref(varno)) return JIT_TYPE_INT;
        else if (is_live_ref(varno)) return JIT_TYPE_CLASS;
        else assert(0);
        return JIT_TYPE_INT; }
    bool no_unknowns() { return true; }
    unsigned num_live_refs() { return _refs->bits_set_without_regs(); }
    void fill_in_lr_array(unsigned *array) { _refs->fill_in_index_array_no_regs(array); }
    unsigned numbits() { return _refs->numbits();}
};

class Bit_Vector_Group_Elim : public Bit_Vector_Group
{
public:
    Bit_Vector_Group *clone(Mem_Manager& mm, bool set_all) {
        return new(mm) Bit_Vector_Group_Elim(_dead->numbits(), mm,set_all); }
    Bit_Vector_Group_Elim(unsigned n_bit, Mem_Manager& mm, bool set_all) {
        _dead = new(mm) Bit_Vector(n_bit, mm,true);
    }
    void init(bool set_all) {
        _dead->set_all();
    }
    void copy_from(Bit_Vector_Group *src_in) {
        Bit_Vector_Group_Elim *src = (Bit_Vector_Group_Elim *) src_in;
        _dead->copy_from(src->_dead);
    }
    void mark_live(unsigned varno, O3_Jit_Type type) { _dead->reset(varno); }
    void mark_dead(unsigned varno) { _dead->set(varno); }
    unsigned is_live_var(unsigned varno) { return (!_dead->is_set(varno)); }
    unsigned is_live_ref(unsigned varno) { return 0; }
    unsigned is_live_nonref(unsigned varno) { return 0; }
    unsigned is_unknown(unsigned varno) { return is_live_var(varno); }
    void merge_incoming(Bit_Vector_Group *incoming_in, Bit_Vector *scratch,
                        MergeType mt) {
        if (incoming_in == NULL) return;
        Bit_Vector_Group_Elim *incoming = (Bit_Vector_Group_Elim *) incoming_in;
        _dead->intersect_from(incoming->_dead); }
    void merge_two(Bit_Vector_Group *live1, Bit_Vector_Group *live2,
                   MergeType mt) {
        assert(mt == merge_default);
        Bit_Vector_Group_Elim *l1 = (Bit_Vector_Group_Elim *) live1;
        Bit_Vector_Group_Elim *l2 = (Bit_Vector_Group_Elim *) live2;
        _dead->intersect_two(l1->_dead, l2->_dead);
    }
    void fixup_final(MergeType mt) {}
    bool equals(Bit_Vector_Group *src_in) {
        Bit_Vector_Group_Elim *src = (Bit_Vector_Group_Elim *) src_in;
        return _dead->equals(src->_dead); }
    unsigned available_physical_registers(unsigned n_reg) {
        return (_dead->first_word() & ((1u << n_reg) - 1)); }
    unsigned live_refs_regs() { assert(0); return 0; }
    O3_Jit_Type approx_type(unsigned varno) {
        assert(is_live_var(varno)); return JIT_TYPE_INT; }
    unsigned num_live_refs() { assert(0); return (unsigned)-1; }
    void fill_in_lr_array(unsigned *array) { assert(0); }
    unsigned num_live();
    void fill_in_live_array(unsigned *array) {
        _dead->fill_in_index_array_inverted(array); }
    unsigned numbits() { return _dead->numbits();}
private: 
    Bit_Vector *_dead;
};

#endif _BIT_VECTOR_GROUP_H
