// Copyright (C)  2001 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/dump/ia32_dump.cpp,v 1.14 2001/12/13 07:31:39 xhshi Exp $
//


#include "stdio.h"
#include "string.h"
#include "stdlib.h"
#include "assert.h"

#ifdef TRACE_O3
#define ARROW "-+"
#else
#define ARROW "->"
#endif

const char * spaces = "                                                     ";

typedef int (*handler_proto) (const unsigned char *, char *);

typedef struct {
    const char * name;
    handler_proto handler;
    const char * name0f;
    handler_proto handler0f;
} dump_handler;

const char * reg_names8[] =
{
"al","cl","dl","bl","ah","ch","dh","bh"
};



const char * reg_names16[] =
{
"ax","cx","dx","bx","sp","bp","si","di"
};

const char * reg_names32[] =
{
"eax","ecx","edx","ebx","esp","ebp","esi","edi"
};

const char ** regs_long = reg_names32;


const char * cond_names[] =
{
    "o","no","b","nb","e","ne","be","nbe",
    "s","ns","p","np","l","ge","le","g"
};

inline int is_small(int val)
{
    return (val > -10000) && (val < 10000);
}


int get_s4(const unsigned char * code)
{
    int answer = 0;    
    answer += code[3];
    answer = (answer<<8) + code[2];
    answer = (answer<<8) + code[1];
    answer = (answer<<8) + code[0];   
    return answer;
}


int get_slong(const unsigned char * code)
{
    int answer = 0;
    if ( regs_long == reg_names32 )
    {
        answer += code[3];
        answer = (answer<<8) + code[2];
        answer = (answer<<8) + code[1];
        answer = (answer<<8) + code[0];
    }
    else
    {
        answer += (signed int)code[1];
        answer = (answer<<8) + code[0];
    }
    return answer;
}

int get_slong_len()
{
    if ( regs_long == reg_names32 )
        return 4;
    else
        return 2;
}


char sibbuf[80];
char displacestr[15];

char * sib(int w, int mod, int rm, const unsigned char *code, int * length)
{
    
    const char ** rnames = w ? regs_long : reg_names8;
    int displacement = 0;
    int sib = (rm == 4);
    
    if ( mod == 0 )
    {
        if ( rm == 5 )
        {
            sprintf(sibbuf,"[%p]",get_s4(code));
            *length += 4;
            return sibbuf;
        }
        
        if ( !sib )
        {
            sprintf(sibbuf,"[%s]",reg_names32[rm]);
            return sibbuf;
        }
    }
    else if ( mod == 1)
    {
        displacement = (signed char)code[sib ? 1: 0];
        *length += 1;
        if ( !sib )
        {
            sprintf(sibbuf,"[%s%+d]",reg_names32[rm],displacement);
            return sibbuf;
        }
    }
    else if ( mod == 2)
    {
        displacement = get_s4(&code[sib ? 1: 0]);
        *length += 4;
        if ( !sib )
        {
            sprintf(sibbuf,"[%s%+d]",reg_names32[rm],displacement);
            return sibbuf;
        }       
    }
    else if ( mod == 3)
    {
        sprintf(sibbuf,rnames[rm]);
        return sibbuf;
    }
    
    // let's decode sib - we already know displacement
    *length += 1;   
    sib = code[0];
    int scale = (sib >> 6) & 3;
    int index = (sib>>3)&7;
    int base = sib&7;
    
    const char * scalestr;  
    
    if ( displacement == 0 )
        sprintf(displacestr,"");
    else
        sprintf(displacestr,"%+d",displacement);
    
    if ( scale == 0 )
        scalestr = "";
    else if (scale == 1)
        scalestr = "*2";
    else if ( scale == 2)
        scalestr = "*4";
    else 
        scalestr = "*8";
        
    
    
    if ( index == 4 )
    {
        sprintf(sibbuf,"[%s%s]",reg_names32[base],displacestr);
        return sibbuf;
    }
    else
    {
        sprintf(sibbuf,"[%s%s+%s%s]", reg_names32[base],displacestr,reg_names32[index],scalestr);
        return sibbuf;
    }
    // fixme what with funny sib ebp addressing ?*/
}


char * esib(const unsigned char * code, int * length)
{
    int w = (code[0] & 1);
    int mod = (code[1]>>6)&3;
    int rm = (code[1])&7;
    return sib(w,mod,rm,&code[2],length);
}


int reg0(const unsigned char * code, char * buf)
{
    sprintf(buf,regs_long[(*code)&7]);
    return 1;
}

int reg1a_to_b(const unsigned char * code, char * buf)
{
    const char ** rnames = (code[0] & 1) ? regs_long : reg_names8;
    int reg = (code[1]>>3)&7;
    int length = 2;
    sprintf(buf,"%s %s %s",rnames[reg],ARROW,esib(code,&length));
    return length;
}

int reg1b_to_a(const unsigned char * code, char * buf)
{
    const char ** rnames = (code[0] & 1) ? regs_long : reg_names8;
    int reg = (code[1]>>3)&7;
    int length = 2;
    sprintf(buf,"%s %s %s",esib(code,&length),ARROW, rnames[reg]);
    return length;
}

int movxx(const unsigned char * code, char * buf)
{
    int length = 2;
    int w = code[0] & 1;
    int mod = (code[1] >> 6) & 3;
    int rm = (code[1])&7;
    int reg = ((code[1])>>3)&7;
    if ( mod == 3 )
    {
        if ( !w )
        {
            sprintf(buf, "%s %s %s", reg_names8[rm],ARROW, regs_long[reg]);
        }
        else
        {
            sprintf(buf, "%s %s %s", reg_names16[rm], ARROW, regs_long[reg]);
        }         
    }
    else
    {
        if ( !w )
        {              
            sprintf(buf,"byte %s %s %s",esib(code,&length), ARROW, regs_long[reg]);
        }
        else
        {
            sprintf(buf,"short %s %s %s",esib(code,&length),ARROW, regs_long[reg]);
        }
    }
    return length;
 
}

int lea1b_to_a(const unsigned char * code, char * buf)
{
    int length = reg1b_to_a(code,buf);
    char * c = strchr(buf,'[');
    if ( c )
        *c='(';
    c = strchr(buf,']');
    if ( c )
        *c = ')';
    return length;
}    

int imm_eax(const unsigned char * code, char * buf)
{
    int w = code[0]&1;
    if ( w )
    {
        sprintf(buf,"%#x %s %s", get_slong(&code[1]), ARROW,regs_long[0]);
        return 1+ get_slong_len();
    }
    else
    {
        sprintf(buf,"%#x %s %s", (signed int)code[1], ARROW, reg_names8[0]);
        return 2;
    }
}

int imm_to_reg(const unsigned char * code, char * buf)
{
    int w = (code[0]&0x8);
    const char ** rnames =  w ? regs_long : reg_names8;
    int imm = 0;
    int length = 1;
    if ( w )
    {
        imm = get_slong(&code[1]);
        length += get_slong_len();        
    }
    else
    {
        imm = (signed int)(code[1]);
        length++;
    }
    sprintf(buf,"%#x %s %s", imm, ARROW, rnames[code[0]&7]);
    return length;
}

int imm_to_rm(const unsigned char * code, char * buf)
{
    const char ** rnames =  (code[0]&1) ? regs_long : reg_names8;
    int length = 2;
    char * s = esib(code,&length);
    int val = get_slong(&code[length]);
    if ( is_small(val) )
        sprintf(buf, "%d %s %s",val,ARROW,s);
    else
        sprintf(buf,"%#x %s %s",val,ARROW,s);    
    length += get_slong_len();
    return length;
}

int mul_imm8(const unsigned char * code, char * buf)
{
    int length = 2;
    char * str = esib(code,&length);
    int reg = ((code[1])>>3)&7;
    int s = code[0]&0x2;
    int imm;
    if ( s )
        imm = (signed char)code[length];
    else
        imm = code[length];
    length++;    
    sprintf(buf,"%s*%d %s %s", str, imm, ARROW, regs_long[reg]);
    return length;        
}

int mul_imm32(const unsigned char * code, char * buf)
{
    int length = 2;
    char * str = esib(code,&length);
    int reg = ((code[1])>>3)&7;
    int s = code[0]&0x2;
    int imm = get_slong(&code[2]);
    length += get_slong_len();    
    sprintf(buf,"%s*%d %s %s", str, imm, ARROW,regs_long[reg]);
    return length;        
}


int imm16(const unsigned char * code, char * buf)
{
    int i = code[1] + (code[2]<<8);
    if ( code[2] & 0x80 )
    {
        i = 0xffff0000 | i;
    }   
    sprintf(buf,"%d",i);
    return 3;
}

int imm32(const unsigned char * code, char * buf)
{
    int val = get_slong(&code[1]);
    if ( is_small(val) )
        sprintf(buf, "%d", val);
    else
        sprintf(buf,"%#x",val);
    return 1 + get_slong_len();
}

int ip8offset(const unsigned char * code, char * buf)
{
    sprintf(buf,"%p",((int)code) + 2 + ((signed char)code[1]));
    return 2;
}


int ip32offset(const unsigned char * code, char * buf)
{
    sprintf(buf,"%p",((int)code) + 5 +  get_slong(&code[1]));
    return 1+get_slong_len();
}



int x80_group(const unsigned char * code, char * buf)
{
    int mod = (code[1]>>6)&3;
    int type = (code[1]>>3)&7;
    int rm = (code[1])&7;
    int w = code[0]&1;
    int s = code[0]&2;    
    int length = 2;
    char tmpbuf[80];
    
    static const char * group_80_names[] =
    {
        "add","or","adc","sbb",
        "and","sub","xor","cmp"
    };
    
    sprintf ( tmpbuf, "%s %%d %s %s", group_80_names[type], ARROW,esib(code,&length));       
    
    int imm = 0;    
    
    if ( !s )
    {
        imm = get_slong(&code[length]);
        length += get_slong_len();
    }
    else
    {
        imm = (signed int) code[length];
        length++;
    }
    
    sprintf(buf,tmpbuf,imm);
    return length;
}

int shift_group(const unsigned char * code, char * buf)
{
    int mod = (code[1]>>6)&3;
    int type = (code[1]>>3)&7;
    int rm = (code[1])&7;
    int w = code[0]&1;
    int length = 2;
    const char ** rnames =  w ? regs_long : reg_names8;
    static const char * group_shift_names[] = 
    {
        "rol","ror","rcl","rcr",
        "sal","shr","???","sar"
    };
    int shift = code[0] & 0x12;
    const char * strsib = esib(code,&length);
    
    if ( shift == 0x10 )
    {    
        sprintf(buf,"%s 1 %s %s", group_shift_names[type], ARROW,strsib);
    }
    else if ( shift == 0x12 )
    {
        sprintf(buf,"%s cl %s %s", group_shift_names[type], ARROW,strsib);
    }
    else if ( shift == 0x00 )
    {
        sprintf(buf,"%s %d %s %s", group_shift_names[type], code[length], ARROW,strsib);
        length++;
    }
    else
    {
        printf("Unknown shift type");
        sprintf(buf,"unknown");
    }
    return length;
                
}

int xf6_group(const unsigned char * code, char * buf)
{
    int mod = (code[1]>>6)&3;
    int type = (code[1]>>3)&7;
    int rm = (code[1])&7;
    int w = code[0]&1;
    int length = 2;
    const char ** rnames =  w ? regs_long : reg_names8;
    
    static const char * group_f6_names[] = 
    {
        "test","???","not","neg",
        "mul","imul","div","idiv"
    };
    
    switch (type)
    {
        case 3:
            sprintf(buf, "%s %s", group_f6_names[type],esib(code,&length));
            return length;
                          
        case 4:
            sprintf(buf, "%s %s", group_f6_names[type],esib(code,&length));
            return length;
                          
        case 7:
            sprintf(buf,"%s %s/%s %s %s,%s",group_f6_names[type],rnames[0],esib(code,&length), ARROW, rnames[0],rnames[2]);
            return length;
                          
        default:
            printf("ia_dump32:Uknown f6-group, type: %d\n",type);
            return 2;
    }
    
}



int xff_group(const unsigned char * code, char * buf)
{
    int mod = (code[1]>>6)&3;
    int type = (code[1]>>3)&7;
    int rm = (code[1])&7;
    int w = code[0]&1; // only for inc/dec
    int length = 2;
    
    static const char * group_ff_names[] = 
    {
        "inc","dec","call","callfar",
        "jmp","jmpfar","push","???"
    };
    
    switch (type)
    {
        case 0:
        case 1:
            sprintf(buf, "%s %s", group_ff_names[type],esib(code,&length));
            return length;        
              
        case 2:
        case 4:
        case 6:
            sprintf(buf,"%s %s",group_ff_names[type],sib(1,mod,rm,&code[2],&length));
            return length;
                          
        default:
            printf("ia_dump32:Uknown ff-group, type: %d\n",type);
            return 2;
    }
    
}

int x8f(const unsigned char * code, char * buf)
{
    int length = 2;
    int rm = (code[1])&7;
    sprintf(buf,"%s %s","pop",sib(1,1,rm,&code[2],&length));
    return length;
}

const char * fp_d9[] =
{
    "fchs","fabs","?","?",
    "ftst","fxam","?","?",
    "fld1","fldl2t","lfdl2e","fldpi",
    "fldlg2","fldln2","fldz","?",
    "f2xm1","fyl2x","fptan","fpatan",
    "fxtract","fprem","fdecstp","fincstp",
    "fprem","fyl2xp1","fsqrt","fsincos",
    "frndint","fscale","fsin","fcos"
};

const char * fp_db[] =
{
    "?","?","?","?",
    "?","?","?","?",
    "?","?","?","?",
    "?","?","?","?",
    "?","?","?","?",
    "?","?","?","?",
    "?","?","?","?",
    "?","?","?","?"
};

#if 0
const char * fp_st0[] =
{
    "fadd r32","fmul r32","fcom r32","fcomp r32","fsub r32","fsubr r32","fdiv r32","fdivr r32", // d8
//    "?","?","?","?","?","?","?","?", // d9
    "","?","?","?","?","?","?","?", // d9
    "fiadd i64","fimul i64","ficom i64","ficomp i64","fisub i64","fisubr i64","fidiv i64","fidivr i64", // da
    "?","?","?","?","?"
	,"?","?","?", // db
    "fadd r64","fmul r64","fcom r64","fcomp r64","fsub r64","fsubr r64","fdiv r64","fdivr r64", // dc
    "?","?","?","?","?","?","?","?", // dd
    "fiadd i32","fimul i32","ficom i32","ficomp i32","fisub i32","fisubr i32","fidiv i32","fidivr i32", // de
    "?","?","?","?","?","?","?","?"  // df
};
#endif

#if 1
const char * fp_st0[] =
{
    "fadd","fmul","fcom","fcomp","fsub","fsubr","fdiv","fdivr", // d8
    "fld","?","fst","fstp","fldenv","fldcw","fnstenv","fnstcw", // d9
    "fiadd","fimul","ficom","ficomp","fisub","fisubr","fidiv","fidivr", // da
    "fild","?","fist","fistp","?","?","?","?", // db
    "fadd","fmul","fcom","fcomp","fsub","fsubr","fdiv","fdivr", // dc
    "fld","?","fst","fstp","fldenv","fldcw","fnstenv","fnstcw", // dd
    "fiadd","fimul","ficom","ficomp","fisub","fisubr","fidiv","fidivr", // de
    "fild","?","fist","fistp","?","fild","?","?"  // df
};
#endif

const char * fp_sti[] =
{
    "fadd","fmul","fcom","fcomp","fsub","fsubr","fdiv","fdivr", //d8
    "fld","fxch","?","?","?","?","?","?", //d9
    "?","?","?","?","?","?","?","?", //da
    "?","?","?","?","?","?","?","?", //db
    "fadd","fmul","?","?","fsubr","fsub","fdivr","fdiv", //dc
    "1ffree","?","1fst","1fstp","?","?","?","?", //dd
    "faddp","fmulp","?","fcompp","fsubrp","fsubp","fdivrp","fdivp", //de fcompp:de d9(51)
    "?","?","?","?","fnstsw","?","fcomip","?" //df fnstsw:df e0(60)

};


int fpu(const unsigned char * code, char * buf)
{
    int opcode = (code[0] <<8) | code[1];
    if (code[0] == 0xd9) 
	{
		int xxx = 0;
	}
	int length =2;
    int mod = (opcode &0x00c0) >> 6;

    if ( mod != 3 )
    {
        int rm = (opcode &0x0007);
//        int entry = ((opcode &0x0700) >> 8) | ((opcode & 0x0038)>>3);
        int entry = ((opcode &0x0700) >> 5) | ((opcode & 0x0038)>>3);
        //::printf(" using entry %d\n", entry);
        sprintf(buf, "%s %s %s st(0)", fp_st0[entry],sib(1,mod,rm,&code[2],&length), ARROW );
        if ( buf[0] == '?' )
            printf("Unknown fp_st0 entry %d", entry);
        return length;                
    }
            
    if ( (opcode & 0xffe0) == 0xd9e0 )
    {
        sprintf(buf,fp_d9[opcode&0x1f]);
        if ( buf[0] == '?' )
            printf("unknown fpu d9 code %x\n",opcode&0x1f);
        return length;        
    }
    else if ( (opcode & 0xffe0) == 0xdbe0 )
    {
        sprintf(buf,fp_db[opcode&0x1f]);
        if ( buf[0] == '?' )
            printf("unknown fpu db code %x\n",opcode&0x1f);
        return length;
    }
    else
    {
        int dpabrr = ((code[0]&7)<<3) | ((code[1]>>3)&7);
        int sti = code[1]&7;
        const char * txt = fp_sti[dpabrr];
        if (*txt == '?')
        {        
            //::printf("unknown fpu dpabrr code %x %x\n",code[0], code[1]);
            length = 6;
            sprintf(buf,"unknownfp");
            return length;
        }
        
        if ( *txt == '1' )
        {
            sprintf(buf,"%s st(i)", txt+1,sti);
        }        
        else if ( code[0]&4)
        {
            sprintf(buf, "%s st(0) %s st(%d)", txt, ARROW, sti);
        }
        else
        {
            sprintf(buf, "%s st(%d) %s st(0)", txt, sti, ARROW);
        }
        return length;        
    }    
}

dump_handler basic_table[256] =
{
    // 0x
    {"add",reg1a_to_b, /*|*/ 0,0},
    {"add",reg1a_to_b, /*|*/ 0,0},
    {"add",reg1b_to_a, /*|*/ 0,0},
    {"add",reg1b_to_a, /*|*/ 0,0},
    {"add",imm_eax, /*|*/ 0,0},
    {"add",imm_eax, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {"or",reg1a_to_b, /*|*/ 0,0},
    {"or",reg1a_to_b, /*|*/ 0,0},
    {"or",reg1b_to_a, /*|*/ 0,0},
    {"or",reg1b_to_a, /*|*/ 0,0},
    {"or",imm_eax, /*|*/ 0,0},
    {"or",imm_eax, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},  
    // 1x
    {"adc",reg1a_to_b, /*|*/ 0,0},
    {"adc",reg1a_to_b, /*|*/ 0,0},
    {"adc",reg1b_to_a, /*|*/ 0,0},
    {"adc",reg1b_to_a, /*|*/ 0,0},
    {"adc",imm_eax, /*|*/ 0,0},
    {"adc",imm_eax, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {"sbb",reg1a_to_b, /*|*/ 0,0},
    {"sbb",reg1a_to_b, /*|*/ 0,0},
    {"sbb",reg1b_to_a, /*|*/ 0,0},
    {"sbb",reg1b_to_a, /*|*/ 0,0},
    {"sbb",imm_eax, /*|*/ 0,0},
    {"sbb",imm_eax, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    // 2x
    {"and",reg1a_to_b, /*|*/ 0,0},
    {"and",reg1a_to_b, /*|*/ 0,0},
    {"and",reg1b_to_a, /*|*/ 0,0},
    {"and",reg1b_to_a, /*|*/ 0,0},
    {"and",imm_eax, /*|*/ 0,0},
    {"and",imm_eax, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {"sub",reg1a_to_b, /*|*/ 0,0},
    {"sub",reg1a_to_b, /*|*/ 0,0},
    {"sub",reg1b_to_a, /*|*/ 0,0},
    {"sub",reg1b_to_a, /*|*/ 0,0},
    {"sub",imm_eax, /*|*/ 0,0},
    {"sub",imm_eax, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    // 3x
    {"xor",reg1a_to_b, /*|*/ 0,0},
    {"xor",reg1a_to_b, /*|*/ 0,0},
    {"xor",reg1b_to_a, /*|*/ 0,0},
    {"xor",reg1b_to_a, /*|*/ 0,0},
    {"xor",imm_eax, /*|*/ 0,0},
    {"xor",imm_eax, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {"cmp",reg1a_to_b, /*|*/ 0,0},
    {"cmp",reg1a_to_b, /*|*/ 0,0},
    {"cmp",reg1b_to_a, /*|*/ 0,0},
    {"cmp",reg1b_to_a, /*|*/ 0,0},
    {"cmp",imm_eax, /*|*/ 0,0},
    {"cmp",imm_eax, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    // 4x
    {"inc",reg0, /*|*/ "cmovo",reg1b_to_a},
    {"inc",reg0, /*|*/ "cmovno",reg1b_to_a},
    {"inc",reg0, /*|*/ "cmovb",reg1b_to_a},
    {"inc",reg0, /*|*/ "cmovnb",reg1b_to_a},
    {"inc",reg0, /*|*/ "cmove",reg1b_to_a},
    {"inc",reg0, /*|*/ "cmovne",reg1b_to_a},
    {"inc",reg0, /*|*/ "cmovbe",reg1b_to_a},
    {"inc",reg0, /*|*/ "cmova",reg1b_to_a},
    {"dec",reg0, /*|*/ "cmovs",reg1b_to_a},
    {"dec",reg0, /*|*/ "cmovns",reg1b_to_a},
    {"dec",reg0, /*|*/ "cmovp",reg1b_to_a},
    {"dec",reg0, /*|*/ "cmovnp",reg1b_to_a},
    {"dec",reg0, /*|*/ "cmovl",reg1b_to_a},
    {"dec",reg0, /*|*/ "cmovge",reg1b_to_a},
    {"dec",reg0, /*|*/ "cmovle",reg1b_to_a},
    {"dec",reg0, /*|*/ "cmovg",reg1b_to_a},
    // 5x
    {"push",reg0, /*|*/ 0,0},
    {"push",reg0, /*|*/ 0,0},
    {"push",reg0, /*|*/ 0,0},
    {"push",reg0, /*|*/ 0,0},
    {"push",reg0, /*|*/ 0,0},
    {"push",reg0, /*|*/ 0,0},
    {"push",reg0, /*|*/ 0,0},
    {"push",reg0, /*|*/ 0,0},
    {"pop",reg0, /*|*/ 0,0},
    {"pop",reg0, /*|*/ 0,0},
    {"pop",reg0, /*|*/ 0,0},
    {"pop",reg0, /*|*/ 0,0},
    {"pop",reg0, /*|*/ 0,0},
    {"pop",reg0, /*|*/ 0,0},
    {"pop",reg0, /*|*/ 0,0},
    {"pop",reg0, /*|*/ 0,0},
    // 6x
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {"push",imm32, /*|*/ 0,0},
    {"imul",mul_imm32, /*|*/ 0,0},
    {"push",imm32, /*|*/ 0,0},
    {"imul",mul_imm8, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    // 7x
    {"jo",ip8offset, /*|*/ 0,0},
    {"jno",ip8offset, /*|*/ 0,0},
    {"jb",ip8offset, /*|*/ 0,0},
    {"jnb",ip8offset, /*|*/ 0,0},
    {"je",ip8offset, /*|*/ 0,0},
    {"jne",ip8offset, /*|*/ 0,0},
    {"jbe",ip8offset, /*|*/ 0,0},
    {"ja",ip8offset, /*|*/ 0,0},
    {"js",ip8offset, /*|*/ 0,0},
    {"jns",ip8offset, /*|*/ 0,0},
    {"jp",ip8offset, /*|*/ 0,0},
    {"jnp",ip8offset, /*|*/ 0,0},
    {"jl",ip8offset, /*|*/ 0,0},
    {"jge",ip8offset, /*|*/ 0,0},
    {"jle",ip8offset, /*|*/ 0,0},
    {"jg",ip8offset, /*|*/ 0,0},
    // 8x
    {0,x80_group, /*|*/ "jo",ip32offset},
    {0,x80_group, /*|*/ "jno",ip32offset},
    {0,x80_group, /*|*/ "jb",ip32offset},
    {0,x80_group, /*|*/ "jnb",ip32offset},
    {"test",reg1b_to_a, /*|*/ "je",ip32offset},
    {"test",reg1b_to_a, /*|*/ "jne",ip32offset},
    {0,0, /*|*/ "jbe",ip32offset},
    {0,0, /*|*/ "ja",ip32offset},
    {"mov",reg1a_to_b, /*|*/ "js",ip32offset},
    {"mov",reg1a_to_b, /*|*/ "jns",ip32offset},
    {"mov",reg1b_to_a, /*|*/ "jp",ip32offset},
    {"mov",reg1b_to_a, /*|*/ "jnp",ip32offset},
    {0,0, /*|*/ "jl",ip32offset},
    {"lea",lea1b_to_a, /*|*/ "jge",ip32offset},
    {0,0, /*|*/ "jle",ip32offset},
    {0,x8f, /*|*/ "jg",ip32offset},
    // 9x
    {"nop",0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {"cwde",0, /*|*/ 0,0},
    {"cwd",0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {"sahf",0, /*|*/ 0,0}, //9e SAHF length=1
    {0,0, /*|*/ 0,0},
    // ax
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ "imul",reg1b_to_a},
    // bx
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ "movzx",movxx},
    {"mov",imm_to_reg, /*|*/ "movzx",movxx},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ 0,0},
    {"mov",imm_to_reg, /*|*/ "movsx",movxx},
    {"mov",imm_to_reg, /*|*/ "movsx",movxx},
    // cx
    {0,shift_group, /*|*/ 0,0},
    {0,shift_group, /*|*/ 0,0},
    {"ret",imm16, /*|*/ 0,0},
    {"ret",0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {"mov",imm_to_rm, /*|*/ 0,0},
    {"mov",imm_to_rm, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    // dx
    {0,shift_group, /*|*/ 0,0},
    {0,shift_group, /*|*/ 0,0},
    {0,shift_group, /*|*/ 0,0},
    {0,shift_group, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,fpu, /*|*/ 0,0},
    {0,fpu, /*|*/ 0,0},
    {0,fpu, /*|*/ 0,0},
    {0,fpu, /*|*/ 0,0},
    {0,fpu, /*|*/ 0,0},
    {0,fpu, /*|*/ 0,0},
    {0,fpu, /*|*/ 0,0},
    {0,fpu, /*|*/ 0,0},
    // ex
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {"call",ip32offset, /*|*/ 0,0},
    {"jmp",ip32offset, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {"jmp",ip8offset, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    // fx
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,xf6_group, /*|*/ 0,0},
    {0,xf6_group, /*|*/ 0,0},  
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,0, /*|*/ 0,0},
    {0,xff_group}
};
int inited = 0;



void init_tables()
{
    inited = 1;

}


// function returns length of opcode
int ia32_dump(const unsigned char *code, char * out_buffer,bool addr_and_bytes, bool instructions)
{
    static int success = 0;    
    static int failed = 0;
    
    if ( !inited )
        init_tables();
        
    const char * opcode_name = "unknown";
    const char * params = "";
    char param_buf[50];
    int length = 0;

   

    
    if ( code[length] == 0x66 )
    {
        regs_long = reg_names16;
        length++;
    }
    else
    {
        regs_long = reg_names32;
    }
    
    if ( code[length] == 0x0f)
    {
        length++;
        dump_handler * dh = &basic_table[code[length]];
        if (  dh->name0f || dh->handler0f)
        {
            opcode_name = dh->name0f;
            if ( dh->handler0f )
            {
                length += (*dh->handler0f)(code+length,param_buf);
                params = param_buf;
            }
            else
            {
                length++;
            }  
            
            if ( !opcode_name )
            {
                opcode_name = params;
                params = "";
            }
            success++;
        }
        else
        {
            //::printf ( "0x0f handler for %x not-found\n",code[1]);
            length = 6;
            failed++;
        }    
    }    
    else 
    {
		dump_handler * dh = &basic_table[code[length]];        
        if (  dh->name || dh->handler)
        {
            opcode_name = dh->name;
            if ( dh->handler )
            {
                length += (*dh->handler)(code+length,param_buf);
                params = param_buf;
            }
            else
            {
                length++;                
            }
            
            if ( !opcode_name )
            {
                opcode_name = params;
                params = "";
            }
            success++;
        }
        else
        {
            //::printf("Handler for %x %x %x not found \n", code[0], code[1], code[2]);
            length = 6;
            failed++;
        }
    }
    
    //printf("+%d -%d %f%%\n",success,failed, (success*100.0)/(success+failed));
    
    if (out_buffer)
    {
        sprintf(out_buffer,"%p %s %s", code, opcode_name, params);
        strcat(out_buffer,&spaces[strlen(out_buffer)]);
        int i;
        for ( i =0; i< length; i++)
        {
            sprintf(&out_buffer[strlen(out_buffer)]," %2.2x",code[i]);
        }   

	//::for dot files
	if (addr_and_bytes && !instructions)
	{
		out_buffer += sprintf(out_buffer, "%X ", (unsigned)code);
		for (i=0; i<length; i++)
			out_buffer += sprintf(out_buffer, "%02X", (code[i] & 0xff));
		for (; i<9; i++)
			out_buffer += sprintf(out_buffer, "  ");
			out_buffer += sprintf(out_buffer, " ");
		}
		if(instructions && !addr_and_bytes)
		{
	        out_buffer += sprintf(out_buffer,"%s", opcode_name);
			for(i = strlen(opcode_name) ; i < 9 ; i++)
				out_buffer += sprintf(out_buffer," ") ;
			out_buffer += sprintf(out_buffer,"%s", params) ;
		}
		
    }
    return length;
}
