/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.ruby.rubyproject;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.Charset;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectManager;
import org.netbeans.modules.ruby.rubyproject.ui.customizer.RubyProjectProperties;
import org.netbeans.modules.ruby.spi.project.support.rake.RakeProjectHelper;
import org.netbeans.modules.ruby.spi.project.support.rake.EditableProperties;
import org.netbeans.modules.ruby.spi.project.support.rake.ProjectGenerator;
import org.netbeans.api.queries.FileEncodingQuery;
import org.netbeans.modules.ruby.spi.project.support.rake.ReferenceHelper;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.filesystems.Repository;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.util.Mutex;
import org.openide.util.MutexException;
import org.openide.ErrorManager;
import org.openide.filesystems.FileAlreadyLockedException;
import org.openide.filesystems.FileLock;
import org.openide.util.NbBundle;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * Creates a RubyProject from scratch according to some initial configuration.
 */
public class RubyProjectGenerator {
    
    public static final String DEFAULT_SRC_NAME = "src.dir"; // NOI18N
    public static final String DEFAULT_TEST_SRC_NAME = "test.src.dir"; // NOI18N

    private RubyProjectGenerator() {}
    
    /**
     * Create a new empty Ruby project.
     * @param dir the top-level directory (need not yet exist but if it does it must be empty)
     * @param name the name for the project
     * @return the helper object permitting it to be further customized
     * @throws IOException in case something went wrong
     */
    public static RakeProjectHelper createProject(File dir, String name, String mainClass) throws IOException {
        FileObject dirFO = FileUtil.createFolder(dir);
        RakeProjectHelper h = createProject(dirFO, name, "lib", "test", mainClass, false); //NOI18N
        Project p = ProjectManager.getDefault().findProject(dirFO);
        ProjectManager.getDefault().saveProject(p);
        FileObject srcFolder = dirFO.createFolder("lib"); // NOI18N
        dirFO.createFolder("test"); // NOI18N
        if ( mainClass != null ) {
            createFromTemplate( mainClass, srcFolder, "Templates/Ruby/main.rb" ); // NOI18N
        }
        DataObject rakeFileDO = createFromTemplate( "Rakefile.rb", dirFO, "Templates/Ruby/rakefile.rb" ); // NOI18N
        if (rakeFileDO != null) {
            rename(rakeFileDO.getPrimaryFile(), "Rakefile", null); // NOI18N
        }
        FileObject readme = dirFO.createData("README"); // NOI18N
        writeLines(readme, NbBundle.getMessage(RubyProjectGenerator.class, "TXT_README_Content", name));
        
        // Run Rake -T silently to determine the available targets and write into private area
        RakeTargetsAction.refreshTargets(p);
        
        return h;
    }

    public static RakeProjectHelper createProject(final File dir, final String name,
            final File[] sourceFolders, final File[] testFolders) throws IOException {
        assert sourceFolders != null && testFolders != null: "Package roots can't be null";   //NOI18N
        final FileObject dirFO = FileUtil.createFolder(dir);
        // this constructor creates only java application type
        final RakeProjectHelper h = createProject(dirFO, name, null, null, null, false);
        final RubyProject p = (RubyProject) ProjectManager.getDefault().findProject(dirFO);
        final ReferenceHelper refHelper = p.getReferenceHelper();
        try {
        ProjectManager.mutex().writeAccess(new Mutex.ExceptionAction<Void>() {
            public Void run() throws Exception {
                Element data = h.getPrimaryConfigurationData(true);
                Document doc = data.getOwnerDocument();
                NodeList nl = data.getElementsByTagNameNS(RubyProjectType.PROJECT_CONFIGURATION_NAMESPACE,"source-roots"); // NOI18N
                assert nl.getLength() == 1;
                Element sourceRoots = (Element) nl.item(0);
                nl = data.getElementsByTagNameNS(RubyProjectType.PROJECT_CONFIGURATION_NAMESPACE,"test-roots");  //NOI18N
                assert nl.getLength() == 1;
                Element testRoots = (Element) nl.item(0);
                for (int i=0; i<sourceFolders.length; i++) {
                    String propName;
                    if (i == 0) {
                        //Name the first src root src.dir to be compatible with NB 4.0
                        propName = "src.dir";       //NOI18N
                    }
                    else {
                        String name = sourceFolders[i].getName();
                        propName = name + ".dir";    //NOI18N
                    }
                    
                    int rootIndex = 1;
                    EditableProperties props = h.getProperties(RakeProjectHelper.PROJECT_PROPERTIES_PATH);
                    while (props.containsKey(propName)) {
                        rootIndex++;
                        propName = name + rootIndex + ".dir";   //NOI18N
                    }
                    String srcReference = refHelper.createForeignFileReference(sourceFolders[i], RubyProject.SOURCES_TYPE_RUBY);
                    Element root = doc.createElementNS (RubyProjectType.PROJECT_CONFIGURATION_NAMESPACE,"root");   //NOI18N
                    root.setAttribute ("id",propName);   //NOI18N
                    sourceRoots.appendChild(root);
                    props = h.getProperties(RakeProjectHelper.PROJECT_PROPERTIES_PATH);
                    props.put(propName,srcReference);
                    h.putProperties(RakeProjectHelper.PROJECT_PROPERTIES_PATH, props); // #47609
                }                 
                for (int i = 0; i < testFolders.length; i++) {
                    if (!testFolders[i].exists()) {
                        testFolders[i].mkdirs();
                    }
                    String propName;
                    if (i == 0) {
                        //Name the first test root test.src.dir to be compatible with NB 4.0
                        propName = "test.src.dir";  //NOI18N
                    }
                    else {
                        String name = testFolders[i].getName();
                        propName = "test." + name + ".dir"; // NOI18N
                    }                    
                    int rootIndex = 1;
                    EditableProperties props = h.getProperties(RakeProjectHelper.PROJECT_PROPERTIES_PATH);
                    while (props.containsKey(propName)) {
                        rootIndex++;
                        propName = "test." + name + rootIndex + ".dir"; // NOI18N
                    }
                    String testReference = refHelper.createForeignFileReference(testFolders[i], RubyProject.SOURCES_TYPE_RUBY);
                    Element root = doc.createElementNS(RubyProjectType.PROJECT_CONFIGURATION_NAMESPACE, "root"); // NOI18N
                    root.setAttribute("id", propName); // NOI18N
                    testRoots.appendChild(root);
                    props = h.getProperties(RakeProjectHelper.PROJECT_PROPERTIES_PATH); // #47609
                    props.put(propName, testReference);
                    h.putProperties(RakeProjectHelper.PROJECT_PROPERTIES_PATH, props);
                }
                h.putPrimaryConfigurationData(data,true);
                ProjectManager.getDefault().saveProject (p);
                return null;
            }
        });
        } catch (MutexException me ) {
            ErrorManager.getDefault().notify (me);
        }
        return h;
    }

    private static RakeProjectHelper createProject(FileObject dirFO, String name,
                                                  String srcRoot, String testRoot, String mainClass, boolean isLibrary) throws IOException {
        RakeProjectHelper h = ProjectGenerator.createProject(dirFO, RubyProjectType.TYPE);
        Element data = h.getPrimaryConfigurationData(true);
        Document doc = data.getOwnerDocument();
        Element nameEl = doc.createElementNS(RubyProjectType.PROJECT_CONFIGURATION_NAMESPACE, "name"); // NOI18N
        nameEl.appendChild(doc.createTextNode(name));
        data.appendChild(nameEl);
//        Element minant = doc.createElementNS(RubyProjectType.PROJECT_CONFIGURATION_NAMESPACE, "minimum-ant-version"); // NOI18N
//        minant.appendChild(doc.createTextNode(MINIMUM_ANT_VERSION)); // NOI18N
//        data.appendChild(minant);
        EditableProperties ep = h.getProperties(RakeProjectHelper.PROJECT_PROPERTIES_PATH);
        Element sourceRoots = doc.createElementNS(RubyProjectType.PROJECT_CONFIGURATION_NAMESPACE,"source-roots");  //NOI18N
        if (srcRoot != null) {
            Element root = doc.createElementNS (RubyProjectType.PROJECT_CONFIGURATION_NAMESPACE,"root");   //NOI18N
            root.setAttribute ("id","src.dir");   //NOI18N
            sourceRoots.appendChild(root);
            ep.setProperty("src.dir", srcRoot); // NOI18N
        }
        data.appendChild (sourceRoots);
        Element testRoots = doc.createElementNS(RubyProjectType.PROJECT_CONFIGURATION_NAMESPACE,"test-roots");  //NOI18N
        if (testRoot != null) {
            Element root = doc.createElementNS (RubyProjectType.PROJECT_CONFIGURATION_NAMESPACE,"root");   //NOI18N
            root.setAttribute ("id","test.src.dir");   //NOI18N
            testRoots.appendChild (root);
            ep.setProperty("test.src.dir", testRoot); // NOI18N
        }
        data.appendChild (testRoots);
        h.putPrimaryConfigurationData(data, true);

        Charset enc = FileEncodingQuery.getDefaultEncoding();
        ep.setProperty(RubyProjectProperties.SOURCE_ENCODING, enc.name());        
//        ep.setProperty("dist.dir", "dist"); // NOI18N
//        ep.setComment("dist.dir", new String[] {"# " + NbBundle.getMessage(RubyProjectGenerator.class, "COMMENT_dist.dir")}, false); // NOI18N
//        ep.setProperty("dist.jar", "${dist.dir}/" + PropertyUtils.getUsablePropertyName(name) + ".jar"); // NOI18N
//        ep.setProperty("javac.classpath", new String[0]); // NOI18N
//        ep.setProperty("build.sysclasspath", "ignore"); // NOI18N
//        ep.setComment("build.sysclasspath", new String[] {"# " + NbBundle.getMessage(RubyProjectGenerator.class, "COMMENT_build.sysclasspath")}, false); // NOI18N
//        ep.setProperty("run.classpath", new String[] { // NOI18N
//            "${javac.classpath}:", // NOI18N
//            "${build.classes.dir}", // NOI18N
//        });
//        ep.setProperty("debug.classpath", new String[] { // NOI18N
//            "${run.classpath}", // NOI18N
//        });        
//        ep.setProperty("jar.compress", "false"); // NOI18N
        if (!isLibrary) {
            ep.setProperty(RubyProjectProperties.MAIN_CLASS, mainClass == null ? "" : mainClass); // NOI18N
        }
        
//        ep.setProperty("javac.compilerargs", ""); // NOI18N
//        ep.setComment("javac.compilerargs", new String[] {
//            "# " + NbBundle.getMessage(RubyProjectGenerator.class, "COMMENT_javac.compilerargs"), // NOI18N
//        }, false);
//        SpecificationVersion sourceLevel = getDefaultSourceLevel();
//        ep.setProperty("javac.source", sourceLevel.toString()); // NOI18N
//        ep.setProperty("javac.target", sourceLevel.toString()); // NOI18N
//        ep.setProperty("javac.deprecation", "false"); // NOI18N
//        ep.setProperty("javac.test.classpath", new String[] { // NOI18N
//            "${javac.classpath}:", // NOI18N
//            "${build.classes.dir}:", // NOI18N
//            "${libs.junit.classpath}", // NOI18N
//        });
//        ep.setProperty("run.test.classpath", new String[] { // NOI18N
//            "${javac.test.classpath}:", // NOI18N
//            "${build.test.classes.dir}", // NOI18N
//        });
//        ep.setProperty("debug.test.classpath", new String[] { // NOI18N
//            "${run.test.classpath}", // NOI18N
//        });
//
//        ep.setProperty("build.generated.dir", "${build.dir}/generated"); // NOI18N
//        ep.setProperty("meta.inf.dir", "${src.dir}/META-INF"); // NOI18N
//        
//        ep.setProperty("build.dir", "build"); // NOI18N
//        ep.setComment("build.dir", new String[] {"# " + NbBundle.getMessage(RubyProjectGenerator.class, "COMMENT_build.dir")}, false); // NOI18N
//        ep.setProperty("build.classes.dir", "${build.dir}/classes"); // NOI18N
//        ep.setProperty("build.test.classes.dir", "${build.dir}/test/classes"); // NOI18N
//        ep.setProperty("build.test.results.dir", "${build.dir}/test/results"); // NOI18N
//        ep.setProperty("build.classes.excludes", "**/*.java,**/*.form"); // NOI18N
//        ep.setProperty("dist.javadoc.dir", "${dist.dir}/javadoc"); // NOI18N
//        ep.setProperty("platform.active", "default_platform"); // NOI18N
//
//        ep.setProperty("run.jvmargs", ""); // NOI18N
//        ep.setComment("run.jvmargs", new String[] {
//            "# " + NbBundle.getMessage(RubyProjectGenerator.class, "COMMENT_run.jvmargs"), // NOI18N
//            "# " + NbBundle.getMessage(RubyProjectGenerator.class, "COMMENT_run.jvmargs_2"), // NOI18N
//            "# " + NbBundle.getMessage(RubyProjectGenerator.class, "COMMENT_run.jvmargs_3"), // NOI18N
//        }, false);

//        ep.setProperty(RubyProjectProperties.JAVADOC_PRIVATE, "false"); // NOI18N
//        ep.setProperty(RubyProjectProperties.JAVADOC_NO_TREE, "false"); // NOI18N
//        ep.setProperty(RubyProjectProperties.JAVADOC_USE, "true"); // NOI18N
//        ep.setProperty(RubyProjectProperties.JAVADOC_NO_NAVBAR, "false"); // NOI18N
//        ep.setProperty(RubyProjectProperties.JAVADOC_NO_INDEX, "false"); // NOI18N
//        ep.setProperty(RubyProjectProperties.JAVADOC_SPLIT_INDEX, "true"); // NOI18N
//        ep.setProperty(RubyProjectProperties.JAVADOC_AUTHOR, "false"); // NOI18N
//        ep.setProperty(RubyProjectProperties.JAVADOC_VERSION, "false"); // NOI18N
//        ep.setProperty(RubyProjectProperties.JAVADOC_WINDOW_TITLE, ""); // NOI18N
//        ep.setProperty(RubyProjectProperties.JAVADOC_ENCODING, ""); // NOI18N
//        ep.setProperty(RubyProjectProperties.JAVADOC_ADDITIONALPARAM, ""); // NOI18N
        
        h.putProperties(RakeProjectHelper.PROJECT_PROPERTIES_PATH, ep);        
        return h;
    }

    private static DataObject createFromTemplate( String mainClassName, FileObject srcFolder, String templateName ) throws IOException {
        int lastDotIdx = mainClassName.lastIndexOf( '/' );
        String mName, pName;
        if ( lastDotIdx == -1 ) {
            mName = mainClassName.trim();
            pName = null;
        }
        else {
            mName = mainClassName.substring( lastDotIdx + 1 ).trim();
            pName = mainClassName.substring( 0, lastDotIdx ).trim();
        }
        
        if ( mName.length() == 0 ) {
            return null;
        }
        
        FileObject mainTemplate = Repository.getDefault().getDefaultFileSystem().findResource( templateName );

        if ( mainTemplate == null ) {
            return null; // Don't know the template
        }
                
        DataObject mt = DataObject.find( mainTemplate );
        
        FileObject pkgFolder = srcFolder;
        if ( pName != null ) {
            String fName = pName.replace( '.', '/' ); // NOI18N
            pkgFolder = FileUtil.createFolder( srcFolder, fName );        
        }
        DataFolder pDf = DataFolder.findFolder( pkgFolder );
        // BEGIN SEMPLICE MODIFICATIONS
        int extension = mName.lastIndexOf('.');
        if (extension != -1) {
            mName = mName.substring(0, extension);
        }
        // END SEMPLICE MODIFICATIONS
        return mt.createFromTemplate(pDf, mName);
    }

    // TODO: use FileUtils when #118088 is fixed
    private static void rename(final FileObject rakeFileFO, final String name, final String ext) throws IOException {
        FileLock lock = null;
        try {
            lock = rakeFileFO.lock();
            rakeFileFO.rename(lock, name, ext);
        } finally {
            if (lock != null) {
                lock.releaseLock();
            }
        }
    }

    // TODO: use FileUtils when #118087 is fixed
    private static void writeLines(final FileObject readme, final String... lines) throws FileAlreadyLockedException, IOException {
        PrintWriter readmeW = new PrintWriter(readme.getOutputStream());
        for (String line : lines) {
            readmeW.println(line);
        }
        readmeW.close();
    }
    
}