/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */ 
package org.netbeans.microedition.testme.bt;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import javax.bluetooth.BluetoothStateException;
import javax.bluetooth.RemoteDevice;
import javax.bluetooth.ServiceRecord;
import javax.bluetooth.UUID;
import javax.microedition.lcdui.Choice;
import javax.microedition.lcdui.Command;
import javax.microedition.lcdui.CommandListener;
import javax.microedition.lcdui.Display;
import javax.microedition.lcdui.Displayable;
import javax.microedition.lcdui.Form;
import javax.microedition.lcdui.Gauge;
import javax.microedition.lcdui.List;
import javax.microedition.midlet.MIDlet;
import javax.microedition.rms.RecordStore;
import javax.microedition.rms.RecordStoreException;

/**
 * Guides the user through a selection of available bluetooth devices providing
 * a serial-port connection. The wizard consists of two steps - device lookup and
 * selection and service lookup and selection.
 * A virtual bluetooth COM port(s) must be created on
 * each such device before this discovery is started.
 *
 * @author PavelBenes
 */
final class ConnectionWizard {
    private static final Command SELECT                = new Command( "Select", Command.ITEM, 1);
    private static final Command CANCEL                = new Command( "Cancel", Command.CANCEL, 1);
    private static final String SERIAL_PROTOCOL_PREFIX = "btspp://";
    private static final String REC_STORE_NAME         = "BTURLSTR";
    private static final int    REC_RECORD_ID          = 1;

    /**
     * A callback interface used to inform the wizard user that connection
     * selection is done - either with success or failure.
     */
    public interface ConnectionListener {
        public void connectionSelected(ConnectionInfo conInfo, String msg);
    }
    
    /**
     * A helper class holding information about established bluetooth connection,
     * like connection URL, device friendly name and bluetooth address.
     */
    final static class ConnectionInfo {
        public final String url;
        public final String name;
        public final String address;
        
        public ConnectionInfo(String url, String name, String address) {
            this.url     = url != null ? url : "";
            this.name    = name != null ? name : "";
            this.address = address != null ? address : "";
        }

        public ConnectionInfo(DataInputStream in) throws IOException {
            url     = in.readUTF();
            name    = in.readUTF();
            address = in.readUTF();
        }
        
        public void serialize(DataOutputStream out) throws IOException {
            out.writeUTF(url);
            out.writeUTF(name);
            out.writeUTF(address);
        }
        
        public boolean equals(Object o) {
            if ( o != null && o instanceof ConnectionInfo) {
                ConnectionInfo ci = (ConnectionInfo) o;
                if ( url.equals(ci.url) && name.equals(ci.name) && address.equals(ci.address)) {
                    return true;
                }
            }
            return false;
        }
        
        public String toString() {
            if (name.length() == 0) {
                return address;
            } else {
                return name + "[" + address + "]";
            }
        }
    }
    
    /**
     * A superclass containing common stuff for device and service selectors.
     */
    private abstract class Selector extends List implements CommandListener, Runnable {
        protected Selector(String title) {
            super( title, Choice.IMPLICIT);   
            setSelectCommand(SELECT);
            addCommand(CANCEL);
            setCommandListener(this);            
        }
        
       public void start() {
            new Thread(this).start();
        }        
    }
    
    /**
     * Handles and provides a visual info about device lookup progress and result.
     */
    private final class DeviceSelector extends Selector {
        private RemoteDevice [] devices;
        
        public DeviceSelector() {
            super("Devices");
        }
        
        public void run() {
            try {
                devices = BluetoothManager.instance().findRemoteDevices();
                
                if (devices == null || devices.length == 0) {
                    selectionCompleted(null, "No device found." );
                } else {
                    for (int i = 0; i < devices.length; i++) {
                        String name = BluetoothManager.getName(devices[i]);
                        if (name == null) {
                            // if friendly name could not be retrieved, use the BT address instead
                            name = devices[i].getBluetoothAddress();
                        }
                        append( name, null);
                    }                    
                    show( this);
                }
            } catch (BluetoothStateException e) {
                selectionCompleted(null, "Could not find devices.");
            }        
        }
        
        public void commandAction(Command command, Displayable displayable) {
            if (command == CANCEL) {
                selectionCompleted(null, "Canceled.");
            } else {
                // it is select command
                deviceSelected( devices[getSelectedIndex()]);
            }
        }           
    }
                
    /**
     * Handles and provides a visual info about service lookup progress and result.
     */
    private final class ServiceSelector extends Selector {
        private final RemoteDevice device;
        private ServiceRecord []   services;
        
        public ServiceSelector(RemoteDevice device) {
            super("Connections");
            this.device = device;
            _deviceName = BluetoothManager.getName(device);
            _deviceAddr = device.getBluetoothAddress();  
            
            setSelectCommand( SELECT);
            addCommand(CANCEL);
            setCommandListener(this);
        }

        public void commandAction(Command command, Displayable displayable) {
            if (command == CANCEL) {
                selectionCompleted(null, "Canceled.");
            } else if (command == SELECT) {
                ConnectionInfo ci = new ConnectionInfo( BluetoothManager.getURL(services[getSelectedIndex()]), _deviceName, _deviceAddr);
                selectionCompleted( ci, "Connected to " + ci.toString());
            }
        }   
        
        public void run() {
            try {
                services = BluetoothManager.instance().findServices(device, new UUID[] { BluetoothManager.UUID_SERIAL});
            } catch (BluetoothStateException e) {
                selectionCompleted(null, "Could not find services.");
                return;
            }

            if (services == null || services.length == 0) {
                selectionCompleted(null, "No service found.");
            } else if (services.length == 1) {
                ConnectionInfo ci = new ConnectionInfo( BluetoothManager.getURL(services[0]), _deviceName, _deviceAddr);
                selectionCompleted( ci, "Connected to " + ci.toString());
            } else {
               for (int i = 0; i < services.length; i++) {
                    String url = BluetoothManager.getURL(services[i]);
                    if (url != null) {
                        if (url.startsWith(SERIAL_PROTOCOL_PREFIX)) {
                            url = url.substring( SERIAL_PROTOCOL_PREFIX.length());
                        }
                    }
                    append( url, null);
                }                
                show( this);
            }            
        }
    }
    
    private final MIDlet             _midlet;
    private final ConnectionListener _listener;
    private ConnectionInfo           _rmsConn;
    private String                   _deviceName;
    private String                   _deviceAddr;
    private final Form               _logger;
    
    public ConnectionWizard(MIDlet midlet, ConnectionListener listener, Form logger) {
        this._midlet   = midlet;
        this._listener = listener;
        _logger = logger;
    }
    
    public void startSelection() {
        Form form = new Form("BT Logger");
        form.append( new Gauge("Looking for devices", false, Gauge.INDEFINITE, Gauge.CONTINUOUS_RUNNING));
        show(form);
        try {
            _rmsConn = retrieveConnection();
        } catch (Exception ex) {
            btError("Read from RMS failed", ex);
            _rmsConn = null;
        }
        DeviceSelector devSelector = new DeviceSelector();
        devSelector.start();
    }
    
    protected void deviceSelected(RemoteDevice device) {
        Form form  = new Form("BT Logger");
        form.append( new Gauge("Discovering services", false, Gauge.INDEFINITE, Gauge.CONTINUOUS_RUNNING));
        show(form);

        ServiceSelector servSelector = new ServiceSelector(device);
        servSelector.start();
    }

    protected void selectionCompleted(ConnectionInfo conInfo, String msg) {
        if (conInfo != null) { 
            btLog("Comparing the url");
            if ( !conInfo.equals(_rmsConn)) {
                btLog("Storing url");
                storeConnectionInfo(conInfo);
            }
        } else {
            btLog("No url");
        }
        _listener.connectionSelected(conInfo, msg);
    }
    
    protected void show(Displayable screen) {
        Display.getDisplay(_midlet).setCurrent(screen);
    }
     
    /**
     * Store bluetooth URL into persistent RMS storage for later use. If the storage
     * contains valid URL, it is possible to reuse it and skip the device and service 
     * lookup.
     */
    protected synchronized void storeConnectionInfo(ConnectionInfo connInfo) {
        RecordStore rs = null;
        try {            
            rs = RecordStore.openRecordStore(REC_STORE_NAME, true);
            ByteArrayOutputStream buff = new ByteArrayOutputStream();
            DataOutputStream      out  = new DataOutputStream(buff);
            connInfo.serialize(out);
            out.close();
            byte [] data = buff.toByteArray();
            if (rs.getNumRecords() == 0) {
                rs.addRecord(data, 0, data.length);
            } else {
                rs.setRecord( REC_RECORD_ID, data, 0, data.length);
            }
        } catch( Exception e) {
            btError("Write to RMS failed:", e);
            //BlueDict.logError("Write to RMS failed", e);
        } finally {
            if (rs != null) {
                try {
                    rs.closeRecordStore();
                } catch (Exception e) {}
            }
        }
    }
    
    /**
     * Get bluetooth URL from persistent RMS storage for quick-connect.
     */
    public synchronized static ConnectionInfo retrieveConnection() throws Exception {
        ConnectionInfo conInfo = null;
        RecordStore    rs      = null;
        
        try {            
            rs = RecordStore.openRecordStore(REC_STORE_NAME, false);
            byte [] data = rs.getRecord( REC_RECORD_ID);
            ByteArrayInputStream buff = new ByteArrayInputStream(data);
            DataInputStream      in   = new DataInputStream(buff);
            conInfo = new ConnectionInfo(in);
            in.close();
        } finally {
            if (rs != null) {
                try {
                    rs.closeRecordStore();
                } catch (Exception e) {}
            }
        }
        
        return conInfo;
    }

    /**
     * Remove URL from RMS storage to prevent the quick-connect. It is typicaly
     * done when connection to a different device/port is required.
     */
    public synchronized static void resetURL() {
        try {
            RecordStore.deleteRecordStore(REC_STORE_NAME);
        } catch (RecordStoreException ex) {}
    }
    
    private void btLog(String msg) {
        _logger.append(msg + "\n");
    }
    
    private void btError(String msg, Exception e) {
        _logger.append( msg + " - " + e.getClass().getName() + "[" + e.getMessage() + "]\n");
        
    }    
}
