/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.sql.framework.highlighter;

import java.io.IOException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;

/**
 * Run the Syntax Highlighting as a separate thread. Things that need to be colored are
 * messaged to the thread and put in a list.
 * 
 * @author Ahimanikya Satapathy
 * @version $Revision: 1.2 $
 */
public class SQLTextColorer extends Thread {

    /**
     * A simple wrapper representing something that needs to be colored. Placed into an
     * object so that it can be stored in a Vector.
     */
    private class RecolorEvent {
        public int adjustment;
        public int position;

        public RecolorEvent(int position, int adjustment) {
            this.position = position;
            this.adjustment = adjustment;
        }
    }

    private volatile boolean asleep = false;

    /**
     * The amount of change that has occurred before the place in the sqlDocument that we
     * are currently highlighting (lastPosition).
     */
    private volatile int change = 0;

    /**
     * Keep a list of places in the file that it is safe to restart the highlighting. This
     * happens whenever the lexer reports that it has returned to its initial state. Since
     * this list needs to be sorted and we need to be able to retrieve ranges from it, it
     * is stored in a balanced tree.
     */
    private TreeSet iniPositions = new TreeSet(new DocPositionComparator());

    /**
     * The last position colored
     */
    private volatile int lastPosition = -1;

    /**
     * When accessing the vector, we need to create a critical section. we will
     * synchronize on this object to ensure that we don't get unsafe thread behavior.
     */
    private Object lock = new Object();

    /**
     * As we go through and remove invalid positions we will also be finding new valid
     * positions. Since the position list cannot be deleted from and written to at the
     * same time, we will keep a list of the new positions and simply add it to the list
     * of positions once all the old positions have been removed.
     */
    private HashSet newPositions = new HashSet();

    private final SQLEditorPane sqlPane;

    private volatile boolean stop = false;

    /**
     * Vector that stores the communication between the two threads.
     */
    private volatile Vector v = new Vector();

    /**
     * @param SQLEditorDemo
     */
    SQLTextColorer(SQLEditorPane sqlPane) {
        this.setName("SQLTextColorer");
        this.sqlPane = sqlPane;
    }

    /**
     * Tell the Syntax Highlighting thread to take another look at this section of the
     * sqlDocument. It will process this as a FIFO. This method should be done inside a
     * doclock.
     */
    public void color(int position, int adjustment) {
        // figure out if this adjustment effects the current run.
        // if it does, then adjust the place in the sqlDocument
        // that gets highlighted.
        if (position < lastPosition) {
            if (lastPosition < position - adjustment) {
                change -= lastPosition - position;
            } else {
                change += adjustment;
            }
        }
        synchronized (lock) {
            v.add(new RecolorEvent(position, adjustment));
            if (asleep) {
                this.interrupt();
            }
        }
    }

    /**
     * The colorer runs forever and may sleep for long periods of time. It should be
     * interrupted every time there is something for it to do.
     */
    public void run() {
        int position = -1;
        int adjustment = 0;
        // if we just finish, we can't go to sleep until we
        // ensure there is nothing else for us to do.
        // use try again to keep track of this.
        boolean tryAgain = false;
        for (;;) { // forever
            if (stop)
                break;
            synchronized (lock) {
                if (v.size() > 0) {
                    RecolorEvent re = (RecolorEvent) (v.elementAt(0));
                    v.removeElementAt(0);
                    position = re.position;
                    adjustment = re.adjustment;
                } else {
                    tryAgain = false;
                    position = -1;
                    adjustment = 0;
                }
            }
            if (position != -1) {
                SortedSet workingSet;
                Iterator workingIt;
                DocPosition startRequest = new DocPosition(position);
                DocPosition endRequest = new DocPosition(position + ((adjustment >= 0) ? adjustment : -adjustment));
                DocPosition dp;
                DocPosition dpStart = null;
                DocPosition dpEnd = null;

                // find the starting position. We must start at least one
                // token before the current position
                try {
                    // all the good positions before
                    workingSet = iniPositions.headSet(startRequest);
                    // the last of the stuff before
                    dpStart = ((DocPosition) workingSet.last());
                } catch (NoSuchElementException x) {
                    // if there were no good positions before the requested start,
                    // we can always start at the very beginning.
                    dpStart = new DocPosition(0);
                }

                // if stuff was removed, take any removed positions off the list.
                if (adjustment < 0) {
                    workingSet = iniPositions.subSet(startRequest, endRequest);
                    workingIt = workingSet.iterator();
                    while (workingIt.hasNext()) {
                        workingIt.next();
                        workingIt.remove();
                    }
                }

                // adjust the positions of everything after the insertion/removal.
                workingSet = iniPositions.tailSet(startRequest);
                workingIt = workingSet.iterator();
                while (workingIt.hasNext()) {
                    ((DocPosition) workingIt.next()).adjustPosition(adjustment);
                }

                // now go through and highlight as much as needed
                workingSet = iniPositions.tailSet(dpStart);
                workingIt = workingSet.iterator();
                dp = null;
                if (workingIt.hasNext()) {
                    dp = (DocPosition) workingIt.next();
                }
                try {
                    Token t;
                    boolean done = false;
                    dpEnd = dpStart;
                    synchronized (this.sqlPane.getDoclock()) {
                        // we are playing some games with the lexer for efficiency.
                        // we could just create a new lexer each time here, but
                        // instead, we will just reset it so that it thinks it is
                        // starting at the beginning of the sqlDocument but reporting a
                        // funny start position. Reseting the lexer causes the
                        // close() method on the reader to be called but because the
                        // close() method has no effect on the DocumentReader, we can
                        // do this.
                        this.sqlPane.getSyntaxLexer().reset(this.sqlPane.getDocumentReader(), 0, dpStart.getPosition(), 0);
                        // After the lexer has been set up, scroll the reader so that
                        // it is in the correct spot as well.
                        this.sqlPane.getDocumentReader().seek(dpStart.getPosition());
                        // we will highlight tokens until we reach a good stopping
                        // place. the first obvious stopping place is the end of the
                        // sqlDocument. the lexer will return null at the end of the
                        // sqlDocument and we need to stop there.
                        t = this.sqlPane.getSyntaxLexer().getNextToken();
                    }
                    newPositions.add(dpStart);
                    while (!done && t != null) {
                        // this is the actual command that colors the stuff.
                        // Color stuff with the description of the style matched
                        // to the hash table that has been set up ahead of time.
                        synchronized (this.sqlPane.getDoclock()) {
                            if (t.getCharEnd() <= this.sqlPane.getSqlDocument().getLength()) {

                                String type = t.getDescription();
                                if (t.getID() == SQLToken.IDENTIFIER) {
                                    final String data = t.getContents();
                                    String qType = this.sqlPane.getIdentifierType(data);
                                    if (qType != null) {
                                        type = qType;
                                    }

                                    //if (isDataType(data)) {
                                    //    type = StyleHelper.DATA_TYPE;
                                    //}
                                }

                                this.sqlPane.getSqlDocument().setCharacterAttributes(t.getCharBegin() + change, t.getCharEnd() - t.getCharBegin(),
                                    this.sqlPane.getCustomStyle(type), true);
                                // record the position of the last bit of text that
                                // we colored
                                dpEnd = new DocPosition(t.getCharEnd());
                            }
                            lastPosition = (t.getCharEnd() + change);
                        }
                        // The other more complicated reason for doing no more
                        // highlighting is that all the colors are the same from here
                        // on out anyway. We can detect this by seeing if the place
                        // that the lexer returned to the initial state last time we
                        // highlighted is the same as the place that returned to the
                        // initial state this time. As long as that place is after
                        // the last changed text, everything from there on is fine
                        // already.
                        if (t.getState() == Token.INITIAL_STATE) {
                            // look at all the positions from last time that are less
                            // than or equal to the current position
                            while (dp != null && dp.getPosition() <= t.getCharEnd()) {
                                if (dp.getPosition() == t.getCharEnd() && dp.getPosition() >= endRequest.getPosition()) {
                                    // we have found a state that is the same
                                    done = true;
                                    dp = null;
                                } else if (workingIt.hasNext()) {
                                    // didn't find it, try again.
                                    dp = (DocPosition) workingIt.next();
                                } else {
                                    // didn't find it, and there is no more info from
                                    // last time. This means that we will just
                                    // continue until the end of the sqlDocument.
                                    dp = null;
                                }
                            }
                            // so that we can do this check next time, record all the
                            // initial states from this time.
                            newPositions.add(dpEnd);
                        }
                        synchronized (this.sqlPane.getDoclock()) {
                            t = this.sqlPane.getSyntaxLexer().getNextToken();
                        }
                    }

                    // remove all the old initial positions from the place where
                    // we started doing the highlighting right up through the last
                    // bit of text we touched.
                    workingIt = iniPositions.subSet(dpStart, dpEnd).iterator();
                    while (workingIt.hasNext()) {
                        workingIt.next();
                        workingIt.remove();
                    }

                    // Remove all the positions that are after the end of the file.:
                    workingIt = iniPositions.tailSet(new DocPosition(this.sqlPane.getSqlDocument().getLength())).iterator();
                    while (workingIt.hasNext()) {
                        workingIt.next();
                        workingIt.remove();
                    }

                    // and put the new initial positions that we have found on the
                    // list.
                    iniPositions.addAll(newPositions);
                    newPositions.clear();

                } catch (IOException x) {
                }
                synchronized (this.sqlPane.getDoclock()) {
                    lastPosition = -1;
                    change = 0;
                }
                // since we did something, we should check that there is
                // nothing else to do before going back to sleep.
                tryAgain = true;
            }
            asleep = true;
            if (!tryAgain) {
                try {
                    sleep(0xffffff);
                } catch (InterruptedException x) {
                }

            }
            asleep = false;
        }
    }

    public void stopThread() {
        this.stop = true;
        synchronized (lock) {
            if (asleep) {
                this.interrupt();
            }
        }
    }
}
