/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.bpel.xpath.view.model.tree;

import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.WeakHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import org.netbeans.modules.bpel.model.api.BpelModel;
import org.netbeans.modules.bpel.model.api.Process;
import org.netbeans.modules.bpel.xpath.model.nodes.Node;
import org.netbeans.modules.bpel.xpath.model.nodes.TreeNode;
import org.netbeans.modules.bpel.xpath.view.expression.impl.PredicatesManager;

import org.openide.util.NbBundle;


/**
 * Data model that may be used in a tree view to represent a business process
 * model. This class implements the javax.swing.tree.TreeModel interface.
 *
 * @author sbyn
 * @version 
 */
public class BusinessProcessTreeModel
    implements TreeModel {

    /**
     * The root node.
     */
    protected Object mRoot;


    protected Process mProcess;
    
    protected BpelModel mDocument;
    
    
    /**
     * List of listeners.
     */
    protected Vector mTreeModelListeners;

    /**
     * The logger.
     */
    private static final Logger mLogger = Logger.getLogger(BusinessProcessTreeModel.class.getName());

    /**
     * Debug flag.
     */
    private static final boolean DEBUG = false;

    private static final String NULL_NODE = NbBundle.
        getMessage(BusinessProcessTreeModel.class, "STR_NULL_NODE");    // NOI18N
     
    /** this map keeps the Part-MNode cache so that subsequent calls return the same tree node
     * structures. It is NOT for cache reasons, but more becase of the links that are
     * drawn. If the cache is not maintained then the links are not properly recognized with the 
     * corresponding tree nodes. Asking the factory doesnot guarantee the same MNode for the 
     * same part. 
     * This cache should be invalidated when the wsdl model notifies of a change.
     */
    private WeakHashMap mMNodeMap = new WeakHashMap();
    
    

    private PredicatesManager mPredicatesManager;
    
    /**
     * Creates a new instance of BusinessProcessTreeModel for the given
     * business process.
     * @param businessProcess the business process
     * @param view BPView
     */
    public BusinessProcessTreeModel(BpelModel document) {
        mTreeModelListeners = new Vector();
        this.mDocument = document;
        mProcess = mDocument.getProcess();
        mRoot = null;
    }
    

    public void setPredicatesManager(PredicatesManager predicatesManager) {
    	this.mPredicatesManager = predicatesManager;
    }
    
    public PredicatesManager getPredicatesManager() {
    	return this.mPredicatesManager;
    }
    
    public BpelModel getBPELDocument() {
    	return this.mDocument;
    }
    
    /**
     * Gets the root node.
     * @return the root node which is the business process
     */
    public Object getRoot() {
        return mRoot;
    }

    /**
     * Sets the root node of the tree model.
     * All listeners are notified of the change.
     * @param newRoot the new root node
     */
    public void setRoot(Object newRoot) {
        Object oldRoot = getRoot();        
        mRoot = newRoot;
        if (oldRoot != null) {
            fireTreeStructureChanged(oldRoot);
        }
    }

    /**
     * @see javax.swing.tree.TreeModel#getChild(java.lang.Object, int)
     */
    public Object getChild(Object parent, int index) {
        Object result = null;
		if(parent instanceof TreeNode) {
			TreeNode node = (TreeNode) parent;
			List children = node.getChildren();
			if(index < children.size()) {
				result = children.get(index);
			}
		}
        
        return result;
    }


    /**
     * @see javax.swing.tree.TreeModel#getChildCount(java.lang.Object)
     */
    public int getChildCount(Object parent) {
        int result = 0;
        if(parent instanceof TreeNode) {
        	TreeNode node = (TreeNode) parent;
        	result = node.getChildren().size();
        }
        return result;
    }

    
    /**
     * @see javax.swing.tree.TreeModel#getIndexOfChild(java.lang.Object, java.lang.Object)
     */
    public int getIndexOfChild(Object parent, Object child) {
        if (parent == null || child == null) {
            return -1;
        }

        int numChildren = getChildCount(parent);
        if (numChildren == 0) {
            return -1;
        }

        for (int i = 0; i < numChildren; i++) {
            Object obj = getChild(parent, i);
            if (obj == child) {
                return i;
            }
        }

        return -1;
    }


    /**
     * Generates a TreePath for the given string format.
     * @param path the treepath in string format
     * @return the TreePath
     */
    public TreePath getTreePath(String path) {
        if (path == null) {
            return null;
        }

        StringTokenizer st = new StringTokenizer(path, "/");
        Vector vec = new Vector();
        while (st.hasMoreTokens()) {
            vec.add(st.nextToken());
        }

        Object[] os = new Object[vec.size()];
        Object node = getRoot();
        String name = (String) vec.elementAt(0);

        if (name.compareTo(node.toString()) != 0) {
            return null;
        }

        os[0] = node;
        for (int i = 1; i < vec.size(); i++) {
            boolean found = false;
            Object n = os[i - 1];
            name = (String) vec.elementAt(i);
            int k = getChildCount(n);
            for (int j = 0; j < k; j++) {
                Object m = getChild(n, j);
                if (name.compareTo(m.toString()) == 0) {
                    found = true;
                    os[i] = m;
                    break;
                }
            }

            if (!found) {
                return null;
            }
        }

        return new TreePath(os);
    }


    /**
     * @see javax.swing.tree.TreeModel#isLeaf(java.lang.Object)
     */
    public boolean isLeaf(Object node) {
        // System.out.println("calling getChild for node " + node);
        boolean val =  (getChildCount(node) == 0);
        // System.out.println("CALLED getChild for node " + node + " and got " + val);
        return val;
    }


    /**
     * @see javax.swing.tree.TreeModel#addTreeModelListener(javax.swing.event.TreeModelListener)
     */
    public void addTreeModelListener(TreeModelListener treeModelListener) {
        mTreeModelListeners.addElement(treeModelListener);
    }


    /**
     * @see javax.swing.tree.TreeModel#removeTreeModelListener(javax.swing.event.TreeModelListener)
     */
    public void removeTreeModelListener(TreeModelListener treeModelListener) {
        mTreeModelListeners.removeElement(treeModelListener);
    }


    /**
     * @see javax.swing.tree.TreeModel#valueForPathChanged(javax.swing.tree.TreePath, java.lang.Object)
     */
    public void valueForPathChanged(TreePath treePath, Object newValue) {
        // Empty method.
    }


    /**
     * Notifier the listeners that the tree structure has changed.
     * @param oldRoot the old root node
     */
    protected void fireTreeStructureChanged(Object oldRoot) {
        int len = mTreeModelListeners.size();
        TreeModelEvent e = new TreeModelEvent(this, new Object[]{oldRoot});

        TreeModelListener listener;
        for (int i = 0; i < len; i++) {
            listener = (TreeModelListener) mTreeModelListeners.elementAt(i);
            listener.treeStructureChanged(e);
        }
    }

    /**
     * @param treePath TreePath of the node whose structure is changed
     */
    public void fireTreeStructureChanged(TreePath treePath) {
        int len = mTreeModelListeners.size();
        TreeModelEvent e = new TreeModelEvent(this, treePath);

        TreeModelListener listener;
        for (int i = 0; i < len; i++) {
            listener = (TreeModelListener) mTreeModelListeners.elementAt(i);
            listener.treeStructureChanged(e);
        }
    }

    /**
     * @param treePath TreePath of the node which has changed
     */
    public void fireTreeNodesChanged(TreePath treePath) {
        int len = mTreeModelListeners.size();
        TreeModelEvent e = new TreeModelEvent(this, treePath);

        TreeModelListener listener;
        for (int i = 0; i < len; i++) {
            listener = (TreeModelListener) mTreeModelListeners.elementAt(i);
            listener.treeNodesChanged(e);
        }
    }

    


}
