/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.bpel.xpath.view;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import org.netbeans.modules.bpel.model.api.BpelEntity;
import org.netbeans.modules.bpel.model.api.BpelModel;
import org.netbeans.modules.bpel.xpath.model.node.visitor.XPathGeneratorVisitor;
import org.netbeans.modules.bpel.xpath.model.nodes.CanvasNode;
import org.netbeans.modules.bpel.xpath.model.nodes.Node;
import org.netbeans.modules.bpel.xpath.model.nodes.NodeHelper;
import org.netbeans.modules.bpel.xpath.model.nodes.TreeNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XPathOperatorNode;
import org.netbeans.modules.bpel.xpath.view.expression.IMappingModeConstants;
import org.netbeans.modules.bpel.xpath.view.expression.impl.MapperUtil;
import org.netbeans.modules.bpel.xpath.view.output.ResultWindow;
import org.netbeans.modules.soa.mapper.common.IMapperGroupNode;
import org.netbeans.modules.soa.mapper.common.IMapperLink;
import org.netbeans.modules.soa.mapper.common.IMapperNode;
import org.netbeans.modules.soa.mapper.common.basicmapper.IBasicMapper;
import org.netbeans.modules.soa.mapper.common.basicmapper.methoid.IFieldNode;
import org.netbeans.modules.soa.mapper.common.basicmapper.tree.IMapperTreeNode;

/**
 *
 * @author radval
 *
 */
public abstract class AbstractXPathExpressionBuilder {
    
    protected static final Logger LOGGER = Logger.getLogger(AbstractXPathExpressionBuilder.class.getName());
    
    private IBasicMapper mMapper;
    
    private BpelModel mModel;
    
    private BpelEntity mHoldingEntity;
    
    public AbstractXPathExpressionBuilder(IBasicMapper mapper, BpelModel model) {
        this.mMapper = mapper;
        this.mModel = model;
    }
    
    public AbstractXPathExpressionBuilder(IBasicMapper mapper, BpelEntity holdingEntity) {
        this.mMapper = mapper;
        this.mHoldingEntity = holdingEntity;
        this.mModel = holdingEntity.getBpelModel();
    }
    
     public BpelEntity getBPELElement() {
        return this.mHoldingEntity;
    }
   
    public BpelModel getBPELDocument() {
        return this.mModel;
    }
    
    /**
     * This method is called when a link is deleted which is not
     * a direct link to target tree node.
     *
     * Ex: ST-->OP1-->OP2-->TT
     * if link deleted is OP1-->OP2
     * then links passes to this method is OP2-->TT
     * Implementator of this method should update the xpath expression
     * of the object stored in the link.
     *
     * @param links
     */
    public abstract void handleLinkCreationToNonTargetTreeNode(List links);
    
    /**
     * This method is called when a link is deleted which is a direct link
     * to target tree node.
     *
     * Ex:
     * ST-->OP1-->OP2-->TT
     *
     * if link is deleted is OP2-->TT
     * then links passed to this method is OP2-->TT.
     * Implementator of this method should deleted the link
     * object which is stored in the link.
     * @param links
     */
    public abstract void handleLinkCreationToTargetTreeNode(
            List links,
            String sourceExpression,
            String targetExpression,
            Node targetTreeNode);
    
    
    
    public IBasicMapper getMapper() {
        return this.mMapper;
    }
    
    public String getSelectedMapMode() {
        //TODO: see BpelMapperToolBar for map mode.
        return IMappingModeConstants.CONTAINERS;
    }
    
    protected void connectTrees(IMapperTreeNode source, IMapperTreeNode target) {
        Node sourceTreeNode = (Node) source.getPath().getLastPathComponent();
        Node targetTreeNode = (Node) target.getPath().getLastPathComponent();
        
        sourceTreeNode.setMapperNode(source);
        source.setNodeObject(sourceTreeNode);
        
        targetTreeNode.setMapperNode(target);
        target.setNodeObject(targetTreeNode);
    }
    
    protected void connectFieldToTree(IFieldNode source, IMapperTreeNode target) {
        TreeNode targetModelNode   = (TreeNode) target.getPath().getLastPathComponent();
        CanvasNode sourceModelNode = MapperUtil.getFieldNodeObject(source);
        
        sourceModelNode.addOutput(targetModelNode);
        IMapperGroupNode sourceGroupNode = source.getGroupNode();
        sourceModelNode.setMapperNode(sourceGroupNode);
        
        targetModelNode.setMapperNode(target);
        target.setNodeObject(targetModelNode);
    }
    
    protected void connectTreeToField(IMapperTreeNode source, IFieldNode target) {
        CanvasNode targetModelNode = MapperUtil.getFieldNodeObject(target);
        TreeNode sourceModelNode   = (TreeNode) source.getPath().getLastPathComponent();
        if (sourceModelNode != null && targetModelNode != null) {
            
            sourceModelNode.setMapperNode(source);
            source.setNodeObject(sourceModelNode);
            
            IMapperGroupNode targetGroupNode = target.getGroupNode();
            int targetInputFieldIndex = MapperUtil.findFieldIndex(targetGroupNode, target);
            // The targetModelNode must be an operator node because we connected
            // a link to its input. Literal nodes will not have an input.
            XPathOperatorNode targetOpNode = (XPathOperatorNode) targetModelNode;
            targetOpNode.addInput(targetInputFieldIndex, sourceModelNode);
            targetOpNode.setMapperNode(targetGroupNode);
        }
    }
    
    protected void connectFields(IFieldNode source, IFieldNode target) {
        CanvasNode sourceModelNode = MapperUtil.getFieldNodeObject(source);
        CanvasNode targetModelNode = MapperUtil.getFieldNodeObject(target);
        if (sourceModelNode != null && targetModelNode != null) {
            
            sourceModelNode.addOutput(targetModelNode);
            IMapperGroupNode sourceGroupNode = source.getGroupNode();
            sourceModelNode.setMapperNode(sourceGroupNode);
            
            IMapperGroupNode targetGroupNode = target.getGroupNode();
            int targetInputFieldIndex = MapperUtil.findFieldIndex(targetGroupNode, target);
            // The targetModelNode must be an operator node because we connected
            // a link to its input. Literal nodes will not have an input.
            XPathOperatorNode targetOpNode = (XPathOperatorNode) targetModelNode;
            targetOpNode.addInput(targetInputFieldIndex, sourceModelNode);
            targetOpNode.setMapperNode(targetGroupNode);
        }
    }

    void addMapping(IMapperNode srcNode, IMapperNode destNode, IMapperLink link) {
        if (srcNode instanceof IMapperTreeNode && destNode instanceof IMapperTreeNode) {
            addTreeMapping((IMapperTreeNode) srcNode, (IMapperTreeNode) destNode, link);
        } else if (srcNode instanceof IFieldNode && destNode instanceof IMapperTreeNode) {
            addFieldTreeMapping((IFieldNode) srcNode, (IMapperTreeNode) destNode, link);
        } else if (srcNode instanceof IMapperTreeNode && destNode instanceof IFieldNode) {
            // This addMapping uses getContainerData function for all mapping
            addTreeFieldMapping((IMapperTreeNode) srcNode, (IFieldNode) destNode, link);
        } else if (srcNode instanceof IFieldNode && destNode instanceof IFieldNode) {
            addFieldMapping((IFieldNode) srcNode, (IFieldNode) destNode, link);
        }
    }
    
    /**
     * Add mapping when link is created from source tree node to target tree node.
     * @param source TreePath
     * @param target TreePath
     * @param link IMapperLink
     */
    protected void addTreeMapping(IMapperTreeNode source, IMapperTreeNode target, IMapperLink link) {
        List directLinksToTargetTreeNodes = new ArrayList();
        directLinksToTargetTreeNodes.add(link);
        
        Node sourceTreeNode = (Node) source.getPath().getLastPathComponent();
        Node targetTreeNode = (Node) target.getPath().getLastPathComponent();
        connectTrees(source, target);
        
        XPathGeneratorVisitor srcGenerator = new XPathGeneratorVisitor(getBPELElement());
        sourceTreeNode.accept(srcGenerator);
        String sourceExpression = srcGenerator.getXPath();
        
        XPathGeneratorVisitor targetGenerator = new XPathGeneratorVisitor(getBPELElement());
        targetTreeNode.accept(targetGenerator);
        String targetExpression = targetGenerator.getXPath();
        
        setSynchronizationListenerEnable(false);
        handleLinkCreationToTargetTreeNode(directLinksToTargetTreeNodes, sourceExpression, targetExpression, targetTreeNode);
        srcGenerator.addNewPrefixToProcess();
        targetGenerator.addNewPrefixToProcess();
        setSynchronizationListenerEnable(true);
    }
    
    /**
     * Add mapping when link is created from source operation node (ie. output field node of
     * an operator) to target tree node.
     * @param srcNode IFieldNode
     * @param target TreePath
     * @param link IMapperLink
     */
    protected void addFieldTreeMapping(IFieldNode source, IMapperTreeNode target, IMapperLink link) {
        
        List directLinksToTargetTreeNodes = new ArrayList();
        directLinksToTargetTreeNodes.add(link);
        
        CanvasNode sourceFieldNode = (CanvasNode) source.getNodeObject();
        Node targetTreeNode        = (Node) target.getPath().getLastPathComponent();
        connectFieldToTree(source, target);
        
        XPathGeneratorVisitor srcGenerator = new XPathGeneratorVisitor(getBPELElement());
        sourceFieldNode.accept(srcGenerator);
        String sourceExpression = srcGenerator.getXPath();
        
        XPathGeneratorVisitor targetGenerator = new XPathGeneratorVisitor(getBPELElement());
        targetTreeNode.accept(targetGenerator);
        String targetExpression = targetGenerator.getXPath();
        
        setSynchronizationListenerEnable(false);
        handleLinkCreationToTargetTreeNode(directLinksToTargetTreeNodes, sourceExpression, targetExpression, targetTreeNode);
        srcGenerator.addNewPrefixToProcess();
        targetGenerator.addNewPrefixToProcess();
        
        setSynchronizationListenerEnable(true);
    }
    
    void addTreeFieldMapping(IMapperTreeNode source, IFieldNode target, IMapperLink link) {
        IMapperGroupNode targetGroupNode = target.getGroupNode();
        XPathOperatorNode targetOperatorNode = (XPathOperatorNode) targetGroupNode.getNodeObject();
        Node sourceModelNode       = (Node) source.getPath().getLastPathComponent();
        if (sourceModelNode != null && targetOperatorNode != null) {
            connectTreeToField(source, target);
            int targetInputFieldIndex = MapperUtil.findFieldIndex(targetGroupNode, target);
            updateLinkObjectXPathExpression(sourceModelNode, targetOperatorNode, targetInputFieldIndex);
        }
    }
    
    void addFieldMapping(IFieldNode source, IFieldNode target, IMapperLink link) {
        CanvasNode sourceModelNode = (CanvasNode) source.getNodeObject();
        IMapperGroupNode targetGroupNode = target.getGroupNode();
        XPathOperatorNode targetNode = (XPathOperatorNode) targetGroupNode.getNodeObject();
        if (sourceModelNode != null && targetNode != null) {
            connectFields(source, target);
            int targetInputFieldIndex = MapperUtil.findFieldIndex(targetGroupNode, target);
            updateLinkObjectXPathExpression(sourceModelNode, targetNode, targetInputFieldIndex);
        }
    }
    
    private void updateLinkObjectXPathExpression(
            Node sourceNode,
            XPathOperatorNode targetNode,
            int targetInputFieldIndex) {
        List indirectLinksToTargetTreeNodes = NodeHelper.findSubsequentLinksToTargetTree(targetNode, this.mMapper.getMapperModel().getSelectedViewModel());
        if (indirectLinksToTargetTreeNodes != null) {
            setSynchronizationListenerEnable(false);
            handleLinkCreationToNonTargetTreeNode(indirectLinksToTargetTreeNodes);
            setSynchronizationListenerEnable(true);
        }
    }
    
    protected void setSynchronizationListenerEnable(boolean isEnabled) {
        ResultWindow.getInstance().setSynchronizationListenerEnable(isEnabled);
    }
}
