/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.bpel.xpath.model.node.visitor;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.modules.bpel.model.api.BpelEntity;
import org.netbeans.modules.bpel.model.api.BpelModel;
import org.netbeans.modules.bpel.model.api.Variable;
import org.netbeans.modules.bpel.xpath.model.nodes.Node;
import org.netbeans.modules.bpel.xpath.model.nodes.PartNode;
import org.netbeans.modules.bpel.xpath.model.nodes.TreeNode;
import org.netbeans.modules.bpel.xpath.model.nodes.VariableNode;
import org.netbeans.modules.bpel.xpath.model.nodes.VariablesNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XPathLiteralNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XPathLocationStepNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XPathOperatorNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XPathPredicateNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XPathPredicatesNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XSDAttributeNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XSDComplexTypeNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XSDElementNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XSDSimpleTypeNode;
import org.netbeans.modules.bpel.xpath.model.nodes.impl.OperandPlaceHolderNodeImpl;
import org.netbeans.modules.bpel.xpath.model.xsd.visitor.XSDXPathGeneratorVisitor;
import org.netbeans.modules.bpel.xpath.util.Util;
import org.netbeans.modules.xml.schema.model.Attribute;
import org.netbeans.modules.xml.schema.model.AttributeReference;
import org.netbeans.modules.xml.schema.model.Element;
import org.netbeans.modules.xml.schema.model.ElementReference;
import org.netbeans.modules.xml.schema.model.Form;
import org.netbeans.modules.xml.schema.model.GlobalAttribute;
import org.netbeans.modules.xml.schema.model.GlobalElement;
import org.netbeans.modules.xml.schema.model.LocalAttribute;
import org.netbeans.modules.xml.schema.model.LocalElement;
import org.netbeans.modules.xml.schema.model.Schema;
import org.netbeans.modules.xml.schema.model.SchemaModel;
import org.netbeans.modules.xml.wsdl.model.Part;
import org.netbeans.modules.xml.xam.dom.AbstractDocumentComponent;
import org.netbeans.modules.xml.xam.dom.NamedComponentReference;
import org.netbeans.modules.xml.xpath.AbstractXPathModelHelper;
import org.netbeans.modules.xml.xpath.LocationStep;
import org.netbeans.modules.xml.xpath.XPathCoreOperation;
import org.netbeans.modules.xml.xpath.XPathException;
import org.netbeans.modules.xml.xpath.XPathExpression;
import org.netbeans.modules.xml.xpath.XPathModel;
import org.netbeans.modules.xml.xpath.XPathOperationOrFuntion;
import org.openide.util.NbBundle;

/**
 *
 * @author radval
 *  Generation of xpath expression may result in need to add new prefix to namespace
 *  mapping, since now xpath are namespace qualified.
 *  User who uses XPathGenerator  to get expression and then set to a bpel element
 *  must call addNewPrefixToProcess to add any new prefix at the process element
 *
 */
public class XPathGeneratorVisitor extends AbstractNodeVisitor {
    
    private static final Logger LOGGER = Logger.getLogger(XPathGeneratorVisitor.class.getName());
    
    private List mPathList = new ArrayList();
    private Node mRootExpressionNode;
    
    private BpelModel mModel;
    private BpelEntity mHoldingEntity;
    
    private Map<String, String> mPrefixToNamespaceMap = new HashMap();
    
    public XPathGeneratorVisitor(BpelEntity holdingEntity) {
        this.mHoldingEntity = holdingEntity;
        this.mModel = holdingEntity.getBpelModel();
        
    }
    
    public String getXPath() {
        if (this.mRootExpressionNode != null) {
            XPathExpression exp = (XPathExpression) this.mRootExpressionNode.getDataObject();
            return exp.getExpressionString();
        }
        return createPath();
    }
    
    public Map getPrefixToNamespaceMap() {
        return this.mPrefixToNamespaceMap;
    }
    
    /**
     * Generation of xpath expression may result in need to add new prefix to namespace
     * mapping, since now xpath are namespace qualified.
     * User who uses XPathGenerator  to get expression and then set to a bpel element
     *must call addNewPrefixToProcess to add any new prefix at the process element
     */
    public void addNewPrefixToProcess() {
        Iterator<String> it = this.mPrefixToNamespaceMap.keySet().iterator();
        while(it.hasNext()) {
            String prefix = it.next();
            String ns = this.mPrefixToNamespaceMap.get(prefix);
            ((AbstractDocumentComponent)this.mModel.getProcess()).addPrefix(prefix, ns);
        }
    }

    public void visit(VariablesNode node) {}
    
    public void visit(VariableNode node) {
        this.mPathList.add(0, "$" + node.getName());    // NOI18N
        visitParent(node);
    }
    
    
    public void visit(PartNode node) {
        Part part = node.getPart();
        this.mPathList.add(0, "." + part.getName());    // NOI18N
        visitParent(node);
    }
    
    public void visit(XSDAttributeNode node) {
        Attribute attribute = node.getAttribute();
        
        String prefix = getOrCreateAttributePrefix(attribute);
        
        XSDXPathGeneratorVisitor xGen = new XSDXPathGeneratorVisitor();
        attribute.accept(xGen);
        String step = xGen.getXPath();
        if (step != null) {
            if(prefix != null) {
                this.mPathList.add(0, "/@"+ prefix + ":" + step); //NOI18N
            } else {
                this.mPathList.add(0, "/@" + step);
            }
        }
        visitParent(node);
    }
    
    public void visit(XSDElementNode node) {
        Element element = node.getElement();
        String prefix = getOrCreateElementPrefix( element);
        
        XSDXPathGeneratorVisitor xGen = new XSDXPathGeneratorVisitor();
        element.accept(xGen);
        String step = xGen.getXPath();
        if (step != null) {
            if(prefix != null) {
               this.mPathList.add(0, "/" + prefix + ":"+ step);  // NOI18N 
            } else {
                this.mPathList.add(0, "/" + step);  // NOI18N
            }
        }
        visitParent(node);
    }
    
    public void visit(XSDComplexTypeNode node) {}
    
    public void visit(XSDSimpleTypeNode node) {}
    
    public void visit(XPathLiteralNode node) {
        //handle case when we only have literal
        if(mRootExpressionNode == null) {
            this.mRootExpressionNode = node;
        }
    }
    
    public void visit(XPathOperatorNode node) {
        if (mRootExpressionNode == null) {
            this.mRootExpressionNode = node;
        }
        XPathOperationOrFuntion operator = node.getOperationOrFuntion();
        operator.clearChildren();
        
        List inputs = node.getInputs();
        Iterator it = inputs.iterator();
        int unconnectedCount = 1;
        while (it.hasNext()) {
            Node input = (Node) it.next();
            if(input instanceof TreeNode) {
                input.accept(this);
                String path = createPath();
                XPathExpression expression = null;
                XPathModel model = AbstractXPathModelHelper.getInstance().newXPathModel();
                try {
                    expression = model.parseExpression(path);
                    operator.addChild(expression);
                    clearPathList();
                } catch (XPathException ex) {
                    LOGGER.log(Level.SEVERE,
                            NbBundle.getMessage(XPathGeneratorVisitor.class,
                                "STR_FAILED_TO_PARSE_XPATH") + path,    // NOI18N
                            ex);
                }
                
            } else {
                input.accept(this);
                Object obj = input.getDataObject();
                if(obj != null && obj instanceof XPathExpression) {
                    operator.addChild((XPathExpression)input.getDataObject());
                } else if (obj == OperandPlaceHolderNodeImpl.UNCONNECTED_PLACEHOLDER){
                    operator.addChild(createPlaceholderExpression(unconnectedCount++));
                }
            }
        }
        
        if (node.getDataObject() instanceof XPathCoreOperation) {
            // Operators need placeholders when one of their operands
            // is missing, otherwise the entire expression becomes unparsable.
            XPathCoreOperation expr = (XPathCoreOperation) node.getDataObject();
            if (expr.getOperator() == XPathCoreOperation.OP_NEGATIVE) {
                if (expr.getChildCount() == 0) {
                    expr.addChild(createPlaceholderExpression(1));
                }
            } else {
                if (expr.getChildCount() == 0) {
                    expr.addChild(createPlaceholderExpression(1));
                    expr.addChild(createPlaceholderExpression(2));
                } else if (expr.getChildCount() == 1) {
                    expr.addChild(createPlaceholderExpression(1));
                }
            }
        }
    }
    
    // The placeholder expression is a variable. This causes the BPEL
    // validator to fail (yet the XPath expression to parse), so that
    // the user knows that an operand is left unconnected.
    private XPathExpression createPlaceholderExpression(int id) {
        XPathModel xpImpl = AbstractXPathModelHelper.getInstance().newXPathModel();
        try {
            return xpImpl.parseExpression(OperandPlaceHolderNodeImpl.UNCONNECTED_PLACEHOLDER + id);   // NOI18N
        } catch (XPathException e) {
        }
        return null;
    }
    
    public void visit(XPathPredicateNode node) {
    }
    
    public void visit(XPathPredicatesNode node) {
    }
    
    public void visit(XPathLocationStepNode node) {
        LocationStep step = node.getLocationStep();
        String expression = step.getExpressionString();
        this.mPathList.add(0, expression);
        visitParent(node);
    }
    
    private void visitParent(TreeNode node) {
        TreeNode parent = node.getParent();
        if (parent != null) {
            parent.accept(this);
        }
    }
    
    private void clearPathList() {
        this.mPathList.clear();
    }
    
    private String createPath() {
        StringBuffer path = new StringBuffer(10);
        Iterator it = this.mPathList.iterator();
        
        while (it.hasNext()) {
            String step = (String) it.next();
            path.append(step);
        }
        
        return path.toString();
    }
    
    private String findPrefix(SchemaModel sModel) {
        String prefix = null;
        
        if(sModel != null) {
            Schema schema = sModel.getSchema();
            if(schema != null) {
                String targetNamesapce = schema.getTargetNamespace();
                if(targetNamesapce != null) {
                   AbstractDocumentComponent adc = (AbstractDocumentComponent) this.mHoldingEntity;
                   prefix = adc.lookupPrefix(targetNamesapce);
                    
                }
            }
        }
        return prefix;
    }
    
    private Map createPrefixToNamespace(SchemaModel sModel) {
        Map prefixToNamespace = new HashMap();
        if(sModel != null) {
            Schema schema = sModel.getSchema();
            if(schema != null) {
                String targetNamesapce = schema.getTargetNamespace();
                if(targetNamesapce != null) {
                   AbstractDocumentComponent adc = (AbstractDocumentComponent) this.mHoldingEntity;
                   String prefix = Util.generatePrefix(targetNamesapce, adc);
                   if(prefix != null) {
                       prefixToNamespace.put(prefix, targetNamesapce);
                   }

                }
            }
        }
        return prefixToNamespace;
    }
    
    private String getOrCreateElementPrefix(Element element) {
        if(element == null) {
            return null;
        }
        
        SchemaModel sModel = null;
        
        if(element instanceof ElementReference) {
            ElementReference elementRef = (ElementReference) element;
            NamedComponentReference<GlobalElement> geRef = elementRef.getRef();
            if(geRef != null && geRef.get() != null) {
                sModel = geRef.get().getModel();
            }
        } else {
            sModel = element.getModel();
        }
        
        if(sModel == null || sModel.getSchema() == null) {
            return null;
        }
        
        //see http://www.zvon.org/xxl/XMLSchemaTutorial/Output/ser_elementFormDefault.html
        Schema schema = sModel.getSchema();
        Form form = schema.getElementFormDefaultEffective();
        
        //for global element we always generate namespace prefix
        if (element instanceof GlobalElement) {
            return getOrCreatePrefix(sModel);
        }
        else if (form.equals(Form.UNQUALIFIED)) {
            if(element instanceof LocalElement) {
                LocalElement le = (LocalElement) element;
                form = le.getFormEffective();

                if (form.equals(Form.QUALIFIED)) {
                    return getOrCreatePrefix(sModel);
                }
            }
            // vlv
            else if (element instanceof ElementReference) {
                return getOrCreatePrefix(sModel);
            }
        }
        else if (form.equals(Form.QUALIFIED)) {
            //if elementFormDefault is qualified
            //then we need to generate prefix
            return getOrCreatePrefix(sModel);
        }
        return null;
    }
    
    private String getOrCreateAttributePrefix(Attribute attribute) {
        if(attribute == null) {
            return null;
        }
        
        SchemaModel sModel = null;
        
        if(attribute instanceof AttributeReference) {
            AttributeReference attrRef = (AttributeReference) attribute;
            NamedComponentReference<GlobalAttribute> gaRef = attrRef.getRef();
            if(gaRef != null && gaRef.get() != null) {
                sModel = gaRef.get().getModel();
            }
        } else {
            sModel = attribute.getModel();
        }
        
        if(sModel == null || sModel.getSchema() == null) {
            return null;
        }
        
        //using recommendation from
        //http://www.zvon.org/xxl/XMLSchemaTutorial/Output/ser_attributeFormDefault.html
        Schema schema  = sModel.getSchema();
        String targetNamespace = schema.getTargetNamespace();
        if(targetNamespace != null) {
            Form form = schema.getAttributeFormDefaultEffective();
            if(form.equals(Form.UNQUALIFIED)) {
                if(attribute instanceof LocalAttribute) {
                   LocalAttribute le = (LocalAttribute) attribute;
                    form = le.getFormEffective();
                    if(form.equals(Form.QUALIFIED)) {
                        return getOrCreatePrefix(sModel);
                    }
                }
            }else if(form.equals(Form.QUALIFIED)) {
                //if elementFormDefault is qualified
                //then we need to generate prefix
                return getOrCreatePrefix(sModel);
            }
        } 
        return null;
    }
    
    private String getOrCreatePrefix(SchemaModel sModel) {
        
        if(sModel == null || sModel.getSchema() == null) {
            return null;
        }
            String prefix = findPrefix(sModel);
            if(prefix == null) {
                Map prefixToNamespace = createPrefixToNamespace(sModel);
                mPrefixToNamespaceMap.putAll(prefixToNamespace);
                if(prefixToNamespace.size() != 0) {
                    prefix = mPrefixToNamespaceMap.keySet().iterator().next();
                }
            }
        return prefix;
    }
}
