/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.bpel.model.validation.runtime;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.channels.FileChannel;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.ResourceBundle;
import java.util.regex.Pattern;
import junit.framework.TestSuite;
import org.netbeans.modules.bpel.model.api.BpelModel;
import org.netbeans.modules.bpel.model.api.support.ActivityDescriptor;
import org.netbeans.modules.xml.xam.spi.Validation;
import org.netbeans.modules.xml.xam.spi.Validation.ValidationType;
import org.netbeans.modules.xml.xam.spi.ValidationResult;
import org.netbeans.modules.bpel.model.api.support.TBoolean;
import org.netbeans.modules.bpel.model.validation.common.TestCatalogModel;
import org.netbeans.modules.xml.xam.spi.Validator.ResultItem;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;

/**
 *
 * @author apepin
 */
public class BPELRuntimeValidatorMultiTest extends MultiTestSuite{
    
    private static final ResourceBundle mMessages = ResourceBundle
        .getBundle(BPELRuntimeNotSupportedValidator.class.getPackage().getName()+
            ".Bundle");
    private static final String FILENAME = "test_sample.bpel";
    private static final String DEFAULT_FILE_NAME = 
        "/org/netbeans/modules/bpel/model/validation/runtime/resources/"+FILENAME;
    
//    static final String WORKDIR = System.getProperty("xtest.workdir");
    static final String WORKDIR;
    static{
        String workdir = System.getProperty("xtest.workdir");
        WORKDIR = (workdir != null)? workdir: System.getProperty("user.home");
    }
    
    
    /** Creates a new instance of BPELRuntimeNotSupportedMultiTest */
    public BPELRuntimeValidatorMultiTest() {
    }
    
    public static TestSuite suite(){
        return new BPELRuntimeValidatorMultiTest();
    }
    
    //define set of tests to be executed
    protected MultiTestCase[] cases() {
        return new MultiTestCase[]{new ProcessTest(),  new ActivityTest()};
    }
    
    //Process tests
    public class ProcessTest extends RuntimeTest{
        
        ProcessRules rule;
        int index = 0;
        public MultiTestCase create(){
            if(index < ProcessRules.values().length){
                rule = ProcessRules.values()[index++];
                setName("CheckProcess."+rule.toString());
                return this;
            }
            return null;
        }
        public void execute() {
            try{
                //create source file from resource
                File sourceFile = createSourceFileFromResource(this.getClass(), DEFAULT_FILE_NAME);
                //prepare source for rule
                HashSet<String> expectedErrors =  prepareSourceForRule(sourceFile, rule);
                //get model
                BpelModel model = getBpelModel(sourceFile);
                //validate
                validate(model, rule, expectedErrors);
                
                //test complete
                System.out.println(getName()+" complete");
            }catch(Exception e){
                //fail
                e.printStackTrace();
                fail(e.toString());
            }
            finally{
                System.out.println("Performed "+index+" test(s)");
            }
        }

    }
    
    //Activity tests
    public class ActivityTest extends RuntimeTest{
        
        ActivityDescriptor.ActivityType activity;
        TBoolean supress = null;
        int index = 0;
        public MultiTestCase create(){
            //for each activity test supress join failure YES and NO
            if(supress != null){
                switch(supress){
                    case YES:
                        supress = TBoolean.NO;
                        break;
                    case NO:
                        supress = TBoolean.YES;
                        index++;
                        break;
                }
            }
            else{
                //initial
                supress = TBoolean.YES;
            }
            
            if(index < ActivityDescriptor.ActivityType.values().length){
                activity = ActivityDescriptor.ActivityType.values()[index];
                setName("CheckActivity."+activity.toString()+"_"+supress.toString().toUpperCase());
                return this;
            }
            return null;
        }
        
        public void execute() {
            try{
                //create source file from resource
                File sourceFile = createSourceFileFromResource(this.getClass(), DEFAULT_FILE_NAME);
                //prepare source for rule
                HashSet<String> expectedErrors =  createActivity(sourceFile, activity, supress);
                //get model
                BpelModel model = getBpelModel(sourceFile);
                //validate
                validate(model, activity, expectedErrors);
                
                //test complete
                System.out.println(getName()+" complete");
                
            }catch(Exception e){
                //fail
                fail(e.toString());
            }
            finally{
                int k = (supress == TBoolean.YES)?1:2;
                System.out.println("Performed "+(2*index+k)+" test(s)");
            }
        }
    
    }
    
    //base runtimetest
    public abstract class RuntimeTest extends MultiTestCase{
        protected void validate( BpelModel model, Object elementToTest, Collection<String> expectedErrors )
                throws Exception
        {
                ValidationResult result =
                        BPELRuntimeValidatorMultiTest.validate(model);
                //analising the result
                Iterator<ResultItem> it = result.getValidationResult().iterator();
                while (it.hasNext()) {
                    ResultItem item = it.next();
                    assertTrue("Actual Error "+ item.getDescription(),
                            containsExpectedError(expectedErrors,
                            item.getDescription()));
                }
                if (result.getValidationResult().size() != expectedErrors.size()) {
                    fail("Expected " + expectedErrors.size() +
                            " error(s) for "+elementToTest.toString()+" but got "
                            +result.getValidationResult().size()+" errors instead");
                }
        }
    }
    
    //utilities
    private static File createSourceFileFromResource(Class clazz, String resource) throws IOException {
        URL url = clazz.getResource(resource);
        System.out.println("URL: "+url.toString());
        File workfile = null;
        try {
            workfile = copyFileToWorkdir( new File(url.toURI()));
            System.out.println("source file copied");
        } catch (URISyntaxException ex) {
            ex.printStackTrace();
        }
        return workfile;
    }

    private static int test_index = 0;// index to create new file names for each test
    private static File copyFileToWorkdir(File source) {
        System.out.println("source file "+source.getAbsolutePath());
        System.out.println("workdir: "+WORKDIR);
        System.out.println("workdir: "+System.getProperty("user.home"));
        File dest = new File(WORKDIR+File.separator+
                addIndexToFilename(source.getName(), test_index));
        try{
            if(dest.exists())
                dest.delete();
            //create new file
            test_index++;
            dest = new File(WORKDIR+File.separator+
                    addIndexToFilename(source.getName(), test_index));
            System.out.println("dest file "+dest.getAbsolutePath());
            if(dest.exists())
                dest.delete();

            FileChannel srcChannel = new FileInputStream( source ).getChannel();

            FileChannel dstChannel = new FileOutputStream( dest ).getChannel();

            try{
                dstChannel.transferFrom(srcChannel, 0, srcChannel.size());
            }finally{
                srcChannel.close();
                dstChannel.close();
            }

        }catch(IOException e){
            dest = null;
        }
        return dest;
        
    }
    
    private static String addIndexToFilename(String name, int index){
        int ext_pos = name.indexOf('.');
        return (ext_pos<0)?
                name+index:
                name.substring(0, ext_pos)+index+name.substring(ext_pos);
    }

    private static FileObject toFileObject(String absolutePath){
        File file = new File(absolutePath);
        return FileUtil.toFileObject(file);
    }

    private static BpelModel getBpelModel(File sourceFile) throws Exception {
        
        return getBpelModel(sourceFile.toURI());
    }

    private static BpelModel getBpelModel(URI relativePath) throws Exception{
        return TestCatalogModel.getDefault().getBPELModel(relativePath);
        //get bpel model from workdir
//        BPELModelProvider myBpelModelProvider = new BPELModelProviderOutsideIde(){
//          public BpelModel getBPELModel(URI relativePath) throws Exception{
//            ModelSource modelSource = getModelSource( relativePath );
//////            BpelModelFactory factory = (BpelModelFactory) Lookup.getDefault()
//////                    .lookup(BpelModelFactory.class);
//            BpelModel model = (new BpelModelFactoryImpl(){
//                public BpelModel createModel(ModelSource source){
//                    return super.createModel(source);
//                }
//            }).createModel(modelSource);
//            
//            return model;
//          }  
//        };
//        return myBpelModelProvider.getBPELModel(relativePath);
    }
    
    private static BpelModel getDefaultBpelModel(Class clazz) throws Exception{
        URL url = clazz.getResource(DEFAULT_FILE_NAME);
        return getBpelModel(url.toURI());
    }
    
    private static ValidationResult validate(BpelModel model) throws Exception {
        Validation validation = new Validation();
        ValidationType validationType = Validation.ValidationType.COMPLETE;
        BPELRuntimeNotSupportedValidator instance =
                new BPELRuntimeNotSupportedValidator();
        ValidationResult result =
                instance.validate(model, validation, validationType);
        return result;
    }

    private static Pattern p = Pattern.compile("\"?+\\{\\d\\}\"?+");
    private static boolean containsExpectedError(Collection<String> expectedErrors,
                                                String actualError) {
        boolean result = false;
        Iterator<String> it = expectedErrors.iterator();
        while(it.hasNext()) {
            String[] needToMatch = null;
            String expectedError = it.next();
            needToMatch = p.split(expectedError);
            
            //now let see if expected error can be matched with actual error.
            if(needToMatch != null) {
                //assume we have a match unless we found a mismatch below
                boolean foundMatch = true;
                for(int i = 0; i < needToMatch.length; i++) {
                    String match = needToMatch[i];
                    if(!actualError.contains(match)) {
                        //no exact match found.
                        foundMatch = false;
                        break;
                    }
                }
                
                result = foundMatch;
                if(result) {
                    break;
                }
            }
            
        }
        return result;
    }
    
    //Rules for a process
    public enum ProcessRules {NOTHING, 
    PROC_SUPRESS_YES, PROC_SUPRESS_NO, PROC_EXIT_YES, PROC_EXIT_NO,
    PROC_QUERY, PROC_EXPRESSION, VALIDATE, RETHROW, COMPENSATE, PARTNER_LINK,
    COMPENSATION_HANDLER, TERMINATION_HANDLER, VARIABLE, TARGET_CONTAINER,
    SOURCE_CONTAINER, INVOKE_CATCH, INVOKE_CATCH_ALL, EXTENSIBLE_ASSIGN,
    ASSIGN_VALIDATE_YES, ASSIGN_VALIDATE_NO, FROM_DOCS, FROM_LANG
    , FROM_PROP, FROM_LINK, FROM_ENDPOINT, 
    FROM_NAMESPACE, TO_DOCS, TO_PROP, TO_LINK, TO_NAMESPACE,
    FLOW, SCOPE_PLINK_CONT, SCOPE_CORR_CONT, SCOPE_ISOLATED_YES,
    SCOPE_ISOLATED_NO, SCOPE_EXIT_YES, SCOPE_EXIT_NO,
    FOREACH_YES, FOREACH_NO, IMPORT_LOCATION, IMPORT_NS, /*CHILD_NS,*/
    INVOKE_FROM_PART, INVOKE_TO_PART, RECEIVE_FROM_PART, REPLY_TO_PART,
    ONEVENT_FROM_PART, ONMESSAGE_FROM_PART}
    
    private HashSet<String> prepareSourceForRule(File sourceFile,
            ProcessRules rule) throws Exception{
        HashSet<String> expectedErrors = new HashSet<String>();
        //get editor
        XmlEditor editor = new XmlEditor(sourceFile);
        SourceElement process = editor.findSourceElement("process");
        SourceElement component = null;
        SourceElement component1 = null;
        SourceElement component2 = null;
        switch(rule){
            case NOTHING:
                break;
            case PROC_QUERY:
                process.setAttribute("queryLanguage", "query");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case PROC_EXPRESSION:
                process.setAttribute("expressionLanguage", "expression");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case PROC_SUPRESS_YES:
                process.setAttribute("suppressJoinFailure", "yes");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case PROC_SUPRESS_NO:
                process.setAttribute("suppressJoinFailure", "no");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case PROC_EXIT_YES:
                process.setAttribute("exitOnStandardFault", "yes");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case PROC_EXIT_NO:
                process.setAttribute("exitOnStandardFault", "no");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case VALIDATE:
                process.createChildElement("validate");
                expectedErrors.add(mMessages.getString("FIX_Element"));
                break;
            case RETHROW:
                process.createChildElement("rethrow");
                expectedErrors.add(mMessages.getString("FIX_Element"));
                break;
            case COMPENSATE:
                process.createChildElement("compensate");
                expectedErrors.add(mMessages.getString("FIX_Element"));
                break;
            case PARTNER_LINK:
                component = process.createChildElement("partnerLinks");
                component1 = component.createChildElement("partnerLink");
                component1.setAttribute("initializePartnerRole", "yes");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case COMPENSATION_HANDLER:
                component = process.createChildElement("scope");
                component.createChildElement("compensationHandler");
                expectedErrors.add(mMessages.getString("FIX_Element"));
                break;
            case TERMINATION_HANDLER:
                component = process.createChildElement("scope");
                component.createChildElement("terminationHandler");
                expectedErrors.add(mMessages.getString("FIX_Element"));
                break;
            case VARIABLE:
                component = process.createChildElement("variables");
                component1 = component.createChildElement("variable");
                component1.createChildElement("from");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case TARGET_CONTAINER:
                component = process.createChildElement("empty");
                component.createChildElement("targets");
                expectedErrors.add(mMessages.getString("FIX_Element"));
                break;
            case SOURCE_CONTAINER:
                component = process.createChildElement("empty");
                component.createChildElement("sources");
                expectedErrors.add(mMessages.getString("FIX_Element"));
                break;
            case INVOKE_CATCH:
                component = process.createChildElement("invoke");
                component.createChildElement("catch");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case INVOKE_CATCH_ALL:
                component = process.createChildElement("invoke");
                component.createChildElement("catchAll");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case EXTENSIBLE_ASSIGN:
                component = process.createChildElement("assign");
                component.createChildElement("extensibleAssign");
                expectedErrors.add(mMessages.getString("FIX_Element"));
                break;
            case ASSIGN_VALIDATE_YES:
                component = process.createChildElement("assign");
                component.setAttribute("validate", "yes");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case ASSIGN_VALIDATE_NO:
                component = process.createChildElement("assign");
                component.setAttribute("validate", "no");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case FROM_DOCS:
                component = process.createChildElement("assign");
                component1 = component.createChildElement("copy");
                component2 = component1.createChildElement("from");
                component2.createChildElement("documentation");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case FROM_LANG:
                component = process.createChildElement("assign");
                component1 = component.createChildElement("copy");
                component2 = component1.createChildElement("from");
                component2.setAttribute("expressionLanguage", "http://java.sun.com");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case FROM_PROP:
                process.createChildElement("assign").
                    createChildElement("copy").
                        createChildElement("from").
                    setAttribute("property", "tns:propname");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case FROM_LINK:
                process.createChildElement("assign").
                    createChildElement("copy").
                        createChildElement("from").
                    setAttribute("partnerLink", "partner");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case FROM_ENDPOINT:
                process.createChildElement("assign").
                    createChildElement("copy").
                        createChildElement("from").
                    setAttribute("endpointReference", "myRole");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case FROM_NAMESPACE:
                process.createChildElement("assign").
                    createChildElement("copy").
                        createChildElement("from").
                    setAttribute("tns:part", "partname");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case TO_DOCS:
                process.createChildElement("assign").
                    createChildElement("copy").
                        createChildElement("to").
                            createChildElement("documentation");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case TO_PROP:
                process.createChildElement("assign").
                    createChildElement("copy").
                        createChildElement("to").
                    setAttribute("property", "tns:propname");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case TO_LINK:
                process.createChildElement("assign").
                    createChildElement("copy").
                        createChildElement("to").
                    setAttribute("partnerLink", "partner");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case TO_NAMESPACE:
                process.createChildElement("assign").
                    createChildElement("copy").
                        createChildElement("to").
                    setAttribute("tns:part", "partname");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case FLOW:
                process.createChildElement("flow").
                    createChildElement("links");
                expectedErrors.add(mMessages.getString("FIX_Element"));
                break;
            case SCOPE_PLINK_CONT:
                process.createChildElement("scope").
                    createChildElement("partnerLinks");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case SCOPE_CORR_CONT:
                process.createChildElement("scope").
                    createChildElement("correlationSets");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case SCOPE_ISOLATED_YES:
                process.createChildElement("scope").
                    setAttribute("isolated", "yes");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case SCOPE_ISOLATED_NO:
                process.createChildElement("scope").
                    setAttribute("isolated", "no");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case SCOPE_EXIT_YES:
                process.createChildElement("scope").
                    setAttribute("exitOnStandardFault", "yes");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case SCOPE_EXIT_NO:
                process.createChildElement("scope").
                    setAttribute("exitOnStandardFault", "no");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case FOREACH_YES:
                process.createChildElement("forEach").
                    setAttribute("parallel", "yes");
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case FOREACH_NO:
                process.createChildElement("forEach").
                    setAttribute("parallel", "no");
                //attribute is requared and should be set to "no"
                //no error expected
//                expectedErrors.add(mMessages.getString("FIX_Attribute"));
                break;
            case IMPORT_LOCATION:
                process.findChildElement("import").
                        removeAttribute("location");
                expectedErrors.add(mMessages.getString("FIX_Attribute_Required_For_Sun_BpelSE"));
                break;
            case IMPORT_NS:
                process.findChildElement("import").
                        removeAttribute("namespace");
                expectedErrors.add(mMessages.getString("FIX_Attribute_Required_For_Sun_BpelSE"));
                break;
//            case CHILD_NS:
//                process.createChildElement("sequence").
//                        createChildElement("ns1:empty").
//                        setAttribute("namespace", "http://schemas.xmlsoap.org/wsdl/");
//                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
//                break;
            case INVOKE_FROM_PART:
                process.createChildElement("invoke").
                        createChildElement("fromParts").
                        createChildElement("fromPart").
                        setAttribute("name", "xxx");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case INVOKE_TO_PART:
                process.createChildElement("invoke").
                        createChildElement("toParts").
                        createChildElement("toPart").
                        setAttribute("name", "xxx");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case RECEIVE_FROM_PART:
                process.createChildElement("receive").
                        createChildElement("fromParts").
                        createChildElement("fromPart").
                        setAttribute("name", "xxx");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case REPLY_TO_PART:
                process.createChildElement("reply").
                        createChildElement("toParts").
                        createChildElement("toPart").
                        setAttribute("name", "xxx");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case ONEVENT_FROM_PART:
                process.createChildElement("eventHandlers").
                        createChildElement("onEvent").
                        createChildElement("fromParts").
                        createChildElement("fromPart").
                        setAttribute("name", "xxx");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
            case ONMESSAGE_FROM_PART:
                process.createChildElement("pick").
                        createChildElement("onMessage").
                        createChildElement("fromParts").
                        createChildElement("fromPart").
                        setAttribute("name", "xxx");
                expectedErrors.add(mMessages.getString("FIX_ElementInParent"));
                break;
                
            default:    //do nothing
                throw new Exception("Rule "+rule.toString()+" has not been prepared!");
        }
        //save changes
        editor.save();
        //tmp
//        editor.dump();
        return expectedErrors;
    }
    //To check set of rules
//    private HashSet<String> prepareModelForRules(BpelModel model,
//            ProcessRules[] rules) throws Exception{
//        HashSet<String> expectedErrors = new HashSet<String>();
//        for(ProcessRules rule: rules){
//            expectedErrors.addAll(prepareModelForRule(model, rule));
//        }
//        return expectedErrors;
//    }
    
    //create activity by type with given supress join failure attribute
    private HashSet<String> createActivity(File sourceFile,
            ActivityDescriptor.ActivityType activity,
            TBoolean supress) throws Exception{
        
        HashSet<String> expectedErrors = new HashSet<String>();
        //get editor
        XmlEditor editor = new XmlEditor(sourceFile);
        SourceElement process = editor.findSourceElement("process");
        SourceElement component = process.createChildElement(
                normalizeName(activity.toString()));
        component.setAttribute("suppressJoinFailure", normalizeName(supress.toString()));
        editor.save();
//        editor.dump();
        switch(activity){
            case COMPENSATE:
            case VALIDATE:
            case RETHROW:
                expectedErrors.add(mMessages.getString("FIX_Element"));
                break;
            default:
                expectedErrors.add(mMessages.getString("FIX_Attribute"));
        }

        return expectedErrors;
    }
    
    static String normalizeName(String name){
        String _name = name.toLowerCase();
        int _index = -1;
        while((_index=_name.indexOf("_")) >= 0)
            _name = _name.substring(0, _index)+_name.substring(_index+1, _index+2).toUpperCase()+_name.substring(_index+2);
        System.out.println("normalized name: "+_name);
        return _name;
    }

}    
