/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.spi.editor.hints;

import java.beans.XMLDecoder;
import java.beans.XMLEncoder;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.netbeans.modules.editor.hints.options.ProvidersListAccessor;
import org.openide.ErrorManager;
import org.openide.filesystems.FileLock;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.filesystems.Repository;
import org.openide.util.lookup.Lookups;

/**
 *
 * @author Jan Lahoda
 */
public final class ProvidersList {
    
    static {
        ProvidersListAccessor.INSTANCE = new ProvidersListAccessor() {
            public Collection<String> getInstalledProvidersList() {
                return ProvidersList.getInstalledProvidersList();
            }
            public void setProviderEnabled(String providerKey, boolean enabled) {
                ProvidersList.setProviderEnabled(providerKey, enabled);
            }
            public void setSeverity(String providerKey, String errorKey, Severity severity) {
                ProvidersList.setSeverity(providerKey, errorKey, severity);
            }
            public ProviderDescription getProviderDescription(String providerKey) {
                return ProvidersList.getProviderDescription(providerKey);
            }
            public List<ProviderDescription> getDescriptions() {
                return ProvidersList.getDescriptions();
            }
        };
    }
    
    private static final String PROVIDERS_LIST_FOLDER = "EditorHints/"; // NOI18N
    private static final String CONFIG_FILE_NAME      = "editorhintsconfig.xml"; //NOI18N
    
    private static List<ProviderDescription> descriptions = null;
    private static Map<String, Boolean> key2Enabled = null;
    private static Map<String, ProviderDescription> key2Description = null;
    private static Map<String, Map<String, Severity>> key2Error2Severity = null;
    
    private static boolean initialized = false;
    
    /** Creates a new instance of ProvidersList */
    private ProvidersList() {
    }
    
    static synchronized Collection<String> getInstalledProvidersList() {
        if (!initialized) {
            init();
            initialized = true;
        }
        
        return key2Description.keySet();
    }
    
    private static void init() {
        key2Description = new HashMap<String, ProviderDescription>();
        descriptions = new ArrayList<ProviderDescription>();
        
        Collection<? extends ProviderDescription> looked = Lookups.forPath(PROVIDERS_LIST_FOLDER).lookupAll(ProviderDescription.class);
        
        for (ProviderDescription desc : looked) {
            key2Description.put(desc.getKey(), desc);
            descriptions.add(desc);
        }

	FileObject configFile = Repository.getDefault().getDefaultFileSystem().findResource(CONFIG_FILE_NAME);
	
	if (configFile != null) {
	    InputStream ins = null;
	    
	    try {
		ins = configFile.getInputStream();
		
		XMLDecoder decoder = new XMLDecoder(ins);
		
		key2Error2Severity = (Map<String, Map<String, Severity>>) decoder.readObject();
		key2Enabled = (Map<String, Boolean>) decoder.readObject();
		
		decoder.close();
	    } catch (IOException e) {
		ErrorManager.getDefault().notify(e);
	    } finally {
		if (ins != null) {
		    try {
			ins.close();
		    } catch (IOException e) {
			ErrorManager.getDefault().notify(e);
		    }
		}
	    }
	} else {
	    key2Error2Severity = new HashMap<String, Map<String, Severity>>();
	    key2Enabled = new HashMap<String, Boolean>();
	}
    }
    
    private static void save() {
	FileLock     lock = null;
	OutputStream out  = null;
	
	try {
	    FileObject configFile = Repository.getDefault().getDefaultFileSystem().findResource(CONFIG_FILE_NAME);
	    
	    if (configFile == null) {
		FileObject root = Repository.getDefault().getDefaultFileSystem().getRoot();
		
		configFile = FileUtil.createData(root, CONFIG_FILE_NAME);
	    }
	    
	    lock = configFile.lock();
	    out = configFile.getOutputStream(lock);
	    
	    XMLEncoder encoder = new XMLEncoder(out);
	    
	    encoder.writeObject(key2Error2Severity);
	    encoder.writeObject(key2Enabled);
	    
	    encoder.close();
	} catch (IOException e) {
	    ErrorManager.getDefault().notify(e);
	} finally {
	    if (out != null) {
		try {
		    out.close();
		} catch (IOException e) {
		    ErrorManager.getDefault().notify(e);
		}
	    }
	    
	    if (lock != null) {
		lock.releaseLock();
	    }
	}
    }
    
    /**Test if the given provider should be enabled or disabled.
     * 
     * @param providerKey the provider ID to test
     * @return true if the given provider should be enabled
     */
    public static boolean isProviderEnabled(String providerKey) {
        if (!initialized) {
            init();
            initialized = true;
        }
        
        if (!key2Description.containsKey(providerKey)) {
            throw new IllegalArgumentException("Unknown provider key: " + providerKey); // NOI18N
        }
        
        Boolean enabled = (Boolean) key2Enabled.get(providerKey);
        
        if (enabled == null) {
            ProviderDescription desc = getProviderDescription(providerKey);
            
            return desc.getDefaultState();
        }
        
        return enabled.booleanValue();
    }
    
    /**Return severity of the given error of the given provider.
     * 
     * @param providerKey the provider ID to test
     * @param errorKey the error ID to test
     * @return desired severity of the given error
     */
    public static synchronized Severity getErrorSeverity(String providerKey, String errorKey) {
        if (!initialized) {
            init();
            initialized = true;
        }
        
        if (!key2Description.containsKey(providerKey)) {
            throw new IllegalArgumentException("Unknown provider key: " + providerKey); // NOI18N
        }
        
        Map<String, Severity> error2Severity = key2Error2Severity.get(providerKey);
        
        if (error2Severity == null) {
            key2Error2Severity.put(providerKey, error2Severity = new HashMap<String, Severity>());
        }
        
        Severity severity = error2Severity.get(errorKey);
        
        if (severity == null) {
            ProviderDescription desc = getProviderDescription(providerKey);
            
            if (desc.getSupportedErrorKeys().contains(errorKey)) {
                return desc.getErrorDefaultSeverity(errorKey);
            } else {
                throw new IllegalArgumentException("Unknown error key: " + errorKey + " for provider: " + providerKey); // NOI18N
            }
        }
        
        return severity;
    }
    
    /**Probably not for general use, TBD.
     */
    static synchronized void setProviderEnabled(String providerKey, boolean enabled) {
        if (!initialized) {
            init();
            initialized = true;
        }
        
        if (!key2Description.containsKey(providerKey)) {
            throw new IllegalArgumentException("Unknown provider key: " + providerKey); // NOI18N
        }
        
        key2Enabled.put(providerKey, Boolean.valueOf(enabled));
	
	save();
    }

    /**Probably not for general use, TBD.
     */
    static synchronized void setSeverity(String providerKey, String errorKey, Severity severity) {
        if (!initialized) {
            init();
            initialized = true;
        }
        
        if (!getProviderDescription(providerKey).getSupportedErrorKeys().contains(errorKey)) {
            throw new IllegalArgumentException("Unknown error key: " + errorKey + " for provider: " + providerKey); // NOI18N
        }
        
        Map<String, Severity> error2Severity = key2Error2Severity.get(providerKey);
        
        if (error2Severity == null) {
            key2Error2Severity.put(providerKey, error2Severity = new HashMap<String, Severity>());
        }
        
        error2Severity.put(errorKey, severity);
	
	save();
    }
    
    /**Probably not for general use, TBD.
     */
    static synchronized ProviderDescription getProviderDescription(String providerKey) {
        if (!initialized) {
            init();
            initialized = true;
        }
        
        if (!key2Description.containsKey(providerKey)) {
            throw new IllegalArgumentException("Unknown provider key: " + providerKey); // NOI18N
        }
        
        return (ProviderDescription) key2Description.get(providerKey);
    }
    
    /**Probably not for general use, TBD.
     */
    static synchronized List<ProviderDescription> getDescriptions() {
        if (!initialized) {
            init();
            initialized = true;
        }
        
        return descriptions;
    }
    
//    /**HACK, need to store it somewhere and this is a very simple (though incorrect) place, make bette.
//     */
//    public static final int EAGER_ON_PROJECT = 0;
//    public static final int EAGER_ON_DEMAND = 1;
//    public static final int EAGER_LAZY = 2;
//    
//    public static synchronized void setEagerness(int type) {
//        if (!initialized) {
//            init();
//            initialized = true;
//        }
//        
//        key2Enabled.put("eager-hack", new Integer(type));
//        
//        save();
//    }
//    
//    public static synchronized int getEagerness() {
//        if (!initialized) {
//            init();
//            initialized = true;
//        }
//        
//        Integer v = (Integer) key2Enabled.get("eager-hack");
//        
//        if (v == null)
//            return EAGER_LAZY;
//        
//        return v.intValue();
//    }
}
