/* 
 *   Creation Date: <1999/07/25 14:22:18 samuel>
 *   Time-stamp: <2001/09/22 14:21:02 samuel>
 *   
 *	<init.c>
 *	
 *	Kernel module initialization
 *   
 *   Copyright (C) 1999, 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *
 */

#include "compat.h"
#include <linux/config.h>
#include <linux/module.h>

#include <linux/stddef.h>
#include <linux/sys.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/proc_fs.h>
#include <linux/utsname.h>

#include <asm/io.h> 
#include <asm/uaccess.h>
#include <asm/unistd.h>
#include <asm/pgtable.h>
#include <asm/system.h>

#include "reloc.h"
#include "kernel_vars.h"
#include "multiplexer.h"
#include "misc.h"
#include "version.h"
#include "weaksym.h"

MODULE_AUTHOR("Samuel Rydh <samuel@ibrium.se>");
MODULE_DESCRIPTION("Mac-on-Linux kernel module");
//EXPORT_NO_SYMBOLS;

static int		fix_symbols( void );
#ifdef LINUX_24
static void		compat_initialize_24( void );
#define			COMPAT_INITIALIZE() compat_initialize_24()
#else
static void		compat_initialize_22( void );
#define			COMPAT_INITIALIZE() compat_initialize_22()
int 			__current_mm_offs = 0;
#endif

extern long 		sys_call_table[];

static int 		private_mod_count = 0;
int			linux_vers;
char			linux_extravers[16];

session_table_t		*g_sesstab = NULL;


/************************************************************************/
/*	Initialize / Cleanup Module					*/
/************************************************************************/

int 
init_module( void )
{
	int	major, patchlevel, sublevel;
	char 	*ptr;

	/* compile (runtime) version information */
	major = simple_strtoul( system_utsname.release, &ptr, 10 );
	patchlevel = simple_strtoul( ptr+1, &ptr, 10 );
	sublevel = simple_strtoul( ptr+1, &ptr, 10 );
	linux_vers = (major << 16) + (patchlevel <<8) + sublevel;
	strncpy( linux_extravers, ptr, sizeof(linux_extravers) );
	linux_extravers[ sizeof(linux_extravers)-1] = 0;

	if( strcmp( UTS_RELEASE, system_utsname.release ) ) {
		printk("MOL kernel module compiled for %s (Linux kernel version %d.%d.%d%s)\n",
		       UTS_RELEASE, major, patchlevel, sublevel, linux_extravers );
	}
	if( fix_symbols() )
		return 1;

	COMPAT_INITIALIZE();

	if( !(g_sesstab = kmalloc( sizeof(session_table_t), GFP_KERNEL)) )
		return 1;
	memset( g_sesstab, 0, sizeof(session_table_t) );

	if( relocate_code() ) {
		kfree( g_sesstab );
		return 1;
	}

	sys_call_table[__NR_multiplexer] = (long) sys_macos_multiplexer;

	printk( "MOL %d.%d.%d kernel module loaded\n", 
		MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL );
	return 0;
}

void
cleanup_module( void )
{
	int i;
	for(i=0; i<MAX_NUM_SESSIONS; i++ )
		destroy_session( i );

	sys_call_table[__NR_multiplexer] = 0;
	remove_hooks();
	relocation_cleanup();

	kfree(g_sesstab);
}

void
inc_mod_count( void )
{
	private_mod_count++;
	MOD_INC_USE_COUNT;
}

void
dec_mod_count( void )
{
	if( private_mod_count <= 0 )
		return;
	private_mod_count--;
	MOD_DEC_USE_COUNT;
}


/************************************************************************/
/*	Kernel version checking						*/
/************************************************************************/

static int
kver_at_least( int a, int b, int c, int pre )
{
	int p;
	if( pre<=0 )
		pre=99;
	if( linux_vers > KERNEL_VERSION(a,b,c) )
		return 1;
	if( linux_vers < KERNEL_VERSION(a,b,c) )
		return 0;
	p = ( strncmp("-pre", linux_extravers, 4 ) )? 99 : linux_extravers[4]-'0';
	if( p >= pre )
		return 1;
	return 0;
}


/************************************************************************/
/*	2.4 Compatiblity Code						*/
/************************************************************************/

#ifdef LINUX_24
static int	compat_has_mmap_rw_sem;

static void
compat_initialize_24( void )
{
	/* mmap_sem changed to rw semaphore in 2.4.3-pre5 */
	compat_has_mmap_rw_sem = ( kver_at_least( 2,4,3, /*pre*/ 5 ))? 1:0;
}

void
_compat_mmap_sem_up( void *sem )
{
	if( compat_has_mmap_rw_sem )
		up_read( (struct rw_semaphore*)sem );
	else
		up( (struct semaphore*)sem );
}

void
_compat_mmap_sem_down( void *sem )
{
	if( compat_has_mmap_rw_sem )
		down_read( (struct rw_semaphore*)sem );
	else
		down( (struct semaphore*)sem );
}
#else


/************************************************************************/
/*	2.2 Compatibility Code						*/
/************************************************************************/

weak_extern(giveup_altivec);

static void 
compat_initialize_22( void )
{
	int a,b;
	/* HACK: the __current_mm_offs is dependent upon if the altivec registers
	 * are present or not. The following code 'fixes' the offset if the
	 * module is not running on the kernel it was compiled for. 
	 */
	__current_mm_offs = offsetof(struct task_struct, mm);

	/* AltiVec first appeared in 2.2.17-pre3 */
	if( !kver_at_least( 2,2,17, /*pre*/ 3 ) )
		return;

	/* 2.2.17-pre3	 			offsets 856/1400 */
	/* 2.2.19-pre7 - 2.2.20-pre8(+)	 	offsets 864/1416 */

	a = 856, b=1400;
	if( kver_at_least( 2,2,19, /*pre*/ 7 ) )
		a = 864, b=1416;

#ifdef CONFIG_ALTIVEC
	if( !&giveup_altivec ) {
		__current_mm_offs = a;
		printk("MOL: Non-ALTIVEC kernel detected, using hardcoded offset\n");
	}
#else
	if( &giveup_altivec ) {
		__current_mm_offs = b;
		printk("MOL: ALTIVEC kernel detected, using hardcoded offset\n");
	}
#endif
}

#endif


/************************************************************************/
/*	Handle symbol which are not necessarily defined			*/
/************************************************************************/

/* Exported symbols:
 * 
 * 2.4 (old):	handle_mm_fault,	next_mmu_context,	mol_interface
 * 2.4 (new):	handle_mm_fault,	next_mmu_context,	flush_hash_page
 * molsymglue:	msym_handle_mm_fault,	msym_next_mmu_context,	msym_flush_hash_page
 */

ulong				*mc_flush_hash_page;
atomic_t			*mc_next_mmu_context;
molif_t				*mc_molif;
__typeof(mc_handle_mm_fault)	mc_handle_mm_fault;

weak_extern( flush_hash_page );
weak_extern( handle_mm_fault );
weak_extern( next_mmu_context );
weak_extern( mol_interface );

weak_extern( msym_flush_hash_page )
weak_extern( msym_handle_mm_fault );
weak_extern( msym_next_mmu_context );

#define SYM_GET(s) ( &s )? &s : (void*)msym_##s

static int 
fix_symbols( void )
{
	int err=0;
	
	mc_flush_hash_page = SYM_GET( flush_hash_page );
	mc_next_mmu_context = SYM_GET( next_mmu_context );
	mc_handle_mm_fault = SYM_GET( handle_mm_fault );

	if( !mc_handle_mm_fault || !mc_next_mmu_context )
		err++;
	if( !mc_flush_hash_page && !(mc_molif = &mol_interface) )
		err++;

	if( err ) {
		printk("A kernel symbol is undefined\n");
		return 1;
	}
	return 0;
}
